#ifndef WO_H
#define WO_H

/* object's behavior type */
#define LIST_NULL		0
#define LIST_STILL		1
#define LIST_MOBILE		2
#define LIST_INVISIBLE		3
#define LIST_DELETE		4
#define LIST_CART		5

#define NET_VOLATILE		0	// replicatable
#define NET_PERMANENT		1

/* special behavior masks */
#define NO_ELEMENTARY_MOVE	1	// bit 0
#define NO_BBABLE		2	// bit 1
#define NO_SELECTABLE		4	// bit 2
#define INVISIBLE		8	// bit 3
#define PERSISTENT		16	// bit 4
#define TAKABLE			32	// bit 5
#define REMOVABLE		512	// bit 9
/* collide behaviors */
#define COLLIDE_EVER		0	// bits 6,7,8
#define COLLIDE_ONCE		64	// bit 6
#define COLLIDE_NEVER		128	// bit 7
#define COLLIDE_GHOST		256	// bit 8

#define STATE_INACTIVE		1
#define STATE_DROP		2
#define STATE_IN_CARRIER	3
#define STATE_IN_CART		4

/* intersection types */
#define INTER_NO		0
#define INTER_1IN2		1
#define INTER_2IN1		2
#define INTER_INTERSECT		3

/* max space reachable, even values */
#define GRIDL			10
#define GRIDW			10
#define GRIDH			2
#define GRIDdX			6.0
#define GRIDdY			6.0
#define GRIDdZ			1000.0

#define SEP             	" \t\n"	// space and tab

/**
 * WObjectId Class
 *
 * WObjectId identifies a distributed object
 */
class WObjectId {
 public:
  uint32_t src_id;	///< IPaddr src
  uint16_t port_id;	///< port
  uint16_t obj_id;	///< object id -- network format
};

/**
 * Name struct
 *
 * Container for all names
 */
struct Name {
  char class_name[HNAME_LEN];	///< name of an object class
  char given_name[OBJNAME_LEN];	///< explicit name
  char *instance_name;		///< name of an instancied object
  const char *infos;		///< infos of an instancied object
  const char *world_name;	///< name of world where is this object
  char url[URL_LEN];		///< url
  char owner[9];		///< owner
};

/**
 * Win struct
 */
struct Win {
  float x;		// x position
  float y;		// y position
};

/**
 * Pos Class
 *
 * Spatial position and bounding-box
 */
class Pos {
 public:
  float x;		///< x absolute position
  float y;		///< y absolute position
  float z;		///< z absolute position
  float az;		///< angle plan xy axis z
  float ay;		///< angle plan xz axis y
  float ax;		///< angle plan yz axis x
  float old_x;		///< x old position
  float old_y;		///< y old position
  float old_z;		///< z old position
  float rel_x;		///< x relative position
  float rel_y;		///< y relative position
  float rel_z;		///< z relative position
  float rel_az;		///< angle plan xy axis z
  float rel_ay;		///< angle plan xz axis y
  float rel_ax;		///< angle plan yz axis x
  V3 bbcenter;		///< Bounding Box's center
  V3 bbsize;		///< Bounding Box's dimension
  uint8_t state;	///< button state
  uint8_t group;	///< group of objects
  //pd Win win;		///< position in the window

};

void initBB(Pos &pos);
int projectMovementOnObject(Pos &mobile, Pos &mobileold, Pos &fixe);

/**
 * Nature struct
 *
 * Physical features of the object
 */
struct Nature {
  float density;	///< density info
  float viscuosity;	///< viscuosity info
};

/**
 * State struct
 *
 * Current state of the object
 */
struct State {
  uint8_t state;	///< state of object
  uint8_t collide;	///< collision count
  uint8_t moving;	///< moving state of object
  uint8_t carrying;	///< carrying an other object
  bool ispointed;	///< grid flag
};

/**
 * Move struct
 *
 * Motion parameters of the object
 */
struct Move {
  V3 lspeed;		///< linear speed
  V3 aspeed;		///< angular speed
  time_t sec;		///< timestamp
  time_t usec;		///< timestamp
  time_t perm_sec;	///< time in sec of last permanent movement
  time_t perm_usec;	///< time in usec of last permanent movement
  float ttl;		///< time to live
};

class NetObjectId;
class Payload;
class ObjectList;
class Solid;

typedef class WObject * (WCreator) (char *);
typedef class WObject * (WReplicator) (uint8_t, NetObjectId, Payload *);
typedef void (WBuiltin) ();

/**
 * WClass Class
 *
 * Instance class
 */
class WClass {

  static WClass **objects_table;
  static int    table_size;

 public:
  const uint8_t       type_id;		///< class id
  const char *  const type_name;	///< class name
  WCreator *    const creator;		///< create from file
  WReplicator * const replicator;	///< create from network
  WBuiltin *    const builtin;		///< create from internal

  WClass(uint8_t type_id, const char *type_name, WCreator*, WReplicator* = NULL, WBuiltin* = NULL);

  static WObject * creatorInstance(const uint8_t type_id, char *l);	// file
  static void builtinInstance(const uint8_t type_id);		// builtin
  static class WObject * replicatorInstance(const uint8_t type_id,
					    NetObjectId noid, 
					    Payload *pp);
  static const WClass * getWClass(uint8_t type_id);
  static const WClass * getWClass(const char *type_name);
  static void dumpTable();
};


/**
 * WObject Class
 *
 * Common class for all the objects
 */
class WObject {

 public:
  class NetObject *noh;		///< reserved field for network
  class Solid *soh;		///< reserved field for 3D
  class UBox *guip;		///< reserved field for GUI
  uint8_t type;			///< object type
  uint8_t list_id;		///< PLD object world list
  uint32_t num;			///< object sequence number
  uint32_t behavior;		///< behavior flags
  Name name;			///< names
  class Pos pos;		///< position in the space
  State state;			///< object's state
  Nature nature;		///< object's nature
  Move move;			///< movement specific
  class WObjectId noid;		///< WObject Id
  class WObject *up_p;		///< Parent of grouped objects in trees
  class WObject *next_p;	///< Brother of grouped objects
  class WObject *down_p;	///< Child of grouped objects
  class WObject *nextg;		///< Old grouped objects
  class WObject *prevg;		///< Old grouped objects
  
  WObject();
  /**< Constructor */

  virtual ~WObject();
  /**< Destructor */

  //ELC virtual const WClass* getWClass() = 0;  // abstract really
  virtual const WClass* getWClass() {return NULL;};

  uint8_t getTypeId()		{return getWClass()->type_id;};
  const char* getTypeName()	{return getWClass()->type_name;};
  WCreator* getCreator()	{return getWClass()->creator;};
  WReplicator* getReplicator()  {return getWClass()->replicator;};

  //
  // Instances of general object handlers
  //
  virtual bool isMoving() { return false; }
  /**< Checks if object is moving */

  virtual void changePosition(float lasting) {}
  /**< Changes the position after a triggered movement */

  virtual void changePermanent(float lasting) {}
  /**< Changes the position during a permanent movement */

  virtual void updateTime(time_t sec, time_t usec, float *lasting) {}
  /**< Updates remaining times of the movement */

  virtual bool updateToNetwork(const Pos &oldpos) { return false; }
  /**< Publishes changes to the network */

  virtual void whenIntersect(WObject *pcur, WObject *pold) {}
  /**< Handles an ingoing collision with another object */

  virtual bool whenIntersectOut(WObject *pcur, WObject *pold) {return false;}
  /**< Handles an outgoing collision with another object */

  virtual void whenWallIntersect(WObject *pold, V3 *norm) {}
  /**< Handles collisions with walls */

  virtual void render() {}
  /**< Makes special rendering */

  virtual void click(V3 norm) {}
  /**< Intercepts a click */

  virtual void quit() {}
  /**< Makes all thing when leaving the object */

  virtual int numberAction();
  virtual bool haveAction();

#if 0 //todo
  virtual void (*method) (void *data, time_t sec, time_t usec);
#endif

  void setType(const int type);
  /**< Sets object type */

  int getType() {return type;}
  /**< Gets object type */

  int getNumber() {return num;}
  /**< Gets object number */

  bool isValid();
  /**< Checks if valid object type */

  bool isPermanent();
  /**< Checks if permanent object */

  void toDelete();
  /**< Delete this object latter */

  //
  // VREF parsing
  //
  char * parseObject(char *l);
  /**< Parses the <object ... /> tag */

  char * parseName(char *l);
  /**< Returns a given name */

  char * parseGroup(char *l);

  char * parsePosition(char *ptok);
  /**< Returns the spacial positions x y z az ax */

  char * parseColor(char *ptok);
  /**< Returns a color under r,g,b format */

  char * parseRotation(char *ptok);
  /**< Returns a rotation under r,X,Y,Z format */

  char * parseTranslation(char *ptok);
  /**< Returns a translation under tx,ty,tz format */

  char * parseUrl(char *ptok, char *url);
  /**< Returns an url */
  char * parseWorld(char *ptok, char *url);
  /**< Returns a world url */
  char * parseChannel(char *ptok, char *chan);
  /**< Returns a channel under addr/port/ttl format */

  char * parseString(char *ptok, char *str);
  /**< Returns a string */

  char * parseString(char *ptok, char *str, const char *attrstr);
  /**< Returns a string under attr=string format */

  char * parseQuotedString(char *ptok, char *str);
  /**< Returns a string under "string ..." format */

  char * parseQuotedString(char *ptok, char *str, const char *attrstr);
  /**< Returns a string under attr="string ..." format */

  char * parseInt(char *ptok, int *value);
  /**< Returns an integer */

  char * parseInt(char *ptok, int *value, const char *attrstr);
  /**< Returns an integer under attr=value format */

  char * parseUInt8(char *ptok, uint8_t *value);
  /**< Returns an 8 bits integer */

  char * parseUInt8(char *ptok, uint8_t *value, const char *attrstr);
  /**< Returns an 8 bits integer under attr=value format */

  char * parseUInt16(char *ptok, uint16_t *value);
  /**< Returns an 16 bits integer */

  char * parseUInt16(char *ptok, uint16_t *value, const char *attrstr);
  /**< Returns an 16 bits integer under attr=value format */

  char * parseFloat(char *ptok, float *value);
  /**< Returns a float */

  char * parseFloat(char *ptok, float *value, const char *attrstr);
  /**< Returns a float under attr=value format */

  char * parseVector3f(char *ptok, float *vector);
  /**< Returns a vector under vx,vy,vz format */

  char * parseVector3f(char *ptok, float *vector, const char *attrstr);
  /**< Returns a vector under attr=vx,vy,vz format */

  char * parseGeometry(char *geom);
  /**<
   * Creates a new solid.
   * string 'geom' gives the solid's geometry
   * Returns the solid's handle.
   */

  //
  // noid (Network Object Identifier)
  //
  void setWObjectId();
  /**< Sets the WObjectid.
   * Assigns a unique identifier to each Vreng object
   * whether if be a networked object or not.
   */

  void copyNetObjectId(const NetObjectId _noid);
  /**< Copies the NetObjectid in WObjectId */

  void setSrcId(const uint32_t src_id);
  /**< Sets the SrcId */

  void setPortId(const uint16_t port_id);
  /**< Sets the PortId */

  void setObjId(const uint16_t obj_id);
  /**< Sets the ObjId */

  uint32_t getSrcId();
  /**< Gets the SrcId */

  uint16_t getPortId();
  /**< Gets the PortId */

  uint16_t getObjId();
  /**< Gets the ObjId */

  WObject * getWObjectbyWObject(WObject *po);
  /**< Gets a WObject from the lists */

  //
  // 3D
  //
  void setBB();
  /**< Sets solid's Bounding Box */

  void getBB();
  /**< Gets object's Bounding Box */

  void update3D();
  /**< Updates object in the 3D */

  void updateAll3D();
  /**< Updates object in the 3D */

  void updateCamera();
  /**< Updates camera in the 3D */

  void setObjectToSolid();
  /**< Puts WObject pointer in a Solid */

  //
  // Grid
  //
  void insertObjectIntoGrid();
  /**< Adds an object into the grid */

  void deleteObjectFromGrid();
  /**< Deletes an object from the grid */

  void updateObjectIntoGrid(const Pos &oldpos);
  /**< Updates an object into the grid */

  //
  // Lists
  //
  ObjectList * addObjectToList(ObjectList *pl);
  /**< Adds an object pointer into a list */

  ObjectList * addObjectToListOnce(ObjectList *pl, int id);
  /**< Adds an object pointer into a list indentified with id */

  ObjectList * deleteObjectFromList(ObjectList *pl);
  /**< Deletes an object pointer from a list */

  ObjectList * getVicinityObjectList(const Pos &oldpos);
  /**< Returns list of pointers on objects touching cell where is the object */

  ObjectList * addObjectListToList(ObjectList *l1, ObjectList *l2);
  /**< Concatenation (test of "ispointed") of list pointers on an object */

  //
  // Movements
  //
  void imposedMovement(time_t sec, time_t usec);
  /**< Handles an object movement */

  void elementaryImposedMovement(const float lasting);
  /**< Handles an elementary object movement */

  void enableImposedMovement();
  /**< Enables movement on an object */

  void enablePermanentMovement();
  /**< Enables permanent movement on an object */

  void disablePermanentMovement();
  /**< Disables movement on an object */

  void initImposedMovement(const float lasting);
  /**< Initializes movement on an object */

  float diffTime(time_t sec, time_t usec);
  /**< Updates times on an object */

  void stopImposedMovement();
  /**< Stops a movement on an object */

  void permanentMovement(time_t sec, time_t usec);
  /**< Handles a permanent object movement */

  void elementaryPermanentMovement(const float lasting);
  /**< Handles an elementary  permanent object movement */

  void setMaxLasting(const float maxlast);
  /**< Sets the max lasting time on an object */

  //
  // Names
  //
  //pd static const char * getObjectNameById(const uint8_t type_id);
  static void getObjectNameById(const uint8_t type_id, char *name);
  /**< Gets a name by its id */

  void setObjectName(const char *str);
  /**< Sets an object name */

  void updateNames();
  /**< Updates names */

  //
  // Initializations
  //
  void initializeNetObject(const int oid, const uint8_t nprop, const uint8_t behave);
  /**< Initializes local NetObject */

  void createPermanentNetObject(const uint8_t props, const uint16_t oid);
  /**< Creates local permanent NetObject */

  void createVolatileNetObject(const uint8_t props);
  /**< Creates local volatile NetObject */

  void replicateVolatileNetObject(const uint8_t props, NetObjectId _noid);
  /**< Replicate distant volatile NetObject */

  void initializeObject(const uint8_t list_id);
  /**< Initializes object */

  void enableBehavior(const uint32_t flag);
  /**< Enables behavior */
  void disableBehavior(const uint32_t flag);
  /**< Disables behavior */
  bool isBehavior(const uint32_t flag);
  /**< Checks behavior */
  uint32_t collideBehavior();

  //
  // Collisions
  //
  void generalIntersect(WObject *pold, ObjectList *vicinityList);
  /**< General intersection of objects */

  void copyPosZAndBB(Pos &newpos);
  void copyPosAndBB(Pos &newpos);
  /**< Copy object position and Bounding Box */

  void copyPositionAndBB(WObject *pcur);
  /**< Copy object position and Bounding Box */

  void updateObject(const Pos &oldpos);
  /**< Updating object */

  void wallIntersect(WObject *pold, V3 *norm);
  /**< Intersects withe wall */

  //
  // Properties
  //
  void getProperty(uint8_t prop_id, Payload *pp);
  /**<
   * Gets the local copy property from the payload (pp->data).
   * Il faut renvoyer la position de lecture de la suite,
   * (utiliser getPayload et sa valeur de retour).
   * Typically called after a Delta reception.
   */

  void putProperty(uint8_t prop_id, Payload *pp);
  /**<
   * Puts the local copy property in the payload.
   * Il faut ecrire pp->data et renvoyer la longueur ecrite 
   * Typically called before a Delta emission.
   */

  void deleteReplica();
  /**< Detetes a replicated object */

#if 0 //todo
  void get_xy(Payload *pp);
  void get_z(Payload *pp);
  void get_az(Payload *pp);
  void get_ay(Payload *pp);
  void get_ax(Payload *pp);
  void get_hname(Payload *pp);

  void put_xy(Payload *pp);
  void put_z(Payload *pp);
  void put_az(Payload *pp);
  void put_ay(Payload *pp);
  void put_ax(Payload *pp);
  void put_hname(Payload *pp);
#endif

  virtual void setSqlStatus();
  /**< mysql interface */

  static void resetObjectsNumber();
  /**< Resets object seq. number */

  static uint32_t getObjectsNumber();
  /**< Gets current object sequence number */

  virtual WObject * prev();
  /**<
   * Head linking (will crash if there is no displayable object in tree!)
   */
  virtual WObject * next();
  /**< WObject pointers accessors.
   * To ensure compatibility, group trees can be seen as flat lists
   */

  virtual WObject * getUp_p() const {return up_p;};
  /**< Gets pointer to closest WObject Up in tree */

  virtual WObject * getNext_p() const {return next_p;};
  /**< Gets pointer to closest WObject on the right in tree */

  virtual WObject * getDown_p() const {return down_p;};
  /**< Gets pointer to closest WObject down in tree */

  virtual void setUp_p(WObject * po)  {up_p=po;};
  /**< Sets pointer to closest WObject Up in trees */

  virtual void setNext_p(WObject * po) {next_p=po;};
  /**< Sets pointer to closest WObject on the right in tree */

  virtual void setDown_p(WObject * po) {down_p=po;};
  /**< Sets pointer to closest WObject down in tree */

  virtual void add_son(WObject * p);
  /**< Adds the son to an object */

  virtual void changeComplexePosition();
  /**< Changes the position of the son */

  virtual void translateCoords();
  /**< Translates children coords */

  void traceSolidObject(const char *where);
  /**< test only */
};

/**
 * ObjectList Class
 */
class ObjectList {
 public:
  class WObject *pobject;
  class ObjectList *next;

  ObjectList();
  virtual ~ObjectList();

  virtual void freeObjectList();
  /**< Frees an object list */

  virtual void clearIspointedFlag();
  /**< Clears flags "ispointed" of all objects in a list */

  static ObjectList * getObjectsWithType(int);
  /**< Returns list of pointers on objects with the given type id
   * in the mobile list
   */

  static WObject * findObjectInMobile(uint8_t type_id, uint32_t src_id, uint16_t port_id, uint16_t obj_id);
  /**< Returns the object identified by type and id from the mobile list */

};

bool isValidType(const int type_id);


#include "initobj.h"

/* external variables */
extern ObjectList *stillList, *mobileList, *invisibleList, *deleteList, *cartList;
extern ObjectList *grid[GRIDL][GRIDW][GRIDH];

extern struct GeneralActionList generalActionList[ACTIONSNUMBER][OBJECTSNUMBER+1];
extern struct GeneralActionList2 generalActionList2[ACTIONSNUMBER][OBJECTSNUMBER+1];
extern PropertyFuncList putPropertyFunc, getPropertyFunc;

#if 0
#include "world.h"
#include "names.h"
#endif

void setActionHandler(int type_id, int action_id, void (*action)(void *d, time_t s, time_t u), const char *action_name);
void setActionFunc(int type_id, int action_id, void (*action)(WObject *po, void *d, time_t s, time_t u), const char *action_name);

void get_xy(WObject *po, class Payload *pp);
void get_z(WObject *po, class Payload *pp);
void get_az(WObject *po, class Payload *pp);
void get_ay(WObject *po, class Payload *pp);
void get_ax(WObject *po, class Payload *pp);
void get_hname(WObject *po, class Payload *pp);

void put_xy(WObject *po, class Payload *pp);
void put_z(WObject *po, class Payload *pp);
void put_az(WObject *po, class Payload *pp);
void put_ay(WObject *po, class Payload *pp);
void put_ax(WObject *po, class Payload *pp);
void put_hname(WObject *po, class Payload *pp);

#endif // WO_H
