#include "global.h"
#include "wo.h"
#include "world.h"	// worlds
#include "user.h"	// USER_DEFAULTHEIGHT
#include "http.h"	// urlAbs
#include "vrelet.h"
#include "vjc.h"

#include "gui.h"	// GuiRedirectToVrelet
#include "vgl.h"	// M4_to_GL


const WClass Vrelet::wclass(VRELET_TYPE, "Vrelet", Vrelet::creator);
void vreletInitFuncList(void) {}


/* Vrelet parser */
void Vrelet::parser(char *l)
{
  bool goturl = false;
#if 0 //pdinc
  float incx = 0.01; float incy = 0.01;	// 1 cm
#else
  incrx = 10; incry = 10;	// 1 cm
#endif
  fx = fy = 1.;

  l = parseObject(l);
  l = parsePosition(l);
  l = parseGeometry(l);

  while (l) {
    if (!strncmp(l, "class", 5))
      l = parseString(l, app, "class");
    else if (!strncmp(l, "incrx", 5))
#if 0 //pdinc
      l = parseFloat(l, &incx, "incrx");
#else
      l = parseInt(l, &incrx, "incrx");
#endif
    else if (!strncmp(l, "incry", 5))
#if 0 //pdinc
      l = parseFloat(l, &incy, "incry");
#else
      l = parseInt(l, &incry, "incry");
#endif
    else if (!strncmp(l, "fx", 2))
      l = parseFloat(l, &fx, "fx");
    else if (!strncmp(l, "fy", 2))
      l = parseFloat(l, &fy, "fy");
    else if (!strncmp(l, "base", 4)) {
      // url in the .vre file, try to find out the codebase URL
      l = parseString(l, url, "base");
      goturl = true;
    }
  }

  if (! goturl) { // there's none in the .vre file.
    char l2[URL_LEN];
    char *last = NULL;
    // if the world's url is like [...]/vre/world.vre
    // try to build an url like [...]/jar/vrengapp.jar
    strcpy(l2, World::getCurrentWorld()->url);
    if ((last = strrchr(l2, '/')))
      *last = '\0'; 
    if ((last = strrchr(l2, '/')) && (! strncmp(last, "/vre", 4))) {
      *last = '\0';
      urlAbs(l2, url);
      strcat(url, "/jar/vrengapp.jar");
      trace(DBG_IFC, "Vrelet: computed codebase=%s", url);
    }
    else { // neither of the above methods worked, just put a default url
      sprintf(url, "http://%s%sjar/vrengapp.jar", DEF_HTTP_SERVER, DEF_URL_PREFIX);
      trace(DBG_IFC, "Vrelet: default codebase=%s", url);
    }
  }
#if 0 //pdinc
  incrx = (int) (ceil(incx) * 1000.);
  incry = (int) (ceil(incy) * 1000.);
#endif

  // defaults for the other values
  o2 = NULL;
  wantDelta = false;
  needRedraw = false;
}

/* creation from a file */
WObject * Vrelet::creator(char *l)
{
  return new Vrelet(l);
}

Vrelet::Vrelet(char *l)
{
  parser(l);

  // Vrelet objects can't currently collide with anything - they move through them
  enableBehavior(COLLIDE_GHOST);

  initializeObject(LIST_MOBILE);

  Vjc::startApp(this);
  GuiRedirectToVrelet(this);
} 

/* Send a notification to the child saying a click occured */
void Vrelet::sendClick(int x, int y)
{
  VjcMessage *msg = new VjcMessage(this, VJC_MSGT_CLICK, VJC_MSGV_CLICK);
  msg->put32(x);
  msg->put32(y);
  msg->sendData();
  delete msg;
}

/* Send some object's position to the child */
void Vrelet::sendPos(WObject *po)
{
  VjcMessage *msg = new VjcMessage(this, VJC_MSGT_POS, VJC_MSGV_SET);

  msg->putOID(po);
  msg->putPos(po);
  msg->sendData();
  delete msg;
}

/* Same as sendPos, except that no position is sent */
void Vrelet::sendPosError(int type, int ssrc, int port, int id)
{
  VjcMessage *msg = new VjcMessage(this, VJC_MSGT_POS, VJC_MSGV_ERROR);

  msg->put8(type);
  msg->put32(ssrc);
  msg->put16(port);
  msg->put16(id);
  msg->sendData();
  delete msg;
}

/* Answer a child's query */
void Vrelet::answerTypeQuery(int _type)
{
  VjcMessage *msg = new VjcMessage(this, VJC_MSGT_QUERY, VJC_MSGV_QANS);

  if (_type > 0) {
    ObjectList *list = ObjectList::getObjectsWithType(_type);
    if (list) {
      ObjectList *pl;
      int cnt = 0;
      for (pl = list; pl; pl = pl->next)
        cnt++;
      msg->put32(cnt);
      for (pl = list; pl; pl = pl->next)
        msg->putOID(pl->pobject);
      list->freeObjectList();
      list = NULL;
      trace(DBG_IFC, "answerTypeQuery: send %d replies for %d", cnt, _type);
    }
  }
  else {
    msg->put32(1);
    msg->putOID(worlds->plocaluser);
  }
  msg->sendData();
  delete msg;
}

/* Read a header struct for the raw packet data */
void vreletDumpHeader(vjcHeader hdr)
{
  trace(DBG_IFC, "%x %d %d %d %d %d",
  hdr.proto,
  hdr.version,
  hdr.app_ssrc,
  hdr.msg_type,
  hdr.msg_id,
  hdr.data_len);
}

/* Deal with input from the client app */
void Vrelet::processClient()
{
  V3 color;
  bool processed = false;
  int tag;

  // check if the client has sent anything
  VjcMessage *msg;
  if ((msg = Vjc::getData(this)) == NULL)
    return;
  vjcHeader header = msg->getHeader();

  color.v[0] = 0.;
  color.v[1] = 0.;
  color.v[2] = 0.;

  // Process the client's data
  switch (header.msg_type) {

  case VJC_MSGT_ADD:
    tag = msg->read16();
    color = msg->readPoint3D();

    if ( (header.msg_id == VJC_MSGV_LINE)
    ||   (header.msg_id == VJC_MSGV_LOOP)
    ||   (header.msg_id == VJC_MSGV_FILL)) {
      // create a new line
      o2 = vreletNewObject2D(o2, header.msg_id, tag, color);
      while (msg->hasData(2*sizeof(int32_t))) {
        vreletAddPoint(o2, msg->readPoint2D());
      }
      processed = true;
    }
    else if (header.msg_id == VJC_MSGV_CIRCLE) {
      // Create a new circle
      o2 = vreletNewObject2D(o2, header.msg_id, tag, color);
      vreletAddPoint(o2, msg->readPoint2D());
      vreletAddPoint(o2, msg->readPoint2D());
      processed = true;
    }
    needRedraw = true;
    break;

  case VJC_MSGT_DEL:
    if (header.msg_id == VJC_MSGV_DELALL) {
      vreletFreeObject2DList(o2);
      o2 = NULL;
    }
    else {
      tag = msg->read16();
      o2 = vreletRemoveObject2D(o2, header.msg_id, tag);
    }
    needRedraw = true;
    break;

  case VJC_MSGT_POS: {
    // get the objet ID
    uint8_t  type = msg->read8();
    uint32_t src  = msg->read32();
    uint16_t port = msg->read16();
    uint16_t id   = msg->read16();

    // locate the object
    WObject *who = ((type==0) ? this : ObjectList::findObjectInMobile(type, src, port, id));
    if (! who) {
      // we didn't find anything that matched
      sendPosError(type, src, port, id);
    }
    else if (header.msg_id == VJC_MSGV_ASK) {
      // message was a query - send the data back
      sendPos(who);
      processed = true;
    }
    else if ((header.msg_id == VJC_MSGV_SET)
          || (header.msg_id == VJC_MSGV_UPD)) {
      // message was a move, get the movement information
      CopyV3(&posDelta, msg->readDelta());
      CopyV3(&angDelta, msg->readDelta());

      if (who == this) { // move myself (vrelet object)
        wantDelta = header.msg_id;
      }
      else { // push someone else (vreng object)
        WObject *pold = new WObject();	// needed for collision

        // make the changes
        who->copyPositionAndBB(pold);
        doDeltaPos(who);

        // update the object in the Vreng 3D thingies
        who->updateObjectIntoGrid(pold->pos);
        //pd who->updateAll3D();
        who->update3D();
        who->getBB();

        // check if we need to move the camera (user)
        if (who == worlds->plocaluser) {
          who->updateCamera();
        }

        // propagate the changes
        ObjectList *vicinity = who->getVicinityObjectList(pold->pos);
        who->generalIntersect(pold, vicinity);
        delete pold;
        vicinity->freeObjectList();
      }
      processed = true;
    }
    break;
  }

  case VJC_MSGT_QUERY:
    if (header.msg_id == VJC_MSGV_QTYPE) {
      int type = msg->read32();
      answerTypeQuery(type);
      processed = true;
    }
    break;
  }

  if (! processed)
    trace(DBG_IFC, "message ignore %d %d", header.msg_type, header.msg_id);
  delete msg;
}

/* React to a user click on our surface */
void Vrelet::click(V3 dir)
{
  // calculate the intersection between the click vector
  // and our primary surface

  // get two principal vectors and a normal one
  V3 v, w, norm;
  getSurfVecs(&v, &w, &norm);

  // check if the click comes from the right side of the surface
  float ps = norm.v[0]*dir.v[0] + norm.v[1]*dir.v[1] + norm.v[2]*dir.v[2];
  if (ps < 0) {
    warning("Vrelet::click: bad side!");
    return;
  }
  
  // eye position
  V3 u = V3_New(worlds->plocaluser->pos.x,
                worlds->plocaluser->pos.y,
                worlds->plocaluser->pos.z + 0.5 * USER_DEFAULTHEIGHT);

  // object's center coordinates
  V3 c = V3_New(pos.x, pos.y, pos.z);

  // determine X coord. relativly to our surface
  float det  = DetV3(dir, v, w);
  float varx  = calcVar(c, u, w, dir);
  float alpha = varx/det;

  // determine Y coord. relativly to our surface
  float vary = calcVar(c, u, dir, v);
  float beta = vary/det;

  // scale these values
  int x = (int) (alpha * (float) incrx / fx);
  int y = (int) (beta  * (float) incry / fy);

  // notify the child
  trace(DBG_IFC, "Vrelet::click: x=%d y=%d", x, y);
  sendClick(x, y);
}

/* Dummy: always say we need to move */
void Vrelet::updateTime(time_t sec, time_t usec, float *lasting)
{
  *lasting = 1.;
}

void Vrelet::doSetPos(WObject *po)
{
  po->pos.x = posDelta.v[0];
  po->pos.y = posDelta.v[1];
  po->pos.z = posDelta.v[2];
  po->pos.az = angDelta.v[0];
  po->pos.ay = 0;
  po->pos.ax = angDelta.v[1];
}

void Vrelet::doDeltaPos(WObject *po)
{
  po->pos.x += posDelta.v[0];
  po->pos.y += posDelta.v[1];
  po->pos.z += posDelta.v[2];
  po->pos.az += angDelta.v[0];
  po->pos.ay += 0;
  po->pos.ax += angDelta.v[1];
}

/* Return yes if the child has sent a delta request */
bool Vrelet::isMoving()
{
  processClient();
  return (wantDelta > 0);
}

/* Propagate the last deltas to the object's position */
void Vrelet::changePosition(float lasting)
{
  if (wantDelta == VJC_MSGV_SET)
    doSetPos(this);
  else if (wantDelta == VJC_MSGV_UPD)
    doDeltaPos(this);
  wantDelta = 0;
}

/* Send the client a notification that a thing just entered/exited its BB */
void Vrelet::sendIntersect(WObject *pcur, WObject *pold, int inOrOut)
{
  VjcMessage *msg = new VjcMessage(this, VJC_MSGT_ISEC, inOrOut);

  //printf("sendIntersect: %d name=%s\n", inOrOut, pcur->name.instance_name);
  msg->putOID(pcur);
  msg->putPos(pcur);
  msg->putPos(pold);
  msg->sendData();
  delete msg;
}

/* An ingoing intersection occured */
void Vrelet::whenIntersect(WObject *pcur, WObject *pold)
{
  sendIntersect(pcur, pold, VJC_MSGV_ISECIN);
}

/* An outgoing intersection occured */
bool Vrelet::whenIntersectOut(WObject *pcur, WObject *pold)
{
  sendIntersect(pcur, pold, VJC_MSGV_ISECOUT);
  return true;
}

/* Draws the Vrelet's 2D object list */
void Vrelet::render()
{
  draw2DList();
}

/* Turn the child on */
void Vrelet::quit()
{
  Vjc::stopApp(this);
  GuiRedirectToVrelet(NULL);
}

/*
 * 2D Object manipulation
 */

/* 2D Creation */
Object2D *vreletNewObject2D(Object2D *o2l, int _type, int _tag, V3 _color)
{
  Object2D *o2 = new Object2D[1];
  o2->type = _type;
  o2->tag = _tag;
  CopyV3(&o2->color, _color);
  o2->points = NULL;
  o2->next = o2l;
  return o2;
}

/* Remove one object from a list */
Object2D *vreletRemoveObject2D(Object2D *o2l, int _type, int _tag)
{
  Object2D *ret = o2l;
  Object2D *prev = NULL;

  for (Object2D *pl = o2l; pl; pl = pl->next) {
    if ((pl->type == _type) && (pl->tag == _tag)) {
      if (! prev)
        ret = pl->next;
      else {
        prev->next = pl->next;
        ret = o2l;
      }
      vreletFreeObject2D(pl);
        return ret;
    }
    prev = pl;
  }
  warning("attempt to remove a non-existing 2d object (%d %d)", _type, _tag);
  return o2l;
}

/* Free an object's point list */
void vreletFreePointList(PointList *pts)
{
  PointList *tmp;

  while (pts) {
    tmp = pts->next;
    delete[] pts;
    pts = tmp;
  }
}

/* Free an object2D and its point list */
void vreletFreeObject2D(Object2D *_o2)
{
  if (! _o2)
    warning("attempt to free a null 2d object");
  else {
    vreletFreePointList(_o2->points);
    delete[] _o2;
  }
}

/* Free an object2d list */
void vreletFreeObject2DList(Object2D *o2l)
{
  Object2D *pl = o2l;
  Object2D *todel = o2l;

  while (pl) {
    todel = pl;
    pl = pl->next;
    vreletFreeObject2D(todel);
  }
}

/* Add a point to the point list */
void vreletAddPoint(Object2D *_o2, V3 _point)
{
  PointList *pts = new PointList[1];

  trace(DBG_IFC, "vreletAddPoint: px=%.2f py=%.2f", _point.V_X, _point.V_Y);
  CopyV3(&pts->point, _point);
  pts->next = _o2->points;
  _o2->points = pts;
}

/* Render a list of object */
void Vrelet::draw2DList()
{
  // have things changed since last time we build the dlist ?
  if (needRedraw) {
    needRedraw = false;

    float rx = incrx / fx;
    float ry = incry / fy;

    displaylist = glGenLists(1);
    glNewList(displaylist, GL_COMPILE);
    glTranslatef(0, 0, 0.02); // extra z so that the figures are drawn slightly
  			  // above the surface
    while (o2) {
      trace(DBG_WO, "draw2DList: o2->type=%d", o2->type);
      switch (o2->type) {

      case VJC_MSGV_LINE:
      case VJC_MSGV_LOOP:
      case VJC_MSGV_FILL: {
        PointList *pts = o2->points;
        if (! pts) return;

        glColor3f(o2->color.v[0], o2->color.v[1], o2->color.v[2]);
        //glColor3f(1.0, 1.0, 1.0);

        if (o2->type == VJC_MSGV_LINE)
          glBegin(GL_LINE_STRIP);
        else if (o2->type == VJC_MSGV_LOOP)
          glBegin(GL_LINE_LOOP);
        else 
          glBegin(GL_POLYGON);

        while (pts) {
          glVertex2f(pts->point.v[0]/rx, pts->point.v[1]/ry);
          pts = pts->next;
        }
        glEnd();
        }
        break;

      case VJC_MSGV_CIRCLE: {
        V3 diam = o2->points->point;
        V3 center = o2->points->next->point;
        float arc = M_2PI / 20;

        glBegin(GL_LINE_LOOP);
         glColor3f(o2->color.v[0], o2->color.v[1], o2->color.v[2]);
         for (int i=19; i>=0; i--) { 
           glVertex2f(center.v[0]/rx+diam.v[0]*Cos(i*arc)/rx,
                      center.v[1]/ry+diam.v[1]*Sin(i*arc)/rx);
         }
        glEnd();
        }
        break;

      default:
        break;
      }
      o2 = o2->next;
    }
    glEndList();
  }

  // Do the actual drawing
  float gl_mat[16];
  glLoadName(displaylist);
  glPushMatrix();
  M4_to_GL(gl_mat, &soh->posmat);
  glMultMatrixf(gl_mat);
  glCallList(displaylist);
  glPopMatrix();
}

/* Returns two vectors that describe the object's surface */
void Vrelet::getSurfVecs(V3 *v, V3 *w, V3 *norm)
{
  M4 rot = MulM4(RotateM4(pos.az, UZ), RotateM4(pos.ax, UX));
  V3 vec = V3_New(1, 0, 0);
  MulM4V3(v, &rot, &vec);
  vec = V3_New(0, 1, 0);
  MulM4V3(w, &rot, &vec);
  vec = V3_New(0, 0, 1);
  MulM4V3(norm, &rot, &vec);
}

/* Compute an quantity that's used in the line/surface intersection */
float Vrelet::calcVar(V3 c, V3 u, V3 w, V3 v)
{
  return (  (c.v[0] - u.v[0]) * (v.v[2] * w.v[1] - v.v[1] * w.v[2])
          + (c.v[1] - u.v[1]) * (v.v[0] * w.v[2] - v.v[2] * w.v[0])
          + (c.v[2] - u.v[2]) * (v.v[1] * w.v[0] - v.v[0] * w.v[1]) );
}
