#include "global.h"
#include "wo.h"
#include "vnc.h"
#include "glu.h"	// glu alternative

#include "vgl.h"	// bbsize
#include "gui.h"	// GuiRedirectToVnc GuiLaunchVncConnect


const WClass Vnc::wclass(VNC_TYPE, "Vnc", Vnc::creator);


/**
 * Parse everything concerning the server if defined in the config file.
 * Syntax:  server=servename,port,password
 */
char * Vnc::parseVNCServer(char *ptok)
{
  if (ptok) {
    char *p;
    int vncport = VNC_DEF_PORT;
    char args[256], server[80], passwd[20];

    memset(args, 0, sizeof(args));
    memset(server, 0, sizeof(server));
    memset(passwd, 0, sizeof(passwd));
    memset(passwdFile, 0, sizeof(passwdFile));

    if (sscanf(ptok, "server=%s", args) != 1)
      return ptok;
#if 0
    if ((p = strchr(args, '>')) == 0)
      *p = 0;
#endif
    if ((p = strchr(args, ',')) == 0) {
      warning("parsing: can't get a port from Fileline:\n%s", ptok);
      strcpy(server, args);
    }
    else {
      strncpy(server, args, p-args);
      p++;
      sscanf(p, "%d,%s", &vncport, passwd);
      if (vncport == 0)
        vncport = VNC_DEF_PORT;
    }
    trace(DBG_VNC, "parseVNCServer: %s:%d:%s", server, vncport, passwd);

    strcpy(serverName, server);
    port = vncport;

    sprintf(passwdFile, "%s-%s", vrengpasswdfile, server);
    if (strlen(passwd) > 8)
      passwd[8] = '\0';
    vncEncryptAndStorePasswd(passwd, passwdFile);

    isServerDefined = true;
  }
  ptok = strtok(NULL, SEP);
  return ptok;
}

/* creation from a file */
WObject * Vnc::creator(char *l)
{
  return new Vnc(l);
}

Vnc::Vnc(char *l)
{ 
  isServerDefined = false;
  isConnected = false;
  tex_pixmap = NULL;
  const GLfloat colordef[] = {0.3, 0.3, 0.3};	// grey

  for (int i=0; i<3; i++)
    color[i] = colordef[i];
 
  // get the vnc parameters
  l = parseObject(l);
  l = parsePosition(l);
  l = parseVNCServer(l);
  l = parseGeometry(l);		// bbox
  while (l) {
    if (!strncmp(l, "color", 5))
      parseVector3f(l, color, "color");	// parse screen color
  }

  glColor3fv(color);

  enableBehavior(NO_ELEMENTARY_MOVE);
  enableBehavior(COLLIDE_NEVER);
  initializeObject(LIST_MOBILE);
  
  // texture creation
  glGenTextures(1, &tex_num); // texture number given by OpenGL
  defaultTexture();
  trace(DBG_VNC, "VNC framebuffer: w=%d h=%d tex_num=%d", tex_width, tex_height, tex_num);

  if (isServerDefined)
    connectServer();
} 

void Vnc::defaultTexture()
{
  if (tex_pixmap)
    delete[] tex_pixmap;
  tex_pixmap = NULL;

  tex_width = 2;
  tex_height = 2;

  if ((tex_pixmap = new GLubyte[3 * tex_width * tex_height]) == 0) {
    error("defaultTexture: can't new tex_pixmap");
    return;
  }

  GLubyte *pix = tex_pixmap;
  for (int i=0; i < tex_width; i++) {
    for (int j=0; j < tex_height; j++) {
      *pix++ = (GLubyte)127;
      *pix++ = (GLubyte)127;
      *pix++ = (GLubyte)127;
    }
  }

  glEnable(GL_TEXTURE_2D);	// we need to use a texture
  glBindTexture(GL_TEXTURE_2D, tex_num);	// we use ours.

  // put it in the video memory
  glTexImage2D(GL_TEXTURE_2D, 0, 3, tex_width, tex_height,
               0, GL_RGB, GL_UNSIGNED_BYTE, tex_pixmap);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

  // no interpolation
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

  // no texture repetition
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
  glDisable(GL_TEXTURE_2D);

  u = v = 1;

#if 0 //pd dlist
  drawScreen();
#endif
  drawBorder();
  buildScreen();
}

void Vnc::changePermanent(float lasting)
{
  if (isConnected) {
    FD_ZERO(&rmask);
    FD_SET(vncClient->getSock(), &rmask);
    delay.tv_sec =  0;
    delay.tv_usec = 1;

    int sel = select(32, &rmask, NULL, NULL, &delay);
    if (sel) {
      if (FD_ISSET(vncClient->getSock(), &rmask)) {
        if (!vncClient->HandleRFBServerMessage()) {
          error("could'nt handle RFB server message");
          return;
        }

        // we need to use a texture, we use ours
        glEnable(GL_TEXTURE_2D);
        glBindTexture(GL_TEXTURE_2D, tex_num);

        // put it in the video memory
#if defined(HAVE_LIBGLU) && defined(HAVE_GL_GLU_H)
        gluBuild2DMipmaps(GL_TEXTURE_2D, 3, tex_width, tex_height,
                          GL_RGB, GL_UNSIGNED_BYTE, tex_pixmap);
#else
        glTexImage2D(GL_TEXTURE_2D, 0, 3, tex_width, tex_height,
                     0, GL_RGB, GL_UNSIGNED_BYTE, tex_pixmap);
#endif
        glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

        // no interpolation
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
                        GL_LINEAR_MIPMAP_LINEAR);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

        // no texture repetition
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
        glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
        glDisable(GL_TEXTURE_2D);
      }
    }
  }
}

/** Draw the borders of the screen */
void Vnc::drawBorder()
{
  borderlist = glGenLists(1);
  glNewList(borderlist, GL_COMPILE);
  glBegin(GL_QUAD_STRIP);
   glVertex3fv(vertices+ 0);
   glVertex3fv(vertices+12);
   glVertex3fv(vertices+ 3);
   glVertex3fv(vertices+15);
   glVertex3fv(vertices+ 6);
   glVertex3fv(vertices+18);
   glVertex3fv(vertices+ 9);
   glVertex3fv(vertices+21);
  glEnd();
  glBegin(GL_QUADS);
   glVertex3fv(vertices+24);
   glVertex3fv(vertices+27);
   glVertex3fv(vertices+18);
   glVertex3fv(vertices+15);
  glEnd();
  glBegin(GL_TRIANGLES);
   glVertex3fv(vertices+21);
   glVertex3fv(vertices+18);
   glVertex3fv(vertices+27);
   glVertex3fv(vertices+12);
   glVertex3fv(vertices+24);
   glVertex3fv(vertices+15);
  glEnd();
  glEndList();
}

/** display the screen */
void Vnc::drawScreen()
{
#if 0 //pd dlist
  screenlist = glGenLists(1);
  glNewList(screenlist, GL_COMPILE);
#endif
  glBegin(GL_QUADS);
   glTexCoord2f(u,v); glVertex3fv(vertices+0);
   glTexCoord2f(u,0); glVertex3fv(vertices+3);
   glTexCoord2f(0,0); glVertex3fv(vertices+6);
   glTexCoord2f(0,v); glVertex3fv(vertices+9);
  glEnd();
#if 0 //pd dlist
  glEndList();
#endif
}

void Vnc::buildScreen()
{
  if (! soh)
    return;

  float front = soh->bbsize.v[0];
  float back = -soh->bbsize.v[0];
  float top = soh->bbsize.v[2];
  float bot = -soh->bbsize.v[2];
  float left = soh->bbsize.v[1];
  float right = -soh->bbsize.v[1];
  float depth = 0.9 * soh->bbsize.v[0];
  float height;
  float width;
  float rsol = soh->bbsize.v[1] / soh->bbsize.v[2];
  float rtex = u * tex_width / (v * tex_height);

  if (rtex > rsol) {
    height = -soh->bbsize.v[2] + 2*0.95 * soh->bbsize.v[1]/rtex;
    width = 0.95 * soh->bbsize.v[1];
  }
  else {
    height = (2*0.95-1) * soh->bbsize.v[2];
    width = 0.95 * soh->bbsize.v[2] * rtex;
  }
  vertices[ 0] = depth;  vertices[ 1] = width;  vertices[ 2] = bot;
  vertices[ 3] = depth;  vertices[ 4] = width;  vertices[ 5] = height;
  vertices[ 6] = depth;  vertices[ 7] = -width; vertices[ 8] = height;
  vertices[ 9] = depth;  vertices[10] = -width; vertices[11] = bot;
  vertices[12] = front;  vertices[13] = left;   vertices[14] = bot;
  vertices[15] = front;  vertices[16] = left;   vertices[17] = top;
  vertices[18] = front;  vertices[19] = right;  vertices[20] = top;
  vertices[21] = front;  vertices[22] = right;  vertices[23] = bot;
  vertices[24] = back;   vertices[25] = left;   vertices[26] = bot;
  vertices[27] = back;   vertices[28] = right;  vertices[29] = bot;
}

void Vnc::render()
{
  GLint renderMode;

  glGetIntegerv(GL_RENDER_MODE, &renderMode);
  glEnable(GL_CULL_FACE);
  glEnable(GL_TEXTURE_2D);			// we need to use a texture
  glBindTexture(GL_TEXTURE_2D, tex_num);	// we use ours
   
  update3D();

  GLfloat glmat[16];
  for (int i=0; i < 4; i++)
    for (int j=0; j < 4; j++)
      glmat[i*4+j] = soh->posmat.m[j][i];

  glPushMatrix();
  glMultMatrixf(glmat);

#if 0 //pd dlist
  glCallList(screenlist);	// display screen
#else
  drawScreen();
#endif
 
  glDisable(GL_TEXTURE_2D);	// no use of textures anymore

  glCallList(borderlist);	// display border

  // updating parameters for mouse gestion
  if (renderMode == GL_RENDER) {
    glGetIntegerv(GL_VIEWPORT, viewport);
    glGetDoublev(GL_MODELVIEW_MATRIX, modelMatrix);
    glGetDoublev(GL_PROJECTION_MATRIX, projMatrix);
  }
  glPopMatrix();
}

void Vnc::getVncCoords(int &x, int &y)
{
  y = viewport[3] - y;

  // get intersection with z=0 plan
  GLdouble ox, oy, oz;
  gluUnProject((GLdouble) x, (GLdouble) y, (GLdouble) 0,
               modelMatrix, projMatrix, viewport,
               &ox, &oy, &oz);
  // get intersectiom with z=1 plan
  GLdouble ax, ay, az;
  gluUnProject((GLdouble) x, (GLdouble) y, (GLdouble) 1,
               modelMatrix, projMatrix, viewport,
               &ax, &ay, &az);

  // calculated intersection coordinates in vnc screen plane
  GLdouble dx, dy, dz;
  dx = ax - ox;
  if (dx >= 0) {
    x = -1;
    y = -1;
    return;
  }
  dy = ay - oy;
  dz = az - oz;

  GLdouble t = (vertices[0] - ox) / dx;
  GLdouble ey = t*dy + oy;
  GLdouble ez = t*dz + oz;

  ey = (ey-vertices[10]) / (vertices[4]-vertices[10]);
  ez = (vertices[5]-ez) / (vertices[5]-vertices[11]);

  // built final coordinates
  if ((ey > 0) && (ey < 1) && (ez > 0) && (ez < 1)) {
    x = (int) (ey * vncClient->realScreenWidth);
    y = (int) (ez * vncClient->realScreenHeight);
  }
  else {
    x = -1;
    y = -1;
  }
}

/**
 * vncReconnectServer: open the dialog window
 */
void vncReconnectServer(Vnc *po, void *d, time_t s, time_t u)
{
  if (po->isConnected || po->isServerDefined)
    GuiAlert("VNC: already connected, disconnect first!");
  else
#if WANT_UBIT
    GuiLaunchVncConnect(po);
#endif
    ;
}

/**
 * updates server parameters from UStr from the dialog window
 */
void Vnc::convert(const char *_server, const char *_port, const char *_passwd)
{
  if (!_server || !_port || !_passwd) {
    GuiAlert("VNC connection: wrong arguments");
    return;
  }

  strcpy(serverName, _server);
  port = atoi(_port);

  //if (strlen(passwd) > 8) passwd[8] = '\0';
  vncEncryptAndStorePasswd((char *) _passwd, passwdFile);
  connectServer();
}

/**
 * Access to Gui parameters eventsCaptured and capturingObject
 * give the focus to this object: All events redirected
 */
void vncTakeFocus(Vnc *po, void *d, time_t s, time_t u)
{
  notice("Take Focus");
  GuiRedirectToVnc(po);
}

/** Remove the focus from the object */
void vncLeaveFocus(Vnc *po, void *d, time_t s, time_t u)
{
  notice("Leave Focus");
  GuiRedirectToVnc(NULL);
}

void Vnc::connectServer()
{
  vncClient = new VNCClientTextured(serverName, port, passwdFile);
  
  // VNC client initialisation 
  if (vncClient->VNCInit()) {
    isConnected = true;
    if (! isServerDefined)
      vncTakeFocus(this, NULL, 0L, 0L);

    trace(DBG_VNC, "VNC connectServer: successful");

    vncClient->SendFramebufferUpdateRequest(0, 0,
                                            vncClient->realScreenWidth,
                                            vncClient->realScreenHeight,
                                            false);
    enablePermanentMovement();
  }
  else {
    isServerDefined = false;
    GuiAlert("VNC connectServer: failed");
  }

  // VNCClient init end

  // texture initialisation to the framebuffer
  tex_pixmap = (GLubyte *) vncClient->FrameBuffer;
  tex_width = vncClient->framebufferWidth;
  tex_height = vncClient->framebufferHeight;
 
  glEnable(GL_TEXTURE_2D);		// need to use a texture
  glBindTexture(GL_TEXTURE_2D, tex_num); // we use ours

  // put it in the memory video
  glTexImage2D(GL_TEXTURE_2D, 0, 3, tex_width, tex_height,
               0, GL_RGB, GL_UNSIGNED_BYTE, tex_pixmap);
  glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL);

  // no interpolation
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);

  // no texture repetition
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP);
  glDisable(GL_TEXTURE_2D);

  u = (float) vncClient->realScreenWidth / tex_width;
  v = (float) vncClient->realScreenHeight / tex_height;
}

void vncDisconnectServer(Vnc *po, void *d, time_t s, time_t u)
{
  if (po->isConnected) {
    if (po->vncClient->VNCClose() == false)
      warning("VNC disconnectServer: failed");
    else {
      po->isConnected = false;
      po->isServerDefined = false;
      po->defaultTexture();
      GuiRedirectToVnc(NULL);
    }
  }
}

/**
 * Redirects events to the Vnc object
 */
bool Vnc::redirectEvent(int x, int y, int button)
{
  if (! isConnected) 
    return false;

  const Cardinal card = 4;
  const char *params[card];
  char *p1 = new char[16];
  char *p2 = new char[16];
  char *p3 = new char[16];

  getVncCoords(x, y);  // change coords

  params[0] = "ptr";
  sprintf(p1, "%d", x); params[1] = p1;
  sprintf(p2, "%d", y); params[2] = p2;
  sprintf(p3, "%d", button); params[3] = p3;

  vncClient->SendRFBEvent((char **) params, (Cardinal *) &card); // send to VNC
  delete[] p1;
  delete[] p2;
  delete[] p3;
  return true;
}

bool Vnc::redirectEvent(const char *key, bool is_down)
{
  if (! isConnected)
    return false;

  const Cardinal card = 2;
  const char *params[card];
  char *p1 = new char[16];

  params[0] = (is_down) ? "keydown" : "keyup";
  strcpy(p1, key);
  params[1] = p1;

  vncClient->SendRFBEvent((char **) params, (Cardinal *) &card); // send to VNC
  delete[] p1;
  return true;
}

void Vnc::quit()
{
  if (tex_pixmap)
    delete[] tex_pixmap;
  tex_pixmap = NULL;

  if (isConnected) {
    vncLeaveFocus(this, NULL, 0L, 0L);
    if (vncClient->VNCClose() == false)
      warning("VNC quit: disconnectServer failed");
    else {
      isConnected = false;
      defaultTexture();
      delete vncClient;
      vncClient = NULL;
    }
  }
}

void vncInitFuncList(void)
{
  setActionFunc(VNC_TYPE, 0, WO_ACTION vncTakeFocus,  "Focus");
  setActionFunc(VNC_TYPE, 1, WO_ACTION vncLeaveFocus, "Navigate");
  setActionFunc(VNC_TYPE, 2, WO_ACTION vncDisconnectServer, "Disconnect");
  setActionFunc(VNC_TYPE, 3, WO_ACTION vncReconnectServer,  "Reconnect");
}
