/* Copyright (c) Mark J. Kilgard, 1994. */
/**
 * (c) Copyright 1993, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED 
 * Permission to use, copy, modify, and distribute this software for 
 * any purpose and without fee is hereby granted, provided that the above
 * copyright notice appear in all copies and that both the copyright notice
 * and this permission notice appear in supporting documentation, and that 
 * the name of Silicon Graphics, Inc. not be used in advertising
 * or publicity pertaining to distribution of the software without specific,
 * written prior permission. 
 *
 * THE MATERIAL EMBODIED ON THIS SOFTWARE IS PROVIDED TO YOU "AS-IS"
 * AND WITHOUT WARRANTY OF ANY KIND, EXPRESS, IMPLIED OR OTHERWISE,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF MERCHANTABILITY OR
 * FITNESS FOR A PARTICULAR PURPOSE.  IN NO EVENT SHALL SILICON
 * GRAPHICS, INC.  BE LIABLE TO YOU OR ANYONE ELSE FOR ANY DIRECT,
 * SPECIAL, INCIDENTAL, INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY
 * KIND, OR ANY DAMAGES WHATSOEVER, INCLUDING WITHOUT LIMITATION,
 * LOSS OF PROFIT, LOSS OF USE, SAVINGS OR REVENUE, OR THE CLAIMS OF
 * THIRD PARTIES, WHETHER OR NOT SILICON GRAPHICS, INC.  HAS BEEN
 * ADVISED OF THE POSSIBILITY OF SUCH LOSS, HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, ARISING OUT OF OR IN CONNECTION WITH THE
 * POSSESSION, USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * US Government Users Restricted Rights 
 * Use, duplication, or disclosure by the Government is subject to
 * restrictions set forth in FAR 52.227.19(c)(2) or subparagraph
 * (c)(1)(ii) of the Rights in Technical Data and Computer Software
 * clause at DFARS 252.227-7013 and/or in similar or successor
 * clauses in the FAR or the DOD or NASA FAR Supplement.
 * Unpublished-- rights reserved under the copyright laws of the
 * United States.  Contractor/manufacturer is Silicon Graphics,
 * Inc., 2011 N.  Shoreline Blvd., Mountain View, CA 94039-7311.
 *
 * OpenGL(TM) is a trademark of Silicon Graphics, Inc.
 */

#include "global.h"
#include "wo.h"
#include "stars.h"


const WClass Stars::wclass(STARS_TYPE, "Stars", Stars::creator);
void starsInitFuncList(void) {}


void Stars::newStar(int n, int d)
{
  stars[n].x[0] = (float)(rand() % maxpos - maxpos/2);
  stars[n].y[0] = (float)(rand() % maxpos - maxpos/2);
  stars[n].z[0] = (float) (rand() % maxpos + d);
  stars[n].x[1] = stars[n].x[0];
  stars[n].y[1] = stars[n].y[0];
  stars[n].z[1] = stars[n].z[0];
  if (rand() % 10 == 0) {
    stars[n].offsetX = (float) (rand() % 100 - 100 / 2);
    stars[n].offsetY = (float) (rand() % 100 - 100 / 2);
    stars[n].offsetR = (float) (rand() % 25 - 25 / 2);
  } else {
    stars[n].offsetX = 0.0;
    stars[n].offsetY = 0.0;
    stars[n].offsetR = 0.0;
  }
}

static
void rotatePoint(float *x, float *y, float rotation)
{
  float tx = *x * Cos(rotation) - *y * Sin(rotation);
  float ty = *y * Cos(rotation) + *x * Sin(rotation);
  *x = tx;
  *y = ty;
}

void Stars::moveStars()
{
  float offset = speed * 60.;

  for (int n=0; n < count; n++) {
    stars[n].x[1] = stars[n].x[0];
    stars[n].y[1] = stars[n].y[0];
    stars[n].z[1] = stars[n].z[0];
    stars[n].x[0] += stars[n].offsetX;
    stars[n].y[0] += stars[n].offsetY;
    stars[n].z[0] -= offset;
    stars[n].rotation += stars[n].offsetR;
    if (stars[n].rotation >= STARS_MAXANGLE)
      stars[n].rotation = 0.;
  }
}

int Stars::pointStar(int n)
{
  float x0 = stars[n].x[0] * pos.x * 2 / stars[n].z[0];
  float y0 = stars[n].y[0] * pos.y * 2 / stars[n].z[0];

  rotatePoint(&x0, &y0, stars[n].rotation);
  x0 += pos.x;
  y0 += pos.y;
  if (x0 >= 0. && x0 < pos.x * 2 && y0 >= 0. && y0 < pos.y * 2)
    return true;
  else
    return false;
}

void Stars::showStar(int n)
{
  float stars_color[4] = {1, 1, 1, 0};

  float x0 = stars[n].x[0] * pos.x * 2 / stars[n].z[0];
  float y0 = stars[n].y[0] * pos.y * 2 / stars[n].z[0];
  rotatePoint(&x0, &y0, stars[n].rotation);
  x0 += pos.x;
  y0 += pos.y;

  if (x0 >= 0.0 && x0 < pos.x * 2 && y0 >= 0.0 && y0 < pos.y * 2) {
    float x1 = stars[n].x[1] * pos.x * 2 / stars[n].z[1];
    float y1 = stars[n].y[1] * pos.y * 2 / stars[n].z[1];
    rotatePoint(&x1, &y1, stars[n].rotation);
    x1 += pos.x;
    y1 += pos.y;

#if !defined(WITH_TINYGL)	/* glLineWidth missing */
    glLineWidth(maxpos / 100. / stars[n].z[0] + 1.);
#endif
    stars_color[1] = (float) ((warp - speed) / warp);
    stars_color[2] = (float) ((warp - speed) / warp);
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, stars_color);
    if (fabs(x0 - x1) < 1. && fabs(y0 - y1) < 1.) {
      glBegin(GL_POINTS);
      glVertex2f(x0, y0);
      glEnd();
    } else {
      glBegin(GL_LINES);
      glVertex2f(x0, y0);
      glVertex2f(x1, y1);
      glEnd();
    }
  }
}

void Stars::updateStars()
{
  for (int n=0; n < count; n++) {
    if (stars[n].z[0] > speed || (stars[n].z[0] > 0. && speed < warp)) {
      if (pointStar(n) == false)
        newStar(n, maxpos);
    }
    else
      newStar(n, maxpos);
  }
}

WObject * Stars::creator(char *l)
{
  return new Stars(l);
}

Stars::Stars(char *l)
{
  number = STARS_MAX;
  maxpos = STARS_MAXPOS;
  speed = STARS_SPEED;
  warp = STARS_WARP;

  l = parseObject(l);
  l = parsePosition(l);
  while (l) {
    if (!strncmp(l, "number", 6))
      l = parseUInt16(l, &number, "number");
    else if (!strncmp(l, "speed", 5))
      l = parseFloat(l, &speed, "speed");
    else if (!strncmp(l, "warp", 4))
      l = parseFloat(l, &warp, "warp");
    else if (!strncmp(l, "maxpos", 6))
      l = parseUInt16(l, &maxpos, "maxpos");
  }

  count = number / 2;
  if (pos.x <= 0)
    pos.x = STARS_WIDTH;
  if (pos.y <= 0)
    pos.x = STARS_HEIGHT;
  if (maxpos <= 0)
    maxpos = STARS_MAXPOS;
  if (warp <= speed)
    warp = STARS_WARP;

  stars = new starRec[number];
  srand((uint32_t) time(NULL));

  for (int n=0; n < number; n++)
    newStar(n, 100);	// 100 ?

  enableBehavior(NO_ELEMENTARY_MOVE);
  enableBehavior(NO_SELECTABLE);
  enableBehavior(COLLIDE_NEVER);
  initializeObject(LIST_MOBILE);

  enablePermanentMovement();
}

void Stars::changePermanent(float lasting)
{
  moveStars();
  updateStars();
}

void Stars::render()
{
  glPushMatrix();
  glTranslatef(0, 0, pos.z);
  glRotatef(RADIAN2DEGREE(pos.az), 0, 0, 1);
  glRotatef(RADIAN2DEGREE(pos.ax), 1, 0, 0);

  for (int n=0; n < count; n++) {
    if (stars[n].z[0] > speed || (stars[n].z[0] > 0. && speed < warp))
      showStar(n);
  }
  glPopMatrix();
}

void Stars::quit()
{
  delete[] stars;
}
