/* 
    particle.c
    Nate Robins, 1997
    An example of a simple particle system.
 */

#include "global.h"
#include "wo.h"
#include "particle.h"


const WClass Particle::wclass(PARTICLE_TYPE, "Particle", Particle::creator);
void particleInitFuncList(void) {}

static timeval begin;


/* creation from a file */
WObject * Particle::creator(char *l)
{
  return new Particle(l);
}

Particle::Particle(char *l)
{
  number = PARTICLE_DEF_NUM;
  flow = PARTICLE_DEF_FLOW;
  mode = PARTICLE_WATERFALL;
  speed = PARTICLE_DEF_SPEED;
  pt_size = PARTICLE_DEF_PTSIZE;
  living = 0;
  ground = 0;
  char modestr[16];
  for (int i=0; i<3; i++)
    color[i] = 1;

  l = parseObject(l);
  l = parsePosition(l);

  while (l) {
    if (!strncmp(l, "mode", 4)) {
      l = parseString(l, modestr, "mode");
      if (!strncmp(modestr, "fountain", 8))
        mode = PARTICLE_FOUNTAIN;
      else if (!strncmp(modestr, "rain", 4)) {
        mode = PARTICLE_RAIN;
        for (int i=0; i<3; i++)
          color[i] = 0.5;	// grey
      }
      else if (!strncmp(modestr, "snow", 4)) {
        mode = PARTICLE_SNOW;
        for (int i=0; i<3; i++)
          color[i] = 1.;	// white
      }
      else
        mode = PARTICLE_WATERFALL;
    }
    else if (!strncmp(l, "number", 6))
      l = parseUInt16(l, &number, "number");
    else if (!strncmp(l, "flow", 4))
      l = parseFloat(l, &flow, "flow");
    else if (!strncmp(l, "speed", 5))
      l = parseFloat(l, &speed, "speed");
    else if (!strncmp(l, "ground", 6))
      l = parseFloat(l, &ground, "ground");
    else if (!strncmp(l, "size", 4)) {
      l = parseUInt8(l, &pt_size, "size");
      if (pt_size <= 0)
        pt_size = 1;
    }
    else if (!strncmp(l, "color", 5))
      l = parseVector3f(l, color, "color");
  }

  switch (mode) {
  case PARTICLE_RAIN:
    points = false;
    break;
  case PARTICLE_SNOW:
  case PARTICLE_FOUNTAIN:
  case PARTICLE_WATERFALL:
    points = true;
    break;
  }

  enableBehavior(NO_ELEMENTARY_MOVE);
  enableBehavior(COLLIDE_NEVER);
  enableBehavior(NO_SELECTABLE);
  initializeObject(LIST_MOBILE);
  enablePermanentMovement();

  init();
}

/* timedelta: returns the number of seconds that have elapsed since
   the previous call to the function. */
float timedelta(void)
{
  timeval finish;

  gettimeofday(&finish, NULL);

  double difference = diffDates(begin, finish);
  begin = finish;
  //pd return (float)difference /(float)100;  /* CLK_TCK=1000 */
  return (float)difference / (2. * getRate());
}

bool fequal(float a, float b)
{
  float epsilon = 0.01;		//0.1
  
  return (((a - b) < epsilon) && ((a - b) > -epsilon)) ? true : false;
}

void Particle::timeStep(Particles *p, float dt)
{
  if (! p->alive)
    return;

  p->velocity[0] += 0;
  p->velocity[1] += 0;
  p->velocity[2] += PARTICLE_GRAVITY*dt;

  p->previous[0] = p->position[0];
  p->previous[1] = p->position[1];
  p->previous[2] = p->position[2];
  if (p->position[2] + p->velocity[2]*dt > ground +0.005) {
    p->position[0] += p->velocity[0]*dt * speed/2;
    p->position[1] += p->velocity[1]*dt * speed/2;
    p->position[2] += p->velocity[2]*dt * speed/2;
  }
}

void Particle::newParticle(Particles *p, float dt)
{
  p->position[0] = pos.x;
  p->position[1] = pos.y;
  p->position[2] = pos.z;
  p->previous[0] = p->position[0];
  p->previous[1] = p->position[1];
  p->previous[2] = p->position[2];

  switch (mode) {
  case PARTICLE_WATERFALL:
    p->velocity[0] = 1*(drand48()-0.5);
    p->velocity[1] = 1*(drand48()-0.5);
    p->velocity[2] = 0.5*(drand48()-0.0);	//orig: 0
    p->dampening = 0.45*drand48();
    break;
  case PARTICLE_FOUNTAIN:
  case PARTICLE_RAIN:
  case PARTICLE_SNOW:
    p->velocity[0] = 3*(drand48()-0.5);
    p->velocity[1] = 3*(drand48()-0.5);
    p->velocity[2] = 1*speed;			//orig: 6
    p->dampening = 0.35*drand48();
    break;
  }

  p->alive = 1;
  timeStep(p, 2*dt*drand48());
}

/**
   bounce: the particle has gone past (or exactly hit) the ground
   plane, so calculate the time at which the particle actually
   intersected the ground plane (s).  essentially, this just rolls
   back time to when the particle hit the ground plane, then starts
   time again from then.

   -  -   o A  (previous position)
   |  |    \
   |  s     \   o  (position it _should_ be at) -
   t  |      \ /                                | t - s 
   |  - ------X--------                         -
   |           \
   -            o B  (new position)
               
   A + V*s = 0 or s = -A/V

   to calculate where the particle should be:

   A + V*t + V*(t-s)*d

   where d is a damping factor which accounts for the loss
   of energy due to the bounce.
*/
void Particle::bounce(Particles *p, float dt)
{
  if (! p->alive)
    return;

  /* since we know it is the ground plane, we only need to
     calculate s for a single dimension. */
#if 1 //pdbounce
  float s = -p->previous[2]/p->velocity[2];
#else
  float s = -p->previous[1]/p->velocity[1];
#endif

  p->position[0] = (p->previous[0] + p->velocity[0] * s + 
       	            p->velocity[0] * (dt-s) * p->dampening);
#if 1 //pdbounce
  //pd p->position[2] = -p->velocity[2] * (dt-s) * p->dampening; // reflect
  p->position[1] = (p->previous[1] + p->velocity[1] * s + 
      	            p->velocity[1] * (dt-s) * p->dampening);
#else
  p->position[1] = -p->velocity[1] * (dt-s) * p->dampening; // reflect
  p->position[2] = (p->previous[2] + p->velocity[2] * s + 
      	            p->velocity[2] * (dt-s) * p->dampening);
#endif

  /* damp the reflected velocity (since the particle hit something,
     it lost some energy) */
  p->velocity[0] *=  p->dampening;
#if 1 //pdbounce
  p->velocity[1] *=  p->dampening;
  p->velocity[2] *= -p->dampening;	// reflect
#else
  p->velocity[1] *= -p->dampening;	// reflect
  p->velocity[2] *=  p->dampening;
#endif
}

void Particle::changePermanent(float lasting)
{
  float dt = timedelta();
  /* slow the simulation if we can't keep the frame rate up around 10 fps */
  if (dt > 0.1) {
    speed *= 0.75;
  } else if (dt < 0.1) {
    speed *= 1;
  }
  dt *= speed;

  /* resurrect a few particles */
  for (int i=0; i < flow*dt; i++) {
    newParticle(&particles[living], dt);
    living++;
    if (living >= number)
      living = 0;
  }

  for (int i=0; i < number; i++) {
    timeStep(&particles[i], dt);
    /* collision with ground? */
    if (particles[i].position[2] <= ground) {
      if (mode == PARTICLE_WATERFALL || mode == PARTICLE_FOUNTAIN)
        bounce(&particles[i], dt);
    }

    /* dead particle? */
    if (particles[i].position[2] < ground + 0.005 && fequal(particles[i].velocity[2], 0)) {
      particles[i].alive = 0;
      particles[i].alive = 1;		//pddead
      //pddead particles[i].velocity[2] = 0;	//pddead
    }
  }
}

void Particle::render()
{
  glPushMatrix();
  //glEnable(GL_FOG);
  //glEnable(GL_BLEND);
  glEnable(GL_COLOR_MATERIAL);
  glEnable(GL_POINT_SMOOTH);

  float c;
  glDisable(GL_LIGHTING);

  if (points) {
    glPointSize(pt_size);
    glBegin(GL_POINTS);
    for (uint16_t i=0; i < number; i++) {
      if (! particles[i].alive)
	continue;
      if (mode == PARTICLE_SNOW) {
        //c = 255;
        //glColor3ub((GLubyte) c, (GLubyte) c, (GLubyte) c);
        //glColor3f(1, 1, 1);
        glColor3fv(color);
      }
      else {
        c = particles[i].position[2]/2.1*255;
        glColor3ub((GLubyte) c, (GLubyte) (128+c*0.5), 255);
      }
      glVertex3fv(particles[i].position);
    }
    glEnd();
    glPointSize(1);
  }
  else {
    glLineWidth(pt_size);
    glBegin(GL_LINES);
    for (uint16_t i=0; i < number; i++) {
      if (! particles[i].alive)
	continue;
      if (mode == PARTICLE_RAIN) {
        //c = 128;
        //glColor3ub((GLubyte) c, (GLubyte) c, (GLubyte) c);
        glColor3fv(color);
      }
      else {
        c = particles[i].previous[2]/2.1*255;
        glColor3ub((GLubyte) c, (GLubyte) (128+c*0.5), 255);
      }
      glVertex3fv(particles[i].previous);
      if (mode == PARTICLE_RAIN) {
        //c = 128;
        //glColor3ub((GLubyte) c, (GLubyte) c, (GLubyte) c);
        glColor3fv(color);
      }
      else {
        c = particles[i].position[2]/2.1*255;
        glColor3ub((GLubyte) c, (GLubyte) (128+c*0.5), 255);
      }
      glVertex3fv(particles[i].position);
    }
    glEnd();
  }

  glEnable(GL_LIGHTING);
  glDisable(GL_POINT_SMOOTH);
  glDisable(GL_COLOR_MATERIAL);
  glDisable(GL_BLEND);
  glDisable(GL_FOG);
  glPopMatrix();
}

void Particle::init()
{
  float black[] = { 0, 0, 0, 0 };

  particles =  new Particles[number];
  gettimeofday(&begin, NULL);
  glFogfv(GL_FOG_COLOR, black);
  glFogf(GL_FOG_START, 2.5);
  glFogf(GL_FOG_END, 4);
  glFogi(GL_FOG_MODE, GL_LINEAR);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
}

void Particle::quit()
{
  delete[] particles;
  particles = NULL;
}
