#include "global.h"
#include "wo.h"
#include "world.h"	// setState
#include "http.h"	// Http::Read

enum {
  BEGINSECTION,
  HEADSECTION,
  OBJECTSECTION,
  COMMENTSECTION,
  GROUPSECTION,
  ENDSECTION
};

#define nextToken()	strtok(NULL, SEP)

static char *line = NULL;
static int section_type = 0;
static int grouped = 0;
static bool commented = false;
static bool head = false;


inline
char * skipEqual(char *p)
{
  char *tag = p;
  if ((p = strchr(p, '=')))
    p++;
  else
    error("missing '=' in %s", tag);
  return p;
}

inline
char * skipQuotes(char *p)
{
  if (*p == '"') {
    p++;
    char *q = strchr(p, '"');
    if (q)
      *q = 0;
    else
      error("missing '\"' in %s", p);
  }
  return p;
}

inline
char * nextSpace(char *p)
{
  while (*p && !isspace(*p))
    p++;
  return p;
}

inline
char * skipSpace(char *p)
{
  while (isspace(*p))
    p++;
  return p;
}

inline
char * skipComma(char *p)
{
  if ((p = strchr(p, ',')))
    p++;
  return p;
}

inline
bool isFloat(const char *p)
{
  return (p && (isdigit((int) *p) || *p == '-' || *p == '+' || *p == '.'));
}

inline
bool isInt(const char *p)
{
  return (p && (isdigit((int) *p) || *p == '-' || *p == '+'));
}


/** change of section type */
static
int parseSectionType(char *_line, int *psection_type)
{
  char *p;
  char *line = strdup(_line);
  char *ptok = skipSpace(line);
  const WClass *wclass = NULL;

  if (*ptok == '<') {
    if (commented)
      return COMMENTSECTION;
    ptok++;

    // check <!--
    if ((!strncmp(ptok, "!--", 3))) {
      commented = true;
      free(line);
      return COMMENTSECTION;
    }

    // check <head>
    else if ((!strncmp(ptok, "head", 4))) {
      head = true;
      free(line);
      return HEADSECTION;
    }
    else if ((!strncmp(ptok, "/head", 5))) {
      head = false;
      free(line);
      return HEADSECTION;
    }

    // check group
    else if ((!strncmp(ptok, "group", 5))) {
      grouped = 1;	// group begin
      free(line);
      return GROUPSECTION;	//gplhp
    }

    else if ((!strncmp(ptok, "/group", 6))) {
      grouped = 0;	// no group
      free(line);
      return GROUPSECTION;	//gplhp
    }

    else if (!strncmp(ptok, "vre>", 4)) {
      free(line);
      return BEGINSECTION;
    }
    else if (!strncmp(ptok, "/vre>", 5)) {
      free(line);
      return ENDSECTION;
    }

    else {	// object description follows
      if ((p = nextSpace(ptok)))
        *p = 0;
      if ((wclass = WClass::getWClass(ptok))) {
        *psection_type = wclass->type_id;
        if (! isValidType(*psection_type))
          error("parseSectionType: *psection_type=%d ptok=%s", *psection_type, ptok);
        if (p)
          *p = ' ';
        free(line);
        return OBJECTSECTION;
      }
      free(line);
      return OBJECTSECTION;
    }
  }

  // not a '<'
  if ((!strncmp(ptok, "-->", 3))) {
    return COMMENTSECTION;
  }
  free(line);
  return OBJECTSECTION;
}

void hpdebug(WObject *p)
{
  if (p)
    printf("HPdebug: p:%p\tp->up_p=%p\tp->down_p=%p\tp->next=%p\n",p,p->up_p,p->down_p,p->next_p);
}

/** called by cfgHttpReader */
int parseCfgLines(char *buf, int buflen)
{
  static WObject *father = NULL;	//gplhp

  char *tmpline = NULL;
  int linelen;

  if (! line) {
    if ((tmpline = new char[buflen]) == NULL) {
      error("reading: can't new tmpline");
      return -1;
    }
    linelen = 0;
  }
  else {
    // copy previous line in tmpline
    linelen = strlen(line);
    if ((tmpline = new char[buflen + linelen]) == NULL) {
      error("reading: can't new tmpline");
      return -1;
    }
    strcpy(tmpline, line);
  }

  int ibuf; int i0 = 0;

  // copy buf at the end of tmpline
  for (ibuf=0; ibuf < buflen; ibuf++)
    tmpline[linelen + ibuf] = buf[ibuf];
  linelen += buflen;

  while (1) {
    int i1 = 0;

    for (ibuf = i0; ibuf < linelen; ibuf++) {
      if (tmpline[ibuf] == '\n') {
        if (tmpline[ibuf-1] == '\\') {	// line follows
          tmpline[ibuf-1] = tmpline[ibuf] = ' ';
        }
        else {
#if 0 //pdxml
          tmpline[ibuf] = ' ';
#endif //pdxml
	  i1 = ibuf;
	  ibuf = linelen + 1;	// HUGLY !!! to do linelen + 2
        }
      }
    }

    // HERE WE MUST DETECT END OF OBJECT TAG NOT a NEWLINE

    if (ibuf == linelen + 2) {	// end of line
#if 1
      delete[] line;
#endif
      // next line
      if ((line = new char[i1 - i0 + 1]) == NULL) {
        error("reading: can't new line");
        return -1;
      }

      for (int i=0; i < i1 - i0; i++)
        line[i] = tmpline[i0 + i];
      line[i1 - i0] = '\0';
#ifdef WIN32 /* pour pallier l'absence de mode texte */
      if (i1 - i0 > 0 && line[i1 - i0 - 1] == '\r')
        line[i1 - i0 - 1] = '\0';
#endif /* WIN32 */

      // discard empty lines
      char *pline = line;
      if (*pline == '\0') {
        delete[] line;
        line = NULL;
        i0 = i1 + 1;
        continue;
      }

      // discard comments begining with a '#'
      pline = skipSpace(pline);
      if (*pline == '#') {
        delete[] line;
        line = NULL;
        i0 = i1 + 1;
        continue;
      }

      //trace(DBG_FORCE, "line=%s", line);

      WObject *po = NULL;
      char *p;
      int ret = parseSectionType(line, &section_type);

      switch (ret) {

      case COMMENTSECTION:
	if ((p = strstr(line, "-->")))
          commented = false;
      case BEGINSECTION:
      case HEADSECTION:
        delete[] line;
        line = NULL;
        break;

      case GROUPSECTION:
        if (grouped) {
          po = new WObject();	// creates a group proxy
          po->next_p = NULL;	// which has no brother
          po->down_p = NULL;	// nor children
          po->up_p = father;	// sets its father
          printf("up_p=%p father=%p po=%p\n", po->up_p, father, po);
          if (father) {
            father->add_son(po);	// adds this to its father child list
            po->pos.group = true; // belongs to a group
          }
          father = po;		// p becomes the current father
        }
        else
          father = father->up_p;
        //hpdebug(po);
        break;

      case ENDSECTION:
        delete[] line;
        line = NULL;
        return 0;
        break;

      case OBJECTSECTION:
        //trace(DBG_FORCE, "%d type=%d line=%s", ret, section_type, line);
	if ((p = strstr(line, "-->"))) {
          commented = false;
          *p = 0;
          break;
        }
        p = line;
	// skip <type
        if (isspace(*p))
          p = skipSpace(p);
        if (*p == '<') {
          p = nextSpace(p);
        }
        if (section_type != UNKNOWN_TYPE) {
          if (! isValidType(section_type)) {
            error("parseCfgLines: FIXME type=%d line=%s", section_type, p);
            return -1;
          }
          po = WClass::creatorInstance(section_type, p);
          if (! po) {
            error("parseCfgLines: FIXME po NULL, type=%d line=%s", section_type, line);
            return -1;
          }
          po->pos.group = false;
          po->next_p = NULL;
          po->down_p = NULL;
          po->up_p = father;
          if (father) {
            father->add_son(po); // Adds p to its father child list
            po->pos.group = true; // belongs to a group
          }
          //hpdebug(po);
        }
        break;
      }
      delete[] line;
      line = NULL;
      i0 = i1 + 1;
    }
    else {
      break;	// new Section
    }
  }

  if ((line = new char[linelen - i0 + 1]) == NULL) {
    error("reading: can't new line");
    return -1;
  }

  for (int i=0; i < linelen - i0; i++)
    line[i] = tmpline[i + i0];
  line[linelen - i0] = '\0';

  delete[] tmpline;
  return 1;
}

/** configuration world reader */
void cfgHttpReader(void *avre, Http *http)
{
  if (! http) {
    error("Can't download file, check access to the remote http server");
    exit(1);
  }

  int len;
  char buf[BUFSIZ];

  while ((len = http->Read(buf, sizeof(buf))) > 0) {
    if (len == TCPEOF || len == BADREAD)
      break;	// end of data
    if (parseCfgLines(buf, len) <= 0) {	// error
      //trace(DBG_FORCE, "cfgHttpReader: len=%d", len);
      break;
    }
    if (section_type == END_TYPE)	// end of file
      break;
  }

  World::setState(WORLD_DOWNLOADED);
  trace(DBG_WO, "cfgHttpReader: dowloaded");
  return;
}

/** parse object */
char * WObject::parseObject(char *l)
{
  l = skipSpace(l);
  l = parseName(l);	// FIXME: to be done later

  char *p = strstr(l, "/>");	// />
  if (p)
    *p = 0;
  p = strstr(l, "</");	// </type>
  if (p)
    *p = 0;

  // tokenize line
  return strtok(l, SEP);
}

char * WObject::parseGroup(char *l)
{
  return l;
}

/** parse object name:
 * typically called by parseObject
 * fill name.given_name
 */
char * WObject::parseName(char *l)
{
  l = parseGroup(l);

  if (!strncmp(l, "name", 4)) {
    l = skipEqual(l);
    l = skipSpace(l);
    if (*l == '"')	// double quoted name
      l++;

    int i;
    for (i=0; *l != '"' && i < OBJNAME_LEN; name.given_name[i++] = *l++)
      ;
    if (i == OBJNAME_LEN)
      i--;
    name.given_name[i] = '\0';
    l++;
    l = skipSpace(l);
  }
  return l;
}

/** parse spacial position: x y z az ax */
char * WObject::parsePosition(char *ptok)
{
  if (!strncmp(ptok, "pos", 3)) {
    ptok = skipEqual(ptok);
    ptok = skipQuotes(ptok);

    if (isFloat(ptok)) {
      pos.x = (float) atof(ptok); ptok = skipComma(ptok);
    }
    else
      return nextToken();
    if (isFloat(ptok)) {
      pos.y = (float) atof(ptok); ptok = skipComma(ptok);
    }
    else
      return nextToken();
    if (isFloat(ptok)) {
      pos.z = (float) atof(ptok); ptok = skipComma(ptok);
    }
    else
      return nextToken();
    if (isFloat(ptok)) {
      pos.az = (float) atof(ptok); ptok = skipComma(ptok);
    }
    else
      return nextToken();
    if (isFloat(ptok)) {
      pos.ax = (float) atof(ptok);
    }
    ptok = nextToken();
  }
  return ptok;
}

/** parse colors: r g b a */
char * WObject::parseColor(char *ptok)
{
  if (!strncmp(ptok, "color", 5)) {
    ptok = skipEqual(ptok);
    ptok = skipQuotes(ptok);

    if (isFloat(ptok)) {
      pos.x  = (float) atof(ptok); ptok = skipComma(ptok);
    }
    if (isFloat(ptok)) {
      pos.y  = (float) atof(ptok); ptok = skipComma(ptok);
    }
    if (isFloat(ptok)) {
      pos.z  = (float) atof(ptok); ptok = skipComma(ptok);
    }
    if (isFloat(ptok)) {
      pos.az = (float) atof(ptok);
    }
    ptok = nextToken();
  }
  return ptok;
}

char * WObject::parseRotation(char *ptok)
{
  if (!strncmp(ptok, "rot", 3)) {
    ptok = skipEqual(ptok);
    ptok = skipQuotes(ptok);

    if (isFloat(ptok)) {
      pos.az = (float) atof(ptok); ptok = skipComma(ptok);
    }
    if (isFloat(ptok)) {
      pos.x  = (float) atof(ptok); ptok = skipComma(ptok);
    }
    if (isFloat(ptok)) {
      pos.y  = (float) atof(ptok); ptok = skipComma(ptok);
    }
    if (isFloat(ptok)) {
      pos.z  = (float) atof(ptok);
    }
    ptok = nextToken();
  }
  return ptok;
}

char * WObject::parseTranslation(char *ptok)
{
  if (!strncmp(ptok, "trans", 5)) {
    ptok = skipEqual(ptok);
    ptok = skipQuotes(ptok);

    if (isFloat(ptok)) {
      pos.x  = (float) atof(ptok); ptok = skipComma(ptok);
    }
    if (isFloat(ptok)) {
      pos.y  = (float) atof(ptok); ptok = skipComma(ptok);
    }
    if (isFloat(ptok)) {
      pos.z  = (float) atof(ptok);
    }
    ptok = nextToken();
  }
  return ptok;
}

char * WObject::parseUrl(char *ptok, char *url)
{
  return parseString(ptok, url, "url");
}

char * WObject::parseWorld(char *ptok, char *url)
{
  return parseString(ptok, url, "world");
}

char * WObject::parseChannel(char *ptok, char *channel)
{
  if (!strncmp(ptok, "channel", 7)) {
    ptok = skipEqual(ptok);
    ptok = skipQuotes(ptok);
  }

  if (isdigit((int) *ptok))
    strcpy(channel, ptok);
  return nextToken();
}

char * WObject::parseString(char *ptok, char *str)
{
  strcpy(str, ptok);
  return nextToken();
}

char * WObject::parseString(char *ptok, char *str, const char *keystr)
{
  if (!strncmp(ptok, keystr, strlen(keystr))) {
    ptok = skipEqual(ptok);
    ptok = skipQuotes(ptok);
    return parseString(ptok, str);
  }
  return ptok;
}

char * WObject::parseQuotedString(char *ptok, char *str)
{
  if (*ptok++ == '"') {
    char *p = ptok;
    char *s = str;
    while (*p != '"') {
      if (*p)
        *s++ = *p++;
      else {
        *s++ = ' ';
        p = ptok = nextToken();
      }
    }
    *s = '\0';
    return nextToken();
  }
  return ptok;
}

char * WObject::parseQuotedString(char *ptok, char *str, const char *keystr)
{
  if (ptok) {
    if (!strncmp(ptok, keystr, strlen(keystr))) {
      ptok = skipEqual(ptok);
      return parseQuotedString(ptok, str);
    }
  }
  return ptok;
}

char * WObject::parseInt(char *ptok, int *value)
{
  ptok = skipQuotes(ptok);
  if (isInt(ptok))
    *value = atoi(ptok);
  return nextToken();
}

char * WObject::parseInt(char *ptok, int *value, const char *keystr)
{
  if (ptok) {
    if (!strncmp(ptok, keystr, strlen(keystr))) {
      ptok = skipEqual(ptok);
      return parseInt(ptok, value);
    }
  }
  return ptok;
}

char * WObject::parseUInt8(char *ptok, uint8_t *value)
{
  ptok = skipQuotes(ptok);
  if (ptok && isdigit((int) *ptok))
    *value = atoi(ptok) & 0xff;
  return nextToken();
}

char * WObject::parseUInt8(char *ptok, uint8_t *value, const char *keystr)
{
  if (ptok) {
    if (!strncmp(ptok, keystr, strlen(keystr))) {
      ptok = skipEqual(ptok);
      return parseUInt8(ptok, value);
    }
  }
  return ptok;
}

char * WObject::parseUInt16(char *ptok, uint16_t *value)
{
  ptok = skipQuotes(ptok);
  if (ptok && isdigit((int) *ptok))
    *value = atoi(ptok) & 0xffff;
  return nextToken();
}

char * WObject::parseUInt16(char *ptok, uint16_t *value, const char *keystr)
{
  if (ptok) {
    if (!strncmp(ptok, keystr, strlen(keystr))) {
      ptok = skipEqual(ptok);
      return parseUInt16(ptok, value);
    }
  }
  return ptok;
}

char * WObject::parseFloat(char *ptok, float *value)
{
  ptok = skipQuotes(ptok);
  if (isFloat(ptok))
    *value = (float) atof(ptok);
  return nextToken();
}

char * WObject::parseFloat(char *ptok, float *value, const char *keystr)
{
  if (ptok) {
    if (!strncmp(ptok, keystr, strlen(keystr))) {
      ptok = skipEqual(ptok);
      return parseFloat(ptok, value);
    }
  }
  return ptok;
}

char * WObject::parseVector3f(char *ptok, float *vector)
{
  ptok = skipQuotes(ptok);
  for (int i=0; i<3; i++) {
    if (isFloat(ptok))
      vector[i] = (float) atof(ptok);
    ptok = skipComma(ptok);
  }
  return nextToken();
}

char * WObject::parseVector3f(char *ptok, float *vector, const char *keystr)
{
  if (ptok) {
    if (!strncmp(ptok, keystr, strlen(keystr))) {
      ptok = skipEqual(ptok);
      return parseVector3f(ptok, vector);
    }
  }
  return ptok;
}
