#include "global.h"
#include "wo.h"
#include "movie.h"

#include "texture.h"	// TextureCacheEntry::getCurrentNumber
#include "images.h"	// TEXTURE_INTERNAL_FORMAT
#include "app.h"	// startwget

#if HAVE_LIBMPEG
#include <mpeg.h>
#endif


const WClass Movie::wclass(MOVIE_TYPE, "Movie", Movie::creator);


WObject * Movie::creator(char *l)
{
  return new Movie(l);
}

Movie::Movie(char *l)
{
  fps = MOVIE_FPS;	// 10 fps by default

  l = parseObject(l);
  l = parsePosition(l);
  l = parseGeometry(l);
  l = parseUrl(l, name.url);
  while (l) {
    if (!strncmp(l, "rate", 4))
      l = parseUInt8(l, &fps, "rate");
  }

  enableBehavior(NO_ELEMENTARY_MOVE);
  enableBehavior(COLLIDE_ONCE);
  initializeObject(LIST_MOBILE);

  status = MOVIE_INACTIVE;
  cachempeg = NULL;
  fpmpeg = NULL;
  pixmpeg = NULL;
  pixscreen = NULL;
#if HAVE_LIBMPEG
  imgmpeg = NULL;
#endif

  texnum = TextureCacheEntry::getCurrentNumber();
  trace(DBG_WO, "movie: texnum=%d fps=%d", texnum, fps);
} 

void mpegAbort(int sig)
{
#if HAVE_LIBMPEG
  CloseMPEG();
#endif
}

void Movie::changePermanent(float lasting)
{
#if HAVE_LIBMPEG
  if (status == MOVIE_INACTIVE) return;
  if (status == MOVIE_PAUSE) return;

  if (first) {
    gettimeofday(&start, NULL);
    first = 0;
  }

  int frame_inter = frame;
  struct timeval t;

  gettimeofday(&t, NULL);
  frame = (int) floor(((((float) (t.tv_sec - start.tv_sec) * 1000.) +
	                ((float) (t.tv_usec - start.tv_usec) / 1000.)
                       ) / 1000.) * (float) fps);

 /*
  * get Mpeg frame or not
  */
  for (int f=0; f < (frame - frame_inter); f++) {
    trace(DBG_WO, "fps=%d rate=%d frame=%d frame_inter=%d f=%d", fps, rate, frame, frame_inter, f);
    if (f >= rate)
      break;
    if (GetMPEGFrame((char *)pixmpeg) == false) {
      /* The End */
      if (status == MOVIE_LOOP) {
        RewindMPEG(fpmpeg, imgmpeg);
        frame = 0;
        first = 1;
        return;
      }
      CloseMPEG();
      status = MOVIE_INACTIVE;
      return;
    }
  }

  ColormapEntry *colorMap = imgmpeg->Colormap;
  int woff = (screensize - mpegwidth) / 2;
  int hoff = (screensize - mpegheight) / 2;
  
#define R 0
#define G 1
#define B 2
#define A 3
  if (colorMap) {	// case of Colormap Index
    for (int j=0; j < mpegheight; j++) {
      for (int i=0; i < mpegwidth; i++) {
        int cmapindex = pixmpeg[j * mpegwidth + i];
        ColormapEntry *color = &colorMap[cmapindex];
        int u = IMAGE_RGB * ((j+hoff) * screensize + (i+hoff));
        pixscreen[u+R] = color->red % 255;	
        pixscreen[u+G] = color->green % 255;	
        pixscreen[u+B] = color->blue % 255;
      }
    }
  } else {
    for (int j=0; j < mpegheight; j++) {
      for (int i=0; i < mpegwidth; i++) {
        int u = IMAGE_RGB * (screensize * (j+hoff) + (i+woff));
        int v = IMAGE_RGBA * (mpegwidth * j + i);
#if WORDS_BIGENDIAN
        pixscreen[u+R] = pixmpeg[v+A];
        pixscreen[u+G] = pixmpeg[v+B];
        pixscreen[u+B] = pixmpeg[v+G];
#else
        pixscreen[u+R] = pixmpeg[v+R];
        pixscreen[u+G] = pixmpeg[v+G];
        pixscreen[u+B] = pixmpeg[v+B];
#endif // WORDS_BIGENDIAN
      }
    }
  }
#undef R
#undef G
#undef B
#undef A

  glPushMatrix();
  glEnable(GL_TEXTURE_2D);
  glBindTexture(GL_TEXTURE_2D, texnum);
  glTexImage2D(GL_TEXTURE_2D, 0, TEXTURE_INTERNAL_FORMAT,
               screensize, screensize, 0, GL_RGB, GL_UNSIGNED_BYTE, pixscreen);
  glPopMatrix();

#endif /* HAVE_LIBMPEG */
}

int Movie::mpegInit()
{
#if HAVE_LIBMPEG
  /*
   * download Mpeg file
   */
  if ((cachempeg = new char[MAXHOSTNAMELEN]) == NULL) {
    error("mpegInit: can't alloc cachempeg");
    return 0;
  }

  if (App::startwget(name.url, cachempeg) == 0) {
    error("mpegInit: can't wget %s", cachempeg);
    delete[] cachempeg;
    return 0;
  }

  if ((fpmpeg = fopen(cachempeg, "r")) == NULL) {
    error("fopen mpeg");
    delete[] cachempeg;
    return 0;
  }

  /*
   * init Mpeg
   */
  imgmpeg = new ImageDesc[1];

  SetMPEGOption(MPEG_DITHER, FULL_COLOR_DITHER); //ORDERED_DITHER);
  OpenMPEG(fpmpeg, imgmpeg);
  //pd unlink(cachempeg);

  /*
   * get Mpeg infos
   */
  mpegwidth = imgmpeg->Width;
  mpegheight = imgmpeg->Height;
  rate = imgmpeg->PictureRate;
  if ((pixmpeg = new uint8_t[imgmpeg->Size]) == NULL) {
    error("mpegInit: can't new pixmpeg");
    return 0;
  }

  /*
   * compute screensize
   */
  screensize = MAX(mpegwidth, mpegheight);

  int i, power = 0;
  while ((screensize = screensize/2) > 0)
    power++;
  for (i=0, screensize = 1; i <= power; i++)
    screensize *= 2;

  trace(DBG_WO, "mpegInit: size=%d width=%d height=%d rate=%.2f screen=%d",
                 imgmpeg->Size, mpegwidth, mpegheight, rate, screensize);

  /*
   * alloc screen
   */
  if ((pixscreen = new GLubyte[IMAGE_RGB * screensize * screensize]) == NULL) {
    error("mpegInit: can't new pixscreen");
    delete[] pixmpeg;
    return 0;
  }

  frame = 0;
  first = 1;
  signal(SIGABRT, mpegAbort);
  return 1;
#else /* !HAVE_LIBMPEG */
  return 0;
#endif /* HAVE_LIBMPEG */
} 

void moviePlay(Movie *po, void *data, time_t sec, time_t usec)
{
  if (po->status == MOVIE_INACTIVE) {
    if (! po->mpegInit())
      return;
    po->status = MOVIE_PLAYING;

    po->enablePermanentMovement();	// for frames
  }
}

void movieStop(Movie *po, void *data, time_t sec, time_t usec)
{
  if (po->status != MOVIE_INACTIVE) {
    po->status = MOVIE_INACTIVE;
#if HAVE_LIBMPEG
    CloseMPEG();
#endif
    delete[] po->cachempeg;
    fclose(po->fpmpeg);
  }
  po->disablePermanentMovement();
#if HAVE_LIBMPEG
  if (po->imgmpeg)
    delete[] po->imgmpeg;
  po->imgmpeg = NULL;
#endif
  if (po->pixmpeg)
    delete[] po->pixmpeg;
  po->pixmpeg = NULL;
  if (po->pixscreen)
    delete[] po->pixscreen;
  po->pixscreen = NULL;
  //pd if (po->fpmpeg)
  //pd   App::unlinktmp(po->cachempeg);
  po->fpmpeg = NULL;
}

void moviePause(Movie *po, void *data, time_t sec, time_t usec)
{
  if (po->status == MOVIE_PLAYING || po->status == MOVIE_LOOP)
    po->status = MOVIE_PAUSE;
  else if (po->status == MOVIE_PAUSE)
    po->status = MOVIE_PLAYING;
}

void movieRewind(Movie *po, void *data, time_t sec, time_t usec)
{
#if HAVE_LIBMPEG
  if (po->status != MOVIE_PLAYING && po->status != MOVIE_LOOP && po->fpmpeg != NULL) {
    RewindMPEG(po->fpmpeg, po->imgmpeg);
    po->frame = 0;
    po->first = 1;
    po->status = MOVIE_PLAYING;
  }
#endif
}

void movieLoop(Movie *po, void *data, time_t sec, time_t usec)
{
  if (po->status != MOVIE_INACTIVE)
    po->status = MOVIE_LOOP;
}

void Movie::whenIntersect(WObject *pcur, WObject *pold)
{
  pold->copyPositionAndBB(pcur);
}

void Movie::quit()
{
  movieStop(this, NULL, 0L, 0L);
}

void movieInitFuncList(void)
{
  setActionFunc(MOVIE_TYPE, 0, WO_ACTION moviePlay, "Play");
  setActionFunc(MOVIE_TYPE, 1, WO_ACTION movieStop, "Stop");
  setActionFunc(MOVIE_TYPE, 2, WO_ACTION moviePause, "Pause");
  setActionFunc(MOVIE_TYPE, 3, WO_ACTION movieLoop, "Loop");
  setActionFunc(MOVIE_TYPE, 4, WO_ACTION movieRewind, "Rewind");
}
