/**
* program	: glutmech V1.1
* author	: Simon Parkinson-Bates.
* E-mail	: sapb@yallara.cs.rmit.edu.au
* Copyright Simon Parkinson-Bates.
* "source if freely avaliable to anyone to copy as long as they
*  acknowledge me in their work."
**/

#include "global.h"
#include "wo.h"
#include "mech.h"

#include "vgl.h"	// Draw

/* compilation conditions */
#define MECH_TORSO
#define MECH_HIP
#define MECH_SHOULDER
#define MECH_UPPER_ARM
#define MECH_LOWER_ARM
#define MECH_ROCKET_POD
#define MECH_UPPER_LEG
#define MECH_LOWER_LEG
#define	MECH_VULCANO

/* display list definitions */
#define SOLID_MECH_TORSO       	1
#define SOLID_MECH_HIP          2
#define SOLID_MECH_SHOULDER     3
#define SOLID_MECH_UPPER_ARM    4
#define SOLID_MECH_FOREARM	5
#define SOLID_MECH_UPPER_LEG   	6
#define SOLID_MECH_FOOT        	7
#define SOLID_MECH_ROCKET      	8
#define SOLID_MECH_VULCAN	9

/* motion rate definitions */
#define ANKLE_RATE	3
#define HEEL_RATE	3
#define ROTATE_RATE	10
#define TILT_RATE	10
#define ELBOW_RATE	2
#define SHOULDER_RATE	5
#define LAT_RATE	5
#define CANNON_RATE	40
#define UPPER_LEG_RATE	3
#define UPPER_LEG_RATE_GROIN 10
#define LIGHT_TURN_RATE	10
#define VIEW_TURN_RATE	10


const WClass Mech::wclass(MECH_TYPE, "Mech", Mech::creator);


/* motion variables */
static char leg = 0;
static int shoulder1 = 0, shoulder2 = 0, shoulder3 = 0, shoulder4 = 0;
static int lat1 = 20, lat2 = 20;
static int elbow1 = 0, elbow2 = 0;
static int pivot = 0;
static int tilt = 10;
static int ankle1 = 0, ankle2 = 0;
static int heel1 = 0, heel2 = 0;
static int hip11 = 0, hip12 = 10, hip21 = 0, hip22 = 10;
static int fire = 0;
static int turn = 0, turn1 = 0;
static int lightturn = 0, lightturn1 = 0;
static float elevation = 0.0, distance = 0.0, frame = 3.0;

/* material definitions */
static GLfloat mat_specular[] = {0.628281, 0.555802, 0.366065, 1.0};
static GLfloat mat_ambient[] = {0.24725, 0.1995, 0.0745, 1.0};
static GLfloat mat_diffuse[] = {0.75164, 0.60648, 0.22648, 1.0};
static GLfloat mat_shininess[] = {128.0 * 0.4};
static GLfloat mat_specular2[] = {0.508273, 0.508273, 0.508373, 1.0};
static GLfloat mat_ambient2[] = {0.19225, 0.19225, 0.19225, 1.0};
static GLfloat mat_diffuse2[] = {0.50754, 0.50754, 0.50754, 1.0};
static GLfloat mat_shininess2[] = {128.0 * 0.6};

static GLfloat mat_specular3[] = {0.296648, 0.296648, 0.296648, 1.0};
static GLfloat mat_ambient3[] = {0.25, 0.20725, 0.20725, 1.0};
static GLfloat mat_diffuse3[] = {1, 0.829, 0.829, 1.0};
static GLfloat mat_shininess3[] = {128.0 * 0.088};
static GLfloat mat_specular4[] = {0.633, 0.727811, 0.633, 1.0};
static GLfloat mat_ambient4[] = {0.0215, 0.1745, 0.0215, 1.0};
static GLfloat mat_diffuse4[] = {0.07568, 0.61424, 0.07568, 1.0};
static GLfloat mat_shininess4[] = {128 * 0.6};

/* body motion functions */
void Heel1Add(void) { heel1 = (heel1 + HEEL_RATE) % 360; }
void Heel1Subtract(void) { heel1 = (heel1 - HEEL_RATE) % 360; }
void Heel2Add(void) { heel2 = (heel2 + HEEL_RATE) % 360; }
void Heel2Subtract(void) { heel2 = (heel2 - HEEL_RATE) % 360; }
void Ankle1Add(void) { ankle1 = (ankle1 + ANKLE_RATE) % 360; }
void Ankle1Subtract(void) { ankle1 = (ankle1 - ANKLE_RATE) % 360; }
void Ankle2Add(void) { ankle2 = (ankle2 + ANKLE_RATE) % 360; }
void Ankle2Subtract(void) { ankle2 = (ankle2 - ANKLE_RATE) % 360; }
void RotateAdd(void) { pivot = (pivot + ROTATE_RATE) % 360; }
void RotateSubtract(void) { pivot = (pivot - ROTATE_RATE) % 360; }
void MechTiltSubtract(void) { tilt = (tilt - TILT_RATE) % 360; }
void MechTiltAdd(void) { tilt = (tilt + TILT_RATE) % 360; }
void elbow1Add(void) { elbow1 = (elbow1 + ELBOW_RATE) % 360; }
void elbow1Subtract(void) { elbow1 = (elbow1 - ELBOW_RATE) % 360; }
void elbow2Add(void) { elbow2 = (elbow2 + ELBOW_RATE) % 360; }
void elbow2Subtract(void) { elbow2 = (elbow2 - ELBOW_RATE) % 360; }
void shoulder1Add(void) { shoulder1 = (shoulder1 + SHOULDER_RATE) % 360; }
void shoulder1Subtract(void) { shoulder1 = (shoulder1 - SHOULDER_RATE) % 360; }
void shoulder2Add(void) { shoulder2 = (shoulder2 + SHOULDER_RATE) % 360; }
void shoulder2Subtract(void) { shoulder2 = (shoulder2 - SHOULDER_RATE) % 360; }
void shoulder3Add(void) { shoulder3 = (shoulder3 + SHOULDER_RATE) % 360; }
void shoulder3Subtract(void) { shoulder3 = (shoulder3 - SHOULDER_RATE) % 360; }
void shoulder4Add(void) { shoulder4 = (shoulder4 + SHOULDER_RATE) % 360; }
void shoulder4Subtract(void) { shoulder4 = (shoulder4 - SHOULDER_RATE) % 360; }
void lat1Raise(void) { lat1 = (lat1 + LAT_RATE) % 360; }
void lat1Lower(void) { lat1 = (lat1 - LAT_RATE) % 360; }
void lat2Raise(void) { lat2 = (lat2 + LAT_RATE) % 360; }
void lat2Lower(void) { lat2 = (lat2 - LAT_RATE) % 360; }
void FireCannon(void) { fire = (fire + CANNON_RATE) % 360; }
void RaiseLeg1Forward(void) { hip11 = (hip11 + UPPER_LEG_RATE) % 360; }
void LowerLeg1Backwards(void) { hip11 = (hip11 - UPPER_LEG_RATE) % 360; }
void RaiseLeg1Outwards(void) { hip12 = (hip12 + UPPER_LEG_RATE_GROIN) % 360; }
void LowerLeg1Inwards(void) { hip12 = (hip12 - UPPER_LEG_RATE_GROIN) % 360; }
void RaiseLeg2Forward(void) { hip21 = (hip21 + UPPER_LEG_RATE) % 360; }
void LowerLeg2Backwards(void) { hip21 = (hip21 - UPPER_LEG_RATE) % 360; }
void RaiseLeg2Outwards(void) { hip22 = (hip22 + UPPER_LEG_RATE_GROIN) % 360; }
void LowerLeg2Inwards(void) { hip22 = (hip22 - UPPER_LEG_RATE_GROIN) % 360; }

/* light source position functions */
void TurnRight(void) { turn = (turn - VIEW_TURN_RATE) % 360; }
void TurnLeft(void) { turn = (turn + VIEW_TURN_RATE) % 360; }
void TurnForwards(void) { turn1 = (turn1 - VIEW_TURN_RATE) % 360; }
void TurnBackwards(void) { turn1 = (turn1 + VIEW_TURN_RATE) % 360; }
void LightTurnRight(void) { lightturn = (lightturn + LIGHT_TURN_RATE) % 360; }
void LightTurnLeft(void) { lightturn = (lightturn - LIGHT_TURN_RATE) % 360; }
void LightForwards(void) { lightturn1 = (lightturn1 + LIGHT_TURN_RATE) % 360; }
void LightBackwards(void) { lightturn1 = (lightturn1 - LIGHT_TURN_RATE) % 360; }


/* geometric shape functions */
void drawBox(float width, float height, float depth)
{
  float x = width/2.0, y = height/2.0, z = depth/2.0;
  int j = 0;

  for (int i=0; i<4; i++) {
    glRotatef(90, 0, 0, 1);
    if (j) {
      glBegin(GL_QUADS);
       glNormal3f(-1.0, 0.0, 0.0);
       glVertex3f(-x, y, z);
       glVertex3f(-x, -y, z);
       glVertex3f(-x, -y, -z);
       glVertex3f(-x, y, -z);
      glEnd();
      glBegin(GL_TRIANGLES);
       glNormal3f(0.0, 0.0, 1.0);
       glVertex3f(0.0, 0.0, z);
       glVertex3f(-x, y, z);
       glVertex3f(-x, -y, z);
       glNormal3f(0.0, 0.0, -1.0);
       glVertex3f(0.0, 0.0, -z);
       glVertex3f(-x, -y, -z);
       glVertex3f(-x, y, -z);
      glEnd();
      j = 0;
    } else {
      glBegin(GL_QUADS);
       glNormal3f(-1.0, 0.0, 0.0);
       glVertex3f(-y, x, z);
       glVertex3f(-y, -x, z);
       glVertex3f(-y, -x, -z);
       glVertex3f(-y, x, -z);
      glEnd();
      glBegin(GL_TRIANGLES);
       glNormal3f(0.0, 0.0, 1.0);
       glVertex3f(0.0, 0.0, z);
       glVertex3f(-y, x, z);
       glVertex3f(-y, -x, z);
       glNormal3f(0.0, 0.0, -1.0);
       glVertex3f(0.0, 0.0, -z);
       glVertex3f(-y, -x, -z);
       glVertex3f(-y, x, -z);
      glEnd();
      j = 1;
    }
  }
}

void drawOctagon(float side, float height)
{
  float x = sin(0.785398163) * side, y = side / 2.0, z = height / 2.0, c;

  c = x + y;
  for (int i=0; i<8; i++) {
    glTranslatef(-c, 0.0, 0.0);
    glBegin(GL_QUADS);
     glNormal3f(-1.0, 0.0, 0.0);
     glVertex3f(0.0, -y, z);
     glVertex3f(0.0, y, z);
     glVertex3f(0.0, y, -z);
     glVertex3f(0.0, -y, -z);
    glEnd();
    glTranslatef(c, 0.0, 0.0);
    glBegin(GL_TRIANGLES);
     glNormal3f(0.0, 0.0, 1.0);
     glVertex3f(0.0, 0.0, z);
     glVertex3f(-c, -y, z);
     glVertex3f(-c, y, z);
     glNormal3f(0.0, 0.0, -1.0);
     glVertex3f(0.0, 0.0, -z);
     glVertex3f(-c, y, -z);
     glVertex3f(-c, -y, -z);
    glEnd();
    glRotatef(45.0, 0.0, 0.0, 1.0);
  }
}

void SetMaterial(GLfloat spec[], GLfloat amb[], GLfloat diff[], GLfloat shin[])
{
  glMaterialfv(GL_FRONT, GL_SPECULAR, spec);
  glMaterialfv(GL_FRONT, GL_SHININESS, shin);
  glMaterialfv(GL_FRONT, GL_AMBIENT, amb);
  glMaterialfv(GL_FRONT, GL_DIFFUSE, diff);
}

void drawTorso(void)
{
  glNewList(SOLID_MECH_TORSO, GL_COMPILE);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   drawBox(1.0, 1.0, 3.0);
   glTranslatef(0.75, 0.0, 0.0);
   SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
   drawBox(0.5, 0.6, 2.0);
   glTranslatef(-1.5, 0.0, 0.0);
   drawBox(0.5, 0.6, 2.0);
   glTranslatef(0.75, 0.0, 0.0);
  glEndList();
}

void drawHip(void)
{
  glNewList(SOLID_MECH_HIP, GL_COMPILE);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   drawOctagon(0.7, 0.5);
   for (int i=0; i<2; i++) {
     if (i)
       glScalef(-1, 1, 1);
     glTranslatef(1.0, 0.0, 0.0);
     SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
     Draw::sphere(0.2, 16, 16, 0);
     glTranslatef(-1.0, 0.0, 0.0);
   }
   glScalef(-1, 1, 1);
  glEndList();
}

void drawShoulder(void)
{
  glNewList(SOLID_MECH_SHOULDER, GL_COMPILE);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   drawBox(1.0, 0.5, 0.5);
   glTranslatef(0.9, 0.0, 0.0);
   SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
   Draw::sphere(0.6, 16, 16, 0);
   glTranslatef(-0.9, 0.0, 0.0);
  glEndList();
}

void drawUpperArm(void)
{
  glNewList(SOLID_MECH_UPPER_ARM, GL_COMPILE);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   drawBox(1.0, 2.0, 1.0);
   glTranslatef(0.0, -0.95, 0.0);
   glRotatef(90, 1, 0, 0);
   SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
   Draw::cylinder(0.4, 0.4, 1.5, 16, 10, 0);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   glRotatef(-90, 1, 0, 0);
   glTranslatef(-0.4, -1.85, 0.0);
   glRotatef(90, 0, 1, 0);
   for (int i=0; i<2; i++) {
     if (i)
       Draw::cylinder(0.5, 0.5, 0.8, 16, 10, 0);
     else
       Draw::cylinder(0.2, 0.2, 0.8, 16, 10, 0);
   }
   for (int i=0; i<2; i++) {
     if (i)
       glScalef(-1, 1, 1);
     if (i)
       glTranslatef(0.0, 0.0, 0.8);
     Draw::disk(0.2, 0.5, 16, 10, 0);
     if (i)
       glTranslatef(0.0, 0.0, -0.8);
   }
   glScalef(-1, 1, 1);
   glRotatef(-90, 0, 1, 0);
   glTranslatef(0.4, 2.9, 0.0);
  glEndList();
}

void drawVulcanGun(void)
{
#ifdef MECH_VULCANO
  glNewList(SOLID_MECH_VULCAN, GL_COMPILE);
  SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
  Draw::cylinder(0.5, 0.5, 0.5, 16, 10, 0);
  glTranslatef(0.0, 0.0, 0.5);
  Draw::disk(0.0, 0.5, 16, 10, 0);

  for (int i=0; i<5; i++) {
    glRotatef(72.0, 0.0, 0.0, 1.0);
    glTranslatef(0.0, 0.3, 0.0);
    Draw::cylinder(0.15, 0.15, 2.0, 16, 10, 0);
    Draw::cylinder(0.06, 0.06, 2.0, 16, 10, 0);
    glTranslatef(0.0, 0.0, 2.0);
    Draw::disk(0.1, 0.15, 16, 10, 0);
    Draw::cylinder(0.1, 0.1, 0.1, 16, 5, 0);
    glTranslatef(0.0, 0.0, 0.1);
    Draw::disk(0.06, 0.1, 16, 5, 0);
    glTranslatef(0.0, -0.3, -2.1);
  }
  glEndList();
#endif
}

void drawForeArm(void)
{
  glNewList(SOLID_MECH_FOREARM, GL_COMPILE);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   for (int i=0; i<5; i++) {
     glTranslatef(0.0, -0.1, -0.15);
     drawBox(0.6, 0.8, 0.2);
     glTranslatef(0.0, 0.1, -0.15);
     drawBox(0.4, 0.6, 0.1);
   }
   glTranslatef(0.0, 0.0, 2.45);
   drawBox(1.0, 1.0, 2.0);
   glTranslatef(0.0, 0.0, -1.0);
  glEndList();
}

void drawUpperLeg(void)
{
  glNewList(SOLID_MECH_UPPER_LEG, GL_COMPILE);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   glTranslatef(0.0, -1.0, 0.0);
   drawBox(0.4, 1.0, 0.7);
   glTranslatef(0.0, -0.65, 0.0);
   for (int i=0; i<5; i++) {
     drawBox(1.2, 0.3, 1.2);
     glTranslatef(0.0, -0.2, 0.0);
     drawBox(1.0, 0.1, 1.0);
     glTranslatef(0.0, -0.2, 0.0);
   }
   glTranslatef(0.0, -0.15, -0.4);
   drawBox(2.0, 0.5, 2.0);
   glTranslatef(0.0, -0.3, -0.2);
   glRotatef(90, 1, 0, 0);
   SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
   Draw::cylinder(0.6, 0.6, 3.0, 16, 10, 0);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   glRotatef(-90, 1, 0, 0);
   glTranslatef(0.0, -1.5, 1.0);
   drawBox(1.5, 3.0, 0.5);
   glTranslatef(0.0, -1.75, -0.8);
   drawBox(2.0, 0.5, 2.0);
   glTranslatef(0.0, -0.9, -0.85);
   SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
   Draw::cylinder(0.8, 0.8, 1.8, 16, 10, 0);
   for (int i=0; i<2; i++) {
     if (i)
       glScalef(-1, 1, 1);
     if (i)
       glTranslatef(0.0, 0.0, 1.8);
     Draw::disk(0.0, 0.8, 16, 10, 0);
     if (i)
       glTranslatef(0.0, 0.0, -1.8);
   }
   glScalef(-1, 1, 1);
  glEndList();
}

void drawFoot(void)
{
  glNewList(SOLID_MECH_FOOT, GL_COMPILE);
   SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
   glRotatef(90, 1, 0, 0);
   drawOctagon(1.5, 0.6);
   glRotatef(-90, 1, 0, 0);
  glEndList();
}

void drawLowerLeg(void)
{
  SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
  for (float k=0.0; k<2.0; k++) {
    for (float l=0.0; l<2.0; l++) {
      glPushMatrix();
      glTranslatef(k, 0.0, l);
      SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
      drawBox(1.0, 0.5, 1.0);
      glTranslatef(0.0, -0.45, 0.0);
      SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
      Draw::sphere(0.2, 16, 16, 0); 
      if (leg)
        glRotatef((GLfloat) heel1, 1.0, 0.0, 0.0);
      else
        glRotatef((GLfloat) heel2, 1.0, 0.0, 0.0);
      /* glTranslatef(0.0, -0.2, 0.0); */
      glTranslatef(0.0, -1.7, 0.0);
      SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
      drawBox(0.25, 3.0, 0.25);
      glTranslatef(0.0, -1.7, 0.0);
      SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
      Draw::sphere(0.2, 16, 16, 0);
      if (leg)
        glRotatef((GLfloat) - heel1, 1.0, 0.0, 0.0);
      else
        glRotatef((GLfloat) - heel2, 1.0, 0.0, 0.0);
      glTranslatef(0.0, -0.45, 0.0);
      SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
      drawBox(1.0, 0.5, 1.0);
      if (!k && !l) {
        int j;

        glTranslatef(-0.4, -0.8, 0.5);
        if (leg)
          glRotatef((GLfloat) ankle1, 1.0, 0.0, 0.0);
        else
          glRotatef((GLfloat) ankle2, 1.0, 0.0, 0.0);
        glRotatef(90, 0, 1, 0);
        Draw::cylinder(0.8, 0.8, 1.8, 16, 10, 0);
        for (j=0; j<2; j++) {
          if (j) {
            glScalef(-1, 1, 1);
            glTranslatef(0.0, 0.0, 1.8);
          }
          Draw::disk(0.0, 0.8, 16, 10, 0);
          if (j)
            glTranslatef(0.0, 0.0, -1.8);
        }
        glScalef(-1, 1, 1);
        glRotatef(-90, 0, 1, 0);
        glTranslatef(0.95, -0.8, 0.0);
        glCallList(SOLID_MECH_FOOT);
      }
      glPopMatrix();
    }
  }
}

void drawRocketPod(void)
{
  glNewList(SOLID_MECH_ROCKET, GL_COMPILE);
   SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
   glScalef(0.4, 0.4, 0.4);
   /* Neck1 */
   glRotatef(45, 0, 0, 1);
   glTranslatef(1, 0, 0);
   drawBox(2.0, 0.5, 3.0);
   /* Neck2 */
   glTranslatef(1, 0, 0);
   glRotatef(45, 0, 0, 1);
   glTranslatef(0.5, 0.0, 0.0);
   drawBox(1.2, 0.5, 3.0);
   /* Head */
   glTranslatef(2.1, 0.0, 0.0);
   glRotatef(-90, 0, 0, 1);
   SetMaterial(mat_specular, mat_ambient, mat_diffuse, mat_shininess);
   drawBox(2.0, 3.0, 4.0);
   glTranslatef(-0.5, -1.0, 1.3);
   /* Eyes */
   for (int i=0; i<2; i++) {
     int k = 0;
     /* DIST for (j=0; j<3; j++) { */
     for (int j=2; j<3; j++) {
       glTranslatef(i, j, 0.6);
       SetMaterial(mat_specular3, mat_ambient3, mat_diffuse3, mat_shininess3);
       Draw::cylinder(0.4, 0.4, 0.3, 16, 10, 0);
       glTranslatef(0.0, 0.0, 0.3);
       SetMaterial(mat_specular4, mat_ambient4, mat_diffuse4, mat_shininess4);
       Draw::cylinder(0.4, 0.0, 0.5, 16, 10, 0);
       k++;
       glTranslatef(-i, -j, -0.9);
     }
   }
  glEndList();
}

void displayMech(void)
{
  glPushMatrix();
  glScalef(0.5, 0.5, 0.5);
  glPushMatrix();
  glTranslatef(0.0, -0.75, 0.0);
  glRotatef((GLfloat) tilt, 1.0, 0.0, 0.0);
  glRotatef(90, 1, 0, 0);
  glCallList(SOLID_MECH_HIP);
  glRotatef(-90, 1, 0, 0);

  glTranslatef(0.0, 0.75, 0.0);
  glPushMatrix();
  glRotatef((GLfloat) pivot, 0.0, 1.0, 0.0);
  glPushMatrix();
  glCallList(SOLID_MECH_TORSO);
  glPopMatrix();
  glPushMatrix();
  glTranslatef(0.5, 0.5, 0.0);
  glCallList(SOLID_MECH_ROCKET);
  glPopMatrix();
  for (int i=0; i<2; i++) {
    glPushMatrix();
    if (i)
      glScalef(-1, 1, 1);
    glTranslatef(1.5, 0.0, 0.0);
    glCallList(SOLID_MECH_SHOULDER);
    glTranslatef(0.9, 0.0, 0.0);
    if (i) {
      glRotatef((GLfloat) lat1, 0.0, 0.0, 1.0);
      glRotatef((GLfloat) shoulder1, 1.0, 0.0, 0.0);
      glRotatef((GLfloat) shoulder3, 0.0, 1.0, 0.0);
    } else {
      glRotatef((GLfloat) lat2, 0.0, 0.0, 1.0);
      glRotatef((GLfloat) shoulder2, 1.0, 0.0, 0.0);
      glRotatef((GLfloat) shoulder4, 0.0, 1.0, 0.0);
    }
    glTranslatef(0.0, -1.4, 0.0);
    glCallList(SOLID_MECH_UPPER_ARM);
    glTranslatef(0.0, -2.9, 0.0);
    if (i)
      glRotatef((GLfloat) elbow1, 1.0, 0.0, 0.0);
    else
      glRotatef((GLfloat) elbow2, 1.0, 0.0, 0.0);
    glTranslatef(0.0, -0.9, -0.2);
    glCallList(SOLID_MECH_FOREARM);
    glPushMatrix();
    glTranslatef(0.0, 0.0, 2.0);
    glRotatef((GLfloat) fire, 0.0, 0.0, 1.0);
#ifdef MECH_VULCANO
    glCallList(SOLID_MECH_VULCAN);
#endif
    glPopMatrix();
    glPopMatrix();
  }
  glPopMatrix();
  glPopMatrix();

  for (int j=0; j<2; j++) {
    glPushMatrix();
    if (j) {
      glScalef(-0.5, 0.5, 0.5);
      leg = 1;
    } else {
      glScalef(0.5, 0.5, 0.5);
      leg = 0;
    }
    glTranslatef(2.0, -1.5, 0.0);
    if (j) {
      glRotatef((GLfloat) hip11, 1.0, 0.0, 0.0);
      glRotatef((GLfloat) hip12, 0.0, 0.0, 1.0);
    } else {
      glRotatef((GLfloat) hip21, 1.0, 0.0, 0.0);
      glRotatef((GLfloat) hip22, 0.0, 0.0, 1.0);
    }
    glTranslatef(0.0, 0.3, 0.0);
    glPushMatrix();
    glCallList(SOLID_MECH_UPPER_LEG);
    glPopMatrix();
    glTranslatef(0.0, -8.3, -0.4);
    if (j)
      glRotatef((GLfloat) - hip12, 0.0, 0.0, 1.0);
    else
      glRotatef((GLfloat) - hip22, 0.0, 0.0, 1.0);
    glTranslatef(-0.5, -0.85, -0.5);
    drawLowerLeg();
    glPopMatrix();
  }
  glPopMatrix();
}

void drawMech(void)
{
  SetMaterial(mat_specular2, mat_ambient2, mat_diffuse2, mat_shininess2);
  drawTorso();
  drawHip();
  drawShoulder();
  drawRocketPod();
  drawUpperArm();
  drawForeArm();
  drawUpperLeg();
  drawFoot();
  drawVulcanGun();
}

/* system of equations handling permanent motion */
void Mech::changePermanent(float lasting)
{
  float angle;
  static int step;

  if (!anim)
    return;

  if (step == 0 || step == 2) {
    /* for(frame=3.0; frame<=21.0; frame=frame+3.0){ */
    if (frame >= 0.0 && frame <= 21.0) {
      if (frame == 0.0)
        frame = 3.0;
      angle = (180 / M_PI) * (acos(((cos((M_PI / 180) * frame) * 2.043) + 1.1625) / 3.2059));
      if (frame > 0)
        elevation = -(3.2055 - (cos((M_PI / 180) * angle) * 3.2055));
      else
        elevation = 0.0;

      if (step == 0) {
        hip11 = (int) -(frame * 1.7);
        if (1.7 * frame > 15)
          heel1 = (int) (frame * 1.7);
        heel2 = 0;
        ankle1 = (int) (frame * 1.7);
        if (frame > 0)
          hip21 = (int) angle;
        else
          hip21 = 0;
        ankle2 = -hip21;
        shoulder1 = (int) (frame * 1.5);
        shoulder2 = (int) -(frame * 1.5);
        elbow1 = (int) frame;
        elbow2 = (int) -frame;
      } else {
        hip21 = (int) -(frame * 1.7);
        if (1.7 * frame > 15)
          heel2 = (int) (frame * 1.7);
        heel1 = 0;
        ankle2 = (int) (frame * 1.7);
        if (frame > 0)
          hip11 = (int) angle;
        else
          hip11 = 0;
        ankle1 = -hip11;
        shoulder1 = (int) -(frame * 1.5);
        shoulder2 = (int) (frame * 1.5);
        elbow1 = (int) -frame;
        elbow2 = (int) frame;
      }
      if (frame == 21)
        step++;
      if (frame < 21)
        frame = frame + 3.0;
    }
  }
  if (step == 1 || step == 3) {
    /* for(x=21.0; x>=0.0; x=x-3.0){ */
    if (frame <= 21.0 && frame >= 0.0) {
      angle = (180 / M_PI) * (acos(((cos((M_PI / 180) * frame) * 2.043) + 1.1625) / 3.2029));
      if (frame > 0)
        elevation = -(3.2055 - (cos((M_PI / 180) * angle) * 3.2055));
      else
        elevation = 0.0;
      if (step == 1) {
        elbow2 = hip11 = (int) -frame;
        elbow1 = heel1 = (int) frame;
        heel2 = 15;
        ankle1 = (int) frame;
        if (frame > 0)
          hip21 = (int) angle;
        else
          hip21 = 0;
        ankle2 = -hip21;
        shoulder1 = (int) (1.5 * frame);
        shoulder2 = (int) -(frame * 1.5);
      } else {
        elbow1 = hip21 = (int) -frame;
        elbow2 = heel2 = (int) frame;
        heel1 = 15;
        ankle2 = (int) frame;
        if (frame > 0)
          hip11 = (int) angle;
        else
          hip11 = 0;
        ankle1 = -hip11;
        shoulder1 = (int) -(frame * 1.5);
        shoulder2 = (int) (frame * 1.5);
      }
      if (frame == 0.0)
        step++;
      if (frame > 0)
        frame = frame - 3.0;
    }
  }
  if (step == 4)
    step = 0;
  distance += 0.1678;
}

/* creation from a file */
WObject * Mech::creator(char *l)
{
  return new Mech(l);
}

Mech::Mech(char *l)
{
  anim = walk = 0;

  l = parseObject(l);
  l = parsePosition(l);
  while (l) {
    if (!strncmp(l, "anim", 4))
      l = parseUInt8(l, &anim, "anim");
    else if (!strncmp(l, "walk", 4))
      l = parseUInt8(l, &walk, "walk");
  }

  char geom[80];
  sprintf(geom,"<solid=bbox,size=%.2f,%.2f,%.2f>", 0.5, 0.5, 1.0);
  parseGeometry(geom);

  enableBehavior(NO_ELEMENTARY_MOVE);
  enableBehavior(COLLIDE_NEVER);
  initializeObject(LIST_MOBILE);

  enablePermanentMovement();	// anim

  rotz = (float) RADIAN2DEGREE(pos.az);
  roty = (float) RADIAN2DEGREE(pos.ax);
  rotx = (float) 90.0;

  drawMech();
}

void Mech::render()
{
  static       float mech_radian  = 3 * M_PI/2;	// radian
  static const float mech_step   = 72;		// steps per tours
  static const float mech_radius = 7.0;		// space unit

  glPushMatrix();
  glTranslatef(pos.x, pos.y, pos.z);
  glScalef(0.5, 0.5, 0.5);
  glRotatef(rotx, 1, 0, 0);
  glRotatef(roty, 0, 1, 0);
  glRotatef(rotz, 0, 0, 1);

  if (walk) {
    mech_radian -= M_2PI / mech_step;
    if (mech_radian <= 0.0)
      mech_radian = M_2PI;
    float dx =  mech_radius * cos(mech_radian);
    float dz = -mech_radius * sin(mech_radian);
    glTranslatef(dx, 0, dz);
    glRotatef((float) RADIAN2DEGREE(mech_radian), 0, 1, 0);
  }
  displayMech();
  glPopMatrix();
}

void mechStill(Mech *po, void *d, time_t s, time_t u)
{ po->anim = 0; po->walk = 0; }

void mechAnim(Mech *po, void *d, time_t s, time_t u)
{ po->anim = 1; }

void mechWalk(Mech *po, void *d, time_t s, time_t u)
{ po->walk = 1; }

void mechInitFuncList(void)
{
  setActionFunc(MECH_TYPE, 0, WO_ACTION mechStill, "Still");
  setActionFunc(MECH_TYPE, 1, WO_ACTION mechAnim, "Anim");
  setActionFunc(MECH_TYPE, 2, WO_ACTION mechWalk, "Walk");
}
