#include "global.h"
#include "wo.h"
#include "world.h"	// worlds
#include "user.h"	// plocaluser
#include "lift.h"

#include "net.h"	// NetObject


const WClass Lift::wclass(LIFT_TYPE, "Lift", Lift::creator);
const uint8_t Lift::props = LIFT_PROPS;

static uint16_t oid = 0;


WObject * Lift::creator(char *l)
{
  return new Lift(l);
}

Lift::Lift(char *l)
{
  enableBehavior(PERSISTENT);
  speed = LIFT_DEF_SPEED;
  height = LIFT_DEF_HEIGHT;
  top = LIFT_DEF_TOP;
  bot = LIFT_DEF_BOT;

  l = parseObject(l);
  l = parsePosition(l);
  l = parseGeometry(l);
  while (l) {
    if (!strncmp(l, "height", 6))
      l = parseFloat(l, &height, "height");
    else if (!strncmp(l, "speed", 5))
      l = parseFloat(l, &speed, "speed");
    else if (!strncmp(l, "top", 3))
      l = parseInt(l, &top, "top");
    else if (!strncmp(l, "bot", 3))
      l = parseInt(l, &bot, "bot");
  }

  initializeObject(LIST_MOBILE);
  setMaxLasting(LIFT_LASTING);
  createPermanentNetObject(props, ++oid);

  dest = pos.z;		// initial position
}

/* Updates towards the network */
bool Lift::updateToNetwork(const Pos &oldpos)
{
  bool change = false;
  
  if (pos.z != oldpos.z) {
    noh->declareObjDelta(LIFT_PROPZ);
    trace(DBG_WO, "declareObjDelta: l=%.2f", pos.z);
    change = true;
  }
  return change;
}

bool Lift::isMoving()
{
  return (move.ttl > 0.0005);
}

void Lift::updateTime(time_t sec, time_t usec, float *lasting)
{
  *lasting = diffTime(sec, usec);
  if (*lasting < move.ttl) {
    move.ttl -= *lasting;
    move.sec = sec;
    move.usec = usec;
  }
  else {
    *lasting = move.ttl;
    stopImposedMovement();
  }
  trace(DBG_WO, "updateTime: l=%.2f lasting=%.2f ttl=%.2f", pos.z, *lasting, move.ttl);
}

void Lift::changePosition(float lasting)
{
  if (ABSF(dest - pos.z) <= 0.001)	// same level
    return;

  int sign = (dest > pos.z) ? 1 : -1;
  pos.z += sign * lasting * speed;
  if (pos.state == 1) {
    worlds->plocaluser->pos.z += sign * lasting * speed; // user follows the lift
    worlds->plocaluser->update3D();
    worlds->plocaluser->getBB();
    worlds->plocaluser->updateObjectIntoGrid(worlds->plocaluser->pos);
    worlds->plocaluser->updateCamera();
    trace(DBG_WO, "userChangePosition: feet=%.2f l=%.2f", worlds->plocaluser->pos.z-USER_DEFAULTHEIGHT/2, pos.z);
  }
  trace(DBG_WO, "changePosition: dest=%.3f l=%.2f", dest, pos.z);
}

void liftUp(Lift *po, void *data, time_t sec, time_t usec)
{
  po->dest = MIN(po->pos.z + po->height, po->height * po->top);
  po->initImposedMovement((po->dest - po->pos.z) / po->speed);
}

void liftDown(Lift *po, void *data, time_t sec, time_t usec)
{
  po->dest = MAX(po->pos.z - po->height, po->height * po->bot);;
  po->initImposedMovement((po->dest - po->pos.z) / po->speed);
}

void liftCall(Lift *po, void *data, time_t sec, time_t usec)
{
  if (! po->pos.state) {	// call only from outside
    po->dest = worlds->plocaluser->pos.z - USER_DEFAULTHEIGHT/2 - LIFT_STEP;
    trace(DBG_WO, "liftCall: feet=%.2f l=%.2f dest=%.2f", worlds->plocaluser->pos.z-USER_DEFAULTHEIGHT/2, po->pos.z, po->dest);
    po->initImposedMovement((po->dest - po->pos.z) / po->speed);
  }
}

/* somebody intersects */
void Lift::whenIntersect(WObject *pcur, WObject *pold)
{
  switch (pcur->type) {
  case USER_TYPE:
    if (! pos.state) {	// user comes in
      trace(DBG_WO, "whenIntersect: pos.state=1 l=%.2f", pos.z);
      pos.state = 1;
      if (pcur->pos.z >= pold->pos.z) {	// same level
        trace(DBG_WO, "Enter: feet=%.2f-%.2f l=%.2f", pcur->pos.z-USER_DEFAULTHEIGHT/2, pold->pos.z-USER_DEFAULTHEIGHT/2, pos.z);
        pcur->pos.z += LIFT_STEP;	// going on top
        pcur->update3D();
        pcur->getBB();
        pcur->updateObjectIntoGrid(pcur->pos);
      }
      else {	// inside from up
        pold->copyPositionAndBB(pcur);
      }
    }
    else {	// user already inside
      trace(DBG_WO, "Inside: feet=%.2f l=%.2f", pcur->pos.z-USER_DEFAULTHEIGHT/2, pos.z);
      pold->copyPositionAndBB(pcur);
    }
    break;
  default:
    trace(DBG_WO, "not a user: name=%s", pcur->name.instance_name);
    pold->copyPositionAndBB(pcur);
    break;
  }
}

bool Lift::whenIntersectOut(WObject *pcur, WObject *pold)
{
  //trace(DBG_WO, "Outside: user=%.2f-%.2f l=%.2f", pcur->pos.z-USER_DEFAULTHEIGHT/2, pold->pos.z-USER_DEFAULTHEIGHT/2, pos.z);
  if (pos.state == 1) {
    trace(DBG_WO, "whenIntersectOut: l=%.2f", pos.z);
    pcur->update3D();
    pcur->getBB();
    pcur->updateObjectIntoGrid(pcur->pos);
    pos.state = 0;
  }
  return true;
}

void Lift::quit()
{
  oid = 0;
}

void liftInitFuncList(void)
{
  getPropertyFunc[LIFT_PROPZ][LIFT_TYPE].pf = WO_PAYLOAD get_z;
  getPropertyFunc[LIFT_PROPHNAME][LIFT_TYPE].pf = WO_PAYLOAD get_hname;

  putPropertyFunc[LIFT_PROPZ][LIFT_TYPE].pf = WO_PAYLOAD put_z;
  putPropertyFunc[LIFT_PROPHNAME][LIFT_TYPE].pf = WO_PAYLOAD put_hname;

  setActionFunc(LIFT_TYPE, 0, WO_ACTION liftUp, "Up");
  setActionFunc(LIFT_TYPE, 1, WO_ACTION liftDown, "Down");
  setActionFunc(LIFT_TYPE, 2, WO_ACTION liftCall, "Call");
}
