#include "global.h"
#include "wo.h"
#include "http.h"	// httpOpen
#include "guy.h"

#include "vgl.h"	// Draw


/* for cylinder models */
#define UPPER_LEG_SIZE 0.4
#define LOWER_LEG_SIZE 0.4
#define LEG_GIRTH 0.064
#define UPPER_LEG_GIRTH 0.08
#define UPPER_LEG_TAPER 0.64
#define LOWER_LEG_TAPER 0.64
#define LOWER_LEG_GIRTH 0.056
#define UPPER_ARM_SIZE 0.36
#define LOWER_ARM_SIZE 0.36
#define ARM_GIRTH 0.04
#define UPPER_ARM_GIRTH 0.04
#define LOWER_ARM_GIRTH 0.032
#define UPPER_ARM_TAPER 0.64
#define LOWER_ARM_TAPER 0.64
#define HIP_JOINT_SIZE 0.08
#define KNEE_JOINT_SIZE 0.072
#define SHOULDER_JOINT_SIZE 0.04
#define ELBOW_JOINT_SIZE 0.036
#define HEAD_SIZE 0.16
#define FOOT_SIZE 0.12
#define TORSO_HEIGHT 0.64
#define TORSO_WIDTH 0.28
#define TORSO_TAPER 0.56

#define STACKS 16
#define SLICES 16
#define GUY_NUM_BODY_PARTS 7


const WClass Guy::wclass(GUY_TYPE, "Guy", Guy::creator);


void Guy::draw_head()
{ 
  glPushMatrix();
   glTranslatef(0, TORSO_HEIGHT+HEAD_SIZE, 0);
   glScalef(HEAD_SIZE, HEAD_SIZE, UPPER_LEG_GIRTH);
   Draw::sphere(1, SLICES, STACKS, 0);
  glPopMatrix();
}

void Guy::draw_torso()
{
  glPushMatrix();
   glScalef(TORSO_WIDTH, TORSO_HEIGHT, UPPER_LEG_GIRTH);
   glRotatef(-90, 1, 0, 0);
   Draw::cylinder(TORSO_TAPER, 1, 1, SLICES, STACKS, 0);
  glPopMatrix();
}

void Guy::draw_uleg()
{
  glPushMatrix();
   glTranslatef(0, -(HIP_JOINT_SIZE+UPPER_LEG_SIZE), 0);
   Draw::sphere(KNEE_JOINT_SIZE, SLICES, STACKS, 0);
  glPopMatrix();
   glTranslatef(0, -HIP_JOINT_SIZE, 0);
   Draw::sphere(HIP_JOINT_SIZE, SLICES, STACKS, 0);
  glPushMatrix();
   glScalef(UPPER_LEG_GIRTH, UPPER_LEG_SIZE, UPPER_LEG_GIRTH);
   glRotatef(90, 1, 0, 0);
   Draw::cylinder(1, UPPER_LEG_TAPER, 1, SLICES, STACKS, 0);
  glPopMatrix();
}
  
void Guy::draw_lleg()
{
  glPushMatrix();
   glScalef(LOWER_LEG_GIRTH, LOWER_LEG_SIZE, LOWER_LEG_GIRTH);
   glRotatef(90, 1, 0, 0);
   Draw::cylinder(1, LOWER_LEG_TAPER, 1, SLICES, STACKS, 0);
  glPopMatrix();
}
  
void Guy::draw_foot()
{
  glPushMatrix();
   glTranslatef(0, 0, -FOOT_SIZE/2.);
   glScalef(LOWER_LEG_GIRTH, LOWER_LEG_GIRTH, FOOT_SIZE);
   glRotatef(90, 1, 0, 0);
   Draw::cylinder(1, 1, 1, SLICES, STACKS, 0);
  glPopMatrix();
}
  
void Guy::draw_uarm()
{
  glPushMatrix();
   glTranslatef(0, -(SHOULDER_JOINT_SIZE+UPPER_ARM_SIZE), 0);
   Draw::sphere(ELBOW_JOINT_SIZE, SLICES, STACKS, 0);
   glPopMatrix();
   glTranslatef(0, -SHOULDER_JOINT_SIZE, 0);
   Draw::sphere(SHOULDER_JOINT_SIZE, SLICES, STACKS, 0);
  glPushMatrix();
   glScalef(UPPER_ARM_GIRTH, UPPER_ARM_SIZE, UPPER_ARM_GIRTH);
   glRotatef(90, 1, 0, 0);
   Draw::cylinder(1, UPPER_ARM_TAPER, 1, SLICES, STACKS, 0);
  glPopMatrix();
}
 
void Guy::draw_larm()
{
  glPushMatrix();
   glScalef(LOWER_ARM_GIRTH, LOWER_ARM_SIZE, LOWER_ARM_GIRTH);
   glRotatef(90, 1, 0, 0);
   Draw::cylinder(1, LOWER_ARM_TAPER, 1, SLICES, STACKS, 0);
  glPopMatrix();
}

void Guy::draw()
{
  glEnable(GL_NORMALIZE);

  body_lists = glGenLists(GUY_NUM_BODY_PARTS);
  glNewList(body_lists, GL_COMPILE);   draw_head();  glEndList();
  glNewList(body_lists+1, GL_COMPILE); draw_torso(); glEndList();
  glNewList(body_lists+2, GL_COMPILE); draw_uleg();  glEndList();
  glNewList(body_lists+3, GL_COMPILE); draw_lleg();  glEndList();
  glNewList(body_lists+4, GL_COMPILE); draw_foot();  glEndList();
  glNewList(body_lists+5, GL_COMPILE); draw_uarm();  glEndList();
  glNewList(body_lists+6, GL_COMPILE); draw_larm();  glEndList();
}

void Guy::display_head()
{
  float head_diffuse[] = {0.7, 0.4, 0.4, 1};

  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, head_diffuse);
  glPushMatrix();
  glCallList(body_lists);
  glPopMatrix();
}

void Guy::display_torso()
{
  float torso_diffuse[] = {0, 0.7, 0.7, 1};

  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, torso_diffuse);
  glPushMatrix();
  glCallList(body_lists+1);
  glPopMatrix();
}

void Guy::display_leg(int which)
{
  float leg_diffuse[] = {0.7, 0, 0.7, 1};

  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, leg_diffuse);
  glPushMatrix();
  if (which == 0)
    glTranslatef(TORSO_TAPER*TORSO_WIDTH/2., 0, 0);
  else
    glTranslatef(-TORSO_TAPER*TORSO_WIDTH/2., 0, 0);

  /* UPPER leg: rotates about the x axis only */
  glRotatef(walk_cycle[which][0][step], 1, 0, 0);
  glPushMatrix();
  glCallList(body_lists+2);
  glPopMatrix();

  /* LOWER leg: rotates about the x axis only */
  glTranslatef(0, -(UPPER_LEG_SIZE+KNEE_JOINT_SIZE), 0);
  glRotatef(walk_cycle[which][1][step], 1, 0, 0);
  glPushMatrix();
  glCallList(body_lists+3);
  glPopMatrix();

  /* Foot: rotates about the x axis only */
  glTranslatef(0, -(UPPER_LEG_SIZE+LOWER_LEG_SIZE+LOWER_LEG_GIRTH)/2., 0);
  glRotatef(walk_cycle[which][2][step], 1, 0, 0);
  glPushMatrix();
  glCallList(body_lists+4);
  glPopMatrix();
  glPopMatrix();
}
 
void Guy::display_arm(int which)
{
  float arm_diffuse[] = {0.7, 0.4, 0.4, 1};

  glMaterialfv(GL_FRONT, GL_AMBIENT_AND_DIFFUSE, arm_diffuse);

  glPushMatrix();
  glTranslatef(0, TORSO_HEIGHT, 0);
  if (which == 0)
    glTranslatef(TORSO_WIDTH, 0, 0);
  else
    glTranslatef(-TORSO_WIDTH, 0, 0);
  /* UPPER leg: rotates about the x axis only */
  glRotatef(walk_cycle[which][3][step], 1, 0, 0);
  glPushMatrix();
  glCallList(body_lists+5);
  glPopMatrix();

  /* LOWER leg: rotates about the x axis only */
  glTranslatef(0, -(UPPER_ARM_SIZE+ELBOW_JOINT_SIZE), 0);
  glRotatef(walk_cycle[which][4][step], 1, 0, 0);
  glPushMatrix();
  glCallList(body_lists+6);
  glPopMatrix();
  glPopMatrix();
}

/* Matrix times a vector, dest = m*v */
static
void MultMV(float m[3][4], float v[4], float dest[3])
{
  for (int i=0; i<3; i++) {
    dest[i] = 0;
    for (int j=0; j<4; j++)
      dest[i] += m[i][j] * v[j];
  }
}

/* Matrix multiplication, dest = m1*m2 */
static
void MultM(float m1[3][4], float m2[4][4], float dest[3][4])
{
  for (int i=0; i<3; i++)
    for (int j=0; j<4; j++) {
      dest[i][j] = 0;
      for (int k = 0; k < 4; k++)
	  dest[i][j] += (m1[i][k] * m2[k][j]);
    }
}

void Guy::computeCurve(int joint)
{
  float pointset[3][4];
  float prod[3][4], tm[4], vec[3];
  float tinc = (float) GUY_CYCLE_STEP/GUY_OVERSAMPLE;
  float BBasis[4][4] = {{-1, 3, -3, 1}, {3, -6, 3, 0},
                        {-3, 3,  0, 0}, {1,  0, 0, 0}};

  for (int i=0; i<4; i++)   /* z's are always zero, only 2-d */
    pointset[2][i] = 0; 

  int newindex, lastindex = -1;
  for (int p=0; p < curve[joint].numpoints; p += 3) {
    float t = 0.;
    for (int i=0; i<4; i++)
      pointset[0][i] = curve[joint].xcoords[p + i];
    for (int i=0; i<4; i++)
      pointset[1][i] = curve[joint].angles[p + i];

    MultM(pointset, BBasis, prod);

    while (t <= 1) {
      tm[0] = t*t*t;
      tm[1] = t*t;
      tm[2] = t;
      tm[3] = 1;
      MultMV(prod, tm, vec);
      newindex = (int) (vec[0]*(GUY_CYCLE_SIZE-1));
      if (newindex > lastindex)  {  /* go at least one */
        newindex %= GUY_CYCLE_SIZE;	// avoid out of bounds
	walk_cycle[0][joint][newindex] = vec[1];
	lastindex++;
      } 
      t += tinc;
    }
  }
  for (int i=0; i < GUY_CYCLE_SIZE; i++) {  /* copy to other leg, out-o-phase */
    walk_cycle[1][joint][i] =
      walk_cycle[0][joint][(i+(GUY_CYCLE_SIZE/2))%GUY_CYCLE_SIZE];
  }
}

void Guy::flatCSet()
{
  for (int j=0; j < numjoints; j++) {
    curve[j].numpoints = 4;
    curve[j].xcoords[0] = 0.0;
    curve[j].angles[0]  = 0.0;
    curve[j].xcoords[1] = 0.2;
    curve[j].angles[1]  = 0.0;
    curve[j].xcoords[2] = 0.8;
    curve[j].angles[2]  = 0.0;
    curve[j].xcoords[3] = 1.0;
    curve[j].angles[3]  = 0.0;
  }
}

void csetHttpReader(void *oa, Http *http)
{
  int numjoints = 0, numpoints = 0;
  Guy *po = (Guy *) oa;
  char *l, line[512];

  if (! http)
    goto abort;

  httpClearBuf();
  http->GetLine(line);
  if ((po->numjoints = numjoints = atoi(line)) >= GUY_MAX_JOINTS)
    goto abort;
  if ((po->curve = new tControlPts[numjoints]) == NULL) {
    error("csetHttpReader: can't new curve");
    return;
  }
  http->GetLine(line);	// mirror_flag
  for (int j=0; j < po->numjoints; j++) {
    http->GetLine(line);
    numpoints = atoi(line);
    if (numpoints < 4 || numpoints > GUY_MAX_CPOINTS)
      goto abort;
    po->curve[j].numpoints = numpoints;
    http->GetLine(line);
    l = strtok(line, " ");
    for (int p=0; p < numpoints; p++) {
      po->curve[j].xcoords[p] = (float) atof(l);
      l = strtok(NULL, " ");
    }
    http->GetLine(line);
    l = strtok(line, " ");
    for (int p=0; p < numpoints; p++) {
      po->curve[j].angles[p] = (float) atof(l);
      l = strtok(NULL, " ");
    }
  }
  return;

abort:
  error("Something went wrong while reading CSet file, numjoints = %d, numpoint = %d", numjoints, numpoints);
  if (numjoints == 0) {
    numjoints = po->numjoints = GUY_NUM_JOINTS;
    if ((po->curve = new tControlPts[numjoints]) == NULL) {
      error("csetHttpReader: can't new curve");
      return;
    }
  }
  po->flatCSet();
  return;
}

/* creaton from a file */
WObject * Guy::creator(char *l)
{
  return new Guy(l);
}

Guy::Guy(char *l)
{
  anim = walk = 0;

  l = parseObject(l);
  l = parsePosition(l);
  l = parseUrl(l, name.url);
  while (l) {
    if (!strncmp(l, "anim", 4))
      l = parseUInt8(l, &anim, "anim");
    else if (!strncmp(l, "walk", 4))
      l = parseUInt8(l, &walk, "walk");
  }

  char geom[80];
  sprintf(geom,"<solid=bbox,size=%.2f,%.2f,%.2f>",
          TORSO_WIDTH/2,
          UPPER_LEG_GIRTH,
          (HEAD_SIZE/2 + TORSO_HEIGHT + UPPER_LEG_TAPER + LOWER_LEG_TAPER)/2);
  parseGeometry(geom);

  enableBehavior(NO_ELEMENTARY_MOVE);
  enableBehavior(COLLIDE_NEVER);
  initializeObject(LIST_MOBILE);

  enablePermanentMovement();	// anim

  rotz = RADIAN2DEGREE(pos.az);
  rotx = RADIAN2DEGREE(pos.ax);
  roty = 90;
  step = GUY_CYCLE_SIZE/2;
  incstep = 1.;
  accel = false;

  draw();

  httpOpen(name.url, csetHttpReader, this, THREAD_NO_BLOCK);

  for (int j=0; j < numjoints; j++)
    computeCurve(j);
}

void Guy::changePermanent(float lasting)
{
  if (anim) {
    if (accel || checkRate(GUY_RATE))
      step = (int) fmod((double) step + incstep, GUY_CYCLE_SIZE);
  }
}

void Guy::render()
{
  float dx, dz;
  static       float guy_radian = 3 * M_PI/2;	// radian
  static const float guy_step   = 72;		// number of steps
  static const float guy_radius = 1.5;		// space unit

  glPushMatrix();

  glTranslatef(pos.x, pos.y, pos.z);
  glRotatef(rotz, 0, 0, 1);
  glRotatef(rotx, 0, 1, 0);
  glRotatef(roty, 1, 0, 0);

  if (checkRate(GUY_RATE))
    guy_radian -= M_2PI / guy_step;
  if (guy_radian <= 0.)
    guy_radian = M_2PI;
  if (walk == 1) {
    dx =  guy_radius * cos(guy_radian);
    dz = -guy_radius * sin(guy_radian);
    glTranslatef(-dx, 0, -dz);
    glRotatef(RADIAN2DEGREE(guy_radian), 0, 1, 0);
  }
  if (walk == 2)
    glRotatef(RADIAN2DEGREE(guy_radian), 0, 1, 0);
  if (walk == 3)
    glRotatef(RADIAN2DEGREE(guy_radian), 0, 0, 1);
  if (walk == 4)
    glRotatef(RADIAN2DEGREE(guy_radian), 1, 0, 0);

  display_head();
  display_torso();
  display_leg(0);
  display_leg(1);
  display_arm(0);
  display_arm(1);

  glPopMatrix();
}

void Guy::quit()
{
  glDisable(GL_NORMALIZE);
  if (curve)
    delete[] curve;
  curve = NULL;
}

void guyAnim(Guy *po, void *d, time_t s, time_t u)
{ po->anim = 1 - po->anim; }

void guyWalk(Guy *po, void *d, time_t s, time_t u)
{ po->walk = 1 - po->walk; }

void guyAccel(Guy *po, void *d, time_t s, time_t u)
{ po->accel = 1 - po->accel; }

void guyRotZMore(Guy *po, void *d, time_t s, time_t u)
{ po->rotz += 15; }

void guyRotZLess(Guy *po, void *d, time_t s, time_t u)
{ po->rotz -= 15; }

void guyRotXMore(Guy *po, void *d, time_t s, time_t u)
{ po->rotx += 15; }

void guyRotXLess(Guy *po, void *d, time_t s, time_t u)
{ po->rotx -= 15; }

void guyRotYMore(Guy *po, void *d, time_t s, time_t u)
{ po->roty += 15; }

void guyRotYLess(Guy *po, void *d, time_t s, time_t u)
{ po->roty -= 15; }

void guyInitFuncList(void)
{
  setActionFunc(GUY_TYPE, 0, WO_ACTION guyAnim, "Anim");
  setActionFunc(GUY_TYPE, 1, WO_ACTION guyWalk, "Walk");
  setActionFunc(GUY_TYPE, 2, WO_ACTION guyAccel, "Accel");
  setActionFunc(GUY_TYPE, 3, WO_ACTION guyRotZMore, "RotZ+");
  setActionFunc(GUY_TYPE, 4, WO_ACTION guyRotZLess, "RotZ-");
  setActionFunc(GUY_TYPE, 5, WO_ACTION guyRotXMore, "RotX+");
  setActionFunc(GUY_TYPE, 6, WO_ACTION guyRotXLess, "RotX-");
  setActionFunc(GUY_TYPE, 7, WO_ACTION guyRotYMore, "RotY+");
  setActionFunc(GUY_TYPE, 8, WO_ACTION guyRotYLess, "RotY-");
}
