#include "global.h"
#include "wo.h"
#include "world.h"
#include "door.h"
#include "user.h"	// USER-TYPE

#include "net.h"	// NetObject
#include "vgl.h"	// bbsize
#include "app.h"	// audio

#include "vrmysql.h"	// VRSql

const WClass Door::wclass(DOOR_TYPE, "Door", Door::creator);
const uint8_t Door::props = DOOR_PROPS;

static uint16_t oid = 0;


WObject * Door::creator(char *l)
{
  return new Door(l);
}

Door::Door(char *l)
{
  enableBehavior(PERSISTENT);

  l = parseObject(l);
  l = parsePosition(l);
  l = parseGeometry(l);
  while (l) {
    if (!strncmp(l, "open", 4))
      l = parseFloat(l, &aopen, "open");
    else if (!strncmp(l, "close", 5))
      l = parseFloat(l, &aclose, "close");
    else if (!strncmp(l, "speed", 5))
      l = parseFloat(l, &aspeed, "speed");
  }

  center.v[0] = pos.x;	// hinge
  center.v[1] = pos.y;
  center.v[2] = pos.z;
  size.v[0] = soh->bbsize.v[0];

  status = (aclose == pos.az) ? DOOR_CLOSED : DOOR_OPENED;

#if WITH_MYSQL && defined(HAVE_LIBMYSQLCLIENT)
  /* calls persistency MySql server to know the door state */
  setSqlStatus();	// updates state.state
  status = state.state;
  pos.az = (status == DOOR_CLOSED) ? aclose : aopen;
  trace(DBG_WO, "Door: sql status: %d %d %.2f", status, state.state, pos.az);
#endif

  pos.x = center.v[0] + size.v[0] * Cos(pos.az);
  pos.y = center.v[1] + size.v[0] * Sin(pos.az);
  pos.z = center.v[2];

  initializeObject(LIST_MOBILE);
  setMaxLasting(DOOR_LASTING);
  createPermanentNetObject(props, ++oid);
} 

void Door::updateTime(time_t sec, time_t usec, float *lasting)
{
  if (move.aspeed.v[0] == aspeed)
    move.ttl = MIN(ABSF(deltaAngle(pos.az,aopen)/move.aspeed.v[0]), move.ttl);
  else 
    move.ttl = MIN(ABSF(deltaAngle(pos.az,aclose)/move.aspeed.v[0]), move.ttl);

  *lasting = diffTime(sec, usec);
  if (*lasting < move.ttl) {
    move.ttl -= *lasting;
    move.sec = sec;
    move.usec = usec;
  }
  else {
    *lasting = move.ttl;
    if (status == DOOR_CLOSED)
      App::playSound(DOORCLOSESND);
    stopImposedMovement();
  }
}

void Door::changePosition(float lasting)
{
  pos.az += lasting * move.aspeed.v[0];
  pos.x = center.v[0] + size.v[0] * Cos(pos.az);
  pos.y = center.v[1] + size.v[0] * Sin(pos.az);
  pos.z = center.v[2];
}

bool Door::isMoving()
{
  return (move.ttl > 0.0005);
}

/* Updates towards the network */
bool Door::updateToNetwork(const Pos &oldpos)
{
  bool change = false;
  
  if ((pos.x != oldpos.x) || (pos.y != oldpos.y)) {
    noh->declareObjDelta(DOOR_PROPXY);
    change = true;
  }
  if (pos.az != oldpos.az) {
    noh->declareObjDelta(DOOR_PROPAZ);
    change = true;
  }
  return change;
}

void doorOpen(Door *po, void *d, time_t s, time_t u)
{
  if (po->status == DOOR_OPENED || po->status == DOOR_LOCKED)
    return;
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->initImposedMovement(ABSF(deltaAngle(po->pos.az, po->aopen) / po->move.aspeed.v[0]));
  App::playSound(DOOROPENSND);
  po->status = DOOR_OPENED;
#if WITH_MYSQL && defined(HAVE_LIBMYSQLCLIENT)
  VRSql::getVRSql()->updateStateObject(World::getCurrentName(), po->name.instance_name, po->status);
#endif

  /* here call persistency manager to inform change state */
  // noh->NetDeclarePersistSet();
}

void doorClose(Door *po, void *d, time_t s, time_t u)
{
  if (po->status == DOOR_CLOSED)
    return;
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = -po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->initImposedMovement(ABSF(deltaAngle(po->pos.az, po->aclose) / po->move.aspeed.v[0]));

  po->status = DOOR_CLOSED;
#if WITH_MYSQL && defined(HAVE_LIBMYSQLCLIENT)
  VRSql::getVRSql()->updateStateObject(World::getCurrentName(), po->name.instance_name, po->status);
#endif
}

void doorUnlock(Door *po, void *d, time_t s, time_t u)
{
  switch (po->status) {
  case DOOR_LOCKED:
    po->status = DOOR_UNLOCKED;
#if WITH_MYSQL && defined(HAVE_LIBMYSQLCLIENT)
  VRSql::getVRSql()->updateStateObject(World::getCurrentName(), po->name.instance_name, po->status);
#endif
  default:
    break;
  }
}

void doorLock(Door *po, void *d, time_t s, time_t u)
{
  switch (po->status) {
  case DOOR_UNLOCKED:
  case DOOR_CLOSED:
    po->status = DOOR_LOCKED;
#if WITH_MYSQL && defined(HAVE_LIBMYSQLCLIENT)
  VRSql::getVRSql()->updateStateObject(World::getCurrentName(), po->name.instance_name, po->status);
#endif
  default:
    break;
  }
}

/* Intersects with an object */
void Door::whenIntersect(WObject *pcur, WObject *pold)
{
  switch (pcur->type) {
  case USER_TYPE:
    if (projectMovementOnObject(pcur->pos, pold->pos, pos))
      pcur->updateObject(pold->pos);
    else
      pold->copyPositionAndBB(pcur);
    break;
  default:
    pold->copyPositionAndBB(pcur);
    break;
  }
}

void Door::quit()
{
  oid = 0;
}

void doorInitFuncList(void)
{
  getPropertyFunc[DOOR_PROPXY][DOOR_TYPE].pf = WO_PAYLOAD get_xy;
  getPropertyFunc[DOOR_PROPZ][DOOR_TYPE].pf = WO_PAYLOAD get_z;
  getPropertyFunc[DOOR_PROPAZ][DOOR_TYPE].pf = WO_PAYLOAD get_az;
  getPropertyFunc[DOOR_PROPHNAME][DOOR_TYPE].pf = WO_PAYLOAD get_hname;

  putPropertyFunc[DOOR_PROPXY][DOOR_TYPE].pf = WO_PAYLOAD put_xy;
  putPropertyFunc[DOOR_PROPZ][DOOR_TYPE].pf = WO_PAYLOAD put_z;
  putPropertyFunc[DOOR_PROPAZ][DOOR_TYPE].pf = WO_PAYLOAD put_az;
  putPropertyFunc[DOOR_PROPHNAME][DOOR_TYPE].pf = WO_PAYLOAD put_hname;

  setActionFunc(DOOR_TYPE, 0, WO_ACTION doorOpen, "Open");
  setActionFunc(DOOR_TYPE, 1, WO_ACTION doorClose, "Close");
  setActionFunc(DOOR_TYPE, 2, WO_ACTION doorUnlock, "Unlock");
  setActionFunc(DOOR_TYPE, 3, WO_ACTION doorLock, "Lock");
}
