#include "global.h"
#include "wo.h"
#include "world.h"	// worlds
#include "user.h"	// User
#include "caul.h"	// Cauldron
#include "move.h"	// GRAVITY
#include "ball.h"

#include "net.h"	// NetObject
#include "vgl.h"	// Solid
#include "gui.h"	// clearObjectBar


const WClass Ball::wclass(BALL_TYPE, "Ball", NULL, Ball::replicator);
const uint8_t Ball::props = BALL_PROPS;


void Ball::updateTime(time_t sec, time_t usec, float *lasting)
{
  *lasting = diffTime(sec, usec);
  if (*lasting < move.ttl) {
    move.ttl -= *lasting;
    move.sec = sec;
    move.usec = usec;
  }
  else {
    *lasting = move.ttl;
    stopImposedMovement();
    ttl = BALL_TTL;
  }
}

bool Ball::isMoving()
{
  if (taken)
    return true;
  return (move.ttl > 0.0005);
}

/* sollicited movements */
void Ball::changePosition(float lasting)
{
  if (! taken) {
    float ratio = MAX(getRate() / 20., 1);
    pos.x  += lasting * move.lspeed.v[0] * ratio;
    pos.y  += lasting * move.lspeed.v[1] * ratio;
    pos.z  += lasting * move.lspeed.v[2] * ratio;
    pos.az += lasting * move.aspeed.v[0] * ratio;
    pos.ax += lasting * move.aspeed.v[2] * ratio;

    // reduce speed
    move.lspeed.v[0] /= 1.02;
    move.lspeed.v[1] /= 1.02;
    move.lspeed.v[2] /= 1.02;
    move.aspeed.v[0] /= 1.02;
    move.aspeed.v[2] /= 1.02;
  }
  else {
    float off = 0.4; //(pos.x > worlds->plocaluser->pos.x) ? 0.4 : -0.4;
    pos.x = worlds->plocaluser->pos.x + off * Cos(worlds->plocaluser->pos.az);
    pos.y = worlds->plocaluser->pos.y + off * Sin(worlds->plocaluser->pos.az);
    pos.z = worlds->plocaluser->pos.z + 0.5;
  }
}

/** permanent movements */
void Ball::changePermanent(float lasting)
{
  pos.z  -= lasting * GRAVITY / 2;
  pos.az += lasting * aspeed;
#if 0
  pos.ax += lasting * aspeed;
#endif
}

/* creation: this method is invisible: called by the cauldron */
void ballCreate(Cauldron *pcauldron, void *data, time_t sec, time_t usec)
{
  new Ball(pcauldron, data, sec, usec);
}

void Ball::setDefaults()
{
  lspeed = BALL_LSPEED;
  zspeed = BALL_ZSPEED;
  aspeed = BALL_ASPEED; // * (float)drand48();
  ttl = BALL_TTL;
  taken = false;
}

Ball::Ball(WObject *pcauldron, void *d, time_t s, time_t u)
{
  /* position */
  srand((uint32_t) time(NULL));
  pos.x = pcauldron->pos.x + (float)drand48() * 2 -1;
  pos.y = pcauldron->pos.y + (float)drand48() * 2 -1;
  pos.z = pcauldron->pos.z + BALL_ORIGZ;

  /* geometry */
  char geom[128];
  sprintf(geom, "solid=\"sphere,radius=%f,diffuse=%s,texture=%s\"",
          BALL_RADIUS, BALL_COLOR, DEF_BALL_GIF);
  parseGeometry(geom);

  /* local creation */
  type = BALL_TYPE;
  initializeObject(LIST_MOBILE);
  setMaxLasting(BALL_LASTING);

  /* network creation */
  createVolatileNetObject(props);
  //pd worlds->plocaluser->noh->declareObjCreation();

  setDefaults();
  origz = pos.z;	// see wall

  enablePermanentMovement();
}

/* Replication from the network */
WObject * Ball::replicator(uint8_t type_id, NetObjectId noid, Payload *pp)
{
  return new Ball(type_id, noid, pp);
}

Ball::Ball(uint8_t type_id, NetObjectId _noid, Payload *pp)
{
  setType(type_id);

  replicateVolatileNetObject(props, _noid);
  noh->getAllProperties(pp);
  copyNetObjectId(_noid);

  char geom[128];
  sprintf(geom, "solid=\"sphere,radius=%f,diffuse=%s,texture=%s\"",
          BALL_RADIUS, BALL_COLOR, DEF_BALL_GIF);
  parseGeometry(geom);

  initializeObject(LIST_MOBILE);
  initBB(pos);

  setDefaults();
} 

/** Updates to the network */
bool Ball::updateToNetwork(const Pos &oldpos)
{
  bool change = false;
  
  if ((pos.x != oldpos.x) || (pos.y != oldpos.y)) {
    noh->declareObjDelta(BALL_PROPXY);
    change = true;
  }
  if (ABSF(pos.z - oldpos.z) > BALL_DELTAZ) {
    noh->declareObjDelta(BALL_PROPZ);
    change = true;
  }
  if (pos.az != oldpos.az) {
    noh->declareObjDelta(BALL_PROPAZ);
    change = true;
  }
  return change;
}

void Ball::whenIntersect(WObject *pcur, WObject *pold)
{
  switch (pcur->type) {
  case BALL_TYPE:
    pcur->pos.x += ((float) drand48() * 2 - 1) * BALL_RADIUS / 2;
    pcur->pos.y += ((float) drand48() * 2 - 1) * BALL_RADIUS / 2;
    break;
  case USER_TYPE:
    if (! taken) {
      /* user pushes the ball on the ground */
      move.lspeed.v[0] = (lspeed / 2) * Cos(worlds->plocaluser->pos.az);
      move.lspeed.v[1] = (lspeed / 2) * Sin(worlds->plocaluser->pos.az);
      ttl = BALL_TTL / 5;
      initImposedMovement(ttl);
    }
    pold->copyPositionAndBB(pcur);
    break;
  default:
    pcur->pos.z += BALL_RADIUS;
    break;
  }
  pcur->update3D();
  pcur->getBB();
}

void ballPush(Ball *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = po->lspeed * Cos(worlds->plocaluser->pos.az);
  po->move.lspeed.v[1] = po->lspeed * Sin(worlds->plocaluser->pos.az);
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->ttl = BALL_TTL;
  po->initImposedMovement(po->ttl);
  po->taken = false;
}

void ballPull(Ball *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = -po->lspeed * Cos(worlds->plocaluser->pos.az);
  po->move.lspeed.v[1] = -po->lspeed * Sin(worlds->plocaluser->pos.az);
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->ttl = BALL_TTL;
  po->initImposedMovement(po->ttl);
  po->taken = false;
}

void ballShoot(Ball *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = po->lspeed * Cos(worlds->plocaluser->pos.az);
  po->move.lspeed.v[1] = po->lspeed * Sin(worlds->plocaluser->pos.az);
  po->move.lspeed.v[2] = po->zspeed;
  po->move.aspeed.v[0] = po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->ttl = BALL_TTL;
  po->initImposedMovement(po->ttl);
  po->taken = false;
}

void ballUp(Ball *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = po->zspeed;
  po->move.aspeed.v[0] = po->aspeed;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = 0;
  po->ttl = BALL_TTL;
  po->initImposedMovement(po->ttl);
}

void ballTake(Ball *po, void *d, time_t s, time_t u)
{
  po->ttl = MAXFLOAT;
  po->initImposedMovement(po->ttl);
  po->disablePermanentMovement();
  po->taken = true;
}

void ballDrop(Ball *po, void *d, time_t s, time_t u)
{
  po->enablePermanentMovement();
  po->ttl = 0.;
  po->initImposedMovement(po->ttl);
  po->taken = false;
}

void ballTurn(Ball *po, void *d, time_t s, time_t u)
{
  po->move.lspeed.v[0] = 0;
  po->move.lspeed.v[1] = 0;
  po->move.lspeed.v[2] = 0;
  po->move.aspeed.v[0] = po->aspeed * 4;
  po->move.aspeed.v[1] = 0;
  po->move.aspeed.v[2] = po->aspeed * 4;
  po->ttl = BALL_TTL;
  po->initImposedMovement(po->ttl);
}

void ballDestroy(Ball *po, void *d, time_t s, time_t u)
{
  po->toDelete();	// delete Wobject
  po->taken = false;
  po->soh->clearObjectBar();
}

void ballInitFuncList(void)
{
  getPropertyFunc[BALL_PROPXY][BALL_TYPE].pf = WO_PAYLOAD get_xy;
  getPropertyFunc[BALL_PROPZ][BALL_TYPE].pf = WO_PAYLOAD get_z;
  getPropertyFunc[BALL_PROPAZ][BALL_TYPE].pf = WO_PAYLOAD get_az;
  getPropertyFunc[BALL_PROPHNAME][BALL_TYPE].pf = WO_PAYLOAD get_hname;

  putPropertyFunc[BALL_PROPXY][BALL_TYPE].pf = WO_PAYLOAD put_xy;
  putPropertyFunc[BALL_PROPZ][BALL_TYPE].pf = WO_PAYLOAD put_z;
  putPropertyFunc[BALL_PROPAZ][BALL_TYPE].pf = WO_PAYLOAD put_az;
  putPropertyFunc[BALL_PROPHNAME][BALL_TYPE].pf = WO_PAYLOAD put_hname;

  setActionFunc(BALL_TYPE, BALL_PUSH, WO_ACTION ballPush, "Push");
  setActionFunc(BALL_TYPE, BALL_PULL, WO_ACTION ballPull, "Pull");
  setActionFunc(BALL_TYPE, BALL_SHOOT, WO_ACTION ballShoot, "Shoot");
  setActionFunc(BALL_TYPE, BALL_UP, WO_ACTION ballUp, "Up");
  setActionFunc(BALL_TYPE, BALL_TAKE, WO_ACTION ballTake, "Take");
  setActionFunc(BALL_TYPE, BALL_DROP, WO_ACTION ballDrop, "Drop");
  setActionFunc(BALL_TYPE, BALL_TURN, WO_ACTION ballTurn, "Turn");
  setActionFunc(BALL_TYPE, BALL_KILL, WO_ACTION ballDestroy, "Destroy");
  setActionFunc(BALL_TYPE, BALL_CREAT, WO_ACTION ballCreate, "");
}
