#include "global.h"
#include "wo.h"
#include "world.h"
#include "http.h"	// httpOpen
#include "android.h"

#include "vnp.h"	// vnpModelHttpReader
#include "net.h"	// getGroup
#include "gui.h"	// options


const WClass Android::wclass(ANDROID_TYPE, "Android", Android::creator);


int Phalanx2::drawPhalanx2()
{
  return 1;
}

Phalanx::Phalanx(Phalanx2 *phalanx2)
{
  angle_bend_phalanx = 0;
  length = PHALANX_LEN;
  pphalanx2 = phalanx2;
}

int Phalanx::drawPhalanx()
{
  return pphalanx2->drawPhalanx2();
}

Finger::Finger(Phalanx *phalanx)
{
  angle_bend_finger = angle_tilt_finger = 0;
  length = FINGER_LEN;
  pphalanx = phalanx;
}

int Finger::drawFinger(Android *po)
{
  return pphalanx->drawPhalanx();
}

Hand::Hand(Finger *fingers[5])
{
  angle_bend_hand = angle_tilt_hand = angle_turn_hand = 0;
  length = HAND_LEN;
  //pfingers[0] = fingers[0];
  pfingers[1] = fingers[1];
  pfingers[2] = fingers[2];
  pfingers[3] = fingers[3];
  pfingers[4] = fingers[4];
}

int Hand::drawHand(Android *po)
{
  return pfingers[4]->drawFinger(po);
}

Forearm::Forearm(Hand *hand)
{
  angle_bend_elbow = angle_turn_elbow = 0;
  length = FOREARM_LEN;
  phand = hand;
}

int Forearm::drawForearm(Android *po)
{
  return phand->drawHand(po);
}

Arm::Arm(Forearm *forearm)
{
  angle_raise_arm = -90;
  angle_rote_arm = 0;
  angle_turn_arm = 90;
  length = ARM_LEN;
}

int Arm::drawArm(Android *po)
{
  return pforearm->drawForearm(po);
}

Shoulder::Shoulder(Arm* arm)
{
  distance_avance_shoulder = distance_raise_shoulder = 0;
  radius = SHOULDER_RAD;
  parm = arm;
}

int Shoulder::drawSoulder(Android *po)
{
  return parm->drawArm(po);
}

Head::Head()
{
  angle_bend_head = angle_tilt_head = angle_turn_head = 0;
  radius = HEAD_RAD;
}

int Head::drawHead()
{
  return 1;
}

Neck::Neck(Head *head)
{
  angle_bend_neck = angle_tilt_neck = angle_turn_neck = 0;
  length = NECK_LEN;
  phead = head;
}

int Neck::drawNeck()
{
  return phead->drawHead();
}

int Foot::drawFoot(Android *po)
{
  return 1;
}

Calf::Calf(Foot *foot)
{
  angle_bend_knee = 0;
  length = CALF_LEN;
  pfoot = foot;
}

int Calf::drawCalf(Android *po)
{
  return pfoot->drawFoot(po);
}

Leg::Leg(Calf *calf)
{
  angle_raise_leg = angle_rote_leg = angle_turn_leg = 0;
  length = LEG_LEN;
  pcalf = calf;
}

int Leg::drawLeg(Android *po)
{
  return pcalf->drawCalf(po);
}

Bust::Bust(Leg *legs[2], Shoulder *pshoulders[2], Neck *pneck,class Android *po)
{
}

int Bust::drawBust(Android *po)
{
  return po->shoulders[1]->drawSoulder(po);
}

/** create the body */
void Android::createBody()
{
  tx = ty = tz = rx = ry = rz = 0;

  // Create human body
  for (int k=0;k<10;k++)
    phalanx2[k] = new Phalanx2();
  for (int k=0;k<10;k++)
    phalanx[k] = new Phalanx(phalanx2[k]);
  for (int k=0;k<5;k++)
    fingers_r[k] = new Finger(phalanx[k]);
  for (int k=0;k<5;k++)
    fingers_l[k] = new Finger(phalanx[k+5]);

  hand_r =       new Hand(fingers_r);
  hand_l =       new Hand(fingers_l);
  forearm_r =    new Forearm(hand_r);
  forearm_l =    new Forearm(hand_l);
  arm_r =        new Arm(forearm_r);
  arm_l =        new Arm(forearm_l);
  shoulders[0] = new Shoulder(arm_r);
  shoulders[1] = new Shoulder(arm_l);
  head =         new Head();
  neck =         new Neck(head);
  foot_r =       new Foot();
  foot_l =       new Foot();
  calf_r =       new Calf(foot_r);
  calf_l =       new Calf(foot_l);
  legs[0] =      new Leg(calf_r);
  legs[1] =      new Leg(calf_l);
  bust =         new Bust(legs, shoulders, neck, this);

  buildModel();

  //set all values to 0
  for (int i=0; i <= NUM_BAPS_V32; i++) {
    bap[i] = 0;
    mask[i] = 0;
  }
  for (int i=0; i <= NUM_FAPS; i++)
    fap[i] = 0;
  status = ANDROID_INACTIVE;
}

///////////////
// MAN_MODEL
///////////////

/** load body's parts */
void Android::loadUrlModels(int numjoints, Http *http)
{
  char line[BUFSIZ];
  struct {
    char keystr[10];
    struct VnpModel *vnp;
    char url[URL_LEN];
  } jointpoints[] = {
      { "mod_head", &head1 },
      { "mod_neck", &neck1 },
      { "mod_bust", &body_up },
      { "mod_tumm", &body_bot },
      { "mod_la",   &la_up },
      { "mod_lm",   &la_mid },
      { "mod_lh",   &la_hand },
      { "mod_ra",   &ra_up },
      { "mod_rm",   &ra_mid },
      { "mod_rh",   &ra_hand },
      { "mod_ll",   &ll_up },
      { "mod_lm",   &ll_mid },
      { "mod_lf",   &ll_foot },
      { "mod_rl",   &rl_up },
      { "mod_rm",   &rl_mid },
      { "mod_rf",   &rl_foot },
      { "",         NULL }
      };

  for (int i=0; i < numjoints; ) {
    char *keystr, *url;

    if (! http->GetLine(line))
      break;
    trace(DBG_MAN, "loadUrlModels: %s", line);

    keystr = strtok(line, "=");
    url = strtok(NULL, " \t");
    if (strcmp(jointpoints[i].keystr, keystr) == 0) {
      trace(DBG_MAN, "loadUrlModels: i=%d key=%s url=%s", i, keystr, url);
      strcpy(jointpoints[i].url, url);
    }
    i++;
  }

  // get faces.index
  while (http->GetLine(line)) {
    char *keystr, *url;

    keystr = strtok(line, "=");
    url = strtok(NULL, " \t");
    if (strcmp("faces", keystr) == 0) {
      strcpy(faces_url, url);
      trace(DBG_MAN, "loadUrlModels: faces_url=%s", faces_url);
    }
  }

  // now we can download VnpModels
  for (int i=0; i < numjoints; i++) {
    trace(DBG_MAN, "loadUrlModels: i=%d url=%s", i, jointpoints[i].url);
    jointpoints[i].vnp->scale = MAN_SCALE;
    httpOpen(jointpoints[i].url, vnpModelHttpReader, jointpoints[i].vnp, THREAD_NO_BLOCK);
  }
}

/** load joint points */
static void jointsHttpReader(void *_po, Http *http)
{
  if (! http)
    return;

  int numjoints = 0;
  Android *po = (Android *) _po;
  char line[BUFSIZ];

  httpClearBuf();
  if (! http->GetLine(line))
    return;
  trace(DBG_MAN, "jointsHttpReader: %s", line);
  numjoints = atoi(line);
  trace(DBG_MAN, "jointsHttpReader: numjoints=%d", numjoints);

  for (int i=0; i < numjoints;) {
    if (! http->GetLine(line))
      break;
    trace(DBG_MAN, "jointsHttpReader: %s", line);

    char *l;
    int number;

    l = strtok(line, " \t");
    number = (int)atoi(l);
    l = strtok(NULL, " \t");
#ifdef VNP_SCALING
    po->jp.x[number] = (float)atof(l) * MAN_SCALE; l = strtok(NULL, " \t");
    po->jp.y[number] = (float)atof(l) * MAN_SCALE; l = strtok(NULL, " \t");
    po->jp.z[number] = (float)atof(l) * MAN_SCALE;
#else
    po->jp.x[number] = (float)atof(l); l = strtok(NULL, " \t");
    po->jp.y[number] = (float)atof(l); l = strtok(NULL, " \t");
    po->jp.z[number] = (float)atof(l);
#endif
    i++;
  }
  po->loadUrlModels(numjoints, http);
}

void Android::buildModel()
{
  httpOpen(name.url, jointsHttpReader, this, THREAD_NO_BLOCK);

  //pd glDisable(GL_NORMALIZE);
  glShadeModel(GL_SMOOTH);

  manlist = glGenLists(16);
  glNewList(manlist,GL_COMPILE);    drawVnpModel(&head1);    glEndList();
  glNewList(manlist+1,GL_COMPILE);  drawVnpModel(&body_up);  glEndList();
  glNewList(manlist+2,GL_COMPILE);  drawVnpModel(&body_bot); glEndList();
  glNewList(manlist+3,GL_COMPILE);  drawVnpModel(&ra_up);    glEndList();
  glNewList(manlist+4,GL_COMPILE);  drawVnpModel(&ra_mid);   glEndList();
  glNewList(manlist+5,GL_COMPILE);  drawVnpModel(&ra_hand);  glEndList();
  glNewList(manlist+6,GL_COMPILE);  drawVnpModel(&la_up);    glEndList();
  glNewList(manlist+7,GL_COMPILE);  drawVnpModel(&la_mid);   glEndList();
  glNewList(manlist+8,GL_COMPILE);  drawVnpModel(&la_hand);  glEndList();
  glNewList(manlist+9,GL_COMPILE);  drawVnpModel(&rl_up);    glEndList();
  glNewList(manlist+10,GL_COMPILE); drawVnpModel(&rl_mid);   glEndList();
  glNewList(manlist+11,GL_COMPILE); drawVnpModel(&rl_foot);  glEndList();
  glNewList(manlist+12,GL_COMPILE); drawVnpModel(&ll_up);    glEndList();
  glNewList(manlist+13,GL_COMPILE); drawVnpModel(&ll_mid);   glEndList();
  glNewList(manlist+14,GL_COMPILE); drawVnpModel(&ll_foot);  glEndList();
  glNewList(manlist+15,GL_COMPILE); drawVnpModel(&neck1);    glEndList();
}

void manLight(void)
{
  const GLfloat light_ambient[] =  {0.2, 0.2, 0.2, 1};
  const GLfloat light_diffuse[] =  {1.0, 1.0, 1.0, 1};
  const GLfloat light_specular[] = {1.0, 1.0, 1.0, 1};
  const GLfloat light_position[] = {50, 50, 20, 1};
  //PD const GLfloat light_position[] = {0, 0, 2, 1};

  glLightfv(GL_LIGHT1, GL_AMBIENT,  light_ambient);
  glLightfv(GL_LIGHT1, GL_DIFFUSE,  light_diffuse);
  glLightfv(GL_LIGHT1, GL_SPECULAR, light_specular);
  glLightfv(GL_LIGHT1, GL_POSITION, light_position);
  glEnable(GL_LIGHT1);
#if 0	//AJ
  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE); // bad: blank triangles
#endif
}

// skin color
static const GLfloat skinnude[] = {1.0, 0.75, 0.7, 1};

/** dispaly body and face */
void Android::manDisplay()
{
  manLight();

#if 0	//AJ
  const GLfloat specular[] = {0.1, 0.1, 0.1, 1};
  glMaterialfv(GL_FRONT, GL_SPECULAR,  specular);	// Bad: white with Mesa
#endif
#if 1	//AJ
  const GLfloat shininess[] = {100};
  glMaterialfv(GL_FRONT, GL_SHININESS, shininess);
#endif
#if 1	//AJ
  glDepthFunc(GL_LESS);	//pd GL_LEQUAL
#endif
#if 0	//AJ
  glColorMaterial(GL_FRONT, GL_DIFFUSE);	// bad: black with Mesa
#endif
  glEnable(GL_COLOR_MATERIAL);

  //  Pelvic (Body Bottom)
  glPushMatrix();	// pelvic
   glTranslatef(jp.x[PELVIC], jp.y[PELVIC], jp.z[PELVIC]);
   glRotatef(bap[PELVIC_TILT], 1,0,0);
   glRotatef(bap[PELVIC_TORSION], 0,0,1);
   glRotatef(bap[PELVIC_ROLL], 0,1,0);
   glTranslatef(-jp.x[PELVIC], -jp.y[PELVIC], -jp.z[PELVIC]);
   glColor4fv(skin);
   glCallList(manlist+2);
   //  Spinal (Body Up)
   glPushMatrix();	// spinal
    glTranslatef(jp.x[SPINAL], jp.y[SPINAL], jp.z[SPINAL]);
    glRotatef(bap[T5ROLL], 0,1,0);
    glRotatef(bap[T5TORSION], 0,0,1);
    glRotatef(bap[T5TILT], 1,0,0);
    glTranslatef(-jp.x[SPINAL], -jp.y[SPINAL], -jp.z[SPINAL]);
    glCallList(manlist+1);
    //  Lower Neck
    glPushMatrix();	// neck
     glTranslatef(jp.x[LOWER_NECK], jp.y[LOWER_NECK], jp.z[LOWER_NECK]);
     glRotatef(bap[C4ROLL], 0,1,0);
     glRotatef(bap[C4TORSION], 0,0,1);
     glRotatef(bap[C4TILT], 1,0,0);
     glTranslatef(-jp.x[LOWER_NECK], -jp.y[LOWER_NECK], -jp.z[LOWER_NECK]);
     glColor4fv(skinnude);
     glCallList(manlist+15);
     //  Upper Neck -> Head
     glPushMatrix();	// head
      glTranslatef(jp.x[UPPER_NECK], jp.y[UPPER_NECK], jp.z[UPPER_NECK]);
      glRotatef(bap[C1ROLL], 0,1,0);
      glRotatef(bap[C1TORSION], 0,0,1);
      glRotatef(bap[C1TILT], 1,0,0);
#if 1
#ifdef VNP_SCALING
      glScalef(FACE_SCALE, FACE_SCALE, FACE_SCALE);     // PD
#else
      glScalef(3.3, 3.3, 3.3);	// YR
#endif
      glTranslatef(0, 0.5, -0.7);	// PD
      glRotatef(90, 1,0,0);		// YR

      // render the face
      faceObject->render();	// YR
#else
      glColor4fv(skinnude);
      glCallList(manlist);
#endif
      glTranslatef(-jp.x[UPPER_NECK], -jp.y[UPPER_NECK], -jp.z[UPPER_NECK]);
     glPopMatrix(); 	// head
    glPopMatrix(); 	// neck
    //  Left Shoulder (Arm Up)
    glPushMatrix();	// l_shoulder
     glTranslatef(jp.x[L_SHOULDER], jp.y[L_SHOULDER], jp.z[L_SHOULDER]);
     glRotatef(-90, 0,0,1);
     glRotatef(-bap[L_SHOULDER_FLEXION], 0,1,0);
     glRotatef(bap[L_SHOULDER_ABDUCT], 1,0,0);
     glRotatef(-bap[L_SHOULDER_TWIST], 0,0,1);
     glTranslatef(-jp.x[L_SHOULDER], -jp.y[L_SHOULDER], -jp.z[L_SHOULDER]);
     glColor4fv(skin);
     glCallList(manlist+6);
     //  Left Elbow (Arm Mid)
     glTranslatef(jp.x[L_ELBOW], jp.y[L_ELBOW], jp.z[L_ELBOW]);
     glRotatef(-bap[L_ELBOW_FLEXION], 0,1,0);
     glRotatef(-bap[L_ELBOW_TWIST], 0,0,1);
     glTranslatef(-jp.x[L_ELBOW], -jp.y[L_ELBOW], -jp.z[L_ELBOW]);
     glColor4fv(skinnude);
     glCallList(manlist+7);
     //  Left Wrist (Arm Hand)
     glTranslatef(jp.x[L_WRIST], jp.y[L_WRIST], jp.z[L_WRIST]);
     glRotatef(bap[L_WRIST_FLEXION], 1,0,0);
     glRotatef(bap[L_WRIST_PIVOT], 0,1,0);
     glRotatef(bap[L_WRIST_TWIST], 0,0,1);
     glTranslatef(-jp.x[L_WRIST], -jp.y[L_WRIST], -jp.z[L_WRIST]);
     glColor4fv(skinnude);
     glCallList(manlist+8);
    glPopMatrix();	// l_shoulder
    //  Right Shoulder (Arm Up)
    glPushMatrix();	// r_shoulder
     glTranslatef(jp.x[R_SHOULDER], jp.y[R_SHOULDER], jp.z[R_SHOULDER]);
     glRotatef(90, 0,0,1);
     glRotatef(bap[R_SHOULDER_FLEXION], 0,1,0);
     glRotatef(-bap[R_SHOULDER_ABDUCT], 1,0,0);
     glRotatef(-bap[R_SHOULDER_TWIST], 0,0,1);
     glTranslatef(-jp.x[R_SHOULDER], -jp.y[R_SHOULDER], -jp.z[R_SHOULDER]);
     glColor4fv(skin);
     glCallList(manlist+3);
     //  Right Elbow (Arm Mid)
     glTranslatef(jp.x[R_ELBOW], jp.y[R_ELBOW], jp.z[R_ELBOW]);
     glRotatef(bap[R_ELBOW_FLEXION], 0,1,0);
     glRotatef(-bap[R_ELBOW_TWIST], 0,0,1);
     glTranslatef(-jp.x[R_ELBOW], -jp.y[R_ELBOW], -jp.z[R_ELBOW]);
     glColor4fv(skinnude);
     glCallList(manlist+4);
     //  Right Wrist (Arm Hand)
     glTranslatef(jp.x[R_WRIST], jp.y[R_WRIST], jp.z[R_WRIST]);
     glRotatef(bap[R_WRIST_FLEXION], 1,0,0);
     glRotatef(bap[R_WRIST_PIVOT], 0,1,0);
     glRotatef(bap[R_WRIST_TWIST], 0,0,1);
     glTranslatef(-jp.x[R_WRIST], -jp.y[R_WRIST], -jp.z[R_WRIST]);
     glColor4fv(skinnude);
     glCallList(manlist+5);
    glPopMatrix();	// r_shoulder
   glPopMatrix();	// spinal
  glPopMatrix();	// pelvic
  //  Left Hip (Leg Up)
  glPushMatrix();	// l_leg
   glTranslatef(jp.x[L_HIP], jp.y[L_HIP], jp.z[L_HIP]);
   glRotatef(bap[L_HIP_FLEXION], 1,0,0);
   glRotatef(bap[L_HIP_ABDUCT], 0,1,0);
   glRotatef(bap[L_HIP_TWIST], 0,0,1);
   glTranslatef(-jp.x[L_HIP], -jp.y[L_HIP], -jp.z[L_HIP]);
   glColor4fv(skin);
   glCallList(manlist+12);
   //  Left Knee (Leg Mid)
   glTranslatef(jp.x[L_KNEE], jp.y[L_KNEE], jp.z[L_KNEE]);
   glRotatef(bap[L_KNEE_FLEXION], 1,0,0);
   glRotatef(bap[L_KNEE_TWIST], 0,0,1);
   glTranslatef(-jp.x[L_KNEE], -jp.y[L_KNEE], -jp.z[L_KNEE]);
   glColor4fv(skin);
   glCallList(manlist+13);
   //  Left Ankle (Leg Foot)
   glTranslatef(jp.x[L_ANKLE], jp.y[L_ANKLE], jp.z[L_ANKLE]);
   glRotatef(bap[L_ANKLE_FLEXION], 1,0,0);
   glRotatef(bap[L_ANKLE_TWIST], 0,0,1);
   glTranslatef(-jp.x[L_ANKLE], -jp.y[L_ANKLE], -jp.z[L_ANKLE]);
   glColor4fv(skin);
   glCallList(manlist+14);
  glPopMatrix();	// l_leg
  //  Right Hip (Leg Up)
  glPushMatrix();	// r_leg
   glTranslatef(jp.x[R_HIP], jp.y[R_HIP], jp.z[R_HIP]);
   glRotatef(bap[R_HIP_FLEXION], 1,0,0);
   glRotatef(bap[R_HIP_ABDUCT], 0,1,0);
   glRotatef(bap[R_HIP_TWIST], 0,0,1);
   glTranslatef(-jp.x[R_HIP], -jp.y[R_HIP], -jp.z[R_HIP]);
   glColor4fv(skin);
   glCallList(manlist+9);
   //  Right Knee (Leg Mid)
   glTranslatef(jp.x[R_KNEE], jp.y[R_KNEE], jp.z[R_KNEE]);
   glRotatef(bap[R_KNEE_FLEXION], 1,0,0);
   glRotatef(bap[R_KNEE_TWIST], 0,0,1);
   glTranslatef(-jp.x[R_KNEE], -jp.y[R_KNEE], -jp.z[R_KNEE]);
   glColor4fv(skin);
   glCallList(manlist+10);
   //  Right Ankle (Leg Foot)
   glTranslatef(jp.x[R_ANKLE], jp.y[R_ANKLE], jp.z[R_ANKLE]);
   glRotatef(bap[R_ANKLE_FLEXION], 1,0,0);
   glRotatef(bap[R_ANKLE_TWIST], 0,0,1);
   glTranslatef(-jp.x[R_ANKLE], -jp.y[R_ANKLE], -jp.z[R_ANKLE]);
   glColor4fv(skin);
   glCallList(manlist+11);
  glPopMatrix();	// r_leg

  glDisable(GL_COLOR_MATERIAL);
}

void Android::render()
{
  glPushMatrix();
   glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_FASTEST); //AJ effect bad tex
   glPushMatrix();
#ifdef VNP_SCALING
   glTranslatef(pos.x+tx, pos.y+ty, pos.z+tz + MAN_HALF_HEIGHT*MAN_SCALE);
#else
   glScalef(MAN_SCALE, MAN_SCALE, MAN_SCALE);	// 1/40 = 68m/1.70m
   glTranslatef(41.5*pos.x + tx, 40*pos.y + ty, pos.z+MAN_HALF_HEIGHT + tz);
#endif
   glRotatef(-90 + ry, 0,1,0);	//PD8  horiz -> vertic axe vre Y
   glRotatef(-90 + rz, 0,0,1);	//PD8  axe vre X
   glRotatef(-90 + rx, 1,0,0);	//PD8  axe vre Y
   glRotatef(rotz - 90, 0,0,1);	//PD6

   manDisplay();
   glPopMatrix();
  glPopMatrix();
}

//== Network

/** init listening */
int Android::initReceiver()
{
  if ((sdudp = socketDatagram()) < 0)
    return 0;

  char group[GROUP_LEN];
  getGroup(World::getChannelName(), group);

  memset(&udpsa, 0, sizeof(udpsa));
  udpsa.sin_family = AF_INET;
  udpsa.sin_port = htons(bap_port);
  if (ipmode == ANDROID_MULTICAST)
    inet4_pton(group, &(udpsa.sin_addr.s_addr));
  else
    udpsa.sin_addr.s_addr = htonl(INADDR_ANY);

  /* bind local port */
  setReuseAddr(sdudp);
  if (bind(sdudp, (struct sockaddr *) &udpsa, sizeof(udpsa)) <0) {
    error("initReceiver: can't bind port=%d", bap_port);
    return 0;
  }
  if (ipmode == ANDROID_MULTICAST) {	// Multicast
    memset(&mreq, 0, sizeof(mreq));
    inet4_pton(group, &(mreq.imr_multiaddr.s_addr));
    mreq.imr_interface.s_addr = htonl(INADDR_ANY);
    if (addMembership(sdudp, &mreq) < 0) {
      error("initReceiver: cannot join multicast group %s", group);
      return -1;
    }
    trace(DBG_MAN, "initReceiver: waiting for Multicast on %x:%d", udpsa.sin_addr.s_addr, udpsa.sin_port);
  }
  else	// Unicast
    error("initReceiver: waiting for Unicast on port %d", bap_port);
  return 1;
}

/** Establishes a TCP connection and send the setup packet */
int Android::connectToBapServer(int _ipmode)
{
  if ((sdtcp = socketStream()) < 0) {
    error("socket bap/fap failed");
    return 0;
  }

  struct hostent *hp;
  if ((hp = my_gethostbyname(vaps, AF_INET)) == NULL)
    return 0;

  memset(&tcpsa, 0, sizeof(tcpsa));
  tcpsa.sin_family = AF_INET;
  tcpsa.sin_port = htons(VAPS_PORT);
  memcpy(&tcpsa.sin_addr, hp->h_addr_list[0], hp->h_length);
  my_free_hostent(hp);

  if (connect(sdtcp, (const struct sockaddr *) &tcpsa, sizeof(tcpsa)) < 0) {
    perror("connect vaps");
    error("Connection failed with the bap server: %s (%s)", vaps,
          inet4_ntop(&tcpsa.sin_addr));
    return 0;
  }
  trace(DBG_MAN, "Connection established with the bap server: %s (%s)", vaps,
        inet4_ntop(&tcpsa.sin_addr));

  char command[128];
  char group[GROUP_LEN];
  getGroup(World::getChannelName(), group);

  ipmode = _ipmode;
  if (ipmode == ANDROID_MULTICAST)
    sprintf(command, "setup a=%s p=%d t=%d r=%.2f ",
            group, bap_port, getTtl(World::getChannelName()), getRate());
  else
    sprintf(command, "setup p=%d r=%.2f ", bap_port, getRate());

  trace(DBG_FORCE, "connectToBapServer: command=%s", command);
  write(sdtcp, command, strlen(command));	// setup packet
  return sdtcp;
}

void Android::disconnectFromBapServer()
{
  if (sdtcp > 0) {
    if (ipmode == ANDROID_MULTICAST)
      dropMembership(sdudp, &mreq);
    if (sdudp > 0)
      close(sdudp);
    write(sdtcp, "stop", 4);
    close(sdtcp);
  }
  sdtcp = sdudp = 0;
}

void AndroidListen(Android *po, void *d, time_t s, time_t us)
{
  if (po->status == ANDROID_INACTIVE) {
    if (! po->connectToBapServer(ANDROID_MULTICAST))
      return;
    if (! po->initReceiver())
      return;
    po->status = ANDROID_LISTENING;
  }
  else
    notice("Listen without effect");
}

/** get a frame from the bap server */
int Android::readBapFrame()
{
  if (sdudp <= 0)
    return 0;	// not connected, usually in Unicast mode

  fd_set set;
  struct timeval tv;

  FD_ZERO(&set);
  FD_SET(sdudp, &set);
  tv.tv_sec = tv.tv_usec = 0;	// set select passing
  if (select(FD_SETSIZE, &set, NULL, NULL, &tv) == 0)
    return 0;	// nothing to read

  socklen_t slen = sizeof(struct sockaddr_in);
  int len;

  /* receive Bap line */
  memset(line, 0, VAPS_BUFSIZ);

  if ((len = recvfrom(sdudp, line, VAPS_BUFSIZ, 0,
                      (struct sockaddr *) &udpsa, &slen)) < 0) {
    error("readBapFrame: err recvfrom");
    return 0;
  }
  if (len == 0) { // eof
    trace(DBG_MAN, "readBapFrame: eof");
    //TODO DON'T DISCONNECT NEXT LINE
    //PD disconnectFromBapServer();	// send stop command
    return 0;
  }

  if (strcmp(line, "") == 0)
    return 0;		// discard empty line
  else if (line[0] == '#')
    return 0;		// discard comment line

  //
  // TODO: mix bap and fap
  // needs header: HBAP, HFAP, B3.1, B3.2, F2.1 F2.0 inserted by vaps server
  // num_faps to be added
  //
  else { // Header or Data is there
    char *l;
    float bap_rate, vreng_rate, ratio_rate;

    trace(DBG_MAN, "%s", line);
    l = strtok(line, " \t");
    if (!strncmp(l, HEAD_BAP_V31, 3)) {		// Bap3.1 Header
      num_baps = NUM_BAPS_V31;
      bfflag = TYPE_BAP_V31;
      l = strtok(NULL, " \t"); l = strtok(NULL, " \t");
      trace(DBG_MAN, "readBapHeader bap3.1: num_baps=%d", num_baps);
    }
    else if (!strncmp(l, HEAD_BAP_V32, 3)) {	// Bap3.2 Header
      num_baps = NUM_BAPS_V32;
      bfflag = TYPE_BAP_V32;
      l = strtok(NULL, " \t"); l = strtok(NULL, " \t");
      bap_rate = (float) atof(l);
      vreng_rate = getRate();
      ratio_rate = vreng_rate / bap_rate;
      trace(DBG_MAN, "readBapHeader bap3.2: num_baps=%d vreng_rate=%.2f bap_rate=%.2f ratio_rate=%.2f", num_baps, vreng_rate, bap_rate, ratio_rate);
    }
    else if (!strncmp(l, HEAD_FAP_V20, 3)) {	// Fap2.0 Header
      num_baps = NUM_FAPS;
      bfflag = TYPE_FAP_V20;
      l = strtok(NULL, " \t"); l = strtok(NULL, " \t");
      trace(DBG_MAN, "readBapHeader fap2.0: num_baps=%d", num_baps);
    }
    else if (!strncmp(l, HEAD_FAP_V21, 3)) {	// Fap2.1 Header
      num_baps = NUM_FAPS;
      bfflag = TYPE_FAP_V21;
      l = strtok(NULL, " \t"); l = strtok(NULL, " \t");
      trace(DBG_MAN, "readBapHeader fap2.1: num_baps=%d", num_baps);
    }

    else {	// Masks and Values
      if (bfflag == TYPE_BAP_V31 || bfflag == TYPE_BAP_V32) {
        for (int i=1; i <= num_baps; i++) {
          mask[i] = atoi(l);	//extract all the mask values
          l = strtok(NULL, " \t");
        }
        trace(DBG_MAN, "readBapHeader: num_frame=%s", l);
        for (int i=1; i <= num_baps; i++) {
          if (mask[i] == 0)
	    continue;
	  if ((l = strtok(NULL, " \t")) == NULL)
            break;

#define TR_DIV		300.f	// I can explain why
#define BAPV32_DIV	555	// 180/1e5
#define BAPV31_DIV	1745	// 180/PI*1e5
#define FAPV20_DIV	20	// I can explain why
#define FAPV21_DIV	1	// I can explain why

          if (i >= TR_VERTICAL && i <= TR_FRONTAL) {	// translations
            bap[i] = atoi(l);	// millimeters ?
          }
          else {	// rotations
            if (num_baps == NUM_BAPS_V32)
              bap[i] = atoi(l) / BAPV32_DIV; //magic formula (555)	//GB
            else
              bap[i] = atoi(l) / BAPV31_DIV; //magic formula (1745)
          }
          trace(DBG_MAN, "readBapHeader: l=%s bap[%d]=%d", l, i, bap[i]);
        }
      }
      else if (bfflag == TYPE_FAP_V20 || bfflag == TYPE_FAP_V21) {
        for (int i=1; i <= num_baps; i++) {
          mask[i] = atoi(l);	//extract all the mask values
          l = strtok(NULL, " \t");
        }
        trace(DBG_MAN, "readBapHeader fap: num_frame=%s", l);
        for (int i=1; i <= num_baps; i++) {
          if (mask[i] == 0)
	    continue;
	  if ((l = strtok(NULL, " \t")) == NULL)
            break;
          if (bfflag == TYPE_FAP_V20)
            fap[i] = atoi(l) / FAPV20_DIV; //unknown formula
          else
            fap[i] = atoi(l) / FAPV21_DIV; //unknown formula
        }
      }
      else {
        error("readBapHeader: unknown type=%d", bfflag);
        return 0;
      }
    }
    return 1;	// frame ready to play
  }
}

/** system of equations handling permanent motion */
void Android::changePermanent(float lasting)
{
  if (readBapFrame()) {
    if (bfflag == TYPE_BAP_V31 || bfflag == TYPE_BAP_V32) {

      if (mask[PELVIC_TILT])    bust->tilt_bust(bap[PELVIC_TILT]); //X
      if (mask[PELVIC_TORSION]) bust->turn_bust(bap[PELVIC_TORSION]); //Y
      if (mask[PELVIC_ROLL])    bust->roll_bust(bap[PELVIC_ROLL]); //Z

      if (mask[L_HIP_FLEXION]) legs[1]->rote_leg(bap[L_HIP_FLEXION]); //X
      if (mask[R_HIP_FLEXION]) legs[0]->rote_leg(bap[R_HIP_FLEXION]);
      if (mask[L_HIP_ABDUCT])  legs[1]->raise_leg(bap[L_HIP_ABDUCT]); //Z
      if (mask[R_HIP_ABDUCT])  legs[0]->raise_leg(bap[R_HIP_ABDUCT]);
      if (mask[L_HIP_TWIST])   legs[1]->turn_leg(bap[L_HIP_TWIST]); //Y
      if (mask[R_HIP_TWIST])   legs[0]->turn_leg(bap[R_HIP_TWIST]);

      if (mask[L_KNEE_FLEXION]) calf_l->bend_knee(bap[L_KNEE_FLEXION]); //X
      if (mask[R_KNEE_FLEXION]) calf_r->bend_knee(bap[R_KNEE_FLEXION]);
      if (mask[L_KNEE_TWIST])   calf_l->turn_knee(bap[L_KNEE_TWIST]); //Y
      if (mask[R_KNEE_TWIST])   calf_r->turn_knee(bap[R_KNEE_TWIST]);

      if (mask[L_ANKLE_FLEXION]) foot_l->bend_foot(bap[L_ANKLE_FLEXION]); //X
      if (mask[R_ANKLE_FLEXION]) foot_r->bend_foot(bap[R_ANKLE_FLEXION]);
      if (mask[L_ANKLE_TWIST])   foot_l->turn_foot(bap[L_ANKLE_TWIST]); //Y
      if (mask[R_ANKLE_TWIST])   foot_r->turn_foot(bap[R_ANKLE_TWIST]);

      if (mask[L_SHOULDER_FLEXION]) arm_l->rote_arm(bap[L_SHOULDER_FLEXION]);//X
      if (mask[R_SHOULDER_FLEXION]) arm_r->rote_arm(bap[R_SHOULDER_FLEXION]);
      if (mask[L_SHOULDER_ABDUCT])  arm_l->raise_arm(bap[L_SHOULDER_ABDUCT]);//Z
      if (mask[R_SHOULDER_ABDUCT])  arm_r->raise_arm(bap[R_SHOULDER_ABDUCT]);
      if (mask[L_SHOULDER_TWIST])   arm_l->turn_arm(bap[L_SHOULDER_TWIST]); //Y
      if (mask[R_SHOULDER_TWIST])   arm_r->turn_arm(bap[R_SHOULDER_TWIST]);

      if (mask[L_ELBOW_FLEXION]) forearm_l->bend_elbow(bap[L_ELBOW_FLEXION]);//X
      if (mask[R_ELBOW_FLEXION]) forearm_r->bend_elbow(bap[R_ELBOW_FLEXION]);

      if (mask[L_WRIST_FLEXION]) hand_l->tilt_hand(bap[L_WRIST_FLEXION]); //Z
      if (mask[R_WRIST_FLEXION]) hand_r->tilt_hand(bap[R_WRIST_FLEXION]);
      if (mask[L_WRIST_PIVOT])   hand_l->bend_hand(bap[L_WRIST_PIVOT]); //X
      if (mask[R_WRIST_PIVOT])   hand_r->bend_hand(bap[R_WRIST_PIVOT]);
      if (mask[L_WRIST_TWIST])   hand_l->turn_hand(bap[L_WRIST_TWIST]); //Y
      if (mask[R_WRIST_TWIST])   hand_r->turn_hand(bap[R_WRIST_TWIST]);

      if (mask[C1ROLL])    head->bend_head(bap[C1ROLL]); //X
      if (mask[C1TORSION]) head->turn_head(bap[C1TORSION]); //Y
      if (mask[C1TILT])    head->tilt_head(bap[C1TILT]); //Z

      if (mask[C4ROLL])    neck->bend_neck(bap[C4ROLL]); //X
      if (mask[C4TORSION]) neck->turn_neck(bap[C4TORSION]); //Y
      if (mask[C4TILT])    neck->tilt_neck(bap[C4TILT]); //Z
#if 0
      if (mask[69]) head->tilt_head(bap[69]);
      if (mask[70]) head->turn_head(bap[70]);
      if (mask[71]) head->bend_head(bap[71]);
#else
      if (mask[69]) neck->tilt_neck(bap[69]);
      if (mask[70]) neck->turn_neck(bap[70]);
      if (mask[71]) neck->bend_neck(bap[71]);
#endif

      if (mask[TR_VERTICAL])  tz = bap[TR_VERTICAL] / TR_DIV;
      if (mask[TR_LATERAL])   ty = bap[TR_LATERAL] / TR_DIV;
      if (mask[TR_FRONTAL])   tx = bap[TR_FRONTAL] / TR_DIV;

      if (mask[RT_BODY_TURN]) ry = bap[RT_BODY_TURN];
      if (mask[RT_BODY_ROLL]) rx = bap[RT_BODY_ROLL];
      if (mask[RT_BODY_TILT]) rz = bap[RT_BODY_TILT];
    }

    else if (bfflag == TYPE_FAP_V20 || bfflag == TYPE_FAP_V21) {
      for (int i=1; i <= NUM_FAPS; i++) {
        if (mask[i] && fap[i])
          faceObject->animate(i, fap[i]);
      }
    }
  }
  else {
    //TODO: predictive interpollation
    ;
  }
  faceObject->animate();
}

static uint16_t bap_offset_port = VAPS_OFFSET_PORT;

/** create from a fileline */
WObject * Android::creator(char *l)
{
  return new Android(l);
}

Android::Android(char *l)
{
  char face_url[URL_LEN];

  memset(faces_url, 0, sizeof(faces_url));
  memset(face_url, 0, sizeof(face_url));
  for (int i=0; i<4; i++)
    skin[i] = skinnude[i];
  strcpy(vaps, DEF_VAPS_SERVER);	// default

  l = parseObject(l);
  l = parsePosition(l);

  while (l) {
    if (!strncmp(l, "body", 4))
      l = parseString(l, name.url, "body");	// parse body
    else if (!strncmp(l, "face", 4))
      l = parseString(l, face_url, "face");	// parse face
    else if (!strncmp(l, "color", 5))
      l = parseVector3f(l, skin, "color");	// parse skin color
    else if (!strncmp(l, "server", 6))
      l = parseString(l, vaps, "server");	// parse vaps_server
  }

  char geom[80];
  sprintf(geom, "<solid=bbox,size=%.2f,%.2f,%.2f>",0.12,0.07,0.85);
  parseGeometry(geom);

  enableBehavior(NO_ELEMENTARY_MOVE);
  enableBehavior(COLLIDE_NEVER);
  initializeObject(LIST_MOBILE);

  enablePermanentMovement();	// bap/fap frames

  rotz = (float) RADIAN2DEGREE(pos.az);
  rotx = (float) RADIAN2DEGREE(pos.ax);
  roty = 90;
  bap_offset_port += 2;
  bap_port = getPort(World::getChannelName()) + bap_offset_port;
  bfflag = TYPE_BAP_NULL;
  sdudp = 0;

  createBody();

  faceObject = new Face(faces_url);	// PD
 
  if (*face_url) {
    trace(DBG_MAN, "Android: face_url=%s", face_url);
    faceObject->loadFace(face_url);
  }

  if (! options->reflector)
    AndroidListen(this, NULL, 0, 0);
}

/** send a play command to the bap server */
void Android::sendPlayToBapServer(const char *bap_name)
{
  if (status == ANDROID_INACTIVE) {
    // Ask an Unicast connection if Multicast is disabled
    if (! connectToBapServer(ANDROID_UNICAST))
      return;
  }
  if (sdtcp <= 0) {
    error("sendPlayToBapServer: not connected");
    return;
  }
  if (status == ANDROID_INACTIVE) {
    if (! initReceiver()) {
      perror("initReceiver");
      return;
    }
    status = ANDROID_PLAYING;
  }

  char command[80];
  sprintf(command, "play f=%s ", bap_name);
  trace(DBG_MAN, "sendPlayToBapServer: command=%s status=%d", command, status);
  write(sdtcp, command, strlen(command));	// TCP play packet
}

void Android::quit()
{
  disconnectFromBapServer();
  bap_offset_port = VAPS_OFFSET_PORT;

  delete faceObject;	// YR
  glDisable(GL_LIGHT2);
}

void AndroidHi(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("hi.bap"); }

void AndroidBye(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("bye.bap"); }

void AndroidAsk(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("ask.bap"); }

void AndroidSit(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("sit.bap"); }

void AndroidReset(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("reset.bap"); }

void AndroidNack(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("nak.bap"); }

void AndroidClap(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("clap.bap"); }

void AndroidEyes(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("eyes.fap"); }

void AndroidJoy(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("joy.fap"); }

void AndroidSad(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("sad.fap"); }

void AndroidSurp(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("surp.fap"); }

void AndroidJag(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("jag.fap"); }

void AndroidTest(Android *po, void *d, time_t s, time_t u)
{ po->sendPlayToBapServer("test.bap"); }

void AndroidStop(Android *po, void *d, time_t s, time_t u)
{
  if (po->status == ANDROID_PLAYING || po->status == ANDROID_LISTENING) {
    po->status = ANDROID_INACTIVE;
    po->disconnectFromBapServer();
  }
}

/** functions initialization */
void androidInitFuncList(void)
{
  setActionFunc(ANDROID_TYPE, 0, WO_ACTION AndroidHi, "Hi");
  setActionFunc(ANDROID_TYPE, 1, WO_ACTION AndroidBye, "Bye");
  setActionFunc(ANDROID_TYPE, 2, WO_ACTION AndroidAsk, "Ask");
  setActionFunc(ANDROID_TYPE, 3, WO_ACTION AndroidSit, "Sit");
  setActionFunc(ANDROID_TYPE, 4, WO_ACTION AndroidReset, "Reset");
  setActionFunc(ANDROID_TYPE, 5, WO_ACTION AndroidNack, "Nack");
  setActionFunc(ANDROID_TYPE, 6, WO_ACTION AndroidClap, "Clap");
  setActionFunc(ANDROID_TYPE, 7, WO_ACTION changeFace, "New");
  setActionFunc(ANDROID_TYPE, 8, WO_ACTION changeMoveYes, "Yes");
  setActionFunc(ANDROID_TYPE, 9, WO_ACTION changeMoveNo, "No");
  setActionFunc(ANDROID_TYPE, 10, WO_ACTION AndroidEyes, "Eyes");
  setActionFunc(ANDROID_TYPE, 11, WO_ACTION AndroidJoy, "Joy");
  setActionFunc(ANDROID_TYPE, 12, WO_ACTION AndroidSad, "Sad");
  setActionFunc(ANDROID_TYPE, 13, WO_ACTION AndroidSurp, "Surp");
  setActionFunc(ANDROID_TYPE, 14, WO_ACTION AndroidJag, "Jag");
  setActionFunc(ANDROID_TYPE, 15, WO_ACTION AndroidTest, "Test");
  setActionFunc(ANDROID_TYPE, 16, WO_ACTION AndroidStop, "Stop");
  setActionFunc(ANDROID_TYPE, 17, WO_ACTION AndroidListen, "Listen");
  setActionFunc(ANDROID_TYPE, 18, WO_ACTION changeMoveEyeL, "EyeL");
  setActionFunc(ANDROID_TYPE, 19, WO_ACTION changeMoveMouth, "Mouth");
}
