/*******************************************************************/
/* Les fonctions suivantes sont utilisees pour gerer des solides 
 * disposes dans le monde.
 * On n'a pas specifie ici:
 * - la gestion des sources lumineuses et des modeles lumineux
 * - la gestion des hierarchies d'objets
 * - les fonctions d'entree/sortie pour les geometries et les scenes
 * - comment gerer plusieurs mondes ou affichages
 */

#ifndef VGL_H
#define VGL_H


/* draw styles */
#define STYLE_FILL	0
#define STYLE_LINES	1
#define STYLE_POINTS	2

#define RENDER_NORMAL		0
#define RENDER_FLASHY_BEGIN	1
#define RENDER_FLASHY_END	2
#define RENDER_REFLEXIVE	3

class VGContext;


/**
 * Solid Class
 */
class Solid {
 private:
  bool		havebb;

 public:
  uint16_t	id;		///< solid id
  uint16_t	type;		///< solid type
  bool		isvisible;
  bool		istransparent;
  bool		isflashy;
  bool		isflashable;
  bool		isreflexive;
  bool		isblinking;
  bool		blink;
  M4		posmat;		///< position matrix
  int32_t	*displaylist;	///< display list
  uint16_t	nbframes;	///< number of frames of this solid
  uint16_t	curframe;	///< current frame
  V3		bbcenter;	///< position center
  V3		bbsize;		///< dimension
  GLfloat	flashcolor[3];	///< flash color
  class WObject *object;	///< WObject pointer
  Solid 	*next, *prev;

  Solid();
  virtual ~Solid();

  void getSolidBB(V3 *center, V3 *size);
  /**<
   * Returns a bounding box for this solid.
   * The bounding box is aligned with the world reference.
   */

  void setSolidBB(V3 *bbmax, V3 *bbmin);
  /**
   * Sets the solid bounding box.
   */

  int solidParser(int shape, V3 *bbmax, V3 *bbmin, int32_t *pdisp_list, uint16_t frame);
  /**
   * Parses the solid.
   * Computes bounding box and builds display list.
   */
   
  void setSolidPosition(M4 *mpos);
  /**<
   * Updates the solid position.
   * The position is defined by the matrix 4x4 'mpos'
   * which gives the transformation to do on local coords towards world coords
   */ 

  void getSolidPosition(M4 *mpos);
  /**< Gives solid's position */

  void setSolidFrame(uint16_t frame);
  /**< Changes the "frame" of the solid (ie its geometry) */

  virtual int renderDisplayList(const int flashy);
  virtual void renderReflexive();
  virtual bool renderBlinking();
  virtual void displayObject();

  void setSolidBBFlag(bool flag);
  void setSolidTransparent(bool flag);
  void setSolidFlashable(bool flag);
  void setSolidFlashy(bool flag);
  void setSolidFlashy(GLfloat *_flashcolor);
  void resetSolidFlashy();
  void setSolidReflexive(bool flag);
  void setSolidBlinking(bool flag);
  void setSolidVisible(bool flag);
  /**<
   * Sets the solid visible (true) or invisible (false).
   * By default solids are visible.
   */

  void deleteSolidFromList();
  /**< Deletes solid from the scene */

  WObject * getObjectFromSolid();
  /**< Gets WObject pointer from a Solid */

  void clearObjectBar();
  /**< Clears the ObjectBar in the GUI */

  void specialAction(int action, void *data, time_t sec, time_t usec);
  /**< Calls methods dedicated to each object.
   * Called by GUI.
   */

  void getObjectHumanName(char **classname, char **instancename, char **actionnames);
  /**< Gives object's class_name & action names.
   * Called by WO.
   */

  void addSolidToList();
  /**<
   * Adds this solid to the list.
   */

  int solidIntersect(Solid *s2);
  /**<
   * Returns True if both solids 's1' et 's2' intersect themself
   * not implemented for the moment: return 'true'
   */

  static Solid * parseGeometry(const char *geometry);
  /**<
   * Creates a new solid and parses its geometry.
   * Returns the solid's handle.
   * Called from each WObject.
   */

  static int getFramesNumber();
  static uint16_t getSolidsNumber();

  static void resetSolid();

  void testBlend(bool mode, float alpha);
  void testTexture(bool mode, int texture);

  int doClickMethod(V3 dir);

};

/**
 * Draw class
 */
class Draw {
 public:

  static void setMinMaxBB(float *v);
  static void vertex3fv(float *v);
  static void vertex3f(float x, float y, float z);

  static void point(float x, float y);
  static void line(float x, float y, float w);
  static void rect(float x, float y, int style);
  static void triangle(float x1, float y1, float z1, float x2, float y2, float z2, float x3, float y3, float z3);
  static void box(float x1, float x2, float y1, float y2, float z1, float z2, int textures[], int style);
  static void octagon(float side, float height, int style);
  static void torus(float rc, int numc, float rt, int numt, int style);
  static void sphere(float radius, int slices, int stacks, int style);
  static void cylinder(float baseR, float topR, float height, int slices, int stacks, int style);
  static void disk(float inR, float outR, int slices, int loops, int style);

};


void VglInit(const bool quality);
/**< quality = 0: normal 3D quality, quality = 1: high 3D quality */

void VglRender(void);
/**< Displays the current scene in the current buffer */

void VglRenderMaterials(void);

void VglQuit(void);
/**< close 3d display */

void VglSetCameraPosition(M4 *mat);
/**<
 * Matrix given the camera position in the world,
 * specify the orthogonal transformation to apply
 * on world coordoninates to screen coordoninates.
 * not fully implemented
 */

void cameraPerspective(float fovy, float near, float far, float ratio);
void cameraProjection(float fovy, float near, float far);
/**<
 * Projection definition.
 * - fovy: angle of field width (in degree)
 * - near: distance between the eye and the projection plan
 * - far : distance between the eye and the clipping plan
 */

//void setFrameBuffer(char *buf, int x0, int y0, int xsize, int ysize);
/**<
 * parameters given the position of the display buffer in the display window
 * (at address 'buf'). 
 * note: 'xsize' and 'x0' must be even
 */

void VglSetViewPort(int xsize, int ysize);
/**<
 * Sets window sizes of the display buffer.
 * note: 'xsize' must be multiple of 16.
 */

void specialRendering(int num = -1);
/**< Special rendering */

Solid * VglGetBufferSelection(int x, int y);
/**<
 * Returns the solid's handle displayed in (x,y) on the screen.
 * We use here coordinates IN THE ZBUFFER and no in the display window.
 */

void VglClickToVector(int, int, V3 *);
/**<
 * To help in handling clicks.
 */

#endif  // VGL_H
