/*
 * (c) 1996-2003 Fabrice Bellard, Philippe Dax
 */

#include "global.h"
#include "vgl.h"
#include "texture.h"	// getFromCache
#include "md2.h"
#include "gcontext.h"	// gcontext
#include "wo.h"		// WObject

/* Solid tokens */
enum {
  STOK_ERR = 0,
  /* shapes */
  STOK_BOX,
  STOK_SPHERE,
  STOK_CONE,
  STOK_TORUS,
  STOK_RECT,
  STOK_DISK,
  STOK_LINE,
  STOK_STATUE,
  STOK_BBOX,
  STOK_CROSS,
  STOK_SPHERE_TORUS,
  STOK_SPHERE_DISK,
  STOK_CONE_DISK,
  /* dimensions */
  STOK_SIZE,
  STOK_RADIUS,
  STOK_RADIUS2,
  STOK_RADIUS3,
  STOK_WIDTH,
  STOK_DEPTH,
  STOK_HEIGHT,
  /* textures */
  STOK_TEXTURE,
  STOK_TEX_XP,
  STOK_TEX_XN,
  STOK_TEX_YP,
  STOK_TEX_YN,
  STOK_TEX_ZP,
  STOK_TEX_ZN,
  /* materials */
  STOK_DIFFUSE,
  STOK_AMBIENT,
  STOK_SPECULAR,
  STOK_EMISSION,
  STOK_SHININESS,
  STOK_ALPHA,
  /* drawing */
  STOK_STYLE,
  STOK_SLICES,
  STOK_STACKS,
  STOK_CYLINDERS,
  STOK_CIRCLES,
  STOK_BLINK,
  /* frames */
  STOK_GEOMETRY,
  STOK_SCALE,
  STOK_FRAMES,
  /* lightings */
  STOK_SPOT_DIRECTION,
  STOK_SPOT_CUTOFF,
  STOK_CONSTANT_ATTENUAT,
  STOK_LINEAR_ATTENUAT,
  STOK_QUADRATIC_ATTENUAT
};

#define DEF_ALPHA		1.
#define DEF_SCALE		1.
#define DEF_SPHERE_SLICES	16
#define DEF_SPHERE_STACKS	16
#define DEF_CONE_SLICES		16
#define DEF_CONE_STACKS		8
#define DEF_TORUS_CYLINDERS	16
#define DEF_TORUS_CIRCLES	16
#define DEF_DISK_SLICES		16
#define DEF_DISK_LOOPS		8

#define FRAME_MAX		256
#define FRAME_BEGIN		'['
#define FRAME_SEP		'-'
#define FRAME_END		']'

/* global */
VGContext *gcontext;	// initialized by VglInit

static int solidCounter = 0;
 
struct _stokens {
  char *tokstr;
  char *tokalias;
  int tokid;
};
static const struct _stokens stokens[] = {
  { "box", "cube", STOK_BOX },
  { "sphere", "", STOK_SPHERE },
  { "cone", "cylinder", STOK_CONE },
  { "torus", "", STOK_TORUS },
  { "rect", "", STOK_RECT },
  { "disk", "", STOK_DISK },
  { "line", "", STOK_LINE },
  { "statue", "", STOK_STATUE },
  { "bbox", "bb", STOK_BBOX },
  { "cross", "cr", STOK_CROSS },
  { "sphere+torus", "dsphere", STOK_SPHERE_TORUS },
  { "sphere+disk", "saucer", STOK_SPHERE_DISK },
  { "cone+disk", "hat", STOK_CONE_DISK },
  { "size", "dim", STOK_SIZE },
  { "radius", "r", STOK_RADIUS },
  { "rb", "ri", STOK_RADIUS },
  { "radius2", "rc", STOK_RADIUS2 },
  { "rt", "re", STOK_RADIUS2 },
  { "radius3", "", STOK_RADIUS3 },
  { "height", "h", STOK_HEIGHT },
  { "texture", "tx", STOK_TEXTURE },
  { "tex_xp", "xp", STOK_TEX_XP },
  { "tex_xn", "xn", STOK_TEX_XN },
  { "tex_yp", "yp", STOK_TEX_YP },
  { "tex_yn", "yn", STOK_TEX_YN },
  { "tex_zp", "zp", STOK_TEX_ZP },
  { "tex_zn", "zn", STOK_TEX_ZN },
  { "diffuse", "dif", STOK_DIFFUSE },
  { "ambient", "amb", STOK_AMBIENT },
  { "specular", "spe", STOK_SPECULAR },
  { "emission", "emi", STOK_EMISSION },
  { "shininess", "shi", STOK_SHININESS },
  { "alpha", "a", STOK_ALPHA },
  { "style", "st", STOK_STYLE },
  { "slices", "sl", STOK_SLICES },
  { "stacks", "st", STOK_STACKS },
  { "cylinders", "cy", STOK_CYLINDERS },
  { "circles", "ci", STOK_CIRCLES },
  { "blink", "bl", STOK_BLINK },
  { "geometry", "geo", STOK_GEOMETRY },
  { "scale", "sc", STOK_SCALE },
  { "frames", "", STOK_FRAMES },
  { "spot_direction", "", STOK_SPOT_DIRECTION },
  { "spot_cutoff", "", STOK_SPOT_CUTOFF },
  { "constant_attenuation", "", STOK_CONSTANT_ATTENUAT },
  { "linear_attenuation", "", STOK_LINEAR_ATTENUAT },
  { "quadratic_attenuation", "", STOK_QUADRATIC_ATTENUAT },
  { NULL, NULL, 0 }
};

/*
 * Parser for solids
 */
static char *begin_ptr, *curr_ptr;
static int ch;

static
void initParser(char *l)
{
  curr_ptr = begin_ptr = l;
  ch = curr_ptr[0];
}

static inline
bool isGeometrySep(char c)
{
#if 1
  return (c == ',') || (c == '\0') || (c == FRAME_END);
#else
  return (c == ' ') || (c == ',') || (c == '\0') || (c == FRAME_END);
#endif
}

static inline
void nextChar(void)
{
  if (ch)
    ch = *++curr_ptr;
}

static inline
void skipSpace(void)
{
  while (isspace(ch))
    nextChar();
}

static inline
void skipValue(void)
{
  while (isdigit(ch))
    nextChar();
  nextChar();
}

static inline
void skipChar(int c)
{
  if (ch != c)
    error("'%c' expected at position %d in '%s'",
	     c, curr_ptr-begin_ptr, begin_ptr);
  nextChar();
}

static inline
float getFloat(void) 
{
  int sign = 1;
  char *p;
  char buf[20];

  if (ch == '-') {
    sign = -1;
    nextChar();
  }
  for (p = buf; isdigit(ch); nextChar())
    *p++ = ch;
  if (ch == '.') {
    *p++ = ch;
    nextChar();
    for ( ; isdigit(ch); nextChar())
      *p++ = ch;
  }
  *p++ = '\0';
  return sign * atof(buf);
}

static inline
uint16_t getInt(void) 
{
  char *p;
  char buf[20];

  for (p = buf; isdigit(ch); nextChar())
    *p++ = ch;
  *p++ = '\0';
  return atoi(buf);
}

static inline
int findChar(char c)
{
  while (isGeometrySep(ch) && (ch != c))
    nextChar();
  return (ch == c);
}

static inline
void getStr(char *buf)
{
  char *p;

  for (p = buf; !isGeometrySep(ch); nextChar())
    *p++ = ch;
  *p++ = '\0';
}

/** get token number
 * 0 if unknown token
 * ch is positionned after token or token=
 */
static
int getToken(void)
{
  char *t;
  char token[16];

  for (t = token; isalnum(ch) || ch == '_' || ch == '+' ; nextChar())
    *t++ = ch;
  *t++ = '\0';	// token null terminated
  if (ch == '=')
    skipChar('=');
  const struct _stokens *ptab;
  for (ptab = stokens; ptab->tokstr ; ptab++) {
    if ((!strcmp(ptab->tokstr, token)) || (!strcmp(ptab->tokalias, token)))
      return ptab->tokid;
  }
  error("getToken: unknown \"%s\" at pos='%d'", token, curr_ptr-begin_ptr);
  return 0;
}

/*
 * Routines to create a solid
 */

void Solid::resetSolid()
{
  solidCounter = 1;	// preserves localUser (id=1)
}

uint16_t Solid::getSolidsNumber()
{
  return solidCounter;
}

/** add this solid */
void Solid::addSolidToList()
{
  prev = gcontext->last_solid;

  if (gcontext->first_solid == NULL)
    gcontext->first_solid = this;
  else 
    gcontext->last_solid->next = this;

  gcontext->last_solid = this;
}

/** delete solid */
Solid::~Solid()
{
  del_solid++;
  if (id == 0) return;		// BUG! HACK!
  if (id != 1)			// !localUser
    deleteSolidFromList();	// 3D
}

/** delete this solid */
void Solid::deleteSolidFromList()
{
  if (next)
    next->prev = prev;
  else
    gcontext->last_solid = prev;
  if (prev)
    prev->next = next;
  else
    gcontext->first_solid = next;

  if (displaylist) {
    //PD for (int i=0; i < nbframes; i++)
    //PD   glDeleteLists(displaylist[i], 1);
    delete[] displaylist;
    displaylist = NULL;
  }

  if (object) {
    if (object->soh == this)
      object->soh = NULL; //ELC
    else if (object->soh == NULL)
      error("deleteSolidFromList: null link object=%p ps=%p name=%s", object, this, object->name.class_name);
    else
      error("deleteSolidFromList: bad link object=%p ps=%p name=%s", object, this, object->name.class_name);
  }
}

/** computes and returns bbmax, bbmin in parseGeometry */
void VGContext::getMinMaxBB(V3 *bbmax, V3 *bbmin, uint16_t frame)
{
  if (frame == 0) {
    *bbmax = bbox_max;
    *bbmin = bbox_min;
  }
  else {	// frames
    for (int j=0; j<3; j++) {
      bbmin->v[j] = MIN(bbox_min.v[j], bbmin->v[j]);
      bbmax->v[j] = MAX(bbox_max.v[j], bbmax->v[j]);
    }    
  }
}

void Solid::testBlend(bool mode, float alpha)
{
  if (alpha < 1) {
    switch (mode) {
    case 0:
      glEnable(GL_BLEND);
      glDepthMask(GL_FALSE);
      break;
    case 1:
      glDepthMask(GL_TRUE);
      glDisable(GL_BLEND);
      setSolidTransparent(true);
      break;
    }
  }
}

void Solid::testTexture(bool mode, int texture)
{
  if (texture > 0) {
    switch (mode) {
    case 0:
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, texture);
      break;
    case 1:
      glDisable(GL_TEXTURE_2D);
      break;
    }
  }
}

/** solid Parser */
int Solid::solidParser(int shape, V3 *bbmax, V3 *bbmin, int32_t *pdisp_list, uint16_t frame)
{
  float radius, radius2, radius3, height;
  float mat_diffuse[] = {1, 1, 1, 1};
  float mat_ambient[] = {1, 1, 1, 1};
  float mat_specular[] = {1, 1, 1, 1};
  float mat_emission[] = {0, 0, 0, 0};
  float mat_shininess[] = {20};
  float light_spot_direction[] = {1, 1, 1, 1};
  float light_spot_cutoff[] = {180};
  float light_constant_attenuation[] = {1};
  float light_linear_attenuation[] = {0};
  float light_quadratic_attenuation[] = {0};
  float scale = DEF_SCALE;
  float alpha = DEF_ALPHA;	// opaque
  uint8_t style = STYLE_FILL;	// default
  uint8_t slices = DEF_SPHERE_SLICES;
  uint8_t stacks = DEF_SPHERE_STACKS;
  uint8_t cylinders = DEF_TORUS_CYLINDERS;
  uint8_t circles = DEF_TORUS_CIRCLES;
  int texture = -1;
  int box_textures[6] = { -1, -1, -1, -1, -1, -1 };

  V3 v;
  V3 size = V3_New(1, 1, 1);

  gcontext->first_bbox = 1;

  switch (shape) {
  case STOK_SPHERE:
    slices = DEF_SPHERE_SLICES;
    stacks = DEF_SPHERE_STACKS;
    break;
  case STOK_CONE:
    slices = DEF_CONE_SLICES;
    stacks = DEF_CONE_STACKS;
    break;
  case STOK_TORUS:
    cylinders = DEF_TORUS_CYLINDERS;
    circles = DEF_TORUS_CIRCLES;
    break;
  case STOK_DISK:
  case STOK_SPHERE_DISK:
  case STOK_CONE_DISK:
    slices = DEF_DISK_SLICES;
    stacks = DEF_DISK_LOOPS;
    break;
  case STOK_SPHERE_TORUS:
    slices = DEF_SPHERE_SLICES;
    stacks = DEF_SPHERE_STACKS;
    cylinders = DEF_TORUS_CYLINDERS;
    circles = DEF_TORUS_CIRCLES;
    break;
  }

  while (1) {
    skipSpace();
    if ((ch == '\0') || (ch == FRAME_END))
      break;	// end of geometry

    if (ch == ',') nextChar();
    int tok = getToken();

    if (! tok) {
      error("solid parser: token skipped");
      skipValue();
      continue;
    }

    switch (tok) {
    case STOK_RADIUS:
      radius = getFloat(); break;
    case STOK_RADIUS2:
      radius2 = getFloat(); break;
    case STOK_RADIUS3:
      radius3 = getFloat(); break;
    case STOK_HEIGHT:
      height = getFloat(); break;
    case STOK_SCALE:
      scale = getFloat(); break;
    case STOK_ALPHA:
      alpha = mat_diffuse[3] = mat_ambient[3] = getFloat(); break;
    case STOK_STYLE:
      style = getInt(); break;
    case STOK_SLICES:
      slices = getInt(); break;
    case STOK_STACKS:
      stacks = getInt(); break;
    case STOK_CYLINDERS:
      cylinders = getInt(); break;
    case STOK_CIRCLES:
      circles = getInt(); break;
    case STOK_SHININESS:
      mat_shininess[0] = getFloat(); break;
    case STOK_SPOT_CUTOFF:
      light_spot_cutoff[0] = getFloat(); break;
    case STOK_CONSTANT_ATTENUAT:
      light_constant_attenuation[0] = getFloat(); break;
    case STOK_LINEAR_ATTENUAT:
      light_linear_attenuation[0] = getFloat(); break;
    case STOK_QUADRATIC_ATTENUAT:
      light_quadratic_attenuation[0] = getFloat(); break;
    case STOK_BLINK:
      setSolidBlinking(true); break;

    case STOK_SIZE:
    case STOK_DIFFUSE:
    case STOK_AMBIENT:
    case STOK_SPECULAR:
    case STOK_EMISSION:
    case STOK_SPOT_DIRECTION:
      v.V_X = getFloat(); skipChar(',');
      v.V_Y = getFloat(); skipChar(',');
      v.V_Z = getFloat();
      switch (tok) {
      case STOK_SIZE:
	size = v; break;
      case STOK_DIFFUSE:
	for (int i=0; i<3; mat_diffuse[i] = mat_ambient[i] = v.v[i++]); break;
      case STOK_AMBIENT:
	for (int i=0; i<3; mat_ambient[i] = v.v[i++]); break;
      case STOK_SPECULAR:
	for (int i=0; i<3; mat_specular[i] = v.v[i++]); break;
      case STOK_EMISSION:
	for (int i=0; i<3; mat_emission[i] = mat_ambient[i] = v.v[i++]); break;
      case STOK_SPOT_DIRECTION:
	for (int i=0; i<3; light_spot_direction[i] = v.v[i++]); break;
      }
      break;

    case STOK_TEXTURE: {
      char *url_tex = new char[URL_LEN];
      getStr(url_tex);
      texture = TextureCacheEntry::getFromCache(url_tex);
      delete[] url_tex;
      break;
    }
    case STOK_TEX_XP:
    case STOK_TEX_XN:
    case STOK_TEX_YP:
    case STOK_TEX_YN:
    case STOK_TEX_ZP:
    case STOK_TEX_ZN: {
      char *url_tex = new char[URL_LEN];
      getStr(url_tex);
      box_textures[tok-STOK_TEX_XP] = TextureCacheEntry::getFromCache(url_tex);
      delete[] url_tex;
      break;
    }
    default:
      error("solid parser: bad token=%d", tok);
      return -1;
    }
  }

  /*
   * draw shapes
   */

  /* displaylist generation */
  int dlist = glGenLists(1);
  glNewList(dlist, GL_COMPILE);

  glCullFace(GL_BACK);
  glShadeModel(GL_SMOOTH);

  glMaterialfv(GL_FRONT, GL_DIFFUSE, mat_diffuse);
  glMaterialfv(GL_FRONT, GL_AMBIENT, mat_ambient);
  glMaterialfv(GL_FRONT, GL_SPECULAR, mat_specular);
  glMaterialfv(GL_FRONT, GL_EMISSION, mat_emission);
  glMaterialfv(GL_FRONT, GL_SHININESS, mat_shininess);

  switch (shape) {

  case STOK_BBOX:
    gcontext->bbox_max = size;
    gcontext->bbox_min.v[0] = -size.v[0];
    gcontext->bbox_min.v[1] = -size.v[1];
    gcontext->bbox_min.v[2] = -size.v[2];
    break;

  case STOK_BOX:
    testBlend(0, alpha);
    Draw::box(-size.V_X, size.V_X, -size.V_Y, size.V_Y, -size.V_Z, size.V_Z,
	      box_textures, style);
    testBlend(1, alpha);
    break;

  case STOK_SPHERE:
    testBlend(0, alpha);
    testTexture(0, texture);
    glDisable(GL_CULL_FACE);
    Draw::sphere(radius, slices, stacks, style);
    glEnable(GL_CULL_FACE);
    testTexture(1, texture);
    testBlend(1, alpha);
    break;

  case STOK_CONE:
    testBlend(0, alpha);
    testTexture(0, texture);
    glDisable(GL_CULL_FACE);
    if (radius == radius2)
      stacks = 1;	// cylinder
    Draw::cylinder(radius, radius2, height, slices, stacks, style);
    glEnable(GL_CULL_FACE);
    testTexture(1, texture);
    testBlend(1, alpha);
    break;

  case STOK_TORUS:
    testBlend(0, alpha);
    testTexture(0, texture);
    glDisable(GL_CULL_FACE);
    Draw::torus(radius2, cylinders, radius, circles, style);
    glEnable(GL_CULL_FACE);
    testTexture(1, texture);
    testBlend(1, alpha);
    break;

  case STOK_DISK:
    testBlend(0, alpha);
    testTexture(0, texture);
    glDisable(GL_CULL_FACE);
    Draw::disk(radius, radius2, slices, stacks, style);
    glEnable(GL_CULL_FACE);
    testTexture(1, texture);
    testBlend(1, alpha);
    break;

  case STOK_RECT:
    testTexture(0, texture);
    glDisable(GL_CULL_FACE);
    Draw::rect(size.V_X, size.V_Y, style);
    glEnable(GL_CULL_FACE);
    testTexture(1, texture);
    break;

  case STOK_LINE:
    if (size.V_Z == 0)
      size.V_Z = 1.;	// lineWidth
    Draw::line(size.V_X, size.V_Y, size.V_Z);
    break;

  case STOK_SPHERE_TORUS:
    testBlend(0, alpha);
    testTexture(0, texture);
    glDisable(GL_CULL_FACE);
    Draw::sphere(radius, slices, stacks, style);
    Draw::torus(0.05, cylinders, radius2, circles, style);
    glEnable(GL_CULL_FACE);
    testTexture(1, texture);
    testBlend(1, alpha);
    break;

  case STOK_CROSS:
    testBlend(0, alpha);
    Draw::box(-size.V_X, size.V_X, -size.V_Y, size.V_Y, -size.V_Z, size.V_Z,
	      box_textures, style);
    Draw::box(-size.V_Z, size.V_Z, -size.V_Y, size.V_Y, -size.V_X, size.V_X,
	      box_textures, style);
    testBlend(1, alpha);
    break;

  case STOK_SPHERE_DISK:
    testBlend(0, alpha);
    testTexture(0, texture);
    glDisable(GL_CULL_FACE);
    Draw::sphere(radius, slices, stacks, style);
    Draw::disk(radius, radius2, slices, stacks, style);
    glEnable(GL_CULL_FACE);
    testTexture(1, texture);
    testBlend(1, alpha);
    break;

  case STOK_CONE_DISK:
    testBlend(0, alpha);
    testTexture(0, texture);
    glDisable(GL_CULL_FACE);
    if (radius == radius3)
      stacks = 1;	// cylinder
    Draw::cylinder(radius, radius3, height, slices, stacks, style);
    Draw::disk(radius, radius2, slices, stacks, style);
    glEnable(GL_CULL_FACE);
    testTexture(1, texture);
    testBlend(1, alpha);
    break;

  }

  glEndList();

  /* returns displaylist number for this frame */
  pdisp_list[frame] = dlist;

  /*
   * bounding boxes
   */
  switch (shape) {
  case STOK_RECT:
  case STOK_LINE:
    // without BBox
    break;
  default:
    // with BBox
    gcontext->getMinMaxBB(bbmax, bbmin, frame);
  }
  return 1;	// only one frame
}

/** returns nbf (number of frames) */
static
int statueParser(V3 *bbmax, V3 *bbmin, int32_t *disp_list)
{
  int nbf = 0, texture = -1, dlist = -1, geom_dlist = -1;
  float scale = DEF_SCALE;
  int32_t frames[FRAME_MAX] = { 0, -1 };
  char *url_geom = NULL;
  
  gcontext->first_bbox = 1;
  
  while (1) {
    skipSpace();
    if ((ch == '\0') || (ch == FRAME_END))
      break;	// end of geom

    if (ch == ',') nextChar();
    int tok = getToken();

    switch (tok) {
    case STOK_TEXTURE: {
      char *url_tex = new char[URL_LEN];
      getStr(url_tex);
      texture = TextureCacheEntry::getFromCache(url_tex);
    } break;
    case STOK_GEOMETRY:
      url_geom = new char[URL_LEN];
      getStr(url_geom);
      dlist = 0;
      break;
    case STOK_SCALE:
      scale = getFloat();
      break;
    case STOK_FRAMES: {
      uint16_t cur_frame, end_frame, f = 0;
      
      nbf = 1;
      skipChar(FRAME_BEGIN);
      while (ch != FRAME_END) {
	cur_frame = getInt();
	if (ch == FRAME_SEP) {
	  skipChar(ch);
	  end_frame = getInt();
	}
        else {
	  end_frame = cur_frame;
	  if (ch != FRAME_END)
	    skipChar(',');
	}
	while (cur_frame <= end_frame)
	  frames[f++] = cur_frame++;
      }
      frames[f] = -1;
      skipChar(FRAME_END);
      break;
    }
    default:
      error("statueParser: bad tok=%d", tok);
      return -1;
    }
  }

  if ((texture == -1) || (dlist == -1))
    return -1;

  if (url_geom) {
    //
    // Md2 Model, todo: test .md2 suffix
    //
    Md2 *md2;
    if ((md2 = Md2::loadMd2Model(url_geom)) == NULL) {
      error("statueParser: can't loadMd2Model");
      delete[] url_geom;
      return -1;
    }
    delete[] url_geom;
  
    for (nbf = 0; frames[nbf] != -1 && nbf < FRAME_MAX; nbf++) {
      if ((geom_dlist = md2->getDlistMd2Model(frames[nbf], frames[nbf], 0., scale)) <0) {
        nbf = -1;
        delete md2;	// delete Md2
        break;
      }
      dlist = glGenLists(1);
      glNewList(dlist, GL_COMPILE);
       glEnable(GL_TEXTURE_2D);
       glBindTexture(GL_TEXTURE_2D, texture);
       glCallList(geom_dlist);
       glDisable(GL_TEXTURE_2D);
      glEndList();
      disp_list[nbf] = dlist;
      gcontext->getMinMaxBB(bbmax, bbmin, 1);
    }
    delete md2;	// delete Md2
  }
  return nbf;
}

int Solid::getFramesNumber()
{
  uint16_t _nbframes = 1;	// 1 frame by default

  if (findChar(FRAME_BEGIN)) {
    // this object has more that one frame
    skipChar(FRAME_BEGIN);
    _nbframes = getInt();
    skipChar(FRAME_END);
  }
  return _nbframes;
}

/** new Solid */
Solid::Solid()
{
  new_solid++;
  type = 0;
  id = ++solidCounter;
  isvisible = true;
  istransparent = false;
  isflashy = false;
  isflashable = false;
  isreflexive = false;
  isblinking = false;
  blink = false;
  havebb = true;
  displaylist = NULL;
  curframe = 0;
  flashcolor[0] = 0;
  flashcolor[1] = 1;	// green
  flashcolor[2] = 0;
  object = NULL;
  next = NULL;
  prev = NULL;
}

/** Interface with WO */
char * WObject::parseGeometry(char *l)
{
  if (strlen(l) == 0) {
    error("parseGeometry: no geometry");
    return NULL;
  }
  if (*l == '<')	// FIXME: to cancel
    l++;
  if (!strncmp(l, "solid", 5)) {
    char *p = l, *q;
    p += 5;
    if (*p != '=') {
      error("missing '=' in %s", l);
      return NULL;
    }
    l = ++p;
    if (*l == '"') {
      l++;
      if ((q = strchr(l, '"')))
        *q = 0;
      else {
        warning("missing '\"' in %s", l-1);
      }
    }
    if ((q = strchr(l, '>')))
      *q = 0;
  }

  soh = new Solid();

  soh->object = this;	// was done by setObjectToSolid
  initParser(l);
  soh->nbframes = soh->getFramesNumber();

  if ((soh->displaylist = new int32_t[soh->nbframes]) == NULL) {
    error("parseGeometry: can't new displaylist");
    return NULL;
  }
  soh->addSolidToList();
  IdM4(&soh->posmat);

  if (haveAction())
    soh->setSolidFlashable(true);

  V3 bbmax, bbmin;	// bounding boxes

  // for each frame
  for (int frame = 0; frame < soh->nbframes; ) {
    int f = 0;

    if (soh->nbframes > 1)	// multi frames
      skipChar(FRAME_BEGIN);

    int shape = soh->type = getToken();
    switch (shape) {

    case STOK_BBOX:
    case STOK_BOX:
    case STOK_SPHERE:
    case STOK_CONE:
    case STOK_TORUS:
    case STOK_RECT:
    case STOK_DISK:
    case STOK_LINE:
    case STOK_SPHERE_TORUS:
    case STOK_CROSS:
    case STOK_SPHERE_DISK:
    case STOK_CONE_DISK:
      f = soh->solidParser(shape, &bbmax, &bbmin, soh->displaylist, frame);
      break;
    case STOK_STATUE:
      f = statueParser(&bbmax, &bbmin, soh->displaylist);
      break;
    default:
      error("parseGeometry error: shape=%d in %s", shape, l);
      delete[] soh->displaylist;
      return NULL;
    }

    if (f == -1) {
      error("parseGeometry error: %s", l);
      soh->deleteSolidFromList();
      return NULL;
    }
    if (soh->nbframes > 1)
      skipChar(FRAME_END);
    frame += f;
  }

  /* BBox in solid: bbcenter and bbsize */
  soh->setSolidBB(&bbmax, &bbmin);

  curr_ptr = strtok(NULL, SEP);	// next token
  return curr_ptr;
}

/** compute bbcenter and bbsize for this solid */
void Solid::setSolidBB(V3 *bbmax, V3 *bbmin)
{
  for (int i=0; i<3; i++) {
    bbcenter.v[i] = (bbmax->v[i] + bbmin->v[i]) / 2;
    bbsize.v[i]   = (bbmax->v[i] - bbmin->v[i]) / 2;
  }
}

/** returns center and size of BB */
void Solid::getSolidBB(V3 *center, V3 *size)
{
  if (!this) {		// keep this !
    error("getSolidBB: BUG this==null");
    // called by getBB <- updateObject <- set_xy <- getProperty
    return;
  }
  if (! havebb)
    return;	// no BB

  V3 p[2], v, v1, qmin, qmax;

  for (int i=0; i<3; i++) {
    p[0].v[i] = bbcenter.v[i] - bbsize.v[i]; // min
    p[1].v[i] = bbcenter.v[i] + bbsize.v[i]; // max
  }
  for (int l=0; l<8; l++) {
    v.V_X = p[l % 2].V_X;
    v.V_Y = p[(l/2) % 2].V_Y;
    v.V_Z = p[(l/4) % 2].V_Z;
    MulM4V3(&v1, &posmat, &v);	// v1 = posmat * v
    if (l == 0)
      qmin = qmax = v1;
    else {
      for (int i=0; i<3; i++) {
	qmin.v[i] = MIN(v1.v[i], qmin.v[i]);
	qmax.v[i] = MAX(v1.v[i], qmax.v[i]);
      }
    }
  }

  for (int i=0; i<3; i++) {
    center->v[i] = (qmax.v[i] + qmin.v[i])/2;
    size->v[i]   = (qmax.v[i] - qmin.v[i])/2;
  }
}

void Solid::setSolidPosition(M4 *mpos)
{
  posmat = *mpos;
}

void Solid::getSolidPosition(M4 *mpos)
{
  *mpos = posmat;
}

void Solid::setSolidVisible(bool _isvisible)
{
  isvisible = _isvisible;
}

void Solid::setSolidTransparent(bool _istransparent)
{
  istransparent = _istransparent;
}

void Solid::setSolidFlashable(bool _isflashable)
{
  isflashable = _isflashable;
}

void Solid::setSolidFlashy(bool _isflashy)
{
  isflashy = _isflashy;
}

void Solid::resetSolidFlashy()
{
  isflashy = false;
}

void Solid::setSolidFlashy(GLfloat *_flashcolor)
{
  isflashy = true;
  flashcolor[0] = _flashcolor[0];
  flashcolor[1] = _flashcolor[1];
  flashcolor[2] = _flashcolor[2];
}

void Solid::setSolidReflexive(bool _isreflexive)
{
  isreflexive = _isreflexive;
}

void Solid::setSolidBlinking(bool _isblinking)
{
  isblinking = _isblinking;
}

void Solid::setSolidBBFlag(bool _havebb)
{
  havebb = _havebb;
}

void Solid::setSolidFrame(uint16_t frame)
{
  curframe = frame % nbframes;
}

/** get a WObject pointer from this Solid */
WObject * Solid::getObjectFromSolid()
{
  WObject *po = this ? object : NULL;
  return po;
}

/**
 * Execute an object's click method if it's defined.
 * Returns >0 if the method existed.
 * By Mathieu Seigneurin
 */
int Solid::doClickMethod(V3 dir)
{
  if (!this) {	// keep this check
    return 0;	// avoids segfault in getObjectFromSolid
  }

  /* get the corresponding WObject */
  WObject *po = getObjectFromSolid();
  if (!po) {
    warning("doClickMethod: unable to find WObject");
    return 0;
  }

  /* execute click method if exists */
  po->click(dir);
  return 1;
}

/* 3D stub for solids intersections */
int Solid::solidIntersect(Solid *s_handle2)
{
  return INTER_INTERSECT;
}
