#include "global.h"
#include "images.h"

typedef struct _ImageRec {
  uint16_t imagic;
  uint16_t type;
  uint16_t dim;
  uint16_t xsize, ysize, zsize;
  uint32_t min, max;
  uint32_t wasteBytes;
  char name[80];
  uint32_t colorMap;
  FILE *fp;
  uint8_t *tmp, *tmpR, *tmpG, *tmpB;
  uint32_t rleEnd;
  uint32_t *rowStart;
  uint32_t *rowSize;
} SgiInfo;


static
void bwtorgba(uint8_t *b, uint8_t *l, int n)
{
  while (n--) {
    l[0] = *b; l[1] = *b; l[2] = *b; l[3] = 0xff;
    l += 4; b++;
  }
}

static
void latorgba(uint8_t *b, uint8_t *a, uint8_t *l, int n)
{
  while (n--) {
    l[0] = *b; l[1] = *b; l[2] = *b; l[3] = *a;
    l += 4; b++; a++;
  }
}

static
void rgbtorgba(uint8_t *r, uint8_t *g, uint8_t *b, uint8_t *l, int n)
{
  while (n--) {
    l[0] = r[0]; l[1] = g[0]; l[2] = b[0]; l[3] = 0xff;
    l += 4; r++; g++; b++;
  }
}

static
void rgbatorgba(uint8_t *r, uint8_t *g, uint8_t *b, uint8_t *a, uint8_t *l, int n)
{
  while (n--) {
    l[0] = r[0]; l[1] = g[0]; l[2] = b[0]; l[3] = a[0];
    l += 4; r++; g++; b++; a++;
  }
}

static
void ConvertShort(uint16_t *array, long length)
{
  uint32_t b1, b2;
  uint8_t *ptr;

  ptr = (uint8_t *) array;
  while (length--) {
    b1 = *ptr++;
    b2 = *ptr++;
    *array++ = (b1 << 8) | (b2);
  }
}

static
void ConvertLong(uint32_t *array, long length)
{
  uint32_t b1, b2, b3, b4;
  uint8_t *ptr;

  ptr = (uint8_t *) array;
  while (length--) {
    b1 = *ptr++;
    b2 = *ptr++;
    b3 = *ptr++;
    b4 = *ptr++;
    *array++ = (b1 << 24) | (b2 << 16) | (b3 << 8) | (b4);
  }
}

static
void ImageGetRow(SgiInfo *sgiInfo, uint8_t *buf, int y, int z)
{
  uint8_t *iPtr, *oPtr, pixel;
  int count;

  if ((sgiInfo->type & 0xFF00) == 0x0100) { // storage=RLE (run length encoding)
    fseek(sgiInfo->fp, (uint32_t) sgiInfo->rowStart[y+z*sgiInfo->ysize], SEEK_SET);
    fread(sgiInfo->tmp, 1, (uint32_t) sgiInfo->rowSize[y+z*sgiInfo->ysize], sgiInfo->fp);

    iPtr = sgiInfo->tmp;
    oPtr = buf;
    for (;;) {
      pixel = *iPtr++;
      count = (int)(pixel & 0x7F);
      if (!count)
	return;
      if (pixel & 0x80) {
	while (count--) *oPtr++ = *iPtr++;
      }
      else {
	pixel = *iPtr++;
	while (count--) *oPtr++ = pixel;
      }
    }
  }
  else {	// storage = VERBATIM
    fseek(sgiInfo->fp, 512+(y*sgiInfo->xsize)+(z*sgiInfo->xsize*sgiInfo->ysize), SEEK_SET);
    fread(buf, 1, sgiInfo->xsize, sgiInfo->fp);
  }
}

/**
 * check that this machine is little endian
 * return 1 si little endian else 0 if big endian
 */
static bool littleEndian(void)
{
  union {
    int testword;
    char testbyte[4];
  } endiantest;

  endiantest.testword = 1;
  return (endiantest.testbyte[0] == 1) ? 1 : 0;
}

Image * loadSGI(void *texc, ImageReader read_func)
{
  TextureCacheEntry *tc = (TextureCacheEntry *) texc;

  SgiInfo *sgiInfo = new SgiInfo[1];

  if (! sgiInfo)
    return NULL;

  ImgReader *ir = new ImgReader(texc, read_func);

  char filename[64];

  if ((sgiInfo->fp = ir->downloadInCache(tc->texurl, filename, 0)) == NULL)
    return NULL;
  delete ir;

  fread(sgiInfo, 1, 12, sgiInfo->fp); // header

  if (littleEndian())
    ConvertShort(&sgiInfo->imagic, 6);
  sgiInfo->tmp  = new uint8_t[sgiInfo->xsize*256];
  sgiInfo->tmpR = new uint8_t[sgiInfo->xsize*256];
  sgiInfo->tmpG = new uint8_t[sgiInfo->xsize*256];
  sgiInfo->tmpB = new uint8_t[sgiInfo->xsize*256];
  if (!sgiInfo->tmp || !sgiInfo->tmpR || !sgiInfo->tmpG || !sgiInfo->tmpB)
    return NULL;

  if ((sgiInfo->type & 0xFF00) == 0x0100) {	// storage = RLE
    int x = sgiInfo->ysize * sgiInfo->zsize * sizeof(uint32_t);
    sgiInfo->rowStart = new uint32_t[x];
    sgiInfo->rowSize  = new uint32_t[x];
    if (!sgiInfo->rowStart || !sgiInfo->rowSize)
      return NULL;
    sgiInfo->rleEnd = 512 + (2 * x);
    fseek(sgiInfo->fp, 512, SEEK_SET);
    fread(sgiInfo->rowStart, 1, x, sgiInfo->fp);
    fread(sgiInfo->rowSize, 1, x, sgiInfo->fp);
    if (littleEndian()) {
      ConvertLong(sgiInfo->rowStart, x/(int)sizeof(uint32_t));
      ConvertLong((uint32_t *)sgiInfo->rowSize, x/(int)sizeof(uint32_t));
    }
  }

  trace(DBG_VGL, "sgi: magic=%x type=%x xsize=%d ysize=%d zsize=%d",
                  sgiInfo->imagic, sgiInfo->type, sgiInfo->xsize, sgiInfo->ysize, sgiInfo->zsize);

  /* we alloc the data */
  Image *image = new Image(sgiInfo->xsize, sgiInfo->ysize, IMAGE_RGBA, IMAGE_FIX);

  uint32_t *base = (uint32_t *) image->pixmap;

  uint8_t *rbuf = new uint8_t[sgiInfo->xsize];
  uint8_t *gbuf = new uint8_t[sgiInfo->xsize];
  uint8_t *bbuf = new uint8_t[sgiInfo->xsize];
  uint8_t *abuf = new uint8_t[sgiInfo->xsize];
  if (!base || !rbuf || !gbuf || !bbuf)
    return NULL;

  /* we read the data */
  uint32_t *lptr = base;
  for (int y=0; y < sgiInfo->ysize; y++) {
    if (sgiInfo->zsize == 4) {		// RGBA
      ImageGetRow(sgiInfo, rbuf, y, 0);
      ImageGetRow(sgiInfo, gbuf, y, 1);
      ImageGetRow(sgiInfo, bbuf, y, 2);
      ImageGetRow(sgiInfo, abuf, y, 3);
      rgbatorgba(rbuf, gbuf, bbuf, abuf, (uint8_t *) lptr, sgiInfo->xsize);
    } else if (sgiInfo->zsize == 3) {	// RGB
      ImageGetRow(sgiInfo, rbuf, y, 0);
      ImageGetRow(sgiInfo, gbuf, y, 1);
      ImageGetRow(sgiInfo, bbuf, y, 2);
      rgbtorgba(rbuf, gbuf, bbuf, (uint8_t *) lptr, sgiInfo->xsize);
    } else if (sgiInfo->zsize == 2) {	// LA
      ImageGetRow(sgiInfo, rbuf, y, 0);
      ImageGetRow(sgiInfo, abuf, y, 1);
      latorgba(rbuf, abuf, (uint8_t *) lptr, sgiInfo->xsize);
    } else {				// BW
      ImageGetRow(sgiInfo, rbuf, y, 0);
      bwtorgba(rbuf, (uint8_t *) lptr, sgiInfo->xsize);
    }
    lptr += sgiInfo->xsize;
  }

  fclose(sgiInfo->fp);
  if (sgiInfo->tmp)  delete[] sgiInfo->tmp;
  if (sgiInfo->tmpR) delete[] sgiInfo->tmpR;
  if (sgiInfo->tmpG) delete[] sgiInfo->tmpG;
  if (sgiInfo->tmpB) delete[] sgiInfo->tmpB;
  delete[] sgiInfo;
  if (rbuf) delete[] rbuf;
  if (gbuf) delete[] gbuf;
  if (bbuf) delete[] bbuf;
  if (abuf) delete[] abuf;

  return image;
}
