#include "global.h"
#include "gcontext.h"	// gcontext
#include "vgl.h"


/**
 * computes bounding box and draws vertex
 */
void VGContext::setMinMaxBB(float *v)
{
  if (first_bbox) {
    for (int i=0; i<3; i++) {
      bbox_min.v[i] = v[i];
      bbox_max.v[i] = v[i];
    }
    first_bbox = 0;
  }
  else {
    for (int i=0; i<3; i++) {
      bbox_min.v[i] = MIN(v[i], bbox_min.v[i]);
      bbox_max.v[i] = MAX(v[i], bbox_max.v[i]);
    }
  }
}

void Draw::vertex3fv(float *v)
{
  gcontext->setMinMaxBB(v);
  glVertex3f(v[0], v[1], v[2]);
}

void Draw::vertex3f(float x, float y, float z)
{
  float v[3];

  v[0] = x; v[1] = y; v[2] = z;
  vertex3fv(v);
}
void Draw::point(float x, float y)
{
  glBegin(GL_POINTS);
  glVertex2f(x, y);
  glEnd();
}

void Draw::line(float x, float y, float w)
{
  glLineWidth(w);
  glBegin(GL_LINES);
   glVertex2f(x, 0);
   glVertex2f(0, y);
  glEnd();
}

void Draw::rect(float x, float y, int style)
{
  switch (style) {
  case STYLE_FILL:
    glBegin(GL_QUADS); break;
  case STYLE_LINES:
    glBegin(GL_LINE_LOOP); break;
  case STYLE_POINTS:
    glBegin(GL_POINTS); break;
  }
   glTexCoord2f(0,0); glVertex3f(-x, -y, 0);
   glTexCoord2f(1,0); glVertex3f(x,-y, 0);
   glTexCoord2f(1,1); glVertex3f(x, y, 0);
   glTexCoord2f(0,1); glVertex3f(-x, y, 0);
  glEnd();
}

void Draw::triangle(float x1, float y1, float z1, float x2, float y2, float z2, float x3, float y3, float z3)
{
  glBegin(GL_POLYGON);
   glVertex3f(x1, y1, z1);
   glVertex3f(x2, y2, z2);
   glVertex3f(x3, y3, z3);
  glEnd();
}

/**
 * Draw a box (taken from Mesa)
 */
void Draw::box(float x1, float x2, float y1, float y2, float z1, float z2, int textures[], int style)
{
  static const float normals[6][3] = {
    {-1., 0., 0.},
    {0., 1., 0.},
    {1., 0., 0.},
    {0., -1., 0.},
    {0., 0., 1.},
    {0., 0., -1.}
  };
  static const int faces[6][4] = {
    { 0, 1, 2, 3 },
    { 3, 2, 6, 7 },
    { 7, 6, 5, 4 },
    { 4, 5, 1, 0 },
    { 5, 6, 2, 1 },
    { 7, 4, 0, 3 }
  };
  static const float texcoords[6][4][2] = {
    { { 0,1 }, { 0,0 }, { 1,0 }, { 1,1 } },
    { { 1,1 }, { 1,0 }, { 0,0 }, { 0,1 } },
    { { 0,1 }, { 0,0 }, { 1,0 }, { 1,1 } },
    { { 1,1 }, { 1,0 }, { 0,0 }, { 0,1 } },
    { { 0,0 }, { 0,1 }, { 1,1 }, { 1,0 } },
    { { 0,0 }, { 0,1 }, { 1,1 }, { 1,0 } }
  };
  float v[8][3], tmp;

  if (x1 > x2) {
    tmp = x1; x1 = x2; x2 = tmp;
  }
  if (y1 > y2) {
    tmp = y1; y1 = y2; y2 = tmp; 
  }
  if (z1 > z2) {
    tmp = z1; z1 = z2; z2 = tmp; 
  }
  v[0][0] = v[1][0] = v[2][0] = v[3][0] = x1;
  v[4][0] = v[5][0] = v[6][0] = v[7][0] = x2;
  v[0][1] = v[1][1] = v[4][1] = v[5][1] = y1;
  v[2][1] = v[3][1] = v[6][1] = v[7][1] = y2;
  v[0][2] = v[3][2] = v[4][2] = v[7][2] = z1;
  v[1][2] = v[2][2] = v[5][2] = v[6][2] = z2;

  int shadetype;
  switch (style) {
  case STYLE_LINES:
    shadetype = GL_LINE_LOOP; break;
  case STYLE_POINTS:
    shadetype = GL_POINTS; break;
  default:
    shadetype = GL_QUADS;
  }
  for (int i=0; i<6; i++) {
    if (textures[i] > 0) {
      glEnable(GL_TEXTURE_2D);
      glBindTexture(GL_TEXTURE_2D, textures[i]);
    }
    glBegin(shadetype);
    for (int j=0; j<4; j++) {
      if (textures[i] > 0)
        glTexCoord2fv(&texcoords[i][j][0]);
      glNormal3fv(&normals[i][0]);
      vertex3fv(&v[faces[i][j]][0]);
    }
    glEnd();
    if (textures[i] > 0)
      glDisable(GL_TEXTURE_2D);
  }
}

void Draw::octagon(float side, float height, int style)
{
  float x = sin(0.785398163) * side, y = side / 2., z = height / 2., c;

  c = x + y;
  for (int j=0; j<8; j++) {
    glTranslatef(-c, 0., 0.);
    if (style != STYLE_FILL)
      glBegin(GL_LINE_LOOP);
    else
      glBegin(GL_QUADS);
     glNormal3f(-1., 0., 0.);
     glVertex3f(0., -y, z);
     glVertex3f(0., y, z);
     glVertex3f(0., y, -z);
     glVertex3f(0., -y, -z);
    glEnd();
    glTranslatef(c, 0., 0.);
    if (style == STYLE_FILL) {
      glBegin(GL_TRIANGLES);
       glNormal3f(0., 0., 1.);
       glVertex3f(0., 0., z);
       glVertex3f(-c, -y, z);
       glVertex3f(-c, y, z);
       glNormal3f(0., 0., -1.);
       glVertex3f(0., 0., -z);
       glVertex3f(-c, y, -z);
       glVertex3f(-c, -y, -z);
      glEnd();
    }
    glRotatef(45, 0, 0, 1);
  }
}

/**
 * Torus drawing (taken from Mesa)
 */
void Draw::torus(float rc, int numc, float rt, int numt, int style)
{
  for (int i=0; i < numc; i++) {
    double s, t, x, y, z;

    glBegin(GL_QUAD_STRIP);
    for (int j=0; j <= numt; j++) {
      for (int k=1; k >= 0; k--) {
        s = (i + k) % numc + 0.5;
        t = j % numt;
        x = Cos(t*M_2PI/numt) * Cos(s*M_2PI/numc);
        y = Sin(t*M_2PI/numt) * Cos(s*M_2PI/numc);
        z = Sin(s*M_2PI/numc);
        glNormal3f(x, y, z);

        x = (rt + rc * Cos(s*M_2PI/numc)) * Cos(t*M_2PI/numt);
        y = (rt + rc * Cos(s*M_2PI/numc)) * Sin(t*M_2PI/numt);
        z = rc * Sin(s*M_2PI/numc);
        vertex3f(x, y, z);
      }
    }
    glEnd();
  }
}

/**
 * Sphere (taken from Mesa)
 */
void Draw::sphere(float radius, int slices, int stacks, int style)
{
  float rho, drho, theta, dtheta;
  float x, y, z;
  float s, t, ds, dt;
  int normals = 1;
  float nsign = 1.;
  
  drho = M_PI / (float) stacks;
  dtheta = M_2PI / (float) slices;

  /* draw +Z end as a triangle fan */
  glBegin(GL_TRIANGLE_FAN);
  glNormal3f(0., 0., 1.);
  glTexCoord2f(0.5, 0.);
  vertex3f(0., 0., nsign*radius);
  for (int j=0; j <= slices; j++) {
    theta = (j==slices) ? 0. : j*dtheta;
    x = -Sin(theta) * Sin(drho);
    y = Cos(theta) * Sin(drho);
    z = nsign * Cos(drho);
    if (normals)
      glNormal3f(x*nsign, y*nsign, z*nsign);
    vertex3f(x*radius, y*radius, z*radius);
  }
  glEnd();

  ds = 1. / slices;
  dt = 1. / stacks;
  t = 1.;  /* because loop now runs from 0 */

  /* draw intermediate stacks as quad strips */
  for (int i=0; i < stacks; i++) {
     rho = i * drho;
     glBegin(GL_QUAD_STRIP);
     s = 0.;
     for (int j=0; j <= slices; j++) {
       theta = (j==slices) ? 0. : j*dtheta;
       x = -Sin(theta) * Sin(rho);
       y = Cos(theta) * Sin(rho);
       z = nsign * Cos(rho);
       if (normals)
         glNormal3f(x*nsign, y*nsign, z*nsign);
       glTexCoord2f(s, 1-t);
       vertex3f(x*radius, y*radius, z*radius);
       x = -Sin(theta) * Sin(rho+drho);
       y = Cos(theta) * Sin(rho+drho);
       z = nsign * Cos(rho+drho);
       if (normals)
         glNormal3f(x*nsign, y*nsign, z*nsign);
       glTexCoord2f(s, 1-(t-dt));
       s += ds;
       vertex3f(x*radius, y*radius, z*radius);
    }
    glEnd();
    t -= dt;
  }
  /* draw -Z end as a triangle fan */
  glBegin(GL_TRIANGLE_FAN);
  glNormal3f(0., 0., -1.);
  glTexCoord2f(0.5, 1.);
  vertex3f(0., 0., -radius*nsign);
  rho = M_PI - drho;
  s = 1.;
  t = dt;
  for (int j=slices; j >= 0; j--) {
    theta = (j==slices) ? 0. : j*dtheta;
    x = -Sin(theta) * Sin(rho);
    y = Cos(theta) * Sin(rho);
    z = nsign * Cos(rho);
    if (normals)
      glNormal3f(x*nsign, y*nsign, z*nsign);
    glTexCoord2f(s, 1-t);
    s -= ds;
    vertex3f(x*radius, y*radius, z*radius);
  }
  glEnd();
}

/**
 * Cylinder (taken from Mesa, Brian Paul)
 */
void Draw::cylinder(float baseR, float topR, float height, int slices, int stacks, int style)
{
  float da, r, dr, dz;
  float z, nz, nsign = 1.;
  float ds = 1. / slices;
  float dt = 1. / stacks;
  float t = 0.;

  da = M_2PI / slices;
  dr = (topR-baseR) / stacks;
  dz = height / stacks;
  nz = (baseR-topR) / height;  // Z component of normal vectors

  switch (style) {

  case STYLE_FILL:
    {
      z = 0.;
      r = baseR;
      for (int j=0; j < stacks; j++) {
        float s = 0.;
        glBegin(GL_QUAD_STRIP);
        for (int i=0; i <= slices; i++) {
          float x, y;
          if (i == slices) {
            x = Sin(0.);
            y = Cos(0.);
          }
          else {
            x = Sin(i*da);
            y = Cos(i*da);
          }
          glNormal3f(x*nsign, y*nsign, nz*nsign);
          glTexCoord2f(s, t);
          vertex3f(x*r, y*r, z);
          glNormal3f(x*nsign, y*nsign, nz*nsign);
          glTexCoord2f(s, t+dt);
          vertex3f(x*(r+dr), y*(r+dr), z+dz);
          s += ds;
        }
        glEnd();
        r += dr;
        t += dt;
        z += dz;
      }
    }
    break;
 
  case STYLE_LINES:
    {
      z = 0.;
      r = baseR;
      for (int j=0; j <= stacks; j++) {
        glBegin(GL_LINE_LOOP);
        for (int i=0; i < slices; i++) {
          float x = Cos(i*da);
          float y = Sin(i*da);
          glNormal3f(x*nsign, y*nsign, nz*nsign);
          glVertex3f(x*r, y*r, z);
        }
        glEnd();
        z += dz;
        r += dr;
      }
      /* draw length lines */
      glBegin(GL_LINES);
      for (int i=0;i<slices;i++) {
        float x = Cos(i*da);
        float y = Sin(i*da);
        glNormal3f(x*nsign, y*nsign, nz*nsign);
        glVertex3f(x*baseR, y*baseR, 0.);
        glVertex3f(x*topR, y*topR, height);
      }
      glEnd();
    }
    break;

  case STYLE_POINTS:
    {
      glBegin(GL_POINTS);
      for (int i=0; i < slices; i++) {
        float x = Cos(i*da);
        float y = Sin(i*da);
        glNormal3f(x*nsign, y*nsign, nz*nsign);
        z = 0.;
        r = baseR;
        for (int j=0; j <= stacks; j++) {
          glVertex3f(x*r, y*r, z);
          z += dz;
          r += dr;
        }
      }
      glEnd();
    }
    break;
  }
}

/**
 * Disk (taken from Mesa, Brian Paul)
 */
void Draw::disk(float innerR, float outerR, int slices, int loops, int style)
{
  /* Normal vectors */
  //glNormal3f(0., 0., +1.);
  glNormal3f(0., 0., -1.);

  float da = M_2PI / slices;
  float dr = (outerR-innerR) / (float) loops;

  switch (style) {
  case STYLE_FILL:
    {
      /* texture of a gluDisk is a cut out of the texture unit square
       * x, y in [-outerR, +outerR]; s, t in [0, 1]
       * (linear mapping)
       */
      float dtc = 2. * outerR;
      float sa, ca;
      float r1 = innerR;
      for (int l=0; l<loops; l++) {
        float r2 = r1 + dr;
        glBegin(GL_QUAD_STRIP);
        for (int s=slices; s>=0 ; s--) {
          float a;
          if (s == slices)
            a = 0.;
          else
            a = s * da;
          sa = Sin(a);
          ca = Cos(a);
          glTexCoord2f(0.5-sa*r2/dtc, 0.5+ca*r2/dtc);
          glVertex3f(r2*sa, r2*ca, 0);
          glTexCoord2f(0.5-sa*r1/dtc, 0.5+ca*r1/dtc);
          glVertex3f(r1*sa, r1*ca, 0);
        }
        glEnd();
        r1 = r2;
      }
    }
    break;
  case STYLE_LINES:
    {
      /* draw loops */
      for (int l=0; l<=loops; l++) {
        float r = innerR + l * dr;
        glBegin(GL_LINE_LOOP);
        for (int s=0; s<slices; s++) {
          float a = s * da;
          glVertex3f(r*Sin(a), r*Cos(a), 0);
        }
        glEnd();
      }
      /* draw spokes */
      for (int s=0; s<slices; s++) {
        float a = s * da;
        float x = Sin(a);
        float y = Cos(a);
        glBegin(GL_LINE_STRIP);
          for (int l=0; l<=loops; l++) {
            float r = innerR + l * dr;
            glVertex3f(r*x, r*y, 0);
          }
        glEnd();
      }
    }
    break;
  case STYLE_POINTS:
    {
      glBegin(GL_POINTS);
      for (int s=0; s<slices; s++) {
        float a = s * da;
        float x = Sin(a);
        float y = Cos(a);
        for (int l=0; l<=loops; l++) {
          float r = innerR * l * dr;
          glVertex3f(r*x, r*y, 0);
        }
      }
      glEnd();
    }
    break;
  }
}
