#include "global.h"

#include "net.h"
#include "netdefs.h"	// SD_R_RTP
#include "channel.h"	// Channel
#include "payload.h"	// isValidPayload
#include "rtpsess.h"	// deleteSourceBySsrc
#include "pkt.h"


/**
 * Sends a packet
 */
int sendPacket(const int fd, const uint8_t *pkt, const int pkt_len, const struct sockaddr_in *to)
{
  if (fd < 0) {
    error("sendPacket: fd=%d", fd);
    return -1;
  }

  int r;
  if ((r = sendto(fd, pkt, pkt_len, 0, (struct sockaddr*) to, sizeof(*to))) < 0) {
#if IPMC_ENABLED
    perror("sendto");
#endif
    return -1;
  }
  statSendPacket(pkt_len);
  return r;
}

/**
 * Sends a payload
 */
int Payload::sendPayload(const struct sockaddr_in *to)
{
  uint8_t pkt[PKTSIZE];
  uint8_t *ppl;
  int pkt_len, fd_send_rtp;
  rtp_hdr_t *rtp_hdr = (rtp_hdr_t *) pkt;

  /*
   * initial checks
   */
  if (! isValidPayload()) {
    error("sendPayload: invalid payload");
    return -1;
  }
  if (len == 0) {
    trace(DBG_NET, "sendPayload: payload empty");
    return -1;
  }
  if (to == NULL) {
    error("sendPayload: socket NULL");
    return -1;
  }

  /*
   * build the RTP Header
   */
  Channel *pchan;
  if ((pchan = Channel::getbysa(to)) == NULL) {
    pchan = Channel::getCurrentChannel();	// hack !!!
    error("sendPayload: pchan NULL, to=%p", to);
  }
  if (pchan->session)
    pchan->session->buildRtpHeader(rtp_hdr, pchan->ssrc);
  else {
    error("sendPayload: session NULL");
    return -1;	//FIXME: session NULL
  }

  /*
   * build the Payload Header
   */
  ppl = pkt + RTP_HDR_SIZE;

  ppl[PAYLOAD_HDR_VERSION] = VREP_VERSION;
  ppl[PAYLOAD_HDR_FLAGS] = 1;	// payload header length in words
#if WANT_IPV6
  ppl[PAYLOAD_HDR_FAMILY] = AF_INET6;
#else // IPV4
  ppl[PAYLOAD_HDR_FAMILY] = AF_INET;
#endif

  ppl[PAYLOAD_HDR_LEN] = (len + 3 + PAYLOAD_HDR_SIZE) >> 2; // words
  pkt_len = RTP_HDR_SIZE + (ppl[PAYLOAD_HDR_LEN] <<2);	// bytes

  memcpy(ppl+PAYLOAD_HDR_SIZE, data, len);
  ppl[PAYLOAD_HDR_SIZE + len] = 0;	// end of payload chunk

  trace(DBG_13, "S: %02x%02x%02x%02x/%02x",
                     ppl[0], ppl[1], ppl[2], ppl[3], ppl[4]);

  /*
   * find the file descriptor
   */
  if ((fd_send_rtp = getFdSendRTP(to)) <= 0)
    fd_send_rtp = pchan->sd[SD_W_RTP];	// hack !!!

  /*
   * send the packet
   */
  sendPacket(fd_send_rtp, pkt, pkt_len, to);
  statSendRTP(pkt_len);

  /* Update SR, TODO: compute RCTP interval */
  pchan->session->sr.psent++;
  pchan->session->sr.osent += pkt_len;

  if (pkts_sent && ((pkts_sent % 100) == 0)) {
    /*
     * send RTCP compound (SR + SDES)
     */
    pchan->session->refreshMySdes();
    pchan->sendSRSDES(to);
  }

  return pkt_len;
}

/**
 * Builds common part of rtcp packet
 */
void buildRTCPcommon(rtcp_common_t *prtcp_hdr, const uint8_t pt)
{
  memset(prtcp_hdr, 0, sizeof(rtcp_common_t));	// clear P:1, count:5
  prtcp_hdr->version = RTP_VERSION;
  prtcp_hdr->pt = pt;
}

/**
 * Builds a SR packet
 */
int Channel::buildSR(rtcp_common_t *prtcp_hdr, uint8_t *pkt)
{
  rtcp_sr_t sr = session->sr;
  int len = 0;

  buildRTCPcommon(prtcp_hdr, RTCP_SR);
  prtcp_hdr->count = 0;	// only one SSRC
  sr.ssrc = htonl(ssrc);

  struct timeval ts;
  gettimeofday(&ts, NULL);
  sr.rtp_ts = htonl(ts.tv_sec*1000 + ts.tv_sec/1000);
  prtcp_hdr->length = htons(sizeof(rtcp_sr_t) >>2);
  memcpy(pkt, prtcp_hdr, sizeof(rtcp_common_t));
  len += sizeof(rtcp_common_t);
  memcpy(pkt+len, &sr, sizeof(rtcp_sr_t));
  len += sizeof(rtcp_sr_t);

  trace(DBG_RTP, "setSR: ssrc=%x len=%d", sr.ssrc, len);
  return len;
}

/**
 * Builds a SDES packet
 */
int Channel::buildSDES(rtcp_common_t *prtcp_hdr, uint8_t* pkt)
{
  uint32_t ssrc = htonl(ssrc);
  uint8_t xitem = 0;
  int len = 0;
  uint8_t items[PKTSIZE-sizeof(rtcp_sr_t)-sizeof(rtcp_common_t)-sizeof(ssrc)];
  SdesItem *pchunk;

  buildRTCPcommon(prtcp_hdr, RTCP_SDES);
  memset(items, 0, sizeof(items));
  prtcp_hdr->count = 1; // Only one SDES with multiple chunks

  /* fill chunks */
  for (pchunk = session->mysdes; pchunk; pchunk = pchunk->si_next) {
    items[xitem++] = pchunk->si_type;
    items[xitem++] = pchunk->si_len;
    memcpy(items+xitem, pchunk->si_str, pchunk->si_len);
    xitem += pchunk->si_len;
  }
  items[xitem++] = RTCP_SDES_END; // End of chunks is indicated by a zero

  while (xitem % 4) // Pad to 4 bytes word boundary and store zeros there
    items[xitem++] = 0;

  /* SDES header */
  prtcp_hdr->length = htons(1 + (xitem >> 2));
  memcpy(pkt, prtcp_hdr, sizeof(rtcp_common_t));
  len += sizeof(rtcp_common_t);
  memcpy(pkt+len, &ssrc, sizeof(ssrc));
  len += sizeof(ssrc);
  memcpy(pkt+len, items, xitem);
  len += xitem;

  trace(DBG_RTP, "setSDES: ssrc=%x len=%d xitem=%d", ssrc, len, xitem);
  return len;
}

/**
 * Builds a BYE packet
 */
int Channel::buildBYE(rtcp_common_t *prtcp_hdr, uint8_t *pkt)
{
  uint32_t ssrc = htonl(ssrc);

  buildRTCPcommon(prtcp_hdr, RTCP_BYE);
  prtcp_hdr->count = 1;	// me only says bye
  prtcp_hdr->length = htons(1);
  memcpy(pkt, prtcp_hdr, sizeof(rtcp_common_t));
  int len = sizeof(rtcp_common_t);
  memcpy(pkt+len, &ssrc, sizeof(ssrc));
  len += sizeof(ssrc);

  trace(DBG_RTP, "setBYE: ssrc=%x len=%d", ssrc, len);
  return len;
}

/**
 * Sends a RTCP packet
 */
int Channel::sendRTCPPacket(const struct sockaddr_in *to, const uint8_t pt)
{
  int pkt_len = 0, r, fd_send_rtcp;
  rtcp_common_t rtcp_hdr;
  struct sockaddr_in *sin_rtcp;
  uint8_t pkt[PKTSIZE];

  switch (pt) {
  case RTCP_SR: // send a SR
    pkt_len = buildSR(&rtcp_hdr, pkt);
    break;
  case RTCP_SDES: // send a SDES
    pkt_len = buildSDES(&rtcp_hdr, pkt);
    break;
  case RTCP_BYE: // send a BYE
    pkt_len = buildBYE(&rtcp_hdr, pkt);
    break;
  }
  if ((sin_rtcp = getSaRTCP(to)) == NULL) {
    error("sendRTCPPacket: sin_rtcp NULL");
    return -1;
  }
  if ((fd_send_rtcp = getFdSendRTCP(sin_rtcp)) < 0) {
    error("sendRTCPPacket: fd_send_rtcp <0");
    return -1;
  }

  r = sendPacket(fd_send_rtcp, pkt, pkt_len, sin_rtcp);
  statSendRTCP(pkt_len);
  return r; 
}

/**
 * Sends a RTCP compound packet (SR + SDES)
 */
int Channel::sendSRSDES(const struct sockaddr_in *to)
{
  int pkt_len = 0, fd_send_rtcp;
  rtcp_common_t rtcp_hdr;
  struct sockaddr_in *sin_rtcp;
  uint8_t pkt[PKTSIZE];

  int len = buildSR(&rtcp_hdr, pkt);
  pkt_len += len;
  len = buildSDES(&rtcp_hdr, pkt+len);
  pkt_len += len;

  if ((sin_rtcp = getSaRTCP(to)) == NULL) {
    error("sendSRSDES: sin_rtcp NULL");
    return -1;
  }
  if ((fd_send_rtcp = getFdSendRTCP(sin_rtcp)) < 0) {
    error("sendSRSDES: fd_send_rtcp <0");
    return -1;
  }

  trace(DBG_RTP, "sendSRSDES: pkt_len=%d", pkt_len);
  int r = sendPacket(fd_send_rtcp, pkt, pkt_len, sin_rtcp);
  statSendRTCP(pkt_len);
  return r; 
}

/**
 * Receives a Payload packet
 */
int Payload::recvPayload(const int fd, struct sockaddr_in *sender)
{
  if (!sender) {
    error("recvPacket: sender NULL");
    return -1;
  }

  uint8_t pkt[PKTSIZE];
  rtp_hdr_t *rtp_hdr = (rtp_hdr_t *) pkt;
  rtcp_common_t *rtcp_hdr = (rtcp_common_t *) pkt;
  int pkt_len;
  socklen_t l = sizeof(struct sockaddr_in);

  memset(sender, 0, sizeof(*sender));
  if ((pkt_len = recvfrom(fd, pkt, sizeof(pkt), 0, (struct sockaddr *)sender, &l)) <0) {
    error("recvfrom: %s on %d", strerror(errno), fd);
    return pkt_len;	// here pkt_len < 0 -> error
  }
  statReceivePacket(pkt_len);

  /*
   * test if the packet was sent by myself
   */
#if NEEDLOOPBACK
  if (getMyHostId() == ntohl(sender->sin_addr.s_addr)) {
    // If two apps are running on the same machine,
    // you need to sort out the packets on something
    // else than just the host ID.
    if (ntohl(rtp_hdr->ssrc) == getMySsrcId())
      return 0; // Loopback from same app
  }
#else
  // TODO: This is probably broken anyway, loopback or not
  if (getMyHostId() == ntohl(sender->sin_addr.s_addr))
     return 0;  // Loopback: ignore it
#endif

  /*
   * test if it is a valid RTP header
   */
  uint8_t rtp_hdr_size;
  if (rtp_hdr->version == RTP_VERSION &&
      (rtp_hdr->pt == PAYLOAD_TYPE || rtp_hdr->pt == PAYLOAD_TYPE_V1)) {
    //
    // it's a RTP packet
    //
    uint16_t seq;	// RTP seq
    Source *pso;

    rtp_hdr_size = RTP_HDR_SIZE;
    seq = ntohs(rtp_hdr->seq);
    if ((pso = Source::getSource(ntohl(rtp_hdr->ssrc))) != NULL) {
      sourceInfos *s;

      pso->rr.lost += seq - pso->rr.last_seq - 1;
      pso->rr.last_seq = seq;
      s = &(pso->s);
      Rtp::updateSeq(s, seq);
      pso->extended_max = s->cycles + s->max_seq;
      pso->expected = pso->extended_max - s->base_seq + 1;
      pso->lost = pso->expected - s->received;	// number of packets lost
    }
  }
  else if (rtcp_hdr->version == RTP_VERSION) {
    //
    // it's a RTCP packet
    //
    switch (rtcp_hdr->pt) {
      case RTCP_SR:
      case RTCP_SDES:
      case RTCP_BYE:
      case RTCP_APP:
        return recvRTCPPacket(sender, pkt, pkt_len);
      case RTCP_RR:	// why send a RR ? we only send SR
        return 0;	// like action done
      default:
        return -3;	// unknown RTCP packet type
    }
  }
  else {
   //
   // it's an UDP packet, old version of VREng <= 1.5.8
   //
    rtp_hdr_size = 0;
  }

  uint8_t *ppl = pkt + RTP_HDR_SIZE;
  trace(DBG_13, "R: %02x%02x%02x%02x/%02x",ppl[0],ppl[1],ppl[2],ppl[3],ppl[4]);

  /*
   * compatibility with older VREP Protocol
   */
  uint8_t idxlen = PAYLOAD_HDR_LEN;
  uint8_t payload_version = VREP_VERSION;
  uint8_t payload_hdr_size = PAYLOAD_HDR_SIZE;
  uint16_t payload_len;

  sender->sin_family = ppl[PAYLOAD_HDR_FAMILY];
  if (ppl[PAYLOAD_HDR_FLAGS] == 1)
    payload_len = ppl[idxlen] << 2; // length in words
  else
    payload_len = ppl[idxlen];	// length in bytes <= 1.5.8 VREP_1

  if (ppl[PAYLOAD_HDR_VERSION] != VREP_VERSION &&
      ppl[PAYLOAD_HDR_FAMILY]  != AF_INET) {
    /* old version <= 1.5.8 VREP_1 */
    payload_version = 1;
    payload_hdr_size = PAYLOAD_HDR_SIZE_V1;
    idxlen = PAYLOAD_HDR_LEN_V1;
    sender->sin_family = AF_INET;
    memcpy(&sender->sin_addr.s_addr, ppl+PAYLOAD_HDR_SRCID_V1, 4);
    memcpy(&sender->sin_port, ppl+PAYLOAD_HDR_PORTID_V1, 2);
  }

  /*
   * check if valid payload header
   */
  if (pkt_len != rtp_hdr_size + payload_len || payload_len < payload_hdr_size) {
    trace(DBG_FORCE, "R: %02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
                         pkt[0], pkt[1], pkt[2], pkt[3],
                         pkt[4], pkt[5], pkt[6], pkt[7],
                         pkt[8], pkt[9], pkt[10], pkt[11]);
    trace(DBG_FORCE, "R: %02x%02x%02x%02x/%02x",
                         ppl[0], ppl[1], ppl[2], ppl[3], ppl[4]);
    trace(DBG_NET, "wrong packet size, pkt_len=%d pl_len=%d",
                   pkt_len, payload_len);
    return -2;	// unknown packet type
  }

  /*
   * fill the Payload
   */
  resetPayload();
  len = payload_len - payload_hdr_size;
  memcpy(data, ppl+payload_hdr_size, len);

  return pkt_len;
}

/**
 * Receives a RTCP packet
 */
int recvRTCPPacket(struct sockaddr_in *sender, uint8_t *pkt, const int pkt_len)
{
  rtcp_common_t rtcp_hdr;
  rtcp_t *end;
#ifdef CHECK_RTCP_VALIDITY
  rtcp_t *r;
#endif
  uint16_t length = 0;
  uint32_t ssrc = 0;
  Channel *pchan = Channel::getCurrentChannel(); // maybe indeterminated

  /* we are supposed to receive compounds RTCP packets */
  for (int len = 0; len < pkt_len ; ) {
    memcpy(&rtcp_hdr, pkt+len, sizeof(rtcp_common_t));
    length = ntohs(rtcp_hdr.length);
    //PD end = (rtcp_t *) ((uint32_t *) &rtcp_hdr + (length<<2) + sizeof(rtcp_common_t));
    len += sizeof(rtcp_common_t);

#ifdef CHECK_RTCP_VALIDITY
    /* Check RTCP validity */
    r = (rtcp_t *) &rtcp_hdr;
    do
      r = (rtcp_t *) ((uint32_t *) r + ntohs(r->common.length + sizeof(rtcp_common_t)));
    while (r < end && r->common.version == RTP_VERSION);
    if (r != end) {
      error("RTCP wrong size: r=%x end=%x length=%d pkt_len=%d sizeof(rtcp_common_t)=%d", r, end, length<<2, pkt_len, sizeof(rtcp_common_t));
      len += (length << 2);
      continue;
    }
#endif

    switch (rtcp_hdr.pt) {

     /* receives a SR */
      case RTCP_SR: {
        rtcp_sr_t sr;
        Source *pso;

        trace(DBG_RTP, "Got SR: length=%d len=%d", length<<2, len);
        memcpy(&sr, pkt+len, sizeof(rtcp_sr_t));
        if ((pso = Source::getSource(ntohl(sr.ssrc))) != NULL)
          pso->sr = sr;
        len += (length << 2);
      }
      break;

      /* receives a SDES */
      case RTCP_SDES: {
        SdesItem *sitem;
        Source *pso;
        uint8_t *p = pkt + len;

        end = (rtcp_t *) ((uint32_t *) p + (length<<2) + sizeof(rtcp_common_t));
        memcpy(&ssrc, p, sizeof(ssrc));
        p += sizeof(ssrc);
        trace(DBG_RTP, "Got SDES: ssrc=%x pkt_len=%d length=%d len=%d p=%x",
              ntohl(ssrc), pkt_len, length<<2, len, p);
        if ((pso = Source::getSource(ntohl(ssrc))) != NULL) { /* source found */
          sitem = &pso->sdes;	// first sitem which ever exists

          /* parses SDES chunks */
          while (p < (uint8_t *) end) {
            trace(DBG_RTP, "SDES: sitem=%p p=%p end=%p", sitem, p, end);
            sitem->si_type = *p++;
            sitem->si_len = *p++;
            if (sitem->si_str == NULL) {	// asumption
              /* alloc string space */
              if ((sitem->si_str = (uint8_t *) calloc(1, sitem->si_len + 1)) == NULL) {
                trace(DBG_RTP, "SDES: can't alloc sitem->si_str");
                return -1;
              }
            }
            memcpy(sitem->si_str, p, sitem->si_len);
            sitem->si_str[sitem->si_len] = 0; // now string null terminated
            trace(DBG_RTP, "SDES: %s [%d]", sitem->si_str, sitem->si_len);
            p += sitem->si_len;	// next item
            if (*p == RTCP_SDES_END) {
              trace(DBG_RTP, "end SDES: p=%p end=%p", p, end);
              break;	// end of SDES packet
            }

            if (sitem->si_next == NULL) {
              /* alloc next item */
              if ((sitem->si_next = (SdesItem *) calloc(1, sizeof(SdesItem))) == NULL) {
                error("SDES: can't alloc next sitem");
                return -1;
              }
            }
            sitem = sitem->si_next;
            trace(DBG_RTP, "SDES: sitem=%p", sitem);
          }
        }
        len += (length << 2);
      }
      break;

     /* receives a BYE */
      case RTCP_BYE: {
        memcpy(&ssrc, pkt+len, sizeof(ssrc));
        trace(DBG_RTP, "Got BYE: ssrc=%x", ntohl(ssrc));
#ifdef DELETESESSION
        Session::deleteSessionBySsrc(ntohl(ssrc));
#else
        pchan->session->deleteSourceBySsrc(ntohl(ssrc));
#endif
        len += (length << 2);
      }
      break;

     /* receives a APP */
      case RTCP_APP:
        trace(DBG_RTP, "Got APP");

     /* receives something unknown */
      default:
        len += (length << 2);
        error("Got RTCP unknown type: pt=%u, pkt_len=%d len=%d",
                         rtcp_hdr.pt, pkt_len, len);
        trace(DBG_FORCE, "%02x%02x%02x%02x %02x%02x%02x%02x %02x%02x%02x%02x",
                         pkt[0], pkt[1], pkt[2], pkt[3],
                         pkt[4], pkt[5], pkt[6], pkt[7],
                         pkt[8], pkt[9], pkt[10], pkt[11]);
        return -3;
    }
  }
  return 0;
}
