#ifndef NET_H
#define NET_H

#include "rtp.h"
#include "payload.h"


/**
 * NetObjectId Class
 *
 * A NetObjectId identifies a distributed netobject on 64 bits
 * (src_id + port_id + obj_id)
 */
class NetObjectId {
 public:
  uint32_t src_id;	///< IPaddr src
  uint16_t port_id;	///< port
  uint16_t obj_id;	///< object id -- network format

  class NetObject * getNetObject();
  /**<
   * Gets a NetObject by name,
   * returns NULL if netobject doesn't exist.
   * Header vers noid se fait en lisant directement dans la struct
   * Naming is done by createNetObject ou createNetObjectFromString.
   */

  bool equalNetObjectId(const NetObjectId noid2);
  /**< Returns 0 if differents, other if equals */

  char * getNetNameById();
  /**< Builds a concatened string name */

  int filterQuery();
  /**< Heuristic to avoid to send bunch of Query */

  void sendQueryNoid(const struct sockaddr_in *sender);

  void sendDeleteNoid(const struct sockaddr_in *sender);
};

void setMySsrcId(const uint32_t ssrc_id);
void setMyMgrSsrcId(const uint32_t ssrc_id);
void setMyHostId(const uint32_t host_id);
void setMyPortId(const uint16_t port_id);
void setMyObjId(const uint16_t obj_id);
uint32_t getMySsrcId(void);
uint32_t getMyMgrSsrcId(void);
uint32_t getMyHostId(void);
uint16_t getMyPortId(void);
uint16_t getMyObjId(void);

/**
 * NetProperty Class
 */
class NetProperty {
 public:
  bool responsible;		///< flag responsibility
  int16_t version;		///< version number
  struct timeval last_seen;	///< last seen date
  struct timeval assume_at;	///< assume at date
  float min_assume_delay;	///< min in secs
  float max_assume_delay;	///< max in secs

  NetProperty();
  virtual ~NetProperty();

  virtual void resetDates();
  /**<
   * Computes a new date for the assume_at of the property
   * and sets the last_seen at "now"
   */

  virtual void setResponsible(bool flag);
};

/**
 * NetObject Class
 */
class NetObject {
 public:
  class NetObjectId noid;	///< NetObject id
  uint8_t type;			///< NetObject type
  uint8_t permanent;		///< permanent or valatile (not a true bool)
  uint8_t nbprop;		///< number of properties
  class WObject *pobject;	///< pointer on the WObject
  class NetProperty *netprop;	///< netobject properties
  class NetObject *next;	///< next
  class NetObject *prev;	///< prev

  NetObject();
  /**< Constructor for local */
  NetObject(WObject *po, uint8_t nprop, uint16_t oid);
  /**< Constructor for local ppermanent with args */
  NetObject(WObject *po, uint8_t nprop);
  /**< Constructor for local volatile with args */
  NetObject(WObject *po, uint8_t nprop, NetObjectId _noid);
  /**< Constructor for replica */

  virtual ~NetObject();
  /**< Destructor */

  void initCommun();
  /**< Common to constructors */

  void deleteNetObjectFromList();
  /**<
   * Removes the NetObject from the list.
   * To do necessarly before the final delete.
   */

  //
  // exports to WO
  //
  void createNetObject(const bool netbehave);
  /**<
   * Creates a new local netobject.
   * Then we can do getNetObject, declareObjDelta.
   * One declareObjCreation is wish latter, when props are set.
   */

  void setNetObjectId();
  /**<
   * Initializes a new NetObject.
   * Assigns a unique identifier to each Vreng local netobject
   * whether if be a networked object or not. 
   * Now we can do getObjId and declareObjDelta.
   * It is preferable (perfs) to do a declareObjCreation
   * when all the properties are initialized.
   */

  void createNetObjectIdFromString(const char *str, bool netbehave);
  /**<
   * Build a NetObject name from a string "scene_id/obj_id", both uint16_t > 0
   * Used by getNetObject and declareObjDelta.
   * A declareObjCreation on such netobject produces a fatal.
   */

  void declareObjCreation();
  /**<
   * We assume the header yet initialized,
   * should (perfs) be called after the NetObject naming createNetObject()
   * and the initialization of properties.
   * To call for each new objects.
   */

  void declareObjDelta(const uint8_t prop_id);
  /**<
   * Update netobject version.
   * To call at each modification, eg. after a property value changes.
   */

  void declareDeletion();
  /**<
   * Destroy the netobject (local copy), netobject must be valid (name).
   * To call when we want destroy the object before a deleteNetObject.
   */

  void setSrcId(const uint32_t src_id);
  void setPortId(const uint16_t port_id);
  void setObjId(const uint16_t obj_id);
  uint32_t getSrcId();
  uint16_t getPortId();
  uint16_t getObjId();

  // Persist
#if 0 //notused
  void declarePersistSet();
  void declarePersistInfo();
  void declarePersistReset();
  void declareWorldSet();
  void declareWorldInfo();
  void declareWorldReset();
#endif //notused

  // Send
  void sendCreate(const struct sockaddr_in *to);
  /**<
   * Send a '0x01' packet to mentionned unicast address for the current object
   * we don't touch version and dates
   * Format: '0x01' (c), netobject type (c), netobject name (n), netbehave (c)
   *         then the initialization payload, obtained by a putAllProperties()
   *         then the version vector (nprop *pn).
   */

  void sendDelta(const uint8_t prop_id);
  /**<
   * Send a multicast packet of type '0x02' = Delta,
   * on the property prop_id of the netobject.
   * We do a resetDates (not version).
   * Format: '0x02' (c), netobject name (n), property number (c), version (h),
   *         netobject's properties.
   */

  void sendQuery(const struct sockaddr_in *to);
  /**<
   * Send a Query '0x03' packet toi the unicast sender.
   * We try to avoid to send too many.
   * Format: '0x03' (c), netobject name (n).
   * Called by incomingDelta.
   */

  void sendDelete(const struct sockaddr_in *to);
  /**<
   * Send a Delete '0x04' packet to the unicast sender.
   * Format: '0x04' (c), netobject name (n).
   */

#if 0 //notused
  void sendPersistSet(const struct sockaddr_in *to);
  void sendPersistInfo(const struct sockaddr_in *to);
  void sendPersistReset(const struct sockaddr_in *to);
  void sendWorldSet(const struct sockaddr_in *to);
  void sendWorldInfo(const struct sockaddr_in *to);
  void sendWorldReset(const struct sockaddr_in *to);
#endif //notused

  void insertNetObject();
  /**< Inserts netobject in head of NetObject list */

  void initProperties(bool responsible);
  /**<
   * Initializes responsibilities (false=no, true=yes).
   * Dates handled by resetDates.
   */

  uint8_t propertiesNumber();
  static uint8_t propertiesNumber(const uint8_t type_id);
  /**<
   * Returns le number of properties for this type.
   * Numbered from 0 to propertiesnumber-1.
   * Typically called after an incomingCreate.
   */

  void setPropertiesNumber(const uint8_t _nbprop);
  static void setPropertiesNumber(const uint8_t type_id, const uint8_t _nbprop);
  /**<
   * Sets the number of properties for this type
   */

  void getProperty(const uint8_t prop, Payload *pp);
  /**<
   * Modifies the property (its local copy) get from Network.
   * The new value is in the payload.
   * We must return the new read position given by getPayload.
   * Typically called after an incomingDelta.
   */

  void putProperty(const uint8_t prop, Payload *pp);
  /**<
   * Puts the property (its local copy) to be sent to Network.
   * Then we must write the payload which return the written length.
   * Typically called to fill a payload before a sendDelta.
   */

  void getAllProperties(Payload *pp);
  /**<
   * Gets all properties from the network.
   */

  void putAllProperties(Payload *pp);
  /**<
   * Puts all properties of this netobject. 
   * The payload is initialized before, and filled here.
   * Called to known the Payload after one declareObjCreation.
   */

  void requestDeletionFromNetwork();
  /**<
   * Supprime object du monde, si object n'est pas le local user
   * The sequence must include deleteNetObject.
   * 1) faire le menage et afficher tout ce qui est necessaire.
   * 2) si la decision a ete prise localement: declareDeletion
   * 3) deleteNetObject
   *    le nom devient invalide, plus aucun declare n'est possible
   * 4) faire le delete object final
   */

  bool isResponsible();
  bool isPermanent();

  static void clearNetObjectsList();
  static NetObject * getNetObjectsList();

  WObject * getWObjectByNoid();
  /**< Finds a WObject pointer by its noid */

  static NetObject *replicateObject(uint8_t type, class NetObjectId noid, Payload *pp);
  /**<
   * Dispatching the replicated object.
   * Creates a replication (local copy) of the object.
   * All its fields will be intilisalized by the WObject itself
   * using appropriated getProperty().
   * Typically called after an incomingCreate().
   */

};


//
// exports to GUI
//

void NetQuit(void);

int NetTimeout(void);
/**<
 * Check if some responsibilities must not to be taken when a timeout occurs.
 * Do the heatbeat refreshing.
 * Check if some netobjects must not to be deleted.
 * Returns the delay after which we want to be awaked (ms) before recall.
 * Exported to GUI.
 */

void NetIncoming(const int fd);
/**<
 * Awake the net when data is available on fd.
 * Exported to GUI.
 */


char * getGroup(const char *chanstr, char *groupstr);
uint16_t getPort(const char *chanstr);
uint8_t getTtl(const char *chanstr);
uint8_t currentTtl(void);
char * createMc(char *chan_str, char *groupstr, uint16_t port, uint8_t ttl);
char * newMc(char *chan_str);

//
// Sockets
//
int socketDatagram();
int socketStream();
int setReuseAddr(const int sock);
int setTcpNoDelay(const int sock);
uint16_t getSrcPort(const int sock);
int handleBlocking(const int sock, bool block);
int setBlocking(const int sock);
int setNoBlocking(const int sock);
int handleLoopback(const int sock, const uint8_t loop);
int setLoopback(const int sock);
int setNoLoopback(const int sock);
int setScope(const int sock, const uint8_t ttl);
int addMembership(const int sock, const void *pmreq);
int dropMembership(const int sock, const void *pmreq);
int createUcastSocket(const uint32_t uni_addr, const uint16_t port);
int createSendSocket(const uint8_t ttl);
bool isMulticastAddress(const uint32_t address);

//
// Sdr
//
void sdrInit(void);

#endif // NET_H
