#include "global.h"

#include "net.h"
#include "netdefs.h"	// SD_R_RTP
#include "pkt.h"	// sendPayload
#include "payload.h"	// Payload
#include "rtpsess.h"	// Session
#include "channel.h"

#include "world.h"	// setChannelName
#include "gui.h"	// GuiAddInputTable


/* local: fds */
static int *tabFd;
static int *tabManagerFd;
static int cntFd;

/* local: channels */
static Channel *currentChannel = NULL;	// current channel
static Channel *managerChannel = NULL;	// manager channel
static Channel *channelList = NULL;	// channel head
static uint8_t channelCount;		// channel count
static Vac     *vacList = NULL;	        // vac head

static char vrum_addr[16] = DEF_VRUM_ADDR;


void initReflector()
{
  /* resolve reflector address */
  struct hostent *hp;
  if ((hp = my_gethostbyname(DEF_VRUM_SERVER, AF_INET)) != NULL) {
    strcpy(vrum_addr, inet4_ntop(hp->h_addr_list[0]));
    my_free_hostent(hp);
  }
  if (options->reflector)
    sdrInit();
}

int Vac::connectVac()
{
  int sdvac;

  if ((sdvac = socketStream()) < 0) {
    error("socket vacs");
    return -1;
  }
  /* resolve vacs address */
  struct hostent *hp;
  if ((hp = my_gethostbyname(DEF_VACS_SERVER, AF_INET)) == NULL) {
    //error("resolve vacs");
    return -1;
  }

  struct sockaddr_in savac;

  memset(&savac, 0, sizeof(savac));
  savac.sin_family = AF_INET;
  savac.sin_port = htons(VACS_PORT);
  memcpy(&savac.sin_addr, hp->h_addr_list[0], hp->h_length);
  my_free_hostent(hp);

  if (connect(sdvac, (const struct sockaddr *) &savac, sizeof(savac)) < 0) {
    //perror("connect vacs");
    return -1;
  }
  return sdvac;
}

void Vac::getList()
{
  int sdvac;

  // connect to the vacs server
  if ((sdvac = Vac::connectVac()) < 0)
    return;

  char request[8] = "list";
  write(sdvac, request, strlen(request));

  vacList = new(Vac);
  strcpy(vacList->url, "manager");
  strcpy(vacList->channel, DEF_MANAGER_CHANNEL);
  vacList->next = NULL;
  Vac *prev = vacList;

  char vacline[URL_LEN + CHAN_LEN + 2];
  int r;
  while ( (r = recv(sdvac, vacline, sizeof(vacline), 0)) > 0 ) {
    //printf("readvac: r=%d l=%s\n", r, vacline);
    char *p = strtok(vacline, " ");
    Vac *vac = new(Vac);
    strcpy(vac->url, p);
    p = strtok(NULL, " ");
    strcpy(vac->channel, p);
    vac->next = NULL;
    prev->next = vac;
    prev = vac;
    //printf("readvac: u=%s c=%s\n", vac->url, vac->channel);
  }
  close(sdvac);
}

bool Vac::resolveWorldUrl(const char *_url, char *chanstr)
{
  int sdvac;

  // connect to the vacs server
  if ((sdvac = Vac::connectVac()) < 0)
    return false;

  // send the request "resolve"
  char request[URL_LEN + 8];
  memset(request, 0, sizeof(request));
  sprintf(request, "resolve %s", _url);
  write(sdvac, request, strlen(request));

  char chanline[CHAN_LEN + 2];
  if (recv(sdvac, chanline, sizeof(chanline), 0) > 0) {
    strcpy(chanstr, chanline);
    bool cached = false;
    if (vacList && vacList->next) {
      Vac *w, *wlast;
      for (w = vacList->next; w ; w = w->next) {
        if (! strcmp(_url, w->url)) {
          cached = true;
          break;
        }
        wlast = w;
      }
      if (! cached) {
        Vac *vac = new(Vac);
        strcpy(vac->url, _url);
        strcpy(vac->channel, chanline);
        vac->next = NULL;
        wlast->next = vac;
        trace(DBG_FORCE, "resolveWorld: channel=%s", vac->channel);
      }
    }
    close(sdvac);
    return true;
  }
  else {
    trace(DBG_FORCE, "resolveWorld: channel NULL");
    strcpy(chanstr, DEF_VRE_CHANNEL);
    close(sdvac);
    return false;
  }
}

bool Vac::getChannel(const char *_url, char *chanstr)
{
  if (vacList && vacList->next) {
    for (Vac *w = vacList->next; w ; w = w->next) {
      if (! strcmp(_url, w->url)) {
        strcpy(chanstr, w->channel);
        return true;
      }
    }
  }
  return false;
}

bool Vac::getUrlAndChannel(const char *name, char *_url, char *chanstr)
{
  if (vacList && vacList->next) {
    for (Vac *w = vacList->next; w ; w = w->next) {
      if (strstr(w->url, name)) {
        strcpy(_url, w->url);
        strcpy(chanstr, w->channel);
        return true;
      }
    }
  }
  return false;
}


/** Network Initialization */
void Channel::initNetwork()
{
  static bool first = true;

  if (first) {
    startTime(&ptime_net);
    initReflector();
    first = false;
  }
  channelList = NULL;
  Session::clearSessionsList();
  NetObject::clearNetObjectsList();
}

void NetQuit(void)
{
  statNetwork();
}

/** Create a new Channel */
Channel::Channel()
{
  new_channel++;
  ssrc = 0;
  group = 0;
  port = 0;
  ttl = 0;
  for (int i=0; i<5; i++)
    sd[i] = -1;
  session = NULL;

  if (channelList == NULL) {
    channelList = this;
    next = NULL;
  }
  else {
    next = channelList;
    channelList = this;
  }
}

/** Join group */
int Channel::joinGroup(const int sock)
{
  memset(&mreq, 0, sizeof(struct ip_mreq));
  mreq.imr_multiaddr.s_addr = group;
  mreq.imr_interface.s_addr = htonl(INADDR_ANY);
  if (addMembership(sock, (void *) &mreq) < 0)
    return -1;
  return sock;
}

/** Leave group */
int Channel::leaveGroup(const int sock)
{
  if (dropMembership(sock, (void *) &mreq) < 0)
    return -1;
  return sock;
}

/** Create a Multicast listen socket on the channel defined by group/port */
int Channel::createMcastRecvSocket(struct sockaddr_in *sa)
{
  int sock;

  if ((sock = socketDatagram()) < 0)
    return -1;

  if (setReuseAddr(sock) < 0)
    perror("reuse failed");
  if (bind(sock, (struct sockaddr *) sa, sizeof(struct sockaddr_in)) < 0)
    error("bind: %s port=%d", strerror(errno), ntohs(sa->sin_port));

#ifdef PROXY_MULTICAST
 /* test if unicast-multicast proxy */
  if (mup.active) {
    if (sendCmdToProxy("join", sock, this, sa) < 0)
      return -1;
    return createProxyDataSocket(sa);
  }
#endif

  joinGroup(sock);
  return sock;
}

void Channel::closeMcastSocket()
{
#ifdef PROXY_MULTICAST
  if (mup.active)
    leaveGroupByProxy(SA_RTP, sd[SD_R_RTP], this);
  else
#endif
    leaveGroup(sd[SD_R_RTP]);

#ifdef PROXY_MULTICAST
  if (mup.active)
    leaveGroupByProxy(SA_RTCP, sd[SD_R_RTCP], this);
  else
#endif
    leaveGroup(sd[SD_R_RTCP]);

  close(sd[SD_R_RTP]);
  sd[SD_R_RTP] = -1;
  close(sd[SD_W_RTP]);
  sd[SD_W_RTP] = -1;
  close(sd[SD_R_RTCP]);
  sd[SD_R_RTCP] = -1;
  close(sd[SD_W_RTCP]);
  sd[SD_W_RTCP] = -1;
}

void Channel::closeUcastSocket()
{
  close(sd[SD_R_UDP]);
  sd[SD_R_UDP] = -1;
}

Channel * Channel::getCurrentChannel()
{
  if (! currentChannel)
    error("getCurrentChannel: NULL");
  return currentChannel;
}

Channel * Channel::getManager()
{
  return managerChannel;
}

/** Decode the string format "group[/port[/ttl]]" */
void Channel::decode(const char *chan_str, uint32_t *group, uint16_t *port, uint8_t *ttl)
{
  char *ttlstr, *portstr, *groupstr;
  char *chanstr = strdup(chan_str);

  groupstr = chanstr;
  portstr = strchr(chanstr, '/');
  if (portstr != NULL)
    *portstr = '\0';
  *group = inet_addr(groupstr);
  if (portstr != NULL) {
    portstr++;
    ttlstr = strchr(portstr, '/');
    if (ttlstr != NULL)
      *ttlstr = '\0';
    *port = atoi(portstr);
    if (ttlstr != NULL) {
      *ttlstr = '\0';
      ttlstr++;
      *ttl = atoi(ttlstr);
    }
  }
  free(chanstr);
}

/** Channel naming */
void Channel::naming()
{
  char hostname[MAXHOSTNAMELEN];
  gethostname(hostname, sizeof(hostname)-1);

  struct hostent *ph;
  if ((ph = my_gethostbyname(hostname, AF_INET)) == NULL)
    return;
  struct in_addr *pa;
  if ((pa = (struct in_addr*) (ph->h_addr)) == NULL) {
    my_free_hostent(ph);
    return;
  }
  setMyHostId(pa->s_addr);   
  my_free_hostent(ph);
 
  /* first packet to learn my_port_id */
  Payload pp;	// dummy payload

  pp.sendPayload(sa[SA_RTP]);	// needed for naming (port number)

#if NEEDLOOPBACK
  setMyPortId((uint16_t) (getMySsrcId() & 0x00007FFF));
#else
  setMyPortId(getSrcPort(sd[SD_W_RTP]));
  if (getMyPortId() == 0)
    setMyPortId((uint16_t) (getMySsrcId()) & 0x00007FFF);
#endif

  pp.sendPayload(sa[SA_RTCP]);	// needed for proxy (source port)
  trace(DBG_NET, "my_port_id = %d", getMyPortId());
  setMyObjId(0);
}

/** Create a Channel */
int Channel::createChannel(const char *chan_str, int **pfds)
{
  cntfds = 0;	// number of fd
  
  if (! chan_str)
    return 0;
  decode(chan_str, &group, &port, &ttl);

  port &= ~1;	// RTP compliant: even port

  /*
   * RTP channel
   */
  memset(&sa_rtp, 0, sizeof(struct sockaddr_in));
  memset(&sa_rtp_r, 0, sizeof(struct sockaddr_in));
  sa_rtp.sin_family = AF_INET; 
  sa_rtp_r.sin_family = AF_INET; 
  if (options->reflector) {
    sa_rtp_r.sin_port = 0;
    sa_rtp_r.sin_addr.s_addr = htonl(INADDR_ANY); 
    sa_rtp.sin_port = htons(port - VRUM_PORT_OFFSET);
    inet4_pton(vrum_addr, &sa_rtp.sin_addr.s_addr);
    sa[SA_RTP_R] = &sa_rtp_r;
  }
  else {	// native Multicast
    sa_rtp.sin_port = htons(port);
    sa_rtp.sin_addr.s_addr = group; 
  }
  sa[SA_RTP] = &sa_rtp;

  if (options->reflector) {
    if ((sd[SD_R_RTP] = createMcastRecvSocket(sa[SA_RTP_R])) < 0) {
      warning("can't open receive socket RTP reflector");
      return 0;
    }
  }
  else {	// native Multicast
    if ((sd[SD_R_RTP] = createMcastRecvSocket(sa[SA_RTP])) < 0) {
      warning("can't open receive socket RTP");
      return 0;
    }
  }

  sd[SD_W_RTP] = createSendSocket(ttl);
  cntfds += 2;
  trace(DBG_RTP, "sa[SA_RTP]=%p", sa[SA_RTP]);
  trace(DBG_RTP, "sd[SD_R_RTP]=%d, sd[SD_W_RTP]=%d",
                  sd[SD_R_RTP], sd[SD_W_RTP]);
  trace(DBG_RTP, "SA_RTP: port=%d", sa[SA_RTP]->sin_port);

  /*
   * RTCP channel
   */
  memset(&sa_rtcp, 0, sizeof(struct sockaddr_in));
  memset(&sa_rtcp_r, 0, sizeof(struct sockaddr_in));
  sa_rtcp.sin_family = AF_INET;
  sa_rtcp_r.sin_family = AF_INET;
  if (options->reflector) {
    sa_rtcp_r.sin_port = 0;
    sa_rtcp_r.sin_addr.s_addr = htonl(INADDR_ANY); 
    sa_rtcp.sin_port = htons(port - VRUM_PORT_OFFSET + 1);
    inet4_pton(vrum_addr, &sa_rtcp.sin_addr.s_addr);
    sa[SA_RTCP_R] = &sa_rtcp_r;
  }
  else {	// native Multicast
    sa_rtcp.sin_port = htons(port + 1);	// not ++port !!!
    sa_rtcp.sin_addr.s_addr = group;
  }
  sa[SA_RTCP] = &sa_rtcp;

  if (options->reflector) {
    if ((sd[SD_R_RTCP] = createMcastRecvSocket(sa[SA_RTCP_R])) < 0) {
      warning("can't open receive socket RTCP reflector");
      return 0;
    }
  }
  else {	// native Multicast
    if ((sd[SD_R_RTCP] = createMcastRecvSocket(sa[SA_RTCP])) < 0) {
      warning("can't open receive socket RTCP");
      return 0;
    }
  }

  sd[SD_W_RTCP] = createSendSocket(ttl);
  cntfds += 2;
  trace(DBG_RTP, "sa[SA_RTCP]=%p", sa[SA_RTCP]);
  trace(DBG_RTP, "sd[SD_R_RTCP]=%d, sd[SD_W_RTCP]=%d",
                  sd[SD_R_RTCP], sd[SD_W_RTCP]);
  trace(DBG_RTP, "SA_RTCP: port=%d", sa[SA_RTCP]->sin_port);

  /* UDP channel */
  sa[SA_UDP] = NULL;

  /*
   * RTP session initialization
   */
  session = new Session();
  if (! session) {
    error("session: can't new");
    return 0;
  }

  uint32_t oldssrc = 0;
  World *world = NULL;
  if ((world = World::getWorldByGroup(group)) != NULL) {
    oldssrc = world->ssrc;
  }
  if ((ssrc = session->createSession(group, port, ttl, oldssrc)) == 0) {
    warning("createChannel: can't create session");
    delete session;
    return 0;
  }

  if (this == managerChannel) {
    setMyMgrSsrcId(ssrc);
    session->mode = MANAGER_MODE;
  }
  else {
    setMySsrcId(ssrc);
    session->mode = WORLD_MODE;

    // WARNING: is world initialized ?
    if (world) {
      world->group = group;
      world->ssrc = ssrc;
    }
    else {
      worlds->group = group;
      worlds->ssrc = ssrc;
    }
  }

  naming();

  *pfds = sd;
  cntFd += cntfds;
  return cntfds;
}

void Channel::switchChannel()
{ 
  sleep(3);
  Channel::getCurrentChannel()->quitChannel();
  Channel::join(World::getChannelName());
}

void Channel::sendBYE()
{
  sendRTCPPacket(sa[SA_RTCP], RTCP_SR);
  sendRTCPPacket(sa[SA_RTCP], RTCP_BYE);
  trace(DBG_RTP, "sendBYE: on channel=%p", this);
}

/** Delete from channelList */
void Channel::deleteFromList()
{
  trace(DBG_NET, "deleteFromList: channel=%p", this);
  if (this == managerChannel)
    return;
  if (channelList == NULL) {
    error("deleteFromList channelList NULL");
    return;
  }

  for (Channel *channel = channelList; channel ; channel = channel->next) {
    if (channel == this) {
      channelList = channel->next;
      currentChannel = channel->next;
      channel->next = NULL;
      if (channel == channelList)
        channelList = NULL;
      return;
    }
  }
  channelList = NULL;
}

void Channel::quitChannel()
{
  /* respect this order! */
  if (this != managerChannel) {
    trace(DBG_RTP, "quitChannel: send BYE on channel=%p", this);
    sendBYE();
    trace(DBG_RTP, "quitChannel: session=%p", session);
    //pd session->statRTP();	// segfault
    delete session;	// delete Session
    session = NULL;
    closeMcastSocket();
    deleteFromList();
  }
#if 0 //pd manager
  else {
    strcpy(worlds->name, "manager");	// HUGLY !
    World::setChannelName(MANAGER_CHANNEL);
    if (session) {
      //pd session->statRTP();	// segfault
      delete session;	// delete Session
      session = NULL;
    }
    closeMcastSocket();
  }
#endif //pd manager
}

/** Channel destructor */
Channel::~Channel()
{
  del_channel++;
  quitChannel();
  GuiRemoveInputTable(cntfds, WORLD_MODE);
  for (int i=0; i < cntfds && channelCount > 0; i++)	{ // debugged by efence
    if (tabFd[i] > 0)
      close(tabFd[i]);
    cntFd--;
  }
  channelCount--;
  trace(DBG_IPMC, "~Channel: channelCount=%d", channelCount);
}

/** join the channel and return the new channel string */
char * Channel::join(const char *chan_str)
{
  static char chanstr[CHAN_LEN];	// FIXME

  currentChannel = new Channel();

  strcpy(chanstr, chan_str);
  newMc(chanstr);
  trace(DBG_NET, "join: currentChannel=%p chan_str=%s -> chanstr=%s", currentChannel, chan_str, chanstr);

  notice("channel: %s", chanstr);
  int cntfd;
  if ((cntfd = currentChannel->createChannel(chanstr, &tabFd)) == 0)
    return NULL;

  GuiAddInputTable(cntfd, tabFd, WORLD_MODE);
  channelCount++;
  trace(DBG_NET, "join: currentChannel=%p channelCount=%d cntfd=%d",
        currentChannel, channelCount, cntfd);
  return chanstr;	// FIXME
}

/** joinManagerChannel: only for vreng client */
char * Channel::joinManager(const char *chan_str)
{
  static char chanstr[CHAN_LEN];

  managerChannel = new Channel();

  strcpy(chanstr, chan_str);
  newMc(chanstr);
  trace(DBG_NET, "joinManager: manager.chan: %s", chanstr);

  int cntfd;
  if ((cntfd = managerChannel->createChannel(chanstr, &tabManagerFd)) == 0)
    return NULL;

  GuiAddInputTable(cntfd, tabManagerFd, MANAGER_MODE);
  trace(DBG_INIT, "joinManager: managerChannel=%p cntfd=%d",
        managerChannel, cntfd);
  return chanstr;
}

#ifdef VRENGD
/** joinDaemon: only for vrengd */
char * Channel::joinDaemon(const char *chan_str)
{
  static char chanstr[CHAN_LEN];

  managerChannel = new Channel();

  strcpy(chanstr, chan_str);
  newMc(chanstr);
  fprintf(stderr, "joinManager: manager.chan: %s\n", chanstr);

  int cntfd;
  if ((cntfd = managerChannel->createChannel(chanstr, &tabManagerFd)) == 0)
    return NULL;

  fprintf(stderr, "joinDaemon: managerChannel=%p cntfd=%d\n",
                  managerChannel, cntfd);
  return chanstr;
}
#endif //VRENGD

/** get a socket fd by sockaddr_in */
int getfdbysa(const struct sockaddr_in *sa, const int i_fd)
{
  int ret = -2;

  if (! sa) {
    error("getfdbysa: sa NULL");
    return -1;
  }
  if (! channelList) {
    error("getfdbysa: channelList NULL");
    return -1;
  }

  for (Channel *pchan = channelList; pchan ; pchan = pchan->next) {
    if (pchan->sa[SA_RTP] == sa || pchan->sa[SA_RTCP] == sa) {
      if (pchan->sd[i_fd] < 0) {
        ret = -1;
        break;
      }
      return(pchan->sd[i_fd]);
    }
  }
  ret = channelList->sd[i_fd]; //octobre
  trace(DBG_NET, "getfdbysa: sa=%x not in table, force fd=%d", sa, ret);
  return ret;
}

int getFdSendRTP(const struct sockaddr_in *sa)
{
  return getfdbysa(sa, SD_W_RTP);
}

int getFdSendRTCP(const struct sockaddr_in *sa)
{
  return getfdbysa(sa, SD_W_RTCP);
}

/** get a socket address by sockaddr_in */
struct sockaddr_in * getsabysa(const struct sockaddr_in *sa, const int i_sa)
{
  if (! sa) {
    error("getsabysa: sa NULL");
    return NULL;
  }
  if (! channelList) {
    error("getsabysa: channelList NULL");
    return NULL;
  }

  Channel *pchan;
  for (pchan = channelList; pchan ; pchan = pchan->next) {
    if (pchan->sa[SA_RTP] == sa || pchan->sa[SA_RTCP] == sa) {
      return pchan->sa[i_sa];
    }
  }
  return NULL;
}

struct sockaddr_in * getSaRTCP(const struct sockaddr_in *sa)
{
  return getsabysa(sa, SA_RTCP);
}

/** get a channel by sockaddr_in */
Channel * Channel::getbysa(const struct sockaddr_in *sa)
{
  Channel *pchan;

  if (! sa) {
    error("getbysa: sa NULL");
    return NULL;
  }
  if (! channelList) {
    error("getbysa: channelList NULL");
    return NULL;
  }

  for (pchan = channelList; pchan ; pchan = pchan->next) {
    if (pchan->sa[SA_RTP] == sa)
      return pchan;
    if (pchan->sa[SA_RTCP] == sa)
      return pchan;
    if (pchan->sa[SA_UDP] == sa)
      return pchan;
  }
  pchan = getCurrentChannel();	// hack
  trace(DBG_NET, "getbysa: sa=%x not in table, force curr channel", sa);
  return pchan;
}
