#include "global.h"

extern "C" {	// MacOSX doesn't declare these functions
extern const char * inet_ntop(int, const void *, char *, size_t);
extern int inet_pton(int, const char *, void *);
}

#undef NEW_MUTEX

#if !defined(VRENGD) && defined(NEW_MUTEX)
static Vpthread_mutex_t new_mutex;
#endif


void initResolve(void)
{
#if !defined(VRENGD) && defined(NEW_MUTEX)
  static bool done = false;

  if (done)
    return;
  initMutex(&new_mutex);
  done = true;
  return;
#endif
}

struct hostent * my_getipnodebyname(const char *hostname, int af)
{
#if HAVE_GETIPNODEBYNAME
  int err;

  struct hostent *hp = getipnodebyname(hostname, af, 0, &err);
  return hp;
#else // !HAVE_GETIPNODEBYNAME
  return NULL;
#endif // !HAVE_GETIPNODEBYNAME
}

struct hostent * my_gethostbyname(const char *hostname, int af)
{
#if HAVE_GETIPNODEBYNAME
  return my_getipnodebyname(hostname, af);
#else // !HAVE_GETIPNODEBYNAME

  struct hostent *hptmp, *hp;

  if ((hptmp = gethostbyname(hostname)) == NULL)
    return NULL;
  trace(DBG_HTTP, "my_gethostbyname: hostname=%s hptmp=%x", hostname, hptmp);

#if defined(NEW_MUTEX)
  lockMutex(&new_mutex);
#endif
  if ((hp =  (struct hostent *) malloc(sizeof(struct hostent))) != NULL) {
    trace(DBG_HTTP, "my_gethostbyname: hp=%x", hp);
    memcpy(hp, hptmp, sizeof(struct hostent));
  }
#if defined(NEW_MUTEX)
  unlockMutex(&new_mutex);
#endif
  return hp;
#endif // !HAVE_GETIPNODEBYNAME
}

struct hostent * my_gethostbyname_r(const char *hostname, int af)
{
#if HAVE_GETIPNODEBYNAME
  return my_getipnodebyname(hostname, af);
#else // !HAVE_GETIPNODEBYNAME

  struct hostent *hptmp = NULL, *hp;

#if HAVE_GETHOSTBYNAME_R
  struct hostent result;
  int err;
  char buf[512];

#if defined(LINUX) || defined(HPUX)
  gethostbyname_r(hostname, (struct hostent *) &result, buf, sizeof(buf), (struct hostent **) &hptmp, &err);
#else
  hptmp = gethostbyname_r(hostname, &result, buf, sizeof(buf), &err);
#endif

#else // !HAVE_GETHOSTBYADDR_R
  hptmp = gethostbyname(hostname);
#endif // !HAVE_GETHOSTBYADDR_R

  if (! hptmp)
    return NULL;
  if ((hp =  (struct hostent *) malloc(sizeof(struct hostent))) != NULL)
    memcpy(hp, hptmp, sizeof(struct hostent));
  return hp;
#endif // !HAVE_GETIPNODEBYNAME
}

struct hostent * my_getipnodebyaddr(const char *addr, int af)
{
#if HAVE_GETIPNODEBYADDR
  int err;

  unsigned int len = (af == AF_INET) ? 4 : 16;
  struct hostent *hp = getipnodebyaddr(addr, len, af, &err);
  return hp;
#else // !HAVE_GETIPNODEBYADDR
  return NULL;
#endif // !HAVE_GETIPNODEBYADDR
}

struct hostent * my_gethostbyaddr(const char *addr, int af)
{
#if HAVE_GETIPNODEBYADDR
  return my_getipnodebyaddr(addr, af);
#else // !HAVE_GETIPNODEBYADDR

  struct hostent *hptmp, *hp;

  if ((hptmp = gethostbyaddr(addr, sizeof(struct in_addr), af)) == NULL)
    return NULL;
  if ((hp =  (struct hostent *) malloc(sizeof(struct hostent))) != NULL)
    memcpy(hp, hptmp, sizeof(struct hostent));
  return hp;
#endif // !HAVE_GETIPNODEBYADDR
}

struct hostent * my_gethostbyaddr_r(const char *addr, int af)
{
#if HAVE_GETIPNODEBYADDR
  return my_getipnodebyaddr(addr, af);
#else // !HAVE_GETIPNODEBYADDR

  struct hostent *hptmp = NULL, *hp;

#if HAVE_GETHOSTBYADDR_R
  struct hostent result;
  int err;
  char buf[512];

#if defined(LINUX) || defined(HPUX)
  gethostbyaddr_r(addr, sizeof(struct in_addr), af, (struct hostent *) &result, buf, sizeof(buf), (struct hostent **) &hptmp, &err);
#else
  hptmp = gethostbyaddr_r(addr, sizeof(struct in_addr), af, &result, buf, sizeof(buf), &err);
#endif

#else // !HAVE_GETHOSTBYADDR_R
  hptmp = gethostbyaddr(addr, sizeof(struct in_addr), af);
#endif // !HAVE_GETHOSTBYADDR_R

  if (! hptmp)
    return NULL;
  if ((hp =  (struct hostent *) malloc(sizeof(struct hostent))) != NULL)
    memcpy(hp, hptmp, sizeof(struct hostent));
  return hp;
#endif // !HAVE_GETIPNODEBYADDR
}

struct servent * my_getservbyname(const char *service)
{
  struct servent *sptmp, *sp;

  if ((sptmp = getservbyname(service, NULL)) == NULL)
    return NULL;
  if ((sp =  (struct servent *) malloc(sizeof(struct servent))) != NULL)
    memcpy(sp, sptmp, sizeof(struct servent));
  return sp;
}

struct servent * my_getservbyname_r(const char *service)
{
  struct servent *sptmp = NULL, *sp;

#if HAVE_GETSERVBYNAME_R
  struct servent s_r;
  char buf[512];
#if defined(LINUX) || defined(HPUX)
  getservbyname_r(service, NULL, &s_r, buf, sizeof(buf), (struct servent **) &sptmp);
#else
  sptmp = getservbyname_r(service, NULL, &s_r, buf, sizeof(buf));
#endif

#else // !HAVE_GETSERVBYNAME_R
  sptmp = getservbyname(service, NULL);
#endif // !HAVE_GETSERVBYNAME_R

  if (! sptmp)
    return NULL;
  if ((sp =  (struct servent *) malloc(sizeof(struct servent))) != NULL)
    memcpy(sp, sptmp, sizeof(struct servent));
  return sp;
}

void my_free_hostent(struct hostent *hp)
{
#if HAVE_FREEHOSTENT
  freehostent(hp);
#else
  free(hp);
#endif
}

/** inet_ntop, inet_ntoa */
const char * my_inet_ntop(const int af, const void *addr)
{
#if HAVE_INET_NTOP
  char dst[MAXHOSTNAMELEN];

  return inet_ntop(af, addr, dst, sizeof(dst));
#else
  return inet_ntoa((struct in_addr &) addr);
#endif
}

/** inet_pton, inet_addr */
int my_inet_pton(const int af, const char *name, void *addr)
{
#if HAVE_INET_PTON
  return inet_pton(af, name, addr);
#else
  *((int *)addr) = inet_addr(name);
  return (int) addr;
#endif
}

const char * inet4_ntop(const void *addr)
{
  return my_inet_ntop(AF_INET, addr);
}

int inet4_pton(const char *name, void *addr)
{
  return my_inet_pton(AF_INET, name, addr);
}

const char * inet6_ntop(const void *addr)
{
  return my_inet_ntop(AF_INET6, addr);
}

int inet6_pton(const char *name, void *addr)
{
  return my_inet_pton(AF_INET6, name, addr);
}
