/*
   File     : http.c
   Author   : Lionel ULMER
   Creation : 19/02/97
   Updates  : 20/12/97 - Philippe Dax

   "utils" network: resolver, connection to a HTTP server, URL parsing.
*/

#include "global.h"
#include "texture.h"	// TextureCacheEntry
#include "http.h"


static uint8_t proxy=0, noproxy=0;
static uint16_t portproxy;
static char *domnoproxy, *hostproxy;


/** Check if http proxy */
static
void httpProxy(void)
{
  static bool done = false;

  if (done)
    return;

  char *p;

  if (httpproxystr)
    p = httpproxystr;
  else
    p = getenv(HTTP_PROXY);	// syntax: http://hostname:port/
  if (p && *p) {
    char envproxy[90];

    hostproxy = new char[strlen(p)];
    if (p[strlen(p) - 1] == '/')
      p[strlen(p) - 1] = '\0';
    strcpy(envproxy, p);
    p = strrchr(envproxy, ':');
    *p = '\0';
    portproxy = atoi(++p);
    if ((p = strrchr(envproxy, '/')) == NULL)
      sprintf(hostproxy, "http://%s", envproxy);
    else
      strcpy(hostproxy, ++p);
    proxy = 1;
    trace(DBG_HTTP, "proxy=%s:%d", hostproxy, portproxy);
  }
  p = (noproxystr) ? noproxystr : getenv(NO_PROXY);
  if (p && *p) {
    domnoproxy = new char[strlen(p)];
    strcpy(domnoproxy, p);
    noproxy = 1;
  }
  initResolve();
  done = true;
}

static
int openFile(const char *path)
{
#if defined(WIN32) && !defined(CYGWIN32)
  return CreateFile(path, GENERIC_READ, 0, NULL, OPEN_EXISTING,
                    FILE_ATTRIBUTE_NORMAL, NULL);
#else
  return open(path, O_RDONLY);
#endif
}

/** Fill buffer answer */
static
int httpAnswer(int s, char *answer, int max)
{
#if defined(WIN32) && !defined(CYGWIN32)
  return recv(s, answer, max, 0);
#else
  return read(s, answer, max);
#endif
}

/** Send request to the http server */
static
int httpSend(const int fd, const char *buf, const int size)
{
  int sent, r=0;

  for (sent = 0; sent < size; sent += r) {
#if defined(WIN32) && !defined(CYGWIN32)
    if ((r = write(fd, buf + sent, size - sent, 0)) == -1) {
#else
    if ((r = write(fd, buf + sent, size - sent)) == -1) {
#endif
      perror("httpSend: write");
      return BADSEND;
    }
  }
  return 0;
}

/** Connect to server defined by sa */
static
int httpConnect(const struct sockaddr_in *sa)
{
  int sdhttp;
  
  if ((sdhttp = socket(PF_INET, SOCK_STREAM, 0)) < 0) {
    perror("httpConnect: socket");
    return BADSOCKET;
  }
  if (connect(sdhttp, (struct sockaddr *) sa, sizeof(struct sockaddr_in)) < 0) {
    perror("httpConnect: connect");
    close(sdhttp);
    return BADCONNECT;
  }
  return sdhttp;
}

/** URL parsing */
static
int httpParseURL(char *url, char *host, char *scheme, char *path)
{
  int urltype;
  char *p;
  static char prev_host[64];
  
  /* parse scheme */
  if (url == 0 || strlen(url) == 0) {
    *scheme = '\0';
    trace(DBG_HTTP, "BAD URL: %s", url);
    return URLBAD;
  }
  if (!strncmp(url, "http://", 7)) {    
    urltype = URLHTTP;
    strcpy(scheme, "http");
    url += 7;
  } else if (!strncmp(url, "file:", 5)) {
    urltype = URLFILE;
    strcpy(scheme, "file");
    url += 5;
#if 0 //not used
  } else if (!strncmp(url, "telnet://", 9)) {
    urltype = URLTELNET;
    strcpy(scheme, "telnet");
    url += 9;
  } else if (!strncmp(url, "sip:", 4)) {
    urltype = URLSIP;
    strcpy(scheme, "sip");
    url += 4;
  } else if (!strncmp(url, "rtsp://", 7)) {
    urltype = URLRTSP;
    strcpy(scheme, "rtsp");
    url += 7;
  } else if (!strncmp(url, "ftp://", 6)) {
    urltype = URLFTP;
    strcpy(scheme, "ftp");
    url += 6;
#endif //not used
  }
  else {	// relative URL
    trace(DBG_HTTP, "relative URL: %s", url);
    strcpy(scheme, "http");
    strcpy(host, prev_host);
    if (*url != '/')
      sprintf(path, "/%s", url);
    else
      strcpy(path, url);
    return URLHTTP;
  }

  /* then parse host's name */
  if (urltype != URLFILE) {
    p = prev_host;
    while ((*url != ':') && (*url != '/') && (*url != '\0')) {
      *host++ = *url;
      *p++ = *url;	// keep current host
      url++;
    }
    *host = '\0';
    *p = '\0';
  }

  /* then parse pathname */
  if (*url == '\0')
    strcpy(path, "/");
  else // file
    strcpy(path, url);
  return urltype;
}

/** Converts host/port into sockaddr */
int resolve(char *hoststr, char *portstr, struct sockaddr_in *sa)
{
  struct hostent *hp = NULL;
  
  if (! strncmp(hoststr, "localhost", 9))
    strcpy(hoststr, "127.0.0.1");
  if (isdigit((int) *hoststr)) {
    // numeric IPv4 address
    char addripv4[4];
    int a0,a1,a2,a3;

    sscanf(hoststr, "%d.%d.%d.%d", &a0,&a1,&a2,&a3);
    addripv4[0]=a0;
    addripv4[1]=a1;
    addripv4[2]=a2;
    addripv4[3]=a3;

    if ((hp = my_gethostbyaddr_r(addripv4, AF_INET)) == NULL) {
      /* Bletcherous hack in case you do not have a nameserver */
      struct hostent host_hack;  
      char *addr_hack;

      hp = &host_hack;
      hp->h_addrtype = AF_INET;
      addr_hack = addripv4;
      hp->h_addr_list = &addr_hack;
      hp->h_length = 4;
    }
  }
  else {
    // hostname
    if ((hp = my_gethostbyname_r(hoststr, AF_INET)) == NULL)
      return BADNAME;
  }

  uint16_t port;

  if (isdigit((int) *portstr))
    port = htons(atoi(portstr));
  else {
#if 1 //pd
    if (!strcmp(portstr, "http"))
      port = htons(HTTPPORT);
    else
      return BADSERV;
#else
    struct servent *sp = NULL;
    if ((sp = my_getservbyname_r(portstr)) == NULL) {
      if (!strcmp(portstr, "http"))
        port = htons(HTTPPORT);
      else
        return BADSERV;
    }
    else {
      port = sp->s_port;
      free(sp);
    }
#endif //pd
  }

  sa->sin_family = hp->h_addrtype;
  memcpy(&sa->sin_addr, hp->h_addr_list[0], hp->h_length);
  sa->sin_port = port;
  my_free_hostent(hp);
  return 0;
}

/** Http dowloading by threads */
void * httpThread(void *_tl)
{
  httpProxy();

  ThreadLaunch *tl = (ThreadLaunch *) _tl;
  trace(DBG_HTTP, "httpThread: url=%s", tl->url);

  Http *http = new Http();
  if (! http) {
    warning("httpThread: can't new http");
    return NULL;
  }

  struct sockaddr_in sa;
  bool eoheader;
  bool headline;
  char host[MAXHOSTNAMELEN], scheme[8], path[MAXBUF], req[MAXBUF];

  memset(host, 0, sizeof(host));
  memset(scheme, 0, sizeof(scheme));
  memset(path, 0, sizeof(path));

  int urltype = httpParseURL(tl->url, host, scheme, path);
  trace(DBG_HTTP, "urltype=%d, scheme=%s host=%s path=%s", urltype,scheme,host,path);

  switch (urltype) {

  case URLFILE:
    if ((http->fd = openFile(path)) < 0)
      tl->httpReader(tl->handle, NULL);
    else {
      http->offset = -1;
      tl->httpReader(tl->handle, http);
    }
    goto fin;

  case URLHTTP:
    trace(DBG_HTTP, "HTTP: %s://%s%s", scheme, host, path);
again:
    if (proxy && (!noproxy || strstr(host, domnoproxy) == 0)) {
      struct hostent *hp;
      if ((hp = my_gethostbyname(hostproxy, AF_INET)) == NULL) {
        trace(DBG_HTTP, "my_gethostbyname hostproxy=%s", hostproxy);
	proxy = 0;
        noproxy = 0;
        goto again;
      }
      memset(&sa, 0, sizeof(sa));
      sa.sin_family = AF_INET;
      sa.sin_port = htons(portproxy);
      memcpy(&sa.sin_addr, hp->h_addr_list[0], hp->h_length);
      my_free_hostent(hp);
    }
    else {
      int res = -1;
      if ((res = resolve(host, scheme, &sa)) != 0)
        trace(DBG_FORCE, "can't resolve %s err=%d", host, res);
      if (res < 0) {	
        tl->httpReader(tl->handle, NULL);
        goto fin;
      }
    }
    if ((http->fd = httpConnect(&sa)) < 0) {
      trace(DBG_FORCE, "can't connect %s fd=%d", host, http->fd);
      tl->httpReader(tl->handle, NULL);
      goto fin;
    }

    /* send GET request with adding useful infos */
    if (proxy && (!noproxy || strstr(host, domnoproxy) == 0))
      sprintf(req, "GET %s?version=%s&target=%s-%s%s HTTP/1.0\r\n\r\n",
        tl->url, PACKAGE_VERSION, machinename, systemname, releasename);
    else
#if 0 //DARWIN
      sprintf(req, "GET %s HTTP/1.0\r\n\r\n", path);
#else
      sprintf(req, "GET %s?version=%s&target=%s-%s%s HTTP/1.0\r\n\r\n",
        path, PACKAGE_VERSION, machinename, systemname, releasename);
#endif

    if (httpSend(http->fd, req, strlen(req)) < 0) {
      trace(DBG_FORCE, "can't httpSend req=%s fd=%d", req, http->fd);
      tl->httpReader(tl->handle, NULL);
      goto fin;
    }

    /* parse header HTTP/1.0 received from the server */
    eoheader = 0;
    headline = 1; // position first line

    do {
#if 0
      signal(SIGALRM, sigalrm);
      alarm(TIMEOUT_HTTP);
      if (setjmp(jmphttp)) {
        trace(DBG_HTTP, "Timeout on server");
        tl->httpReader(tl->handle, NULL);
        goto fin;
      }
#endif
      if ((http->len = httpAnswer(http->fd, http->buf, MAXNCBUF)) <= 0) {
        eoheader = 1;	// end of http header reached
        break;
      }

      for (http->offset = 0; 1; ) {	
        int i = 0;
        char httpline[MAXBUF];

	while ((http->offset < http->len) && (http->buf[http->offset] != '\n'))
	  httpline[i++] = http->buf[http->offset++];

	if (http->offset < http->len) {
	  if (httpline[0] == '\r') { // test end of header
	    http->offset++;
	    eoheader = 1;
	    break;
	  }
	  httpline[i-1] = '\0'; // null before last (\r)
	  http->offset++;
	  trace(DBG_HTTP, "->%s", httpline);
	  if (headline) {
	    /* first line => get error code */
            int httperr, major, minor;
            char httpmess[MAXBUF];

	    sscanf(httpline, "HTTP/%d.%d %d", &major, &minor, &httperr);
	    strcpy(httpmess, httpline+12);
	    trace(DBG_HTTP, "HTTP-Code_err (%d.%d): %d - %s %s",
		   major, minor, httperr, httpmess, tl->url);
	    if (httperr != HTTP_OK) {	      
	      trace(DBG_FORCE, "HTTP-err: %d - %s %s",
		               httperr, httpmess, tl->url);
	      tl->httpReader(tl->handle, NULL);
	      goto fin;
	    }
	    headline = 0; // headline done
	  }
          if (!strncmp(httpline, "Content-Type: ", 14)) {
            char *p;

            if ((p = strchr(httpline, '/')) != NULL) {
              p++;
              trace(DBG_HTTP, "mime=%s %s", p, tl->url);
              /* only for textures */
              if (tl->handle) {
                TextureCacheEntry *tc = (TextureCacheEntry *) tl->handle;
                strcpy(tc->texmime, p);
              }
            }
	  }
          i = 0;
	}
        else {
	  /* get datas by httpRead() by readCfgDatas() in wmgt */
	  break;
	}
      }
    } while (!eoheader);    

    /* Call the httpReader */
    tl->httpReader(tl->handle, http);
    break;

  case URLTELNET:
#if 0
    if ((http->fd = httpConnect(&sa)) < 0) {
      tl->httpReader(tl->handle, NULL);
      goto fin;
    }
    sprintf(req, "\r\n\r\n");
    if (httpSend(http->fd, req, strlen(req)) < 0) {
      tl->httpReader(tl->handle, NULL);
      goto fin;
    }
    break;
#endif
  case URLSIP:
  case URLRTSP:
  case URLFTP:
    warning("TELNET/FTP/SIP/RTSP protocols are not implemented");
    tl->httpReader(tl->handle, NULL);
    goto fin; 

  default:
    if (urltype)
      notice("unknown scheme = %d", urltype);
    tl->httpReader(tl->handle, NULL);
    goto fin;
  }

  /* End */
fin:
  if (tl)
    delete tl;
  tl = NULL;
  if (http)
    delete http;
  http = NULL;
  return NULL;
}

/** Open a new HTTP connection */
int httpOpen(const char *_url, void (*_httpReader)(void *,Http *), void *_infos, int _blocking)
{
  trace(DBG_HTTP, "httpOpen: %s", _url);

  ThreadLaunch *tl = new ThreadLaunch();
  if (! tl) {
    warning("httpOpen: can't new tl");
    return -1;
  }

  tl->handle = _infos;
  tl->httpReader = _httpReader;
  tl->mode = _blocking;
  strcpy(tl->url, _url);
  if (_blocking == THREAD_BLOCK) {
    return tl->fifoThread();
  }
  else {
    httpThread((void *) tl);
    return 0;
  }
}


static uint8_t htbuf[BUFSIZ];
static int32_t htbuf_pos = 0;
static int32_t htbuf_len = 0;

void httpClearBuf(void)
{
  htbuf_pos = htbuf_len = 0;
}

Http::Http()
{
  fd = -1;
  len = offset = 0;
  httpClearBuf();
}

Http::~Http()
{
  close(fd);
}

int Http::Read(char *_pbuf, int _maxl)
{
  int l = 0, length;

  /* Modifie par Fabrice, l= longueur lue, maxl= longueur restante a recevoir */
  if (offset < 0)
    length = 0;
  else
    length = len - offset;
  if (length > 0) {
    if (length > _maxl)
      length = _maxl;
    memcpy(_pbuf, buf + offset, length);
    l += length;
    _maxl -= length;
    offset += length;
  }
  while (_maxl > 0) {
#if defined(WIN32) && !defined(CYGWIN32)
    ReadFile(fd, _pbuf + l, _maxl, &length, NULL);
#else
    length = read(fd, _pbuf + l, _maxl);
#endif
    if (length < 0) {
      trace(DBG_FORCE, "BADREAD length=%d fd=%d l=%d maxl=%d offset=%d len=%d", length, fd, l, _maxl, offset, len);
      perror("Http::Read");
      return BADREAD;
    }
    if (length == 0)
      break;
    l += length;
    _maxl -= length;
  }
  return l;
}

/** returns a byte or an error */
int Http::GetChar()
{
  if (htbuf_pos >= htbuf_len) {	// eob
    htbuf_pos = 0;
    if ((htbuf_len = Read((char *) htbuf, sizeof(htbuf))) == 0)
      return -1;	// eof
    if (htbuf_len < 0)
      return -2;	// err
  }
  return htbuf[htbuf_pos++];
}

/** returns a byte */
uint8_t Http::GetByte()
{
  if (htbuf_pos >= htbuf_len) {
    htbuf_pos = 0;
    htbuf_len = Read((char *) htbuf, sizeof(htbuf));
  }
  return htbuf[htbuf_pos++];
}

/** returns an int */
int32_t Http::GetInt()
{
  int32_t val;

  if (htbuf_pos >= htbuf_len) {
    htbuf_len = Read((char *) htbuf, sizeof(htbuf));
    htbuf_pos = 0;
  }
  val = htbuf[htbuf_pos++];
  if (htbuf_pos >= htbuf_len) {
    htbuf_len = Read((char *) htbuf, sizeof(htbuf));
    htbuf_pos = 0;
  }
  val |= (htbuf[htbuf_pos++] << 8);
  if (htbuf_pos >= htbuf_len) {
    htbuf_len = Read((char *) htbuf, sizeof(htbuf));
    htbuf_pos = 0;
  }
  val |= (htbuf[htbuf_pos++] << 16);
  if (htbuf_pos >= htbuf_len) {
    htbuf_len = Read((char *) htbuf, sizeof(htbuf));
    htbuf_pos = 0;
  }
  val |= (htbuf[htbuf_pos++] << 24);
  return val;
}

/** returns a float */
float Http::GetFloat()
{
  int32_t nb = GetInt();
  return *((float *) &nb);
}

/** returns a line or an error */
int Http::NextLine(char *line)
{
  int i = 0;
  
  while (1) {
    int c = GetChar();

    if (c == '\n')
      break;
    if (c < 0) {
      line[i] = '\0';
      return 0;
    }
    line[i++] = c;
  }
  line[i++] = '\0';
  return 1;
}

static
bool isEmptyLine(char *l)
{
  if (*l == '#')        // commented line
    return true;
  if (strlen(l) == 0)   // empty line
    return true;
  if (isprint(*l))
    return false;
  return true;
}

/** return a line non empty without comments */
int Http::GetLine(char *line)
{
  do {
    if (! NextLine(line))
      return 0;
  } while (isEmptyLine(line)) ;
  return 1;
}

/** returns an item */
int Http::Fread(char *pbuf, int size, int nitems)
{
  int toread, length = nitems * size;

  while (length > 0) {
    if (htbuf_pos >= htbuf_len) {
      if ((htbuf_len = Read((char *) htbuf, sizeof(htbuf))) < 0)
	return (nitems - (length / size));
      htbuf_pos = 0;
    }
    toread = (length < (htbuf_len - htbuf_pos)) ? length : (htbuf_len - htbuf_pos);
    memcpy(pbuf, htbuf + htbuf_pos, toread);
    htbuf_pos += toread;
    pbuf += toread;
    length -= toread;
  }
  return nitems;
}

/** returns a block */
uint32_t Http::GetBuf(char *buffer, int maxlen)
{
  int32_t in_pos = htbuf_len - htbuf_pos;

  if (in_pos >= maxlen) {
    memcpy(buffer, htbuf, maxlen);
    htbuf_pos += maxlen;
    return maxlen;
  }
  else {
    memcpy(buffer, htbuf, in_pos);
    htbuf_pos = htbuf_len;
    return in_pos + Read((char *) buffer+in_pos, maxlen-in_pos);
  }
}

/** skips an offset */
uint32_t Http::Skip(int32_t skiplen)
{
  int32_t in_pos = htbuf_len - htbuf_pos;

  if (in_pos >= skiplen) {
    htbuf_pos += skiplen;
    return 0;
  }
  else {
    skiplen -= in_pos;
    while (skiplen > 0) {
      if ((htbuf_len = Read((char *) htbuf, sizeof(htbuf))) == 0)
        break;
      if (skiplen >= htbuf_len) {
        skiplen -= htbuf_len;
        htbuf_pos = htbuf_len;
      }
      else {
        htbuf_pos = skiplen;
        skiplen = 0;
      }
    }
    return skiplen;
  }
}

void urlAbs(const char *oldurl, char *newurl)
{
  if ((! strncmp(oldurl, "http://", 7)) || (! strncmp(oldurl, "ftp://", 6)))
    strcpy(newurl, oldurl);
  else
    sprintf(newurl, "http://%s%s", DEF_HTTP_SERVER, oldurl);
}

int url2cache(const char *url, char *cachefile)
{
  const char *pfile = strrchr(url, '/');
  if (pfile == NULL) {
    *cachefile = '\0';
    return 0;
  }
  sprintf(cachefile, "%s/%s", vrengcache, ++pfile);
  return 1;
}

void url2filename(const char *_url, char *filename)
{
  int i = 0;
  char *url = strdup(_url);
  for (char *p = url; *p ; p++, i++) {
    if (*p == '/')
      *p = '%';
    filename[i] = *p;
  }
  filename[i] = '\0';
  free(url);
}

void filename2url(const char *filename, char *url)
{
  int i = 0;
  char *fname = strdup(filename);
  for (char *p = fname; *p ; p++, i++) {
    if (*p == '%')
      *p = '/';
    url[i] = *p;
  }
  url[i] = '\0';
  free(fname);
}
