/*
 *  Copyright (C) 1999 AT&T Laboratories Cambridge.  All Rights Reserved.
 *
 *  This is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This software is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 *  USA.
 */

/*
 * rfbproto.cc - functions to deal with client side of RFB protocol.
 */

#include "global.h"
#include "rfbproto.h"
#include "vncauth.h"

int endianTest = 1;
char *serverCutText = 0;
bool newServerCutText = false;

/* note that the CoRRE encoding uses this buffer and assumes it is big enough
   to hold 255 * 255 * 32 bits -> 260100 bytes.  640*480 = 307200 bytes */
/* also hextile assumes it is big enough to hold 16 * 16 * 32 bits */

//#define BUFFER_SIZE (640*480)
//static char buffer[BUFFER_SIZE];

/*
 * Constructors
 */
VNCRFBproto::VNCRFBproto() : VNC_sock() {
  ShareDesktop = true;
  buffered = 0;
  trace(DBG_FORCE, "VNCRFBproto: 1 not used");
}

VNCRFBproto::VNCRFBproto(char *Servername, int Port) : VNC_sock(Servername, Port) {
  ShareDesktop = true;
  buffered = 0;
  trace(DBG_FORCE, "VNCRFBproto: 2 not used server=%s", Servername);
}

VNCRFBproto::VNCRFBproto(char *Servername, int Port, char* pswdFile) : VNC_sock(Servername, Port) {
  ShareDesktop = true;
  buffered = 0;
  strcpy(passwordFile, pswdFile);
//  Encodings=NULL; // Not used (see rfbproto.h)
}

VNCRFBproto::VNCRFBproto(unsigned int IPAddr, int Port, char* pswdFile) : VNC_sock(IPAddr, Port) {
  ShareDesktop = true;
  buffered = 0;
  trace(DBG_FORCE, "VNCRFBproto: 4 not used passwd=%s", pswdFile);
  strcpy(passwordFile, pswdFile);
}

/*
 * ConnectToRFBServer.
 */
bool VNCRFBproto::ConnectToRFBServer()
{
  int sock = this->VNC_sock.ConnectToTcpAddr();
  if (sock < 0) {
    trace(DBG_FORCE, "ConnectToRFBServer: unable to connect to VNC server");
    return false;
  }
  return this->VNC_sock.SetNonBlocking();
}


/*
 * InitialiseRFBConnection.
 */
bool VNCRFBproto::InitialiseRFBConnection()
{
  rfbProtocolVersionMsg pv;
  rfbClientInitMsg ci;
  int major, minor;
  char *reason, *passwd = NULL;
  CARD32 authScheme, reasonLen, authResult;
  CARD8 challenge[CHALLENGESIZE];

  trace(DBG_VNC, "InitialiseRFBConnection: initializing RFB connection");

  if (!VNC_sock.ReadFromRFBServer(pv, sz_rfbProtocolVersionMsg))
    return false;

  pv[sz_rfbProtocolVersionMsg] = 0;

  if (sscanf(pv, rfbProtocolVersionFormat, &major, &minor) != 2) {
    trace(DBG_FORCE, "InitialiseRFBConnection: not a valid VNC server");
    return false;
  }
  trace(DBG_VNC, "InitialiseRFBConnection: VNC server supports protocol version %d.%d (viewer %d.%d)",
	major, minor, rfbProtocolMajorVersion, rfbProtocolMinorVersion);

  major = rfbProtocolMajorVersion;
  minor = rfbProtocolMinorVersion;
  sprintf(pv, rfbProtocolVersionFormat, major, minor);

  if (!VNC_sock.WriteExact(pv, sz_rfbProtocolVersionMsg))
    return false;
  if (!VNC_sock.ReadFromRFBServer((char *) &authScheme, 4))
    return false;

  authScheme = (CARD32) Swap32IfLE(authScheme);
  switch (authScheme) {

  case rfbConnFailed:
    if (!VNC_sock.ReadFromRFBServer((char *) &reasonLen, 4))
      return false;
    reasonLen = Swap32IfLE(reasonLen);
    reason = (char*) malloc(reasonLen);

    if (!VNC_sock.ReadFromRFBServer(reason, reasonLen))
      return false;
    trace(DBG_FORCE, "VNC connection failed: %.*s", (int)reasonLen, reason);
    free(reason);
    return false;

  case rfbNoAuth:
    trace(DBG_VNC, "No authentication needed");
    break;

  case rfbVncAuth:
    if (!VNC_sock.ReadFromRFBServer((char *) challenge, CHALLENGESIZE))
      return false;
   
    if (this->passwordFile) {//look for a password stored in passwordFile   
      passwd = vncDecryptPasswdFromFile(this->passwordFile);
      if (!passwd) {
        trace(DBG_FORCE, "Cannot read valid password from file \"%s\"",
	        this->passwordFile); 
        return false;
      }
    }
    if (! passwd || (! strlen(passwd))) {
      trace(DBG_FORCE, "InitialiseRFBConnection: reading password failed");
      return false;
    }
    if (strlen(passwd) > 8)
      passwd[8] = '\0';

    vncEncryptBytes(challenge, passwd);

    /* Lose the password from memory */
    for (int i = strlen(passwd); i >= 0; i--)
      passwd[i] = '\0';

    if (!VNC_sock.WriteExact((char *) challenge, CHALLENGESIZE))
      return false;
    if (!VNC_sock.ReadFromRFBServer((char *) &authResult, 4))
      return false;

    authResult = Swap32IfLE(authResult);
    switch (authResult) {
    case rfbVncAuthOK:
      trace(DBG_VNC, "InitialiseRFBConnection: VNC authentication succeeded");
      break;
    case rfbVncAuthFailed:
      trace(DBG_FORCE, "InitialiseRFBConnection: VNC authentication failed");
      return false;
    case rfbVncAuthTooMany:
      trace(DBG_FORCE, "VNC authentication failed - too many tries");
      return false;
    default:
      trace(DBG_FORCE, "Unknown VNC authentication result: %d",
	      (int)authResult);
      return false;
    }
    break;

  default:
    trace(DBG_FORCE, "Unknown authentication scheme from VNC server: %d",
	  (int)authScheme);
    return false;
  }

  ci.shared = (this->ShareDesktop ? 1 : 0); 

  if (!VNC_sock.WriteExact((char *)&ci, sz_rfbClientInitMsg))
    return false;
  if (!VNC_sock.ReadFromRFBServer((char *)&this->si, sz_rfbServerInitMsg))
    return false;

  this->si.framebufferWidth = Swap16IfLE(this->si.framebufferWidth);
  this->si.framebufferHeight = Swap16IfLE(this->si.framebufferHeight);
  this->si.format.redMax = Swap16IfLE(this->si.format.redMax);
  this->si.format.greenMax = Swap16IfLE(this->si.format.greenMax);
  this->si.format.blueMax = Swap16IfLE(this->si.format.blueMax);
  this->si.nameLength = Swap32IfLE(this->si.nameLength);

  desktopName = (char *) malloc(this->si.nameLength + 1);
  if (!VNC_sock.ReadFromRFBServer(desktopName, this->si.nameLength)) {
    free(desktopName);
    return false;
  }
  desktopName[this->si.nameLength] = 0;
  notice("Desktop: %s", desktopName);
  free(desktopName);

  trace(DBG_VNC, "InitialiseRFBConnection: connected to VNC server, using protocol version %d.%d",
	rfbProtocolMajorVersion, rfbProtocolMinorVersion);
  trace(DBG_VNC, "InitialiseRFBConnection: VNC server default format:");
  PrintPixelFormat(&this->si.format);

  return true;
}

/*
 * SetFormatAndEncodings.
 */
bool VNCRFBproto::SetFormatAndEncodings()
{ 
  rfbSetPixelFormatMsg spf;
  char buf[sz_rfbSetEncodingsMsg + MAX_ENCODINGS * 4];
  rfbSetEncodingsMsg *se = (rfbSetEncodingsMsg *)buf;
  CARD32 *encs = (CARD32 *)(&buf[sz_rfbSetEncodingsMsg]);
  int len = 0;

  spf.type = rfbSetPixelFormat;
  spf.format = pixFormat;
  spf.format.redMax = Swap16IfLE(spf.format.redMax);
  spf.format.greenMax = Swap16IfLE(spf.format.greenMax);
  spf.format.blueMax = Swap16IfLE(spf.format.blueMax);
 
  if (!VNC_sock.WriteExact((char *)&spf, sz_rfbSetPixelFormatMsg))
    return false;

  se->type = rfbSetEncodings;
  se->nEncodings = 0;

  if (VNC_sock.SameMachine()) {
    trace(DBG_VNC, "Same machine: preferring raw encoding");
    encs[se->nEncodings++] = Swap32IfLE(rfbEncodingRaw);
  }
  encs[se->nEncodings++] = Swap32IfLE(rfbEncodingCopyRect);
  encs[se->nEncodings++] = Swap32IfLE(rfbEncodingHextile);
  encs[se->nEncodings++] = Swap32IfLE(rfbEncodingCoRRE);
  encs[se->nEncodings++] = Swap32IfLE(rfbEncodingRRE);

  len = sz_rfbSetEncodingsMsg + se->nEncodings * 4;
  se->nEncodings = Swap16IfLE(se->nEncodings);

  if (!VNC_sock.WriteExact(buf, len))
    return false;

  return true; 
}

/*
 * SendIncrementalFramebufferUpdateRequest.
 */
bool VNCRFBproto::SendIncrementalFramebufferUpdateRequest()
{
  return this->SendFramebufferUpdateRequest(0, 0, this->si.framebufferWidth,
				      this->si.framebufferHeight, true);
}

/*
 * SendFramebufferUpdateRequest.
 */
bool VNCRFBproto::SendFramebufferUpdateRequest(int x, int y, int w, int h, bool incremental)
{
  rfbFramebufferUpdateRequestMsg fur;

  fur.type = rfbFramebufferUpdateRequest;
  fur.incremental = incremental ? 1 : 0;
  fur.x = Swap16IfLE(x);
  fur.y = Swap16IfLE(y);
  fur.w = Swap16IfLE(w);
  fur.h = Swap16IfLE(h);

  if (!VNC_sock.WriteExact((char *)&fur, sz_rfbFramebufferUpdateRequestMsg))
    return false;

  return true;
}

/*
 * SendPointerEvent.
 */
bool VNCRFBproto::SendPointerEvent(int x, int y, int buttonMask)
{
  rfbPointerEventMsg pe;

  pe.type = rfbPointerEvent;
  pe.buttonMask = buttonMask;
  if (x < 0) x = 0;
  if (y < 0) y = 0;
  pe.x = Swap16IfLE(x);
  pe.y = Swap16IfLE(y);
  return VNC_sock.WriteExact((char *)&pe, sz_rfbPointerEventMsg);
}

/*
 * SendKeyEvent.
 */
bool VNCRFBproto::SendKeyEvent(CARD32 key, bool down)
{
  rfbKeyEventMsg ke;

  ke.type = rfbKeyEvent;
  ke.down = down ? 1 : 0;
  ke.key = Swap32IfLE(key);
  return VNC_sock.WriteExact((char *)&ke, sz_rfbKeyEventMsg);
}

/*
 * SendClientCutText.
 */
bool VNCRFBproto::SendClientCutText(char *str, int len)
{
  rfbClientCutTextMsg cct;

  if (serverCutText)
    free(serverCutText);
  serverCutText = NULL;

  cct.type = rfbClientCutText;
  cct.length = Swap32IfLE(len);

  return  (VNC_sock.WriteExact((char *)&cct, sz_rfbClientCutTextMsg) &&
	   VNC_sock.WriteExact(str, len));
}

/*
 *SetVisual32 
 *initialise pixFormat qui donne le format de pixel que l'on veut que
 *le serveur nous envoie.
 *initialise pour demander le RGB
 *sur 3*8 bits = 24 bits
 *A APPELER DANS LE CONSTRUCTEUR (mto)
 */
void VNCRFBproto::SetVisual32()
{
  pixFormat.bitsPerPixel = 32;
  pixFormat.depth = 24;
  pixFormat.trueColour = 1;
  pixFormat.bigEndian = 1;
  pixFormat.redMax = 255;
  pixFormat.greenMax = 255;
  pixFormat.blueMax = 255;
  pixFormat.redShift = 0;
  pixFormat.greenShift = 8;
  pixFormat.blueShift = 16;
}

void VNCRFBproto::SetVisual8()
{
  pixFormat.bitsPerPixel = 8;
  pixFormat.depth = 8;
  pixFormat.trueColour = 1;
  pixFormat.bigEndian = 1;
  pixFormat.redMax = 7;
  pixFormat.greenMax = 7;
  pixFormat.blueMax = 3;
  pixFormat.redShift = 0;
  pixFormat.greenShift = 3;
  pixFormat.blueShift = 6;
}

void VNCRFBproto::SetVisual16()
{
  pixFormat.bitsPerPixel = 16;
  pixFormat.depth = 16;
  pixFormat.trueColour = 1;
  pixFormat.bigEndian = 1;
  pixFormat.redMax = 63;
  pixFormat.greenMax = 31;
  pixFormat.blueMax = 31;
  pixFormat.redShift = 0;
  pixFormat.greenShift = 6;
  pixFormat.blueShift = 11;
}

/* CONCAT2 concatenates its two arguments.  CONCAT2E does the same but also
   expands its arguments if they are macros */

#define CONCAT2(a,b) a##b
#define CONCAT2E(a,b) CONCAT2(a,b)


/*
 * PrintPixelFormat. For Debugging
 */
void VNCRFBproto::PrintPixelFormat(rfbPixelFormat *format)
{
  if (format->bitsPerPixel == 1) {
    trace(DBG_VNC, " Single bit per pixel");
    trace(DBG_VNC, " %s significant bit in each byte is leftmost on the screen",
	  (format->bigEndian ? "Most" : "Least"));
  }
  else {
    trace(DBG_VNC, " %d bits per pixel", format->bitsPerPixel);
    if (format->bitsPerPixel != 8) {
      trace(DBG_VNC, " %s significant byte first in each pixel",
	    (format->bigEndian ? "Most" : "Least"));
    }
    if (format->trueColour) {
      trace(DBG_VNC, " True color: max red %d green %d blue %d, shift red %d green %d blue %d",
	    format->redMax, format->greenMax, format->blueMax,
	    format->redShift, format->greenShift, format->blueShift);
    }
    else
      trace(DBG_VNC, " Colour map (not true colour)");
  }
}

/*
 * Returns the socket used
 */
int VNCRFBproto::getSock()
{
  return VNC_sock.GetSock();
}
