
// vbview_io.cpp
// load/set images and whatnot
// Copyright (c) 1998-2010 by The VoxBo Development Team

// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
//
// original version written by Dan Kimberg

using namespace std;

#include "vbview.h"
#include "fileview.h"
#include "vbcontrast.h"
#include "qinputdialog.h"
#include "makestatcub.h"

int
VBView::SetImage(Cube &cb)
{
  list<VBLayer> newlayers;
  newlayers.push_back(VBLayer());
  newlayers.front().cube=cb;
  return SetImage(newlayers);
}

int
VBView::LoadImage(string fname)
{
  list<VBLayer> newlayers;

  // get the filenames if needed
  if (!fname.size()) {
    fileview fv;
    fv.ShowDirectoriesAlso(1);
    fv.ShowImageInformation(1);
    fv.Go();
    tokenlist ff=fv.ReturnFiles();
    if (ff.size()==0)
      return 101;
    fname=ff[0];
  }
  q_currentdir=xdirname(fname);
  QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
  // by pushing it onto a temporary list, we can load the cube and
  // know we won't have to copy all the data later in order to splice
  // it into the main list
  newlayers.push_back(VBLayer());
  // read image data
  int err=newlayers.begin()->cube.ReadFile(fname);
  newlayers.begin()->filename=xfilename(fname);
  if (!err) {
    err=SetImage(newlayers);
    QApplication::restoreOverrideCursor();
    return err;
  }
  err=newlayers.begin()->tes.ReadFile(fname);
  if (!err) {
    err=SetImage(newlayers);
    QApplication::restoreOverrideCursor();
    return err;
  }
  QApplication::restoreOverrideCursor();
  return 101;
}



int
VBView::SetImage(list <VBLayer> &newlayers)
{
  VBLayerI tmp=newlayers.begin();
  if (layers.size()==0)
    emit renameme(this,xfilename(tmp->filename));
  // if it's 4d data, grab the first cube
  if (tmp->tes.data) {
    tmp->tes.getCube(0,tmp->cube);
    tmp->dimt=tmp->tes.dimt;
  }
  // set some layer parameters and render -- FIXME!!!
  tmp->q_brightness=50;
  tmp->q_contrast=50;
  tmp->type=VBLayer::vb_struct;
  // FIXME should just use what's in the cube, don't need separate dimxyz
  tmp->dimx=tmp->cube.dimx;
  tmp->dimy=tmp->cube.dimy;
  tmp->dimz=tmp->cube.dimz;
  if (layers.size()==0) {
    tmp->alpha=100;
    tmp->transform.ident();
    base_dimx=tmp->dimx;
    base_dimy=tmp->dimy;
    base_dimz=tmp->dimz;
    base_dims=3;
    if (tmp->tes.data) {
      base_dims=4;
      base_dimt=tmp->dimt;
    }
  }
  else {
    tmp->alpha=50;
    tmp->setAlignment(layers.begin());
  }
  newlayers.front().render();
  // splice it into the list
  layers.splice(layers.end(),newlayers);
  BuildViews(q_viewmode);
  RenderAll();
  updateLayerTable();

  xsliceslider->setRange(0,tmp->dimx-1);
  xsliceslider->setValue(tmp->dimx/2);
  ysliceslider->setRange(0,tmp->dimy-1);
  ysliceslider->setValue(tmp->dimy/2);
  zsliceslider->setRange(0,tmp->dimz-1);
  zsliceslider->setValue(tmp->dimz/2);
  if (tmp->tes.data) {
    tsliceslider->setRange(0,tmp->dimt-1);
    tsliceslider->setValue(0);
  }
  QApplication::restoreOverrideCursor();
  return 0;
}

int
VBView::LoadMask(string fname)
{
  if (layers.size()==0) {
    QMessageBox::warning(this,"No Base Layer","Your have to load an image before you can load a mask.","Okay");
    return 1;
  }
  deque<string> fnames;

  // get the filenames if needed
  if (!fname.size()) {
    fileview fv;
    fv.ShowDirectoriesAlso(1);
    fv.ShowImageInformation(1);
    fv.Go();
    fnames=fv.ReturnFiles();
    if (fnames.size()==0)
      return 101;
  }
  else
    fnames.push_back(fname);

  QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
  vbforeach(string ff,fnames) {
    list<VBLayer> newlayers;
    q_currentdir=xdirname(ff);
    // by pushing it onto a temporary list, we can load the cube and
    // know we won't have to copy all the data later in order to splice
    // it into the main list
    newlayers.push_back(VBLayer());
    // read image data
    if (newlayers.begin()->cube.ReadFile(ff))
      QMessageBox::warning(this,"Mask Error","Error loading one of your masks.","Okay");
     newlayers.begin()->filename=xfilename(ff);
    SetMask(newlayers);
  }
  QApplication::restoreOverrideCursor();
  return 0;
}

int
VBView::SetMask(Cube &mask)
{
  if (layers.size()==0) {
    // FIXME complain with a messagebox maybe
    return 1;
  }
  list<VBLayer> newlayers;
  newlayers.push_back(VBLayer());
  newlayers.front().cube=mask;
  return SetMask(newlayers);
}

int
VBView::SetMask(list<VBLayer> &newlayers)
{
  VBLayerI tmp=newlayers.begin();
  // build a colortable if none exists
  tcolor tc;
  if (tmp->cube.maskspecs.size()==0) {
    for (int i=0; i<tmp->cube.dimx*tmp->cube.dimy*tmp->cube.dimz; i++) {
      uint32 val=tmp->cube.getValue<int32>(i);
      if (val==0) continue;
      if (tmp->cube.maskspecs.count(val)) continue;
      VBMaskSpec ms;
      ms.r=tc.r;
      ms.g=tc.g;
      ms.b=tc.b;
      tc.next();
      ms.name="anon";
      tmp->cube.maskspecs[val]=ms;
    }
  }

  // set some layer parameters and render
  tmp->type=VBLayer::vb_mask;
  if (layers.size()==0) {
    tmp->alpha=100;
    tmp->transform.ident();
  }
  else {
    tmp->alpha=50;
    tmp->setAlignment(layers.begin());
  }
  newlayers.front().render();
  // splice it into the list
  layers.splice(layers.end(),newlayers);
  BuildViews(q_viewmode);
  RenderAll();
  updateLayerTable();
  QApplication::restoreOverrideCursor();
  return 0;


  // kill existing masks
//   ClearMasks();

//   for (int i=0; i<ff.size(); i++) {
//     Cube mask;
//     mask.dimx=cube.dimx;
//     mask.dimy=cube.dimy;
//     mask.dimz=cube.dimz;
//     if (mask.ReadFile(ff[i])) {
//       QApplication::restoreOverrideCursor();
//       QMessageBox::warning(this,"Mask Loading Error","Couldn't load your selected mask.","Okay");

//       return 101;
//     }
//     if (i==0) {
//       // check to see if we're in integral multiples and equal slices
//       if (cube.dimx % mask.dimx || cube.dimy % mask.dimy || cube.dimz % mask.dimz) {
//         QMessageBox::warning(this,"Mask Dimension Error","Your mask dimensions aren't integral divisors of your cube dimensions.","Okay");
//         QApplication::restoreOverrideCursor();
//         return 102;
//       }
//       q_maskxratio=cube.dimx/mask.dimx;
//       q_maskyratio=cube.dimy/mask.dimy;
//       q_maskzratio=cube.dimz/mask.dimz;
//       q_maskx=mask.dimx;
//       q_masky=mask.dimy;
//       q_maskz=mask.dimz;
//       update_status();
//     }
//     else if (q_maskx!=mask.dimx||q_masky!=mask.dimy||q_maskz!=mask.dimz) {
//       QMessageBox::warning(this,"Mask Dimension Error","Skipping mask with inconsistent dimensions.","Okay");
//     }
//     // find all the vb_maskspecs, create those mask widgets
//     tokenlist maskline;
//     for (int i=0; i<(int)mask.header.size(); i++) {
//       maskline.ParseLine(mask.header[i]);
//       if (maskline[0]=="vb_maskspec:" && maskline.size()==6) {
//         VBV_Mask mm;
//         mm.f_masktype='M';
//         mm.f_opacity=50;
//         mm.f_color=qRgb(strtol(maskline[2]),strtol(maskline[3]),strtol(maskline[4]));
//         mm.f_index=strtol(maskline[1]);
//         mm.f_name=maskline[5];
//         NewMaskWidget(mm);
//       }
//     }

//     // rebuild masktable
//     int match;
//     double val;
//     for (int i=0; i<mask.dimx; i++) {
//       for (int j=0; j<mask.dimy; j++) {
//         for (int k=0; k<mask.dimz; k++) {
//           val=mask.GetValue(i,j,k);
//           if (val==0) continue;
//           match=0;
//           for (int m=0; m<(int)q_masks.size(); m++) {
//             if (val==q_masks[m].f_index) {
//               AddMaskIndex(i,j,k,m);
//               match=1;
//             }
//           }
//           if (!match) {
//             VBV_Mask mm;
//             mm.f_masktype='M';
//             mm.f_opacity=50;
//             mm.f_color=qRgb(255,200,200);
//             mm.f_index=(int)val;
//             mm.f_name="anonymous mask";
//             NewMaskWidget(mm);
//             SetMaskValue(i,j,k,(int)val);
//           }
//         }
//       }
//     }
//   }

//   for (int i=0; i<(int)viewlist.size(); i++)
//     PreCalcView(viewlist[i]);
//   RenderAll();
//   if (ff.size()==1)
//     masknamebox->setText(ff(0));
//   else
//     masknamebox->setText("<multiple masks loaded>");
//   QApplication::restoreOverrideCursor();
//   return (0);  // no error!
}









int
VBView::LoadStat(string fname)
{
  if (layers.size()==0) {
    QMessageBox::warning(this,"No Base Layer","Your have to load an image before you can load a stat map.","Okay");
    return 1;
  }
  deque<string> fnames;

  // get the filenames if needed
  if (!fname.size()) {
    fileview fv;
    fv.ShowDirectoriesAlso(1);
    fv.ShowImageInformation(1);
    fv.Go();
    fnames=fv.ReturnFiles();
    if (fnames.size()==0)
      return 101;
  }
  else
    fnames.push_back(fname);

  QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
  vbforeach(string ff,fnames) {
    list<VBLayer> newlayers;
    q_currentdir=xdirname(ff);
    // by pushing it onto a temporary list, we can load the cube and
    // know we won't have to copy all the data later in order to splice
    // it into the main list
    newlayers.push_back(VBLayer());
    VBLayerI tmp=newlayers.begin();
    // read image data
    if (tmp->cube.ReadFile(ff))
      QMessageBox::warning(this,"Map Error","Error loading one of your maps.","Okay");
    newlayers.begin()->filename=xfilename(ff);
    // auto-scale to range
    tmp->q_thresh=3.5;
    tmp->q_high=tmp->cube.get_maximum();
    // set some layer parameters and render
    tmp->type=VBLayer::vb_stat;
    tmp->alpha=40;
    tmp->setAlignment(layers.begin());
    tmp->dimx=tmp->cube.dimx;
    tmp->dimy=tmp->cube.dimy;
    tmp->dimz=tmp->cube.dimz;
    newlayers.front().render();
    // newlayers.front().rethresh();
    // splice it into the list
    layers.splice(layers.end(),newlayers);
  }
  // BuildViews(q_viewmode);
  RenderAll();
  updateLayerTable();
  QApplication::restoreOverrideCursor();
  return 0;
}



int
VBView::LoadCorr(string fname)
{
  if (layers.size()==0) {
    // FIXME complain with a messagebox maybe
    return 1;
  }
  list<VBLayer> newlayers;

  // get the filenames if needed
  if (!fname.size()) {
    fileview fv;
    fv.ShowDirectoriesAlso(1);
    fv.ShowImageInformation(1);
    fv.Go();
    tokenlist ff=fv.ReturnFiles();
    if (ff.size()==0)
      return 101;
    fname=ff[0];
  }
  q_currentdir=xdirname(fname);
  QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
  // by pushing it onto a temporary list, we can load the cube and
  // know we won't have to copy all the data later in order to splice
  // it into the main list
  newlayers.push_back(VBLayer());
  VBLayerI tmp=newlayers.begin();
  // read image data
  int err=tmp->tes.ReadFile(fname);
  if (err) {
    QApplication::restoreOverrideCursor();
    return 101;
  }
  // set some layer parameters and return.  note that we don't render
  // this layer immediately when it's loaded
  tmp->type=VBLayer::vb_corr;
  tmp->filename=xfilename(fname);
  tmp->q_visible=1;
  tmp->alpha=40;
  tmp->q_thresh=0.0;
  tmp->q_high=1.0;
  tmp->dimx=tmp->tes.dimx;
  tmp->dimy=tmp->tes.dimy;
  tmp->dimz=tmp->tes.dimz;
  tmp->tes.getCube(0,tmp->cube);  // so that it inherits alignment info, just in case
  tmp->cube.convert_type(vb_float);
  // tmp->cube.SetVolume(tmp->dimx,tmp->dimy,tmp->dimz,vb_float);
  tmp->setAlignment(layers.begin());
  tmp->render();
  // splice it into the list
  layers.splice(layers.end(),newlayers);
  RenderAll();
  updateLayerTable();
  QApplication::restoreOverrideCursor();
  return 0;
}


int
VBView::LoadAux(string fname)
{
  if (layers.size()==0) {
    // FIXME complain with a messagebox maybe
    return 1;
  }
  list<VBLayer> newlayers;

  // get the filenames if needed
  if (!fname.size()) {
    fileview fv;
    fv.ShowDirectoriesAlso(1);
    fv.ShowImageInformation(1);
    fv.Go();
    tokenlist ff=fv.ReturnFiles();
    if (ff.size()==0)
      return 101;
    fname=ff[0];
  }
  q_currentdir=xdirname(fname);
  QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));
  // by pushing it onto a temporary list, we can load the cube and
  // know we won't have to copy all the data later in order to splice
  // it into the main list
  newlayers.push_back(VBLayer());
  VBLayerI tmp=newlayers.begin();
  // read image data
  int err=tmp->cube.ReadFile(fname);
  if (err) {
    QApplication::restoreOverrideCursor();
    return 101;
  }
  // set some layer parameters and render
  tmp->type=VBLayer::vb_aux;
  tmp->q_visible=0;
  tmp->filename=fname;
  tmp->dimx=tmp->cube.dimx;
  tmp->dimy=tmp->cube.dimy;
  tmp->dimz=tmp->cube.dimz;
  tmp->setAlignment(layers.begin());
  // splice it into the list
  layers.splice(layers.end(),newlayers);
  updateLayerTable();
  QApplication::restoreOverrideCursor();
  return 0;
}




// int
// VBView::SetImage(Cube &newcube)
// {
//   cube=newcube;
//   string fname=cube.GetFileName();
//   tes.invalidate();
//   tes.data_valid=0;
//   volumebox->hide();
//   // and now the rest of the stuff
//   q_update=0;

//   if (cube.dimx < 128 && cube.dimy < 128) {
//     q_mag=400;
//     q_maskx=cube.dimx;
//     q_masky=cube.dimy;
//     q_maskz=cube.dimz;
//     q_maskxratio=1;
//     q_maskyratio=1;
//     q_maskzratio=1;
//   }
//   else {
//     q_mag=100;
//     q_maskx=cube.dimx/4;
//     q_masky=cube.dimy/4;
//     q_maskz=cube.dimz;
//     q_maskxratio=4;
//     q_maskyratio=4;
//     q_maskzratio=1;
//   }

//   ClearMasks();
//   NewMask();
//   update_status();

//   setscale();
//   zsliceslider->setRange(0,cube.dimz-1);
//   zsliceslider->setValue(cube.dimz/2);

//   xsliceslider->setRange(0,cube.dimx-1);
//   xsliceslider->setValue(cube.dimx/2);

//   ysliceslider->setRange(0,cube.dimy-1);
//   ysliceslider->setValue(cube.dimy/2);

//   zslicebox->show();
//   xslicebox->show();
//   yslicebox->show();

//   NewZSlice(cube.dimz/2);
//   if (q_mag==100)
//     magbox->setCurrentText("100%");
//   else if (q_mag==400)
//     magbox->setCurrentText("400%");
//   emit renameme(this,xfilename(fname));
//   imagenamebox->setText(fname.c_str());
//   CopyInfo();
//   q_update=1;

//   // updatescaling() also builds views and renders
//   UpdateScaling();

//   // depopulate the voxel surfing listbox
//   tslist->clear();
//   tspane->clear();
//   return 0;
// }

// int
// VBView::SetImage(Tes &newtes)
// {
//   tes=newtes;
//   string fname=tes.GetFileName();
//   tes.GetCube(0,cube);
//   volumeslider->setRange(0,tes.dimt-1);
//   volumeslider->setValue(0);
//   volumebox->show();
//   q_volume=0;
//   // and now the rest of the stuff
//   q_update=0;

//   if (cube.dimx < 128 && cube.dimy < 128) {
//     q_mag=400;
//     q_maskx=cube.dimx;
//     q_masky=cube.dimy;
//     q_maskz=cube.dimz;
//     q_maskxratio=1;
//     q_maskyratio=1;
//     q_maskzratio=1;
//   }
//   else {
//     q_mag=100;
//     q_maskx=cube.dimx/4;
//     q_masky=cube.dimy/4;
//     q_maskz=cube.dimz;
//     q_maskxratio=4;
//     q_maskyratio=4;
//     q_maskzratio=1;
//   }

//   // BuildViews(q_viewmode);

//   ClearMasks();
//   NewMask();
//   update_status();

//   setscale();
//   zsliceslider->setRange(0,cube.dimz-1);
//   zsliceslider->setValue(cube.dimz/2);

//   xsliceslider->setRange(0,cube.dimx-1);
//   xsliceslider->setValue(cube.dimx/2);

//   ysliceslider->setRange(0,cube.dimy-1);
//   ysliceslider->setValue(cube.dimy/2);

//   zslicebox->show();
//   xslicebox->show();
//   yslicebox->show();

//   NewZSlice(cube.dimz/2);
//   if (q_mag==100)
//     magbox->setCurrentText("100%");
//   else if (q_mag==400)
//     magbox->setCurrentText("400%");
//   emit renameme(this,xfilename(fname));
//   imagenamebox->setText(fname.c_str());
//   CopyInfo();
//   q_update=1;

//   // updatescaling() also builds views and renders
//   UpdateScaling();
//   // populate the voxel surfing listbox
//   tslist->clear();
//   tspane->clear();
//   tslist->insertItem("Raw time series"); 
//   tslist->setSelected(0,1);
//   // remove GLM-specific checkboxes
//   ts_filterbox->hide();
//   ts_removebox->hide();
//   ts_scalebox->hide();
//   return 0;
// }

// int
// VBView::LoadOverlay()
// {
//   QString s;
//   s=QFileDialog::getOpenFileName(q_currentdir.c_str(),"All (*.*)",this,"open statistical overlay","Choose a statistical map to overlay");
//   if (s==QString::null) return (0);
//   return LoadOverlay(s.latin1());
// }

// int
// VBView::LoadOverlay(string fname)
// {
//   Cube tmp;
//   if (tmp.ReadFile(fname)) {
//     QMessageBox::warning(this,
//                          "Couldn't Load Overlay",
//                          "We were unable to load your statistical overlay file.",
//                          "Okay");
//     return 101;
//   }

//   // set up the overlay
//   q_stat.SetVolime(tmp.dimx,tmp.dimy,tmp.dimz,2,vb_double);
//   q_stat.SetCube(0,tmp);
//   tmp.quantize(1);
//   q_stat.SetCube(1,tmp);

//   // check to see if we're in integral multiples and equal slices
//   if (cube.dimx % statmap.dimx || statmap.dimy % statmap.dimy) {
//     QMessageBox::warning(this,"Overlay Dimension Warning","Your overlay dimensions aren't integral divisors of your cube dimensions.","Okay");
//   }
//   else {
//     q_statxratio=cube.dimx/statmap.dimx;
//     q_statyratio=cube.dimy/statmap.dimy;
//     q_statzratio=cube.dimz/statmap.dimz;
//     update_status();
//   }
//   if (cube.dimz != statmap.dimz) {
//     QMessageBox::warning(this,"Overlay Dimension Warning","Your data and your stat map have different numbers of slices.","Okay");
//   }

//   panel_stats->show();
//   ReThreshold();
// //   for (int i=0; i<(int)viewlist.size(); i++)
// //     PreCalcView(viewlist[i]);
//   RenderAll();
//   overlaynamebox->setText(statmap.GetFileName().c_str());
//   return 0;
// }


// int
// VBView::LoadGLM()
// {
//   QString s;
//   QString myfilter="Overlays (*.cub *.img *.nii *.dcm *.prm);;All Files (*.*)";
//   s=QFileDialog::getOpenFileName(q_currentdir.c_str(),myfilter,this,"open statistical overlay","Choose a statistical map or GLM to overlay");
//   if (s==QString::null) return (0);

//   if (xgetextension(s.latin1())!="prm")
//     LoadOverlay(s.latin1());  // FIXME pass file name

//   // DYK

// //   VBQT_GLMSelect glms(this,"",1);
// //   if (glms.exec()==QDialog::Rejected)
// //     return 102;

//   glmi.setup(s.latin1());

//   cout << 1 << endl;
//   VB::VBContrastParamScalingWidget www(this,"hey hey hey");
//   cout << 2 << endl;
//   www.LoadContrastInfo(glmi.stemname);
//   if (www.exec()==QDialog::Rejected)
//     return 102;
//   cout << 3 << endl;
  
//   // init our glmi using info from the glms version
//   glmi.contrast=*(www.selectedContrast());
//   cout << 4 << endl;
//   // find anatomy file
//   if (!glmi.anatomyname.size() && !cube.data) {
//     // find it
//     QString s;
//     s=QFileDialog::getOpenFileName(q_currentdir.c_str(),"All (*.*)",this,"open image file","Choose a file to view");
//     if (s==QString::null)
//       return 120;
//     glmi.anatomyname=s.latin1();
//   }
//   if (glmi.anatomyname.size())
//     SetImage(glmi.anatomyname);
//   int err=glmi.calc_stat_cube();
//   statmap=glmi.statcube;
//   backupmap=statmap;
//   cout << 5 << endl;
//   glmi.statcube.WriteFile("xxx.cub");
//   if (err) {
//     QMessageBox::warning(this,"Bad stat map","Stat map didn't build, sorry!","Okay");
//     return 101;
//   }
//   // set overlay name
//   string tmp="[GLM statmap from "+glmi.stemname+"] contrast: "+glmi.contrast.name;
//   overlaynamebox->setText(tmp.c_str());

//   // check to see if we're in integral multiples and equal slices
//   if (cube.dimx % statmap.dimx || statmap.dimy % statmap.dimy) {
//     QMessageBox::warning(this,"Overlay Dimension Warning","Your overlay dimensions aren't integral divisors of your cube dimensions.","Okay");
//   }
//   else {
//     q_statxratio=cube.dimx/statmap.dimx;
//     q_statyratio=cube.dimy/statmap.dimy;
//     q_statzratio=cube.dimz/statmap.dimz;
//   }
//   if (cube.dimz != statmap.dimz) {
//     QMessageBox::warning(this,"Overlay Dimension Warning","Your data and your stat map have different numbers of slices.","Okay");
//   }

//   // mask should be same shape as functional data and empty
//   q_maskxratio=q_statxratio;
//   q_maskyratio=q_statyratio;
//   q_maskzratio=q_statzratio;
//   q_maskx=statmap.dimx;
//   q_masky=statmap.dimy;
//   q_maskz=statmap.dimz;
//   ClearMasks();
//   NewMask();
//   update_status();

//   panel_stats->show();
//   ReThreshold();
// //   for (int i=0; i<(int)viewlist.size(); i++)
// //     PreCalcView(viewlist[i]);
//   RenderAll();

//   // populate the voxel surfing listbox
//   tslist->clear();
//   tspane->clear();
//   tslist->insertItem("Raw time series");
//   tslist->insertItem("Fitted values");
//   tslist->insertItem("Residuals");
//   for (int i=0; i<(int)glmi.cnames.size(); i++)
//     tslist->insertItem(glmi.cnames[i].c_str()+1);
//   tslist->setSelected(0,1);
//   // populate the trial averaging listbox
//   reload_averages();
//   // make sure GLM-specific applicable buttons are there
//   ts_filterbox->show();
//   ts_removebox->show();
//   ts_scalebox->show();
//   // set detrend and meanscale checkboxes as for glm
//   if (glmi.glmflags & MEANSCALE)
//     ts_meanscalebox->setChecked(1);
//   else
//     ts_meanscalebox->setChecked(0);
//   if (glmi.glmflags & DETREND)
//     ts_detrendbox->setChecked(1);
//   else
//     ts_detrendbox->setChecked(0);
//   trialsets=glmi.trialsets;
  
//   return 0;
// }

