
// dbdata.cpp
// Member function definitions of DBdata
// Copyright (c) 2007-2010 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Dongbo Hu

using namespace std;

#include "dbdata.h"
#include "db_util.h"

/* Hard-code table names
 * If db tables are created in a transactional environment, inputDir is blank;
 * If no environment is available, create them as non-transactional table. */
void DBdata::setDbNames(string inputDir)
{
  // Append "/" at the end of inputDir to make sure path is correct
  if (inputDir.length() && inputDir[inputDir.length() - 1] != '/')
    inputDir.append("/");

  sysDbName = inputDir + "system.db";
  userDbName = inputDir + "user.db";
  scoreNameDbName = inputDir + "scorenames.db";
  scoreValueDbName = inputDir + "scorevalues.db";
  patientScoreDbName = inputDir + "patientscores.sdb";
  sessionDbName = inputDir + "session.db";
  permissionDbName = inputDir + "permissions.db";
  regionDbName = inputDir + "region_name.db";
  synonymDbName = inputDir + "synonym.db";
  regionRelationDbName = inputDir + "region_relation.db";
  namespaceDbName = inputDir + "namespace.db";
  patientDbName = inputDir + "patient.db";
  patientListDbName = inputDir + "patientlist.db";
}

/* Open enviroment and databases files.
 * Returns 0 if db env is already open or everything is ok;
 * returns -1 for db enviroment open errors;
 * returns -2 for any global DBs open errors; */
int DBdata::open()
{
  if (env.open(dirname)) {
    errMsg = "Failed to open DB env";
    return -1;
  }
  
  setDbNames();

  // Open system table, keys ranked lexically
  if (sysDB.open(sysDbName, env, myDB::cmp_lex)) { 
    errMsg = "Failed to open system db";
    return -2;
  }
  // Open user table
  if (userDB.open(userDbName, env)) {
    errMsg = "Failed to open user db";
    return -2;
  }
  // Open permission table: keys ranked lexically, allow duplicate keys 
  if (permDB.open(permissionDbName, env, myDB::cmp_lex, myDB::sort_default)) {
    errMsg = "Failed to open permission db";
    return -2;
  }
  // Open score name table: keys ranked lexically, score name ID no longer exists
  if (scoreNameDB.open(scoreNameDbName, env, myDB::cmp_lex)) { 
    errMsg = "Failed to open score name db";
    return -2;
  }
  // Open session table
  if (sessionDB.open(sessionDbName, env)) {
    errMsg = "Failed to open session db";
    return -2;
  }
  // Open patient table
  if (patientDB.open(patientDbName, env)) {
    errMsg = "Failed to open patient db";
    return -2;
  }
  // Open patient list table
  if (patientListDB.open(patientListDbName, env)) {
    errMsg = "Failed to open patient list db";
    return -2;
  }
  // Open score value table: keys ranked lexically
  if (scoreValueDB.open(scoreValueDbName, env, myDB::cmp_lex)) {
    errMsg = "Failed to open score value db";
    return -2;
  }
  // Open patient score value table (secondary sb based on score value table): 
  // keys ranked numerically, duplicate keys allowed, default sort method used
  if (patientScoreDB.open(patientScoreDbName, env, myDB::cmp_int, myDB::sort_default)) {
    errMsg = "Failed to open patient score value db";
    return -2;
  }
  // Associate secondary db with primary db
  scoreValueDB.getDb().associate(NULL, &patientScoreDB.getDb(), getPID, 0);

  // load the text config files
  // retrieve score names from db
  //   if (getScoreNames(NULL))
  //     return -1;
  int err;
  err = readTypes(dirname+"/types.txt");
  if (err) {
    errMsg = "could not load types.txt, error code: " + strnum(err);
    return -2;
  }
  err = readViews(dirname+"/views.txt");
  if (err) {
    errMsg = "could not load views,txt, error code: " + strnum(err);
    return -2;
  }
  err = readScorenames(dirname+"/scorenames.txt");
  if (err) {
    errMsg = "could not load scorenames.txt, error code: " + strnum(err);
    return -2;
  }

  if (loadRegionData())
    return -3;

  return 0;
}

/* This function closes all open DBs and finally the db enviroment. 
 * Returns 0 if everything is ok;
 * returns -1 for db env close error;
 * returns -2 for any db close error. */
int DBdata::close()
{
  if (closeTables())
    return -2;

  // close enviroment after all dbs have been closed
  if (env.close()) {
    errMsg = "Failed to close db environment";
    return -1;
  }

  return 0;
}

/* Close tables. Returns 0 if all tables are closed cleanly;
 * returns -2 for any db close errors. */
int DBdata::closeTables()
{
  if (sysDB.close()) {
    errMsg = "Failed to close system db";
    return -2;
  }
  if (userDB.close()) {
    errMsg = "Failed to close user db";
    return -2;
  }
  if (permDB.close()) {
    errMsg = "Failed to close permissions db";
    return -2;
  }
  if (scoreNameDB.close()) {
    errMsg = "Failed to close score name db";
    return -2;
  }
  if (sessionDB.close()) {
    errMsg = "Failed to close session db";
    return -2;
  }
  if (patientDB.close()) {
    errMsg = "Failed to close patient db";
    return -2;
  }
  if (patientListDB.close()) {
    errMsg = "Failed to close patient list db";
    return -2;
  }
  if (patientScoreDB.close()) {
    errMsg = "Failed to close patientScore db";
    return -2;
  }
  if (scoreValueDB.close()) {
    errMsg = "Failed to close score values db";
    return -2;
  }
  // close brain region related tables
  if (regionNameDB.close()) {
    errMsg = "Failed to close brain region names db";
    return -2;
  }
  if (synonymDB.close()) {
    errMsg = "Failed to close brain region synonym db";
    return -2;
  }
  if (regionRelationDB.close()) {
    errMsg = "Failed to close brain region relationship db";
    return -2;
  }
  if (namespaceDB.close()) {
    errMsg = "Failed to close namespace db";
    return -2;
  }

  return 0;
}

/* Set maps on server side. 
 * Returns 0 if everything is ok;
 * returns -1 if any global map can not be loaded successfully; */
int DBdata::loadRegionData()
{
  // open namespace db (not used right now)
  if (namespaceDB.open(namespaceDbName, env, myDB::cmp_lex)) {
    errMsg = "Failed to open brain region namespace db";
    return -2;
  }
  
  // set brain region name map
  Dbc *cursorp = NULL;
  Dbt key, data;
  int ret;
  if (regionNameDB.open(regionDbName, env)) {
    errMsg = "Failed to open region name db";
    return -2;
  }

  if (regionNameDB.getDb().cursor(NULL, &cursorp, 0)) { // validate cursor
    errMsg = "Invalid region name db cursor";
    regionNameDB.close();
    return -1;    // validate cursor
  }

  while ((ret = cursorp->get(&key, &data, DB_NEXT)) == 0 ) {
    regionRec rData(data.get_data());
    regionNameMap[rData.getID()] = rData;
  }
  cursorp->close();
  regionNameDB.close();
  // Returns -1 if ret is non-zero and it doesn't reach the end of table 
  if (ret && ret != DB_NOTFOUND) {
    errMsg = "Region name record not found";
    return -1;
  }
  
  // set brain region synoyms map
  if (synonymDB.open(synonymDbName, env)) {
    errMsg = "Failed to open synonym db";
    return -2;
  }
  if (synonymDB.getDb().cursor(NULL, &cursorp, 0)) { // validate cursor
    errMsg = "Invalid synonym db cursor";
    synonymDB.close();
    return -1;
  }
  while ((ret = cursorp->get(&key, &data, DB_NEXT)) == 0 ) {
    synonymRec synData(data.get_data());
    synonymMap[synData.getID()] = synData;
  }
  cursorp->close();
  synonymDB.close();
  // Returns -1 if ret is non-zero and it doesn't reach the end of table 
  if (ret && ret != DB_NOTFOUND) {
    errMsg = "Synonym record not found";
    return -1;
  }

  // set brain region relationship map
  if (regionRelationDB.open(regionRelationDbName, env)) {
    errMsg = "Failed to open region relationship db";
    return -2;
  }
  if (regionRelationDB.getDb().cursor(NULL, &cursorp, 0)) { // validate cursor
    errMsg = "Invalid region relationship db cursor";
    regionRelationDB.close();
    return -1;
  }
  while ((ret = cursorp->get(&key, &data, DB_NEXT)) == 0 ) {
    regionRelationRec rrData(data.get_data());
    regionRelationMap[rrData.getID()] = rrData;
  }
  cursorp->close();
  regionRelationDB.close();
  // Returns -1 if ret is non-zero and it doesn't reach the end of table 
  if (ret && ret != DB_NOTFOUND) {
    errMsg = "Region relationship record not found";
    return -1;
  }

  return 0;
}

// Read type txt file and set type map, written by Dan
int DBdata::readTypes(string fname)
{
  const int MAXLEN = 4096;
  FILE * ifp = fopen(fname.c_str(), "r");
  if (!ifp) {
    printf("%s not available for reading.", fname.c_str());
    return -1;
  }

  char line[MAXLEN];
  int32 lineNo = 0;
  while (fgets(line,MAXLEN,ifp)) {
    lineNo++;
    tokenlist toks;
    toks.ParseLine(line);
    if (toks.size()==0) continue;
    if (toks[0][0]=='#') continue;
    if (toks[0][0]=='%') continue;
    if (toks[0][0]=='!') continue;
    if (toks[0][0]==';') continue;
    // Print out error message if the line is not blank but the number of field is not 2
    if (toks.size()==1) {
      printf("%s line #%d: invalid type specification\n", fname.c_str(), lineNo);
      // FIXME we can continue with an invalid line
      // fclose(ifp); return -2;
    }
    typemap[toks[0]].name = toks[0];
    if (toks[1]=="description" && toks.size()>2)
      typemap[toks[0]].description=toks.Tail(2);
    else
      typemap[toks[0]].values.push_back(toks[1]);
  }
  fclose(ifp);
  return 0;
}

int
DBdata::readScorenames(string fname)
{
  FILE *fp;
  tokenlist toks;
  char buf[1024];
  if ((fp=fopen(fname.c_str(),"r"))==NULL)
    return -1;
  while (fgets(buf,1023,fp)) {
    toks.ParseLine(buf);
    if (toks.size()<2) continue;
    if (toks[0][0]=='#') continue;
    if (toks[0][0]=='%') continue;
    if (toks[0][0]=='!') continue;
    if (toks[0][0]==';') continue;
    DBscorename si;
    si.flags["customizable"]="1";
    si.flags["leaf"]="1";
    si.name=si.screen_name=toks[0];  // FIXME eventually we can have screen name be separate
    si.datatype=toks[1];
    string parentname=scoreparent(si.name);
    // FIXME commented out the condition below, we can use children of "" to find *TESTS*  ok?
    // map parent's id onto our id if there's a parent
    // if (parentname!="")
      scorenamechildren.insert(pair<string,string>(parentname,si.name));
    // map name to id
    // set flags
    if (si.datatype=="stub")
      si.flags.erase("leaf");
    for (int i=2; i<toks.size(); i++) {
      if (toks[i]=="repeating")
	si.flags["repeating"]="1";
      else if (toks[i]=="searchable")
	si.flags["searchable"]="1";
      // if we don't recognize it, assume it takes one argument
      else {
	si.flags[toks[i]]=toks[i+1];
	i++;
      }
    }
    // add it to the main list
    scorenames[si.name]=si;
  }
  fclose(fp);

  return 0;
}


// Read view spec txt file and set a map, written by Dan
// view file includes the following line types:
// myview x x x x x x   [add all the stuff as an entry to myview]
// myview newtab        [add a new tab to myview]

int
DBdata::readViews(string fname)
{
  const int MAXLEN = 4096;
  FILE * ifp = fopen(fname.c_str(), "r");
  if (!ifp) {
    printf("%s not available for reading.",fname.c_str());
    return -1;
  }

  char line[MAXLEN];
  int32 lineNo = 0;
  while (fgets(line,MAXLEN,ifp)) {
    lineNo++;
    tokenlist toks;
    toks.ParseLine(line);
    if (toks.size()==0) continue;
    if (toks[0][0]=='#') continue;
    if (toks[0][0]=='%') continue;
    if (toks[0][0]=='!') continue;
    if (toks[0][0]==';') continue;
    // Print out error message if the line is not blank but the number of field is not >=2
    if (toks.size()<2) {
      printf("%s line #%d: invalid view specification\n",fname.c_str(), lineNo);
      // FIXME we can continue with an invalid line
      // fclose(ifp); return -2;
    }
    viewspecs[toks[0]].name = toks[0];
    viewspecs[toks[0]].entries.push_back(toks.Tail());
  }
  fclose(ifp);
  return 0;
}

/* This function collects all score name information and 
 * put them into an array of DBscorename object. 
 * returns 0 if everything is ok;
 * or returns -1 for db errors. */
// int DBdata::getScoreNames(DbTxn* txn)
// {
//   Dbc *cursorp = NULL;
//   if (scoreNameDB.getDb().cursor(txn, &cursorp, 0)) { // validate curosr
//     errMsg = "Invalid score name db cursor";
//     return -1;
//   }

//   Dbt key, data;
//   int ret;
//   int status = 0;
//   while ((ret = cursorp->get(&key, &data, DB_NEXT)) == 0 ) {
//     scoreNameRec sData(data.get_data());
//     DBscorename tmp_score(sData);
//     add_scorename(tmp_score);
//   }

//   // Returns -1 if ret is non-zero and it doesn't reach the end of table 
//   if (ret && ret != DB_NOTFOUND) {
//     errMsg = "Score name record not found";
//     status = -1;
//   }

//   cursorp->close();
//   return status;
// }

// Add a DBscorename record into scorename maps, written by Dan
void DBdata::add_scorename(const DBscorename &sn)
{
  // add to the master map of scorename ids to scorenames
  scorenames[sn.name]=sn;
  // add to the master map of parents to children
  scorenamechildren.insert(pair<string,string>(scoreparent(sn.name),sn.name));
  // if the parent is zero, it's a "test" -- add to map of test names to ids
  //   if (sn.parentname.size()==0)
  //     testmap[sn.name]=sn.id;
}


/* Utility function written by Dan. */
void DBdata::print_types()
{
  map<string, DBtype>::iterator ti;
  for (ti=typemap.begin(); ti!=typemap.end(); ti++) {
    printf ("TYPE %s\n",ti->first.c_str());
    vector<string>::iterator vi;
    for (vi=ti->second.values.begin(); vi!=ti->second.values.end(); vi++)
      printf("     | %s\n",vi->c_str());
  }
}

/* Initialize tables without envirionment/transaction. 
 * Returns 0 if everything is ok;
 * returns -1 if any db file already exists;
 * returns -2 if any db can not be created; 
 * returns -3 for db close errors;
 * returns -4 for any other db initialization error.*/
int DBdata::initDB(string inputDir, string admin_passwd, uint32 id_start)
{
  setDbNames(inputDir); 
  if (!validateDir())
    return -1;

  // Open system table, keys ranked lexically
  if (sysDB.open(sysDbName, myDB::cmp_lex)) { 
    errMsg = "Failed to open system db";
    return -2;
  }
  // Add two records into system table
  for (int i = 0; i < 2; i++) {
    sysRec sData;
    if (i == 0) {
      sData.setName("Next Unique ID");
      sData.setValue(strnum(id_start + 1));  // id_start will be reserved for admin's user ID
    }
    else {
      sData.setName("last_updated");
      sData.setValue(strnum(time(NULL)));
    }

    Dbt key((char*) sData.getName().c_str(), sData.getName().length() + 1);
    int32 size = sData.getSize();
    char* buff = new char[size];
    sData.serialize(buff);
    Dbt data(buff, size);
    if (sysDB.getDb().put(NULL, &key, &data, 0)) {
      errMsg = string("Failed to add ") + sData.getName();
      delete [] buff;
      return -4;
    }
    delete [] buff;
  }

  // Open user table
  if (userDB.open(userDbName)) {
    errMsg = "Failed to open user db";
    return -2;
  }
  // Add admin account and passwd into user table
  userRec dbUser;
  dbUser.setID(id_start);
  dbUser.setAccount("admin");
  dbUser.setName("Database Administrator");
  char salt[4];
  gnutls_datum verifier;
  if (make_salt_verifier("admin", admin_passwd, salt, &verifier)) {
    errMsg = "Failed to make verifier for admin account";
    return -4;
  }
  dbUser.setSalt(salt);
  dbUser.setVeriSize(verifier.size);
  dbUser.setVerif(verifier.data);
  int32 bufLen = dbUser.getSize();
  char buff[bufLen];
  dbUser.serialize(buff);
  gnutls_free(verifier.data);
  Dbt key(buff, sizeof(int32));
  Dbt data(buff, bufLen);
  if (userDB.getDb().put(NULL, &key, &data, 0)) {
    errMsg = "Failed to add admin into user table";
    return -4;
  }

  // Open permission table: keys ranked lexically, allow duplicate keys 
  if (permDB.open(permissionDbName, myDB::cmp_lex, myDB::sort_default)) {
    errMsg = "Failed to open permission db";
    return -2;
  }
  // Grant full permissions to admin
  permRec pRec;
  pRec.setAccessID(strnum(id_start));
  pRec.setDataID("*");
  pRec.setPermission("rw");
  if (addPerm(permDB, NULL, pRec)) {
    errMsg = "Failed to add admin permission record";
    return -4;
  }
  
  // Open score name table: keys ranked lexically, score name ID no longer exists
  if (scoreNameDB.open(scoreNameDbName, myDB::cmp_lex)) { 
    errMsg = "Failed to open score name db";
    return -2;
  }
  // Open session table
  if (sessionDB.open(sessionDbName)) {
    errMsg = "Failed to open session db";
    return -2;
  }
  // Open patient table
  if (patientDB.open(patientDbName)) {
    errMsg = "Failed to open patient db";
    return -2;
  }
  // Open patient list table
  if (patientListDB.open(patientListDbName)) {
    errMsg = "Failed to open patient list db";
    return -2;
  }
  // Open score value table: keys ranked lexically
  if (scoreValueDB.open(scoreValueDbName, myDB::cmp_lex)) {
    errMsg = "Failed to open score value db";
    return -2;
  }
  // Open patient score value table (secondary sb based on score value table): 
  // keys ranked numerically, duplicate keys allowed, default sort method used
  if (patientScoreDB.open(patientScoreDbName, myDB::cmp_int, myDB::sort_default)) {
    errMsg = "Failed to open patient score value db";
    return -2;
  }
  // Associate secondary db with primary db
  scoreValueDB.getDb().associate(NULL, &patientScoreDB.getDb(), getPID, 0);

  // brain region related tables
  if (regionNameDB.open(regionDbName)) {
    errMsg = "Failed to open region name db";
    return -2;
  }
  if (synonymDB.open(synonymDbName)) {
    errMsg = "Failed to open synonym db";
    return -2;
  }
  if (regionRelationDB.open(regionRelationDbName)) {
    errMsg = "Failed to open region relationship db";
    return -2;
  }
  if (namespaceDB.open(namespaceDbName, myDB::cmp_lex)) {
    errMsg = "Failed to open brain region namespace db";
    return -2;
  }

  // close all tables now
  if (closeTables())
    return -3;

  return 0;
}

/* Returns true is any db file already exists, returns false otherwise. */
bool DBdata::validateDir()
{
  if (vb_fileexists(sysDbName)) {
    errMsg = sysDbName + " exists";
    return false;
  }
  if (vb_fileexists(userDbName)) {
    errMsg = userDbName + " exists";
    return false;
  }
  if (vb_fileexists(scoreNameDbName)) {
    errMsg = scoreNameDbName + " exists";
    return false;
  }
  if (vb_fileexists(scoreValueDbName)) {
    errMsg = scoreValueDbName + " exists";
    return false;
  }
  if (vb_fileexists(patientScoreDbName)) {
    errMsg = patientScoreDbName + " exists";
    return false;
  }
  if (vb_fileexists(sessionDbName)) {
    errMsg = sessionDbName + " exists";
    return false;
  }
  if (vb_fileexists(permissionDbName)) {
    errMsg = permissionDbName + " exists";
    return false;
  }
  if (vb_fileexists(patientDbName)) {
    errMsg = patientDbName + " exists";
    return false;
  }
  if (vb_fileexists(patientListDbName)) {
    errMsg = patientListDbName + " exists";
    return false;
  }
  // brain region related tables
  if (vb_fileexists(regionDbName)) {
    errMsg = regionDbName + " exists";
    return false;
  }
  if (vb_fileexists(synonymDbName)) {
    errMsg = synonymDbName + " exists";
    return false;
  }
  if (vb_fileexists(regionRelationDbName)) {
    errMsg = regionRelationDbName + " exists";
    return false;
  }
  if (vb_fileexists(namespaceDbName)) {
    errMsg = namespaceDbName + " exists";
    return false;
  }

  return true;
}



