/*
 * This file is part of Soprano Project
 *
 * Copyright (C) 2009-2010 Sebastian Trueg <trueg@kde.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "isql.h"

#include <QtCore/QTextStream>

#include <KDebug>


ISql::ISql( QObject* parent )
    : QObject( parent ),
      m_deleteSelf( false ),
      m_tmpFile( 0 )
{
    connect( &m_process, SIGNAL( finished( int, QProcess::ExitStatus ) ),
             this, SLOT( slotProcessFinished( int, QProcess::ExitStatus ) ) );
}


ISql::~ISql()
{
    delete m_tmpFile;
}


void ISql::executeCommand( int port, const QString& command )
{
    // init
    m_port = port;
    m_process.clearProgram();

    // write command file
    delete m_tmpFile;
    m_tmpFile = new KTemporaryFile();
    m_tmpFile->open();
    QTextStream s( m_tmpFile );
    s << command;
    m_tmpFile->close();

    // setup process
    m_process << "isql"
              << QString( "localhost:%1" ).arg( m_port )
              << "dba"
              << "dba"
              << m_tmpFile->fileName();

    // execute
    m_process.start();
}


void ISql::slotProcessFinished( int exitCode, QProcess::ExitStatus exitStatus )
{
    // cleanup
    delete m_tmpFile;
    m_tmpFile = 0;

    emit finished( exitCode == 0 && exitStatus == QProcess::NormalExit );

    if ( m_deleteSelf )
        deleteLater();
}


ISql* ISql::fireAndForgetCommand( int port, const QString& command )
{
    ISql* sql = new ISql();
    sql->m_deleteSelf = true;
    sql->executeCommand( port, command );
    return sql;
}

#include "isql.moc"
