/* $Id: VMM.cpp $ */
/** @file
 * VMM - The Virtual Machine Monitor Core.
 */

/*
 * Copyright (C) 2006-2007 Oracle Corporation
 *
 * This file is part of VirtualBox Open Source Edition (OSE), as
 * available from http://www.virtualbox.org. This file is free software;
 * you can redistribute it and/or modify it under the terms of the GNU
 * General Public License (GPL) as published by the Free Software
 * Foundation, in version 2 as it comes in the "COPYING" file of the
 * VirtualBox OSE distribution. VirtualBox OSE is distributed in the
 * hope that it will be useful, but WITHOUT ANY WARRANTY of any kind.
 */

//#define NO_SUPCALLR0VMM

/** @page pg_vmm        VMM - The Virtual Machine Monitor
 *
 * The VMM component is two things at the moment, it's a component doing a few
 * management and routing tasks, and it's the whole virtual machine monitor
 * thing.  For hysterical reasons, it is not doing all the management that one
 * would expect, this is instead done by @ref pg_vm.  We'll address this
 * misdesign eventually.
 *
 * @see grp_vmm, grp_vm
 *
 *
 * @section sec_vmmstate        VMM State
 *
 * @image html VM_Statechart_Diagram.gif
 *
 * To be written.
 *
 *
 * @subsection  subsec_vmm_init     VMM Initialization
 *
 * To be written.
 *
 *
 * @subsection  subsec_vmm_term     VMM Termination
 *
 * To be written.
 *
 */

/*******************************************************************************
*   Header Files                                                               *
*******************************************************************************/
#define LOG_GROUP LOG_GROUP_VMM
#include <VBox/vmm/vmm.h>
#include <VBox/vmm/vmapi.h>
#include <VBox/vmm/pgm.h>
#include <VBox/vmm/cfgm.h>
#include <VBox/vmm/pdmqueue.h>
#include <VBox/vmm/pdmcritsect.h>
#include <VBox/vmm/pdmapi.h>
#include <VBox/vmm/cpum.h>
#include <VBox/vmm/mm.h>
#include <VBox/vmm/iom.h>
#include <VBox/vmm/trpm.h>
#include <VBox/vmm/selm.h>
#include <VBox/vmm/em.h>
#include <VBox/sup.h>
#include <VBox/vmm/dbgf.h>
#include <VBox/vmm/csam.h>
#include <VBox/vmm/patm.h>
#include <VBox/vmm/rem.h>
#include <VBox/vmm/ssm.h>
#include <VBox/vmm/tm.h>
#include "VMMInternal.h"
#include "VMMSwitcher.h"
#include <VBox/vmm/vm.h>
#include <VBox/vmm/ftm.h>

#include <VBox/err.h>
#include <VBox/param.h>
#include <VBox/version.h>
#include <VBox/x86.h>
#include <VBox/vmm/hwaccm.h>
#include <iprt/assert.h>
#include <iprt/alloc.h>
#include <iprt/asm.h>
#include <iprt/time.h>
#include <iprt/semaphore.h>
#include <iprt/stream.h>
#include <iprt/string.h>
#include <iprt/stdarg.h>
#include <iprt/ctype.h>



/*******************************************************************************
*   Defined Constants And Macros                                               *
*******************************************************************************/
/** The saved state version. */
#define VMM_SAVED_STATE_VERSION     4
/** The saved state version used by v3.0 and earlier. (Teleportation) */
#define VMM_SAVED_STATE_VERSION_3_0 3


/*******************************************************************************
*   Internal Functions                                                         *
*******************************************************************************/
static int                  vmmR3InitStacks(PVM pVM);
static int                  vmmR3InitLoggers(PVM pVM);
static void                 vmmR3InitRegisterStats(PVM pVM);
static DECLCALLBACK(int)    vmmR3Save(PVM pVM, PSSMHANDLE pSSM);
static DECLCALLBACK(int)    vmmR3Load(PVM pVM, PSSMHANDLE pSSM, uint32_t uVersion, uint32_t uPass);
static DECLCALLBACK(void)   vmmR3YieldEMT(PVM pVM, PTMTIMER pTimer, void *pvUser);
static int                  vmmR3ServiceCallRing3Request(PVM pVM, PVMCPU pVCpu);
static DECLCALLBACK(void)   vmmR3InfoFF(PVM pVM, PCDBGFINFOHLP pHlp, const char *pszArgs);


/**
 * Initializes the VMM.
 *
 * @returns VBox status code.
 * @param   pVM         The VM to operate on.
 */
VMMR3_INT_DECL(int) VMMR3Init(PVM pVM)
{
    LogFlow(("VMMR3Init\n"));

    /*
     * Assert alignment, sizes and order.
     */
    AssertMsg(pVM->vmm.s.offVM == 0, ("Already initialized!\n"));
    AssertCompile(sizeof(pVM->vmm.s) <= sizeof(pVM->vmm.padding));
    AssertCompile(sizeof(pVM->aCpus[0].vmm.s) <= sizeof(pVM->aCpus[0].vmm.padding));

    /*
     * Init basic VM VMM members.
     */
    pVM->vmm.s.offVM = RT_OFFSETOF(VM, vmm);
    pVM->vmm.s.pahEvtRendezvousEnterOrdered     = NULL;
    pVM->vmm.s.hEvtRendezvousEnterOneByOne      = NIL_RTSEMEVENT;
    pVM->vmm.s.hEvtMulRendezvousEnterAllAtOnce  = NIL_RTSEMEVENTMULTI;
    pVM->vmm.s.hEvtMulRendezvousDone            = NIL_RTSEMEVENTMULTI;
    pVM->vmm.s.hEvtRendezvousDoneCaller         = NIL_RTSEMEVENT;

    /** @cfgm{YieldEMTInterval, uint32_t, 1, UINT32_MAX, 23, ms}
     * The EMT yield interval.  The EMT yielding is a hack we employ to play a
     * bit nicer with the rest of the system (like for instance the GUI).
     */
    int rc = CFGMR3QueryU32Def(CFGMR3GetRoot(pVM), "YieldEMTInterval", &pVM->vmm.s.cYieldEveryMillies,
                               23 /* Value arrived at after experimenting with the grub boot prompt. */);
    AssertMsgRCReturn(rc, ("Configuration error. Failed to query \"YieldEMTInterval\", rc=%Rrc\n", rc), rc);


    /** @cfgm{VMM/UsePeriodicPreemptionTimers, boolean, true}
     * Controls whether we employ per-cpu preemption timers to limit the time
     * spent executing guest code.  This option is not available on all
     * platforms and we will silently ignore this setting then.  If we are
     * running in VT-x mode, we will use the VMX-preemption timer instead of
     * this one when possible.
     */
    PCFGMNODE pCfgVMM = CFGMR3GetChild(CFGMR3GetRoot(pVM), "VMM");
    rc = CFGMR3QueryBoolDef(pCfgVMM, "UsePeriodicPreemptionTimers", &pVM->vmm.s.fUsePeriodicPreemptionTimers, true);
    AssertMsgRCReturn(rc, ("Configuration error. Failed to query \"VMM/UsePeriodicPreemptionTimers\", rc=%Rrc\n", rc), rc);

    /*
     * Initialize the VMM sync critical section and semaphores.
     */
    rc = RTCritSectInit(&pVM->vmm.s.CritSectSync);
    AssertRCReturn(rc, rc);
    pVM->vmm.s.pahEvtRendezvousEnterOrdered = (PRTSEMEVENT)MMR3HeapAlloc(pVM, MM_TAG_VMM, sizeof(RTSEMEVENT) * pVM->cCpus);
    if (!pVM->vmm.s.pahEvtRendezvousEnterOrdered)
        return VERR_NO_MEMORY;
    for (VMCPUID i = 0; i < pVM->cCpus; i++)
        pVM->vmm.s.pahEvtRendezvousEnterOrdered[i] = NIL_RTSEMEVENT;
    for (VMCPUID i = 0; i < pVM->cCpus; i++)
    {
        rc = RTSemEventCreate(&pVM->vmm.s.pahEvtRendezvousEnterOrdered[i]);
        AssertRCReturn(rc, rc);
    }
    rc = RTSemEventCreate(&pVM->vmm.s.hEvtRendezvousEnterOneByOne);
    AssertRCReturn(rc, rc);
    rc = RTSemEventMultiCreate(&pVM->vmm.s.hEvtMulRendezvousEnterAllAtOnce);
    AssertRCReturn(rc, rc);
    rc = RTSemEventMultiCreate(&pVM->vmm.s.hEvtMulRendezvousDone);
    AssertRCReturn(rc, rc);
    rc = RTSemEventCreate(&pVM->vmm.s.hEvtRendezvousDoneCaller);
    AssertRCReturn(rc, rc);

    /* GC switchers are enabled by default. Turned off by HWACCM. */
    pVM->vmm.s.fSwitcherDisabled = false;

    /*
     * Register the saved state data unit.
     */
    rc = SSMR3RegisterInternal(pVM, "vmm", 1, VMM_SAVED_STATE_VERSION, VMM_STACK_SIZE + sizeof(RTGCPTR),
                               NULL, NULL, NULL,
                               NULL, vmmR3Save, NULL,
                               NULL, vmmR3Load, NULL);
    if (RT_FAILURE(rc))
        return rc;

    /*
     * Register the Ring-0 VM handle with the session for fast ioctl calls.
     */
    rc = SUPR3SetVMForFastIOCtl(pVM->pVMR0);
    if (RT_FAILURE(rc))
        return rc;

    /*
     * Init various sub-components.
     */
    rc = vmmR3SwitcherInit(pVM);
    if (RT_SUCCESS(rc))
    {
        rc = vmmR3InitStacks(pVM);
        if (RT_SUCCESS(rc))
        {
            rc = vmmR3InitLoggers(pVM);

#ifdef VBOX_WITH_NMI
            /*
             * Allocate mapping for the host APIC.
             */
            if (RT_SUCCESS(rc))
            {
                rc = MMR3HyperReserve(pVM, PAGE_SIZE, "Host APIC", &pVM->vmm.s.GCPtrApicBase);
                AssertRC(rc);
            }
#endif
            if (RT_SUCCESS(rc))
            {
                /*
                 * Debug info and statistics.
                 */
                DBGFR3InfoRegisterInternal(pVM, "ff", "Displays the current Forced actions Flags.", vmmR3InfoFF);
                vmmR3InitRegisterStats(pVM);

                return VINF_SUCCESS;
            }
        }
        /** @todo: Need failure cleanup. */

        //more todo in here?
        //if (RT_SUCCESS(rc))
        //{
        //}
        //int rc2 = vmmR3TermCoreCode(pVM);
        //AssertRC(rc2));
    }

    return rc;
}


/**
 * Allocate & setup the VMM RC stack(s) (for EMTs).
 *
 * The stacks are also used for long jumps in Ring-0.
 *
 * @returns VBox status code.
 * @param   pVM     Pointer to the shared VM structure.
 *
 * @remarks The optional guard page gets it protection setup up during R3 init
 *          completion because of init order issues.
 */
static int vmmR3InitStacks(PVM pVM)
{
    int rc = VINF_SUCCESS;
#ifdef VMM_R0_SWITCH_STACK
    uint32_t fFlags = MMHYPER_AONR_FLAGS_KERNEL_MAPPING;
#else
    uint32_t fFlags = 0;
#endif

    for (VMCPUID idCpu = 0; idCpu < pVM->cCpus; idCpu++)
    {
        PVMCPU pVCpu = &pVM->aCpus[idCpu];

#ifdef VBOX_STRICT_VMM_STACK
        rc = MMR3HyperAllocOnceNoRelEx(pVM, PAGE_SIZE + VMM_STACK_SIZE + PAGE_SIZE,
#else
        rc = MMR3HyperAllocOnceNoRelEx(pVM, VMM_STACK_SIZE,
#endif
                                       PAGE_SIZE, MM_TAG_VMM, fFlags, (void **)&pVCpu->vmm.s.pbEMTStackR3);
        if (RT_SUCCESS(rc))
        {
#ifdef VBOX_STRICT_VMM_STACK
            pVCpu->vmm.s.pbEMTStackR3 += PAGE_SIZE;
#endif
#ifdef VBOX_WITH_2X_4GB_ADDR_SPACE
            /* MMHyperR3ToR0 returns R3 when not doing hardware assisted virtualization. */
            if (!VMMIsHwVirtExtForced(pVM))
                pVCpu->vmm.s.CallRing3JmpBufR0.pvSavedStack = NIL_RTR0PTR;
            else
#endif
                pVCpu->vmm.s.CallRing3JmpBufR0.pvSavedStack = MMHyperR3ToR0(pVM, pVCpu->vmm.s.pbEMTStackR3);
            pVCpu->vmm.s.pbEMTStackRC       = MMHyperR3ToRC(pVM, pVCpu->vmm.s.pbEMTStackR3);
            pVCpu->vmm.s.pbEMTStackBottomRC = pVCpu->vmm.s.pbEMTStackRC + VMM_STACK_SIZE;
            AssertRelease(pVCpu->vmm.s.pbEMTStackRC);

            CPUMSetHyperESP(pVCpu, pVCpu->vmm.s.pbEMTStackBottomRC);
        }
    }

    return rc;
}


/**
 * Initialize the loggers.
 *
 * @returns VBox status code.
 * @param   pVM         Pointer to the shared VM structure.
 */
static int vmmR3InitLoggers(PVM pVM)
{
    int rc;

    /*
     * Allocate RC & R0 Logger instances (they are finalized in the relocator).
     */
#ifdef LOG_ENABLED
    PRTLOGGER pLogger = RTLogDefaultInstance();
    if (pLogger)
    {
        pVM->vmm.s.cbRCLogger = RT_OFFSETOF(RTLOGGERRC, afGroups[pLogger->cGroups]);
        rc = MMR3HyperAllocOnceNoRel(pVM, pVM->vmm.s.cbRCLogger, 0, MM_TAG_VMM, (void **)&pVM->vmm.s.pRCLoggerR3);
        if (RT_FAILURE(rc))
            return rc;
        pVM->vmm.s.pRCLoggerRC = MMHyperR3ToRC(pVM, pVM->vmm.s.pRCLoggerR3);

# ifdef VBOX_WITH_R0_LOGGING
        for (VMCPUID i = 0; i < pVM->cCpus; i++)
        {
            PVMCPU pVCpu = &pVM->aCpus[i];

            rc = MMR3HyperAllocOnceNoRelEx(pVM, RT_OFFSETOF(VMMR0LOGGER, Logger.afGroups[pLogger->cGroups]),
                                           0, MM_TAG_VMM, MMHYPER_AONR_FLAGS_KERNEL_MAPPING,
                                           (void **)&pVCpu->vmm.s.pR0LoggerR3);
            if (RT_FAILURE(rc))
                return rc;
            pVCpu->vmm.s.pR0LoggerR3->pVM = pVM->pVMR0;
            //pVCpu->vmm.s.pR0LoggerR3->fCreated = false;
            pVCpu->vmm.s.pR0LoggerR3->cbLogger = RT_OFFSETOF(RTLOGGER, afGroups[pLogger->cGroups]);
            pVCpu->vmm.s.pR0LoggerR0 = MMHyperR3ToR0(pVM, pVCpu->vmm.s.pR0LoggerR3);
        }
# endif
    }
#endif /* LOG_ENABLED */

#ifdef VBOX_WITH_RC_RELEASE_LOGGING
    /*
     * Allocate RC release logger instances (finalized in the relocator).
     */
    PRTLOGGER pRelLogger = RTLogRelDefaultInstance();
    if (pRelLogger)
    {
        pVM->vmm.s.cbRCRelLogger = RT_OFFSETOF(RTLOGGERRC, afGroups[pRelLogger->cGroups]);
        rc = MMR3HyperAllocOnceNoRel(pVM, pVM->vmm.s.cbRCRelLogger, 0, MM_TAG_VMM, (void **)&pVM->vmm.s.pRCRelLoggerR3);
        if (RT_FAILURE(rc))
            return rc;
        pVM->vmm.s.pRCRelLoggerRC = MMHyperR3ToRC(pVM, pVM->vmm.s.pRCRelLoggerR3);
    }
#endif /* VBOX_WITH_RC_RELEASE_LOGGING */
    return VINF_SUCCESS;
}


/**
 * VMMR3Init worker that register the statistics with STAM.
 *
 * @param   pVM         The shared VM structure.
 */
static void vmmR3InitRegisterStats(PVM pVM)
{
    /*
     * Statistics.
     */
    STAM_REG(pVM, &pVM->vmm.s.StatRunRC,                    STAMTYPE_COUNTER, "/VMM/RunRC",                     STAMUNIT_OCCURENCES, "Number of context switches.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetNormal,              STAMTYPE_COUNTER, "/VMM/RZRet/Normal",              STAMUNIT_OCCURENCES, "Number of VINF_SUCCESS returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetInterrupt,           STAMTYPE_COUNTER, "/VMM/RZRet/Interrupt",           STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_INTERRUPT returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetInterruptHyper,      STAMTYPE_COUNTER, "/VMM/RZRet/InterruptHyper",      STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_INTERRUPT_HYPER returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetGuestTrap,           STAMTYPE_COUNTER, "/VMM/RZRet/GuestTrap",           STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_GUEST_TRAP returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetRingSwitch,          STAMTYPE_COUNTER, "/VMM/RZRet/RingSwitch",          STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_RING_SWITCH returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetRingSwitchInt,       STAMTYPE_COUNTER, "/VMM/RZRet/RingSwitchInt",       STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_RING_SWITCH_INT returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetStaleSelector,       STAMTYPE_COUNTER, "/VMM/RZRet/StaleSelector",       STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_STALE_SELECTOR returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetIRETTrap,            STAMTYPE_COUNTER, "/VMM/RZRet/IRETTrap",            STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_IRET_TRAP returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetEmulate,             STAMTYPE_COUNTER, "/VMM/RZRet/Emulate",             STAMUNIT_OCCURENCES, "Number of VINF_EM_EXECUTE_INSTRUCTION returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetIOBlockEmulate,      STAMTYPE_COUNTER, "/VMM/RZRet/EmulateIOBlock",      STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_EMULATE_IO_BLOCK returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPatchEmulate,        STAMTYPE_COUNTER, "/VMM/RZRet/PatchEmulate",        STAMUNIT_OCCURENCES, "Number of VINF_PATCH_EMULATE_INSTR returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetIORead,              STAMTYPE_COUNTER, "/VMM/RZRet/IORead",              STAMUNIT_OCCURENCES, "Number of VINF_IOM_HC_IOPORT_READ returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetIOWrite,             STAMTYPE_COUNTER, "/VMM/RZRet/IOWrite",             STAMUNIT_OCCURENCES, "Number of VINF_IOM_HC_IOPORT_WRITE returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetMMIORead,            STAMTYPE_COUNTER, "/VMM/RZRet/MMIORead",            STAMUNIT_OCCURENCES, "Number of VINF_IOM_HC_MMIO_READ returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetMMIOWrite,           STAMTYPE_COUNTER, "/VMM/RZRet/MMIOWrite",           STAMUNIT_OCCURENCES, "Number of VINF_IOM_HC_MMIO_WRITE returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetMMIOReadWrite,       STAMTYPE_COUNTER, "/VMM/RZRet/MMIOReadWrite",       STAMUNIT_OCCURENCES, "Number of VINF_IOM_HC_MMIO_READ_WRITE returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetMMIOPatchRead,       STAMTYPE_COUNTER, "/VMM/RZRet/MMIOPatchRead",       STAMUNIT_OCCURENCES, "Number of VINF_IOM_HC_MMIO_PATCH_READ returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetMMIOPatchWrite,      STAMTYPE_COUNTER, "/VMM/RZRet/MMIOPatchWrite",      STAMUNIT_OCCURENCES, "Number of VINF_IOM_HC_MMIO_PATCH_WRITE returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetLDTFault,            STAMTYPE_COUNTER, "/VMM/RZRet/LDTFault",            STAMUNIT_OCCURENCES, "Number of VINF_EM_EXECUTE_INSTRUCTION_GDT_FAULT returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetGDTFault,            STAMTYPE_COUNTER, "/VMM/RZRet/GDTFault",            STAMUNIT_OCCURENCES, "Number of VINF_EM_EXECUTE_INSTRUCTION_LDT_FAULT returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetIDTFault,            STAMTYPE_COUNTER, "/VMM/RZRet/IDTFault",            STAMUNIT_OCCURENCES, "Number of VINF_EM_EXECUTE_INSTRUCTION_IDT_FAULT returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetTSSFault,            STAMTYPE_COUNTER, "/VMM/RZRet/TSSFault",            STAMUNIT_OCCURENCES, "Number of VINF_EM_EXECUTE_INSTRUCTION_TSS_FAULT returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPDFault,             STAMTYPE_COUNTER, "/VMM/RZRet/PDFault",             STAMUNIT_OCCURENCES, "Number of VINF_EM_EXECUTE_INSTRUCTION_PD_FAULT returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetCSAMTask,            STAMTYPE_COUNTER, "/VMM/RZRet/CSAMTask",            STAMUNIT_OCCURENCES, "Number of VINF_CSAM_PENDING_ACTION returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetSyncCR3,             STAMTYPE_COUNTER, "/VMM/RZRet/SyncCR",              STAMUNIT_OCCURENCES, "Number of VINF_PGM_SYNC_CR3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetMisc,                STAMTYPE_COUNTER, "/VMM/RZRet/Misc",                STAMUNIT_OCCURENCES, "Number of misc returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPatchInt3,           STAMTYPE_COUNTER, "/VMM/RZRet/PatchInt3",           STAMUNIT_OCCURENCES, "Number of VINF_PATM_PATCH_INT3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPatchPF,             STAMTYPE_COUNTER, "/VMM/RZRet/PatchPF",             STAMUNIT_OCCURENCES, "Number of VINF_PATM_PATCH_TRAP_PF returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPatchGP,             STAMTYPE_COUNTER, "/VMM/RZRet/PatchGP",             STAMUNIT_OCCURENCES, "Number of VINF_PATM_PATCH_TRAP_GP returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPatchIretIRQ,        STAMTYPE_COUNTER, "/VMM/RZRet/PatchIret",           STAMUNIT_OCCURENCES, "Number of VINF_PATM_PENDING_IRQ_AFTER_IRET returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetRescheduleREM,       STAMTYPE_COUNTER, "/VMM/RZRet/ScheduleREM",         STAMUNIT_OCCURENCES, "Number of VINF_EM_RESCHEDULE_REM returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3,                STAMTYPE_COUNTER, "/VMM/RZRet/ToR3",                STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3Unknown,         STAMTYPE_COUNTER, "/VMM/RZRet/ToR3/Unknown",        STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3TMVirt,          STAMTYPE_COUNTER, "/VMM/RZRet/ToR3/TMVirt",         STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3HandyPages,      STAMTYPE_COUNTER, "/VMM/RZRet/ToR3/Handy",          STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3PDMQueues,       STAMTYPE_COUNTER, "/VMM/RZRet/ToR3/PDMQueue",       STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3Rendezvous,      STAMTYPE_COUNTER, "/VMM/RZRet/ToR3/Rendezvous",     STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3Timer,           STAMTYPE_COUNTER, "/VMM/RZRet/ToR3/Timer",          STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3DMA,             STAMTYPE_COUNTER, "/VMM/RZRet/ToR3/DMA",            STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetToR3CritSect,        STAMTYPE_COUNTER, "/VMM/RZRet/ToR3/CritSect",       STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TO_R3 returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetTimerPending,        STAMTYPE_COUNTER, "/VMM/RZRet/TimerPending",        STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_TIMER_PENDING returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetInterruptPending,    STAMTYPE_COUNTER, "/VMM/RZRet/InterruptPending",    STAMUNIT_OCCURENCES, "Number of VINF_EM_RAW_INTERRUPT_PENDING returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPATMDuplicateFn,     STAMTYPE_COUNTER, "/VMM/RZRet/PATMDuplicateFn",     STAMUNIT_OCCURENCES, "Number of VINF_PATM_DUPLICATE_FUNCTION returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPGMChangeMode,       STAMTYPE_COUNTER, "/VMM/RZRet/PGMChangeMode",       STAMUNIT_OCCURENCES, "Number of VINF_PGM_CHANGE_MODE returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPGMFlushPending,     STAMTYPE_COUNTER, "/VMM/RZRet/PGMFlushPending",     STAMUNIT_OCCURENCES, "Number of VINF_PGM_POOL_FLUSH_PENDING returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPendingRequest,      STAMTYPE_COUNTER, "/VMM/RZRet/PendingRequest",      STAMUNIT_OCCURENCES, "Number of VINF_EM_PENDING_REQUEST returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetPatchTPR,            STAMTYPE_COUNTER, "/VMM/RZRet/PatchTPR",            STAMUNIT_OCCURENCES, "Number of VINF_EM_HWACCM_PATCH_TPR_INSTR returns.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZRetCallRing3,           STAMTYPE_COUNTER, "/VMM/RZCallR3/Misc",             STAMUNIT_OCCURENCES, "Number of Other ring-3 calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallPDMLock,            STAMTYPE_COUNTER, "/VMM/RZCallR3/PDMLock",          STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_PDM_LOCK calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallPGMLock,            STAMTYPE_COUNTER, "/VMM/RZCallR3/PGMLock",          STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_PGM_LOCK calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallPGMPoolGrow,        STAMTYPE_COUNTER, "/VMM/RZCallR3/PGMPoolGrow",      STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_PGM_POOL_GROW calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallPGMMapChunk,        STAMTYPE_COUNTER, "/VMM/RZCallR3/PGMMapChunk",      STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_PGM_MAP_CHUNK calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallPGMAllocHandy,      STAMTYPE_COUNTER, "/VMM/RZCallR3/PGMAllocHandy",    STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_PGM_ALLOCATE_HANDY_PAGES calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallRemReplay,          STAMTYPE_COUNTER, "/VMM/RZCallR3/REMReplay",        STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_REM_REPLAY_HANDLER_NOTIFICATIONS calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallLogFlush,           STAMTYPE_COUNTER, "/VMM/RZCallR3/VMMLogFlush",      STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_VMM_LOGGER_FLUSH calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallVMSetError,         STAMTYPE_COUNTER, "/VMM/RZCallR3/VMSetError",       STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_VM_SET_ERROR calls.");
    STAM_REG(pVM, &pVM->vmm.s.StatRZCallVMSetRuntimeError,  STAMTYPE_COUNTER, "/VMM/RZCallR3/VMRuntimeError",   STAMUNIT_OCCURENCES, "Number of VMMCALLRING3_VM_SET_RUNTIME_ERROR calls.");

#ifdef VBOX_WITH_STATISTICS
    for (VMCPUID i = 0; i < pVM->cCpus; i++)
    {
        STAMR3RegisterF(pVM, &pVM->aCpus[i].vmm.s.CallRing3JmpBufR0.cbUsedMax,  STAMTYPE_U32_RESET, STAMVISIBILITY_ALWAYS, STAMUNIT_BYTES,      "Max amount of stack used.", "/VMM/Stack/CPU%u/Max", i);
        STAMR3RegisterF(pVM, &pVM->aCpus[i].vmm.s.CallRing3JmpBufR0.cbUsedAvg,  STAMTYPE_U32,       STAMVISIBILITY_ALWAYS, STAMUNIT_BYTES,      "Average stack usage.",      "/VMM/Stack/CPU%u/Avg", i);
        STAMR3RegisterF(pVM, &pVM->aCpus[i].vmm.s.CallRing3JmpBufR0.cUsedTotal, STAMTYPE_U64,       STAMVISIBILITY_ALWAYS, STAMUNIT_OCCURENCES, "Number of stack usages.",   "/VMM/Stack/CPU%u/Uses", i);
    }
#endif
}


/**
 * Initializes the R0 VMM.
 *
 * @returns VBox status code.
 * @param   pVM         The VM to operate on.
 */
VMMR3_INT_DECL(int) VMMR3InitR0(PVM pVM)
{
    int    rc;
    PVMCPU pVCpu = VMMGetCpu(pVM);
    Assert(pVCpu && pVCpu->idCpu == 0);

#ifdef LOG_ENABLED
    /*
     * Initialize the ring-0 logger if we haven't done so yet.
     */
    if (    pVCpu->vmm.s.pR0LoggerR3
        &&  !pVCpu->vmm.s.pR0LoggerR3->fCreated)
    {
        rc = VMMR3UpdateLoggers(pVM);
        if (RT_FAILURE(rc))
            return rc;
    }
#endif

    /*
     * Call Ring-0 entry with init code.
     */
    for (;;)
    {
#ifdef NO_SUPCALLR0VMM
        //rc = VERR_GENERAL_FAILURE;
        rc = VINF_SUCCESS;
#else
        rc = SUPR3CallVMMR0Ex(pVM->pVMR0, 0 /*idCpu*/, VMMR0_DO_VMMR0_INIT, VMMGetSvnRev(), NULL);
#endif
        /*
         * Flush the logs.
         */
#ifdef LOG_ENABLED
        if (    pVCpu->vmm.s.pR0LoggerR3
            &&  pVCpu->vmm.s.pR0LoggerR3->Logger.offScratch > 0)
            RTLogFlushToLogger(&pVCpu->vmm.s.pR0LoggerR3->Logger, NULL);
#endif
        if (rc != VINF_VMM_CALL_HOST)
            break;
        rc = vmmR3ServiceCallRing3Request(pVM, pVCpu);
        if (RT_FAILURE(rc) || (rc >= VINF_EM_FIRST && rc <= VINF_EM_LAST))
            break;
        /* Resume R0 */
    }

    if (RT_FAILURE(rc) || (rc >= VINF_EM_FIRST && rc <= VINF_EM_LAST))
    {
        LogRel(("R0 init failed, rc=%Rra\n", rc));
        if (RT_SUCCESS(rc))
            rc = VERR_INTERNAL_ERROR;
    }
    return rc;
}


/**
 * Initializes the RC VMM.
 *
 * @returns VBox status code.
 * @param   pVM         The VM to operate on.
 */
VMMR3_INT_DECL(int) VMMR3InitRC(PVM pVM)
{
    PVMCPU pVCpu = VMMGetCpu(pVM);
    Assert(pVCpu && pVCpu->idCpu == 0);

    /* In VMX mode, there's no need to init RC. */
    if (pVM->vmm.s.fSwitcherDisabled)
        return VINF_SUCCESS;

    AssertReturn(pVM->cCpus == 1, VERR_RAW_MODE_INVALID_SMP);

    /*
     * Call VMMGCInit():
     *      -# resolve the address.
     *      -# setup stackframe and EIP to use the trampoline.
     *      -# do a generic hypervisor call.
     */
    RTRCPTR RCPtrEP;
    int rc = PDMR3LdrGetSymbolRC(pVM, VMMGC_MAIN_MODULE_NAME, "VMMGCEntry", &RCPtrEP);
    if (RT_SUCCESS(rc))
    {
        CPUMHyperSetCtxCore(pVCpu, NULL);
        CPUMSetHyperESP(pVCpu, pVCpu->vmm.s.pbEMTStackBottomRC); /* Clear the stack. */
        uint64_t u64TS = RTTimeProgramStartNanoTS();
        CPUMPushHyper(pVCpu, (uint32_t)(u64TS >> 32));    /* Param 3: The program startup TS - Hi. */
        CPUMPushHyper(pVCpu, (uint32_t)u64TS);            /* Param 3: The program startup TS - Lo. */
        CPUMPushHyper(pVCpu, VMMGetSvnRev());             /* Param 2: Version argument. */
        CPUMPushHyper(pVCpu, VMMGC_DO_VMMGC_INIT);        /* Param 1: Operation. */
        CPUMPushHyper(pVCpu, pVM->pVMRC);                 /* Param 0: pVM */
        CPUMPushHyper(pVCpu, 5 * sizeof(RTRCPTR));        /* trampoline param: stacksize.  */
        CPUMPushHyper(pVCpu, RCPtrEP);                    /* Call EIP. */
        CPUMSetHyperEIP(pVCpu, pVM->vmm.s.pfnCallTrampolineRC);
        Assert(CPUMGetHyperCR3(pVCpu) && CPUMGetHyperCR3(pVCpu) == PGMGetHyperCR3(pVCpu));

        for (;;)
        {
#ifdef NO_SUPCALLR0VMM
            //rc = VERR_GENERAL_FAILURE;
            rc = VINF_SUCCESS;
#else
            rc = SUPR3CallVMMR0(pVM->pVMR0, 0 /* VCPU 0 */, VMMR0_DO_CALL_HYPERVISOR, NULL);
#endif
#ifdef LOG_ENABLED
            PRTLOGGERRC pLogger = pVM->vmm.s.pRCLoggerR3;
            if (    pLogger
                &&  pLogger->offScratch > 0)
                RTLogFlushRC(NULL, pLogger);
#endif
#ifdef VBOX_WITH_RC_RELEASE_LOGGING
            PRTLOGGERRC pRelLogger = pVM->vmm.s.pRCRelLoggerR3;
            if (RT_UNLIKELY(pRelLogger && pRelLogger->offScratch > 0))
                RTLogFlushRC(RTLogRelDefaultInstance(), pRelLogger);
#endif
            if (rc != VINF_VMM_CALL_HOST)
                break;
            rc = vmmR3ServiceCallRing3Request(pVM, pVCpu);
            if (RT_FAILURE(rc) || (rc >= VINF_EM_FIRST && rc <= VINF_EM_LAST))
                break;
        }

        if (RT_FAILURE(rc) || (rc >= VINF_EM_FIRST && rc <= VINF_EM_LAST))
        {
            VMMR3FatalDump(pVM, pVCpu, rc);
            if (rc >= VINF_EM_FIRST && rc <= VINF_EM_LAST)
                rc = VERR_INTERNAL_ERROR;
        }
        AssertRC(rc);
    }
    return rc;
}


/**
 * Called when an init phase completes.
 *
 * @returns VBox status code.
 * @param   pVM                 The VM handle.
 * @param   enmWhat             Which init phase.
 */
VMMR3_INT_DECL(int) VMMR3InitCompleted(PVM pVM, VMINITCOMPLETED enmWhat)
{
    int rc = VINF_SUCCESS;

    switch (enmWhat)
    {
        case VMINITCOMPLETED_RING3:
        {
            /*
             * Set page attributes to r/w for stack pages.
             */
            for (VMCPUID idCpu = 0; idCpu < pVM->cCpus; idCpu++)
            {
                rc = PGMMapSetPage(pVM, pVM->aCpus[idCpu].vmm.s.pbEMTStackRC, VMM_STACK_SIZE,
                                   X86_PTE_P | X86_PTE_A | X86_PTE_D | X86_PTE_RW);
                AssertRCReturn(rc, rc);
            }

            /*
             * Create the EMT yield timer.
             */
            rc = TMR3TimerCreateInternal(pVM, TMCLOCK_REAL, vmmR3YieldEMT, NULL, "EMT Yielder", &pVM->vmm.s.pYieldTimer);
            AssertRCReturn(rc, rc);

            rc = TMTimerSetMillies(pVM->vmm.s.pYieldTimer, pVM->vmm.s.cYieldEveryMillies);
            AssertRCReturn(rc, rc);

#ifdef VBOX_WITH_NMI
            /*
             * Map the host APIC into GC - This is AMD/Intel + Host OS specific!
             */
            rc = PGMMap(pVM, pVM->vmm.s.GCPtrApicBase, 0xfee00000, PAGE_SIZE,
                        X86_PTE_P | X86_PTE_RW | X86_PTE_PWT | X86_PTE_PCD | X86_PTE_A | X86_PTE_D);
            AssertRCReturn(rc, rc);
#endif

#ifdef VBOX_STRICT_VMM_STACK
            /*
             * Setup the stack guard pages: Two inaccessible pages at each sides of the
             * stack to catch over/under-flows.
             */
            for (VMCPUID idCpu = 0; idCpu < pVM->cCpus; idCpu++)
            {
                uint8_t *pbEMTStackR3 = pVM->aCpus[idCpu].vmm.s.pbEMTStackR3;

                memset(pbEMTStackR3 - PAGE_SIZE, 0xcc, PAGE_SIZE);
                MMR3HyperSetGuard(pVM, pbEMTStackR3 - PAGE_SIZE, PAGE_SIZE, true /*fSet*/);

                memset(pbEMTStackR3 + VMM_STACK_SIZE, 0xcc, PAGE_SIZE);
                MMR3HyperSetGuard(pVM, pbEMTStackR3 + VMM_STACK_SIZE, PAGE_SIZE, true /*fSet*/);
            }
            pVM->vmm.s.fStackGuardsStationed = true;
#endif
            break;
        }

        case VMINITCOMPLETED_RING0:
        {
            /*
             * Disable the periodic preemption timers if we can use the
             * VMX-preemption timer instead.
             */
            if (   pVM->vmm.s.fUsePeriodicPreemptionTimers
                && HWACCMR3IsVmxPreemptionTimerUsed(pVM))
                pVM->vmm.s.fUsePeriodicPreemptionTimers = false;
            LogRel(("VMM: fUsePeriodicPreemptionTimers=%RTbool\n", pVM->vmm.s.fUsePeriodicPreemptionTimers));
            break;
        }

        default: /* shuts up gcc */
            break;
    }

    return rc;
}


/**
 * Terminate the VMM bits.
 *
 * @returns VINF_SUCCESS.
 * @param   pVM         The VM handle.
 */
VMMR3_INT_DECL(int) VMMR3Term(PVM pVM)
{
    PVMCPU pVCpu = VMMGetCpu(pVM);
    Assert(pVCpu && pVCpu->idCpu == 0);

    /*
     * Call Ring-0 entry with termination code.
     */
    int rc;
    for (;;)
    {
#ifdef NO_SUPCALLR0VMM
        //rc = VERR_GENERAL_FAILURE;
        rc = VINF_SUCCESS;
#else
        rc = SUPR3CallVMMR0Ex(pVM->pVMR0, 0 /*idCpu*/, VMMR0_DO_VMMR0_TERM, 0, NULL);
#endif
        /*
         * Flush the logs.
         */
#ifdef LOG_ENABLED
        if (    pVCpu->vmm.s.pR0LoggerR3
            &&  pVCpu->vmm.s.pR0LoggerR3->Logger.offScratch > 0)
            RTLogFlushToLogger(&pVCpu->vmm.s.pR0LoggerR3->Logger, NULL);
#endif
        if (rc != VINF_VMM_CALL_HOST)
            break;
        rc = vmmR3ServiceCallRing3Request(pVM, pVCpu);
        if (RT_FAILURE(rc) || (rc >= VINF_EM_FIRST && rc <= VINF_EM_LAST))
            break;
        /* Resume R0 */
    }
    if (RT_FAILURE(rc) || (rc >= VINF_EM_FIRST && rc <= VINF_EM_LAST))
    {
        LogRel(("VMMR3Term: R0 term failed, rc=%Rra. (warning)\n", rc));
        if (RT_SUCCESS(rc))
            rc = VERR_INTERNAL_ERROR;
    }

    RTCritSectDelete(&pVM->vmm.s.CritSectSync);
    for (VMCPUID i = 0; i < pVM->cCpus; i++)
    {
        RTSemEventDestroy(pVM->vmm.s.pahEvtRendezvousEnterOrdered[i]);
        pVM->vmm.s.pahEvtRendezvousEnterOrdered[i] = NIL_RTSEMEVENT;
    }
    RTSemEventDestroy(pVM->vmm.s.hEvtRendezvousEnterOneByOne);
    pVM->vmm.s.hEvtRendezvousEnterOneByOne = NIL_RTSEMEVENT;
    RTSemEventMultiDestroy(pVM->vmm.s.hEvtMulRendezvousEnterAllAtOnce);
    pVM->vmm.s.hEvtMulRendezvousEnterAllAtOnce = NIL_RTSEMEVENTMULTI;
    RTSemEventMultiDestroy(pVM->vmm.s.hEvtMulRendezvousDone);
    pVM->vmm.s.hEvtMulRendezvousDone = NIL_RTSEMEVENTMULTI;
    RTSemEventDestroy(pVM->vmm.s.hEvtRendezvousDoneCaller);
    pVM->vmm.s.hEvtRendezvousDoneCaller = NIL_RTSEMEVENT;

#ifdef VBOX_STRICT_VMM_STACK
    /*
     * Make the two stack guard pages present again.
     */
    if (pVM->vmm.s.fStackGuardsStationed)
    {
        for (VMCPUID i = 0; i < pVM->cCpus; i++)
        {
            uint8_t *pbEMTStackR3 = pVM->aCpus[i].vmm.s.pbEMTStackR3;
            MMR3HyperSetGuard(pVM, pbEMTStackR3 - PAGE_SIZE,      PAGE_SIZE, false /*fSet*/);
            MMR3HyperSetGuard(pVM, pbEMTStackR3 + VMM_STACK_SIZE, PAGE_SIZE, false /*fSet*/);
        }
        pVM->vmm.s.fStackGuardsStationed = false;
    }
#endif
    return rc;
}


/**
 * Applies relocations to data and code managed by this
 * component. This function will be called at init and
 * whenever the VMM need to relocate it self inside the GC.
 *
 * The VMM will need to apply relocations to the core code.
 *
 * @param   pVM         The VM handle.
 * @param   offDelta    The relocation delta.
 */
VMMR3_INT_DECL(void) VMMR3Relocate(PVM pVM, RTGCINTPTR offDelta)
{
    LogFlow(("VMMR3Relocate: offDelta=%RGv\n", offDelta));

    /*
     * Recalc the RC address.
     */
#ifdef VBOX_WITH_RAW_MODE
    pVM->vmm.s.pvCoreCodeRC = MMHyperR3ToRC(pVM, pVM->vmm.s.pvCoreCodeR3);
#endif

    /*
     * The stack.
     */
    for (VMCPUID i = 0; i < pVM->cCpus; i++)
    {
        PVMCPU pVCpu = &pVM->aCpus[i];

        CPUMSetHyperESP(pVCpu, CPUMGetHyperESP(pVCpu) + offDelta);

        pVCpu->vmm.s.pbEMTStackRC       = MMHyperR3ToRC(pVM, pVCpu->vmm.s.pbEMTStackR3);
        pVCpu->vmm.s.pbEMTStackBottomRC = pVCpu->vmm.s.pbEMTStackRC + VMM_STACK_SIZE;
    }

    /*
     * All the switchers.
     */
    vmmR3SwitcherRelocate(pVM, offDelta);

    /*
     * Get other RC entry points.
     */
    int rc = PDMR3LdrGetSymbolRC(pVM, VMMGC_MAIN_MODULE_NAME, "CPUMGCResumeGuest", &pVM->vmm.s.pfnCPUMRCResumeGuest);
    AssertReleaseMsgRC(rc, ("CPUMGCResumeGuest not found! rc=%Rra\n", rc));

    rc = PDMR3LdrGetSymbolRC(pVM, VMMGC_MAIN_MODULE_NAME, "CPUMGCResumeGuestV86", &pVM->vmm.s.pfnCPUMRCResumeGuestV86);
    AssertReleaseMsgRC(rc, ("CPUMGCResumeGuestV86 not found! rc=%Rra\n", rc));

    /*
     * Update the logger.
     */
    VMMR3UpdateLoggers(pVM);
}


/**
 * Updates the settings for the RC and R0 loggers.
 *
 * @returns VBox status code.
 * @param   pVM     The VM handle.
 */
VMMR3_INT_DECL(int) VMMR3UpdateLoggers(PVM pVM)
{
    /*
     * Simply clone the logger instance (for RC).
     */
    int rc = VINF_SUCCESS;
    RTRCPTR RCPtrLoggerFlush = 0;

    if (pVM->vmm.s.pRCLoggerR3
#ifdef VBOX_WITH_RC_RELEASE_LOGGING
        || pVM->vmm.s.pRCRelLoggerR3
#endif
       )
    {
        rc = PDMR3LdrGetSymbolRC(pVM, VMMGC_MAIN_MODULE_NAME, "vmmGCLoggerFlush", &RCPtrLoggerFlush);
        AssertReleaseMsgRC(rc, ("vmmGCLoggerFlush not found! rc=%Rra\n", rc));
    }

    if (pVM->vmm.s.pRCLoggerR3)
    {
        RTRCPTR RCPtrLoggerWrapper = 0;
        rc = PDMR3LdrGetSymbolRC(pVM, VMMGC_MAIN_MODULE_NAME, "vmmGCLoggerWrapper", &RCPtrLoggerWrapper);
        AssertReleaseMsgRC(rc, ("vmmGCLoggerWrapper not found! rc=%Rra\n", rc));

        pVM->vmm.s.pRCLoggerRC = MMHyperR3ToRC(pVM, pVM->vmm.s.pRCLoggerR3);
        rc = RTLogCloneRC(NULL /* default */, pVM->vmm.s.pRCLoggerR3, pVM->vmm.s.cbRCLogger,
                          RCPtrLoggerWrapper,  RCPtrLoggerFlush, RTLOGFLAGS_BUFFERED);
        AssertReleaseMsgRC(rc, ("RTLogCloneRC failed! rc=%Rra\n", rc));
    }

#ifdef VBOX_WITH_RC_RELEASE_LOGGING
    if (pVM->vmm.s.pRCRelLoggerR3)
    {
        RTRCPTR RCPtrLoggerWrapper = 0;
        rc = PDMR3LdrGetSymbolRC(pVM, VMMGC_MAIN_MODULE_NAME, "vmmGCRelLoggerWrapper", &RCPtrLoggerWrapper);
        AssertReleaseMsgRC(rc, ("vmmGCRelLoggerWrapper not found! rc=%Rra\n", rc));

        pVM->vmm.s.pRCRelLoggerRC = MMHyperR3ToRC(pVM, pVM->vmm.s.pRCRelLoggerR3);
        rc = RTLogCloneRC(RTLogRelDefaultInstance(), pVM->vmm.s.pRCRelLoggerR3, pVM->vmm.s.cbRCRelLogger,
                          RCPtrLoggerWrapper,  RCPtrLoggerFlush, RTLOGFLAGS_BUFFERED);
        AssertReleaseMsgRC(rc, ("RTLogCloneRC failed! rc=%Rra\n", rc));
    }
#endif /* VBOX_WITH_RC_RELEASE_LOGGING */

#ifdef LOG_ENABLED
    /*
     * For the ring-0 EMT logger, we use a per-thread logger instance
     * in ring-0. Only initialize it once.
     */
    for (VMCPUID i = 0; i < pVM->cCpus; i++)
    {
        PVMCPU       pVCpu = &pVM->aCpus[i];
        PVMMR0LOGGER pR0LoggerR3 = pVCpu->vmm.s.pR0LoggerR3;
        if (pR0LoggerR3)
        {
            if (!pR0LoggerR3->fCreated)
            {
                RTR0PTR pfnLoggerWrapper = NIL_RTR0PTR;
                rc = PDMR3LdrGetSymbolR0(pVM, VMMR0_MAIN_MODULE_NAME, "vmmR0LoggerWrapper", &pfnLoggerWrapper);
                AssertReleaseMsgRCReturn(rc, ("vmmR0LoggerWrapper not found! rc=%Rra\n", rc), rc);

                RTR0PTR pfnLoggerFlush = NIL_RTR0PTR;
                rc = PDMR3LdrGetSymbolR0(pVM, VMMR0_MAIN_MODULE_NAME, "vmmR0LoggerFlush", &pfnLoggerFlush);
                AssertReleaseMsgRCReturn(rc, ("vmmR0LoggerFlush not found! rc=%Rra\n", rc), rc);

                rc = RTLogCreateForR0(&pR0LoggerR3->Logger, pR0LoggerR3->cbLogger,
                                      *(PFNRTLOGGER *)&pfnLoggerWrapper, *(PFNRTLOGFLUSH *)&pfnLoggerFlush,
                                      RTLOGFLAGS_BUFFERED, RTLOGDEST_DUMMY);
                AssertReleaseMsgRCReturn(rc, ("RTLogCreateForR0 failed! rc=%Rra\n", rc), rc);

                RTR0PTR pfnLoggerPrefix = NIL_RTR0PTR;
                rc = PDMR3LdrGetSymbolR0(pVM, VMMR0_MAIN_MODULE_NAME, "vmmR0LoggerPrefix", &pfnLoggerPrefix);
                AssertReleaseMsgRCReturn(rc, ("vmmR0LoggerPrefix not found! rc=%Rra\n", rc), rc);
                rc = RTLogSetCustomPrefixCallback(&pR0LoggerR3->Logger, *(PFNRTLOGPREFIX *)&pfnLoggerPrefix, NULL);
                AssertReleaseMsgRCReturn(rc, ("RTLogSetCustomPrefixCallback failed! rc=%Rra\n", rc), rc);

                pR0LoggerR3->idCpu = i;
                pR0LoggerR3->fCreated = true;
                pR0LoggerR3->fFlushingDisabled = false;

            }

            rc = RTLogCopyGroupsAndFlags(&pR0LoggerR3->Logger, NULL /* default */, pVM->vmm.s.pRCLoggerR3->fFlags, RTLOGFLAGS_BUFFERED);
            AssertRC(rc);
        }
    }
#endif
    return rc;
}


/**
 * Gets the pointer to a buffer containing the R0/RC RTAssertMsg1Weak output.
 *
 * @returns Pointer to the buffer.
 * @param   pVM         The VM handle.
 */
VMMR3DECL(const char *) VMMR3GetRZAssertMsg1(PVM pVM)
{
    if (HWACCMIsEnabled(pVM))
        return pVM->vmm.s.szRing0AssertMsg1;

    RTRCPTR RCPtr;
    int rc = PDMR3LdrGetSymbolRC(pVM, NULL, "g_szRTAssertMsg1", &RCPtr);
    if (RT_SUCCESS(rc))
        return (const char *)MMHyperRCToR3(pVM, RCPtr);

    return NULL;
}


/**
 * Gets the pointer to a buffer containing the R0/RC RTAssertMsg2Weak output.
 *
 * @returns Pointer to the buffer.
 * @param   pVM         The VM handle.
 */
VMMR3DECL(const char *) VMMR3GetRZAssertMsg2(PVM pVM)
{
    if (HWACCMIsEnabled(pVM))
        return pVM->vmm.s.szRing0AssertMsg2;

    RTRCPTR RCPtr;
    int rc = PDMR3LdrGetSymbolRC(pVM, NULL, "g_szRTAssertMsg2", &RCPtr);
    if (RT_SUCCESS(rc))
        return (const char *)MMHyperRCToR3(pVM, RCPtr);

    return NULL;
}


/**
 * Execute state save operation.
 *
 * @returns VBox status code.
 * @param   pVM             VM Handle.
 * @param   pSSM            SSM operation handle.
 */
static DECLCALLBACK(int) vmmR3Save(PVM pVM, PSSMHANDLE pSSM)
{
    LogFlow(("vmmR3Save:\n"));

    /*
     * Save the started/stopped state of all CPUs except 0 as it will always
     * be running. This avoids breaking the saved state version. :-)
     */
    for (VMCPUID i = 1; i < pVM->cCpus; i++)
        SSMR3PutBool(pSSM, VMCPUSTATE_IS_STARTED(VMCPU_GET_STATE(&pVM->aCpus[i])));

    return SSMR3PutU32(pSSM, UINT32_MAX); /* terminator */
}


/**
 * Execute state load operation.
 *
 * @returns VBox status code.
 * @param   pVM             VM Handle.
 * @param   pSSM            SSM operation handle.
 * @param   uVersion        Data layout version.
 * @param   uPass           The data pass.
 */
static DECLCALLBACK(int) vmmR3Load(PVM pVM, PSSMHANDLE pSSM, uint32_t uVersion, uint32_t uPass)
{
    LogFlow(("vmmR3Load:\n"));
    Assert(uPass == SSM_PASS_FINAL); NOREF(uPass);

    /*
     * Validate version.
     */
    if (    uVersion != VMM_SAVED_STATE_VERSION
        &&  uVersion != VMM_SAVED_STATE_VERSION_3_0)
    {
        AssertMsgFailed(("vmmR3Load: Invalid version uVersion=%u!\n", uVersion));
        return VERR_SSM_UNSUPPORTED_DATA_UNIT_VERSION;
    }

    if (uVersion <= VMM_SAVED_STATE_VERSION_3_0)
    {
        /* Ignore the stack bottom, stack pointer and stack bits. */
        RTRCPTR RCPtrIgnored;
        SSMR3GetRCPtr(pSSM, &RCPtrIgnored);
        SSMR3GetRCPtr(pSSM, &RCPtrIgnored);
#ifdef RT_OS_DARWIN
        if (   SSMR3HandleVersion(pSSM)  >= VBOX_FULL_VERSION_MAKE(3,0,0)
            && SSMR3HandleVersion(pSSM)  <  VBOX_FULL_VERSION_MAKE(3,1,0)
            && SSMR3HandleRevision(pSSM) >= 48858
            && (   !strcmp(SSMR3HandleHostOSAndArch(pSSM), "darwin.x86")
                || !strcmp(SSMR3HandleHostOSAndArch(pSSM), "") )
           )
            SSMR3Skip(pSSM, 16384);
        else
            SSMR3Skip(pSSM, 8192);
#else
        SSMR3Skip(pSSM, 8192);
#endif
    }

    /*
     * Restore the VMCPU states. VCPU 0 is always started.
     */
    VMCPU_SET_STATE(&pVM->aCpus[0], VMCPUSTATE_STARTED);
    for (VMCPUID i = 1; i < pVM->cCpus; i++)
    {
        bool fStarted;
        int rc = SSMR3GetBool(pSSM, &fStarted);
        if (RT_FAILURE(rc))
            return rc;
        VMCPU_SET_STATE(&pVM->aCpus[i], fStarted ? VMCPUSTATE_STARTED : VMCPUSTATE_STOPPED);
    }

    /* terminator */
    uint32_t u32;
    int rc = SSMR3GetU32(pSSM, &u32);
    if (RT_FAILURE(rc))
        return rc;
    if (u32 != UINT32_MAX)
    {
        AssertMsgFailed(("u32=%#x\n", u32));
        return VERR_SSM_DATA_UNIT_FORMAT_CHANGED;
    }
    return VINF_SUCCESS;
}


/**
 * Resolve a builtin RC symbol.
 *
 * Called by PDM when loading or relocating RC modules.
 *
 * @returns VBox status
 * @param   pVM             VM Handle.
 * @param   pszSymbol       Symbol to resolv
 * @param   pRCPtrValue     Where to store the symbol value.
 *
 * @remark  This has to work before VMMR3Relocate() is called.
 */
VMMR3_INT_DECL(int) VMMR3GetImportRC(PVM pVM, const char *pszSymbol, PRTRCPTR pRCPtrValue)
{
    if (!strcmp(pszSymbol, "g_Logger"))
    {
        if (pVM->vmm.s.pRCLoggerR3)
            pVM->vmm.s.pRCLoggerRC = MMHyperR3ToRC(pVM, pVM->vmm.s.pRCLoggerR3);
        *pRCPtrValue = pVM->vmm.s.pRCLoggerRC;
    }
    else if (!strcmp(pszSymbol, "g_RelLogger"))
    {
#ifdef VBOX_WITH_RC_RELEASE_LOGGING
        if (pVM->vmm.s.pRCRelLoggerR3)
            pVM->vmm.s.pRCRelLoggerRC = MMHyperR3ToRC(pVM, pVM->vmm.s.pRCRelLoggerR3);
        *pRCPtrValue = pVM->vmm.s.pRCRelLoggerRC;
#else
        *pRCPtrValue = NIL_RTRCPTR;
#endif
    }
    else
        return VERR_SYMBOL_NOT_FOUND;
    return VINF_SUCCESS;
}


/**
 * Suspends the CPU yielder.
 *
 * @param   pVM             The VM handle.
 */
VMMR3_INT_DECL(void) VMMR3YieldSuspend(PVM pVM)
{
    VMCPU_ASSERT_EMT(&pVM->aCpus[0]);
    if (!pVM->vmm.s.cYieldResumeMillies)
    {
        uint64_t u64Now = TMTimerGet(pVM->vmm.s.pYieldTimer);
        uint64_t u64Expire = TMTimerGetExpire(pVM->vmm.s.pYieldTimer);
        if (u64Now >= u64Expire || u64Expire == ~(uint64_t)0)
            pVM->vmm.s.cYieldResumeMillies = pVM->vmm.s.cYieldEveryMillies;
        else
            pVM->vmm.s.cYieldResumeMillies = TMTimerToMilli(pVM->vmm.s.pYieldTimer, u64Expire - u64Now);
        TMTimerStop(pVM->vmm.s.pYieldTimer);
    }
    pVM->vmm.s.u64LastYield = RTTimeNanoTS();
}


/**
 * Stops the CPU yielder.
 *
 * @param   pVM             The VM handle.
 */
VMMR3_INT_DECL(void) VMMR3YieldStop(PVM pVM)
{
    if (!pVM->vmm.s.cYieldResumeMillies)
        TMTimerStop(pVM->vmm.s.pYieldTimer);
    pVM->vmm.s.cYieldResumeMillies = pVM->vmm.s.cYieldEveryMillies;
    pVM->vmm.s.u64LastYield = RTTimeNanoTS();
}


/**
 * Resumes the CPU yielder when it has been a suspended or stopped.
 *
 * @param   pVM             The VM handle.
 */
VMMR3_INT_DECL(void) VMMR3YieldResume(PVM pVM)
{
    if (pVM->vmm.s.cYieldResumeMillies)
    {
        TMTimerSetMillies(pVM->vmm.s.pYieldTimer, pVM->vmm.s.cYieldResumeMillies);
        pVM->vmm.s.cYieldResumeMillies = 0;
    }
}


/**
 * Internal timer callback function.
 *
 * @param   pVM             The VM.
 * @param   pTimer          The timer handle.
 * @param   pvUser          User argument specified upon timer creation.
 */
static DECLCALLBACK(void) vmmR3YieldEMT(PVM pVM, PTMTIMER pTimer, void *pvUser)
{
    /*
     * This really needs some careful tuning. While we shouldn't be too greedy since
     * that'll cause the rest of the system to stop up, we shouldn't be too nice either
     * because that'll cause us to stop up.
     *
     * The current logic is to use the default interval when there is no lag worth
     * mentioning, but when we start accumulating lag we don't bother yielding at all.
     *
     * (This depends on the TMCLOCK_VIRTUAL_SYNC to be scheduled before TMCLOCK_REAL
     * so the lag is up to date.)
     */
    const uint64_t u64Lag = TMVirtualSyncGetLag(pVM);
    if (    u64Lag     <   50000000 /* 50ms */
        ||  (   u64Lag < 1000000000 /*  1s */
             && RTTimeNanoTS() - pVM->vmm.s.u64LastYield < 500000000 /* 500 ms */)
       )
    {
        uint64_t u64Elapsed = RTTimeNanoTS();
        pVM->vmm.s.u64LastYield = u64Elapsed;

        RTThreadYield();

#ifdef LOG_ENABLED
        u64Elapsed = RTTimeNanoTS() - u64Elapsed;
        Log(("vmmR3YieldEMT: %RI64 ns\n", u64Elapsed));
#endif
    }
    TMTimerSetMillies(pTimer, pVM->vmm.s.cYieldEveryMillies);
}


/**
 * Executes guest code in the raw-mode context.
 *
 * @param   pVM         VM handle.
 * @param   pVCpu       The VMCPU to operate on.
 */
VMMR3_INT_DECL(int) VMMR3RawRunGC(PVM pVM, PVMCPU pVCpu)
{
    Log2(("VMMR3RawRunGC: (cs:eip=%04x:%08x)\n", CPUMGetGuestCS(pVCpu), CPUMGetGuestEIP(pVCpu)));

    AssertReturn(pVM->cCpus == 1, VERR_RAW_MODE_INVALID_SMP);

    /*
     * Set the EIP and ESP.
     */
    CPUMSetHyperEIP(pVCpu, CPUMGetGuestEFlags(pVCpu) & X86_EFL_VM
                    ? pVM->vmm.s.pfnCPUMRCResumeGuestV86
                    : pVM->vmm.s.pfnCPUMRCResumeGuest);
    CPUMSetHyperESP(pVCpu, pVCpu->vmm.s.pbEMTStackBottomRC);

    /*
     * We hide log flushes (outer) and hypervisor interrupts (inner).
     */
    for (;;)
    {
#ifdef VBOX_STRICT
        if (RT_UNLIKELY(!CPUMGetHyperCR3(pVCpu) || CPUMGetHyperCR3(pVCpu) != PGMGetHyperCR3(pVCpu)))
            EMR3FatalError(pVCpu, VERR_VMM_HYPER_CR3_MISMATCH);
        PGMMapCheck(pVM);
#endif
        int rc;
        do
        {
#ifdef NO_SUPCALLR0VMM
            rc = VERR_GENERAL_FAILURE;
#else
            rc = SUPR3CallVMMR0Fast(pVM->pVMR0, VMMR0_DO_RAW_RUN, 0);
            if (RT_LIKELY(rc == VINF_SUCCESS))
                rc = pVCpu->vmm.s.iLastGZRc;
#endif
        } while (rc == VINF_EM_RAW_INTERRUPT_HYPER);

        /*
         * Flush the logs.
         */
#ifdef LOG_ENABLED
        PRTLOGGERRC pLogger = pVM->vmm.s.pRCLoggerR3;
        if (    pLogger
            &&  pLogger->offScratch > 0)
            RTLogFlushRC(NULL, pLogger);
#endif
#ifdef VBOX_WITH_RC_RELEASE_LOGGING
        PRTLOGGERRC pRelLogger = pVM->vmm.s.pRCRelLoggerR3;
        if (RT_UNLIKELY(pRelLogger && pRelLogger->offScratch > 0))
            RTLogFlushRC(RTLogRelDefaultInstance(), pRelLogger);
#endif
        if (rc != VINF_VMM_CALL_HOST)
        {
            Log2(("VMMR3RawRunGC: returns %Rrc (cs:eip=%04x:%08x)\n", rc, CPUMGetGuestCS(pVCpu), CPUMGetGuestEIP(pVCpu)));
            return rc;
        }
        rc = vmmR3ServiceCallRing3Request(pVM, pVCpu);
        if (RT_FAILURE(rc))
            return rc;
        /* Resume GC */
    }
}


/**
 * Executes guest code (Intel VT-x and AMD-V).
 *
 * @param   pVM         VM handle.
 * @param   pVCpu       The VMCPU to operate on.
 */
VMMR3_INT_DECL(int) VMMR3HwAccRunGC(PVM pVM, PVMCPU pVCpu)
{
    Log2(("VMMR3HwAccRunGC: (cs:eip=%04x:%08x)\n", CPUMGetGuestCS(pVCpu), CPUMGetGuestEIP(pVCpu)));

    for (;;)
    {
        int rc;
        do
        {
#ifdef NO_SUPCALLR0VMM
            rc = VERR_GENERAL_FAILURE;
#else
            rc = SUPR3CallVMMR0Fast(pVM->pVMR0, VMMR0_DO_HWACC_RUN, pVCpu->idCpu);
            if (RT_LIKELY(rc == VINF_SUCCESS))
                rc = pVCpu->vmm.s.iLastGZRc;
#endif
        } while (rc == VINF_EM_RAW_INTERRUPT_HYPER);

#if 0 /* todo triggers too often */
        Assert(!VMCPU_FF_ISSET(pVCpu, VMCPU_FF_TO_R3));
#endif

#ifdef LOG_ENABLED
        /*
         * Flush the log
         */
        PVMMR0LOGGER pR0LoggerR3 = pVCpu->vmm.s.pR0LoggerR3;
        if (    pR0LoggerR3
            &&  pR0LoggerR3->Logger.offScratch > 0)
            RTLogFlushToLogger(&pR0LoggerR3->Logger, NULL);
#endif /* !LOG_ENABLED */
        if (rc != VINF_VMM_CALL_HOST)
        {
            Log2(("VMMR3HwAccRunGC: returns %Rrc (cs:eip=%04x:%08x)\n", rc, CPUMGetGuestCS(pVCpu), CPUMGetGuestEIP(pVCpu)));
            return rc;
        }
        rc = vmmR3ServiceCallRing3Request(pVM, pVCpu);
        if (RT_FAILURE(rc))
            return rc;
        /* Resume R0 */
    }
}

/**
 * VCPU worker for VMMSendSipi.
 *
 * @param   pVM         The VM to operate on.
 * @param   idCpu       Virtual CPU to perform SIPI on
 * @param   uVector     SIPI vector
 */
DECLCALLBACK(int) vmmR3SendSipi(PVM pVM, VMCPUID idCpu, uint32_t uVector)
{
    PVMCPU pVCpu = VMMGetCpuById(pVM, idCpu);
    VMCPU_ASSERT_EMT(pVCpu);

    /** @todo what are we supposed to do if the processor is already running? */
    if (EMGetState(pVCpu) != EMSTATE_WAIT_SIPI)
        return VERR_ACCESS_DENIED;


    PCPUMCTX pCtx = CPUMQueryGuestCtxPtr(pVCpu);

    pCtx->cs                        = uVector << 8;
    pCtx->csHid.u64Base             = uVector << 12;
    pCtx->csHid.u32Limit            = 0x0000ffff;
    pCtx->rip                       = 0;

    Log(("vmmR3SendSipi for VCPU %d with vector %x\n", uVector));

# if 1 /* If we keep the EMSTATE_WAIT_SIPI method, then move this to EM.cpp. */
    EMSetState(pVCpu, EMSTATE_HALTED);
    return VINF_EM_RESCHEDULE;
# else /* And if we go the VMCPU::enmState way it can stay here. */
    VMCPU_ASSERT_STATE(pVCpu, VMCPUSTATE_STOPPED);
    VMCPU_SET_STATE(pVCpu, VMCPUSTATE_STARTED);
    return VINF_SUCCESS;
# endif
}

DECLCALLBACK(int) vmmR3SendInitIpi(PVM pVM, VMCPUID idCpu)
{
    PVMCPU pVCpu = VMMGetCpuById(pVM, idCpu);
    VMCPU_ASSERT_EMT(pVCpu);

    Log(("vmmR3SendInitIpi for VCPU %d\n", idCpu));
    CPUMR3ResetCpu(pVCpu);
    return VINF_EM_WAIT_SIPI;
}

/**
 * Sends SIPI to the virtual CPU by setting CS:EIP into vector-dependent state
 * and unhalting processor
 *
 * @param   pVM         The VM to operate on.
 * @param   idCpu       Virtual CPU to perform SIPI on
 * @param   uVector     SIPI vector
 */
VMMR3_INT_DECL(void) VMMR3SendSipi(PVM pVM, VMCPUID idCpu,  uint32_t uVector)
{
    AssertReturnVoid(idCpu < pVM->cCpus);

    int rc = VMR3ReqCallNoWaitU(pVM->pUVM, idCpu, (PFNRT)vmmR3SendSipi, 3, pVM, idCpu, uVector);
    AssertRC(rc);
}

/**
 * Sends init IPI to the virtual CPU.
 *
 * @param   pVM         The VM to operate on.
 * @param   idCpu       Virtual CPU to perform int IPI on
 */
VMMR3_INT_DECL(void) VMMR3SendInitIpi(PVM pVM, VMCPUID idCpu)
{
    AssertReturnVoid(idCpu < pVM->cCpus);

    int rc = VMR3ReqCallNoWaitU(pVM->pUVM, idCpu, (PFNRT)vmmR3SendInitIpi, 2, pVM, idCpu);
    AssertRC(rc);
}

/**
 * Registers the guest memory range that can be used for patching
 *
 * @returns VBox status code.
 * @param   pVM         The VM to operate on.
 * @param   pPatchMem   Patch memory range
 * @param   cbPatchMem  Size of the memory range
 */
VMMR3DECL(int) VMMR3RegisterPatchMemory(PVM pVM, RTGCPTR pPatchMem, unsigned cbPatchMem)
{
    if (HWACCMIsEnabled(pVM))
        return HWACMMR3EnablePatching(pVM, pPatchMem, cbPatchMem);

    return VERR_NOT_SUPPORTED;
}

/**
 * Deregisters the guest memory range that can be used for patching
 *
 * @returns VBox status code.
 * @param   pVM         The VM to operate on.
 * @param   pPatchMem   Patch memory range
 * @param   cbPatchMem  Size of the memory range
 */
VMMR3DECL(int) VMMR3DeregisterPatchMemory(PVM pVM, RTGCPTR pPatchMem, unsigned cbPatchMem)
{
    if (HWACCMIsEnabled(pVM))
        return HWACMMR3DisablePatching(pVM, pPatchMem, cbPatchMem);

    return VINF_SUCCESS;
}


/**
 * VCPU worker for VMMR3SynchronizeAllVCpus.
 *
 * @param   pVM         The VM to operate on.
 * @param   idCpu       Virtual CPU to perform SIPI on
 * @param   uVector     SIPI vector
 */
DECLCALLBACK(int) vmmR3SyncVCpu(PVM pVM)
{
    /* Block until the job in the caller has finished. */
    RTCritSectEnter(&pVM->vmm.s.CritSectSync);
    RTCritSectLeave(&pVM->vmm.s.CritSectSync);
    return VINF_SUCCESS;
}


/**
 * Atomically execute a callback handler
 * Note: This is very expensive; avoid using it frequently!
 *
 * @param   pVM         The VM to operate on.
 * @param   pfnHandler  Callback handler
 * @param   pvUser      User specified parameter
 *
 * @thread  EMT
 * @todo    Remove this if not used again soon.
 */
VMMR3DECL(int) VMMR3AtomicExecuteHandler(PVM pVM, PFNATOMICHANDLER pfnHandler, void *pvUser)
{
    int    rc;
    PVMCPU pVCpu = VMMGetCpu(pVM);
    AssertReturn(pVCpu, VERR_VM_THREAD_NOT_EMT);

    /* Shortcut for the uniprocessor case. */
    if (pVM->cCpus == 1)
        return pfnHandler(pVM, pvUser);

    RTCritSectEnter(&pVM->vmm.s.CritSectSync);
    for (VMCPUID idCpu = 0; idCpu < pVM->cCpus; idCpu++)
    {
        if (idCpu != pVCpu->idCpu)
        {
            rc = VMR3ReqCallNoWaitU(pVM->pUVM, idCpu, (PFNRT)vmmR3SyncVCpu, 1, pVM);
            AssertRC(rc);
        }
    }
    /* Wait until all other VCPUs are waiting for us. */
    while (RTCritSectGetWaiters(&pVM->vmm.s.CritSectSync) != (int32_t)(pVM->cCpus - 1))
        RTThreadSleep(1);

    rc = pfnHandler(pVM, pvUser);
    RTCritSectLeave(&pVM->vmm.s.CritSectSync);
    return rc;
}


/**
 * Count returns and have the last non-caller EMT wake up the caller.
 *
 * @returns VBox strict informational status code for EM scheduling. No failures
 *          will be returned here, those are for the caller only.
 *
 * @param   pVM                 The VM handle.
 */
DECL_FORCE_INLINE(int) vmmR3EmtRendezvousNonCallerReturn(PVM pVM)
{
    int rcRet = ASMAtomicReadS32(&pVM->vmm.s.i32RendezvousStatus);
    uint32_t cReturned = ASMAtomicIncU32(&pVM->vmm.s.cRendezvousEmtsReturned);
    if (cReturned == pVM->cCpus - 1U)
    {
        int rc = RTSemEventSignal(pVM->vmm.s.hEvtRendezvousDoneCaller);
        AssertLogRelRC(rc);
    }

    AssertLogRelMsgReturn(   rcRet <= VINF_SUCCESS
                          || (rcRet >= VINF_EM_FIRST && rcRet <= VINF_EM_LAST),
                          ("%Rrc\n", rcRet),
                          VERR_IPE_UNEXPECTED_INFO_STATUS);
    return RT_SUCCESS(rcRet) ? rcRet : VINF_SUCCESS;
}


/**
 * Common worker for VMMR3EmtRendezvous and VMMR3EmtRendezvousFF.
 *
 * @returns VBox strict informational status code for EM scheduling. No failures
 *          will be returned here, those are for the caller only.  When
 *          fIsCaller is set, VINF_SUCCESS is always returned.
 *
 * @param   pVM                 The VM handle.
 * @param   pVCpu               The VMCPU structure for the calling EMT.
 * @param   fIsCaller           Whether we're the VMMR3EmtRendezvous caller or
 *                              not.
 * @param   fFlags              The flags.
 * @param   pfnRendezvous       The callback.
 * @param   pvUser              The user argument for the callback.
 */
static int vmmR3EmtRendezvousCommon(PVM pVM, PVMCPU pVCpu, bool fIsCaller,
                                    uint32_t fFlags, PFNVMMEMTRENDEZVOUS pfnRendezvous, void *pvUser)
{
    int rc;

    /*
     * Enter, the last EMT triggers the next callback phase.
     */
    uint32_t cEntered = ASMAtomicIncU32(&pVM->vmm.s.cRendezvousEmtsEntered);
    if (cEntered != pVM->cCpus)
    {
        if ((fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ONE_BY_ONE)
        {
            /* Wait for our turn. */
            rc = RTSemEventWait(pVM->vmm.s.hEvtRendezvousEnterOneByOne, RT_INDEFINITE_WAIT);
            AssertLogRelRC(rc);
        }
        else if ((fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ALL_AT_ONCE)
        {
            /* Wait for the last EMT to arrive and wake everyone up. */
            rc = RTSemEventMultiWait(pVM->vmm.s.hEvtMulRendezvousEnterAllAtOnce, RT_INDEFINITE_WAIT);
            AssertLogRelRC(rc);
        }
        else if (   (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ASCENDING
                 || (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_DESCENDING)
        {
            /* Wait for our turn. */
            rc = RTSemEventWait(pVM->vmm.s.pahEvtRendezvousEnterOrdered[pVCpu->idCpu], RT_INDEFINITE_WAIT);
            AssertLogRelRC(rc);
        }
        else
        {
            Assert((fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ONCE);

            /*
             * The execute once is handled specially to optimize the code flow.
             *
             * The last EMT to arrive will perform the callback and the other
             * EMTs will wait on the Done/DoneCaller semaphores (instead of
             * the EnterOneByOne/AllAtOnce) in the meanwhile. When the callback
             * returns, that EMT will initiate the normal return sequence.
             */
            if (!fIsCaller)
            {
                rc = RTSemEventMultiWait(pVM->vmm.s.hEvtMulRendezvousDone, RT_INDEFINITE_WAIT);
                AssertLogRelRC(rc);

                return vmmR3EmtRendezvousNonCallerReturn(pVM);
            }
            return VINF_SUCCESS;
        }
    }
    else
    {
        /*
         * All EMTs are waiting, clear the FF and take action according to the
         * execution method.
         */
        VM_FF_CLEAR(pVM, VM_FF_EMT_RENDEZVOUS);

        if ((fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ALL_AT_ONCE)
        {
            /* Wake up everyone. */
            rc = RTSemEventMultiSignal(pVM->vmm.s.hEvtMulRendezvousEnterAllAtOnce);
            AssertLogRelRC(rc);
        }
        else if (   (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ASCENDING
                 || (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_DESCENDING)
        {
            /* Figure out who to wake up and wake it up. If it's ourself, then
               it's easy otherwise wait for our turn. */
            VMCPUID iFirst = (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ASCENDING
                           ? 0
                           : pVM->cCpus - 1U;
            if (pVCpu->idCpu != iFirst)
            {
                rc = RTSemEventSignal(pVM->vmm.s.pahEvtRendezvousEnterOrdered[iFirst]);
                AssertLogRelRC(rc);
                rc = RTSemEventWait(pVM->vmm.s.pahEvtRendezvousEnterOrdered[pVCpu->idCpu], RT_INDEFINITE_WAIT);
                AssertLogRelRC(rc);
            }
        }
        /* else: execute the handler on the current EMT and wake up one or more threads afterwards. */
    }


    /*
     * Do the callback and update the status if necessary.
     */
    if (    !(fFlags & VMMEMTRENDEZVOUS_FLAGS_STOP_ON_ERROR)
        ||  RT_SUCCESS(ASMAtomicUoReadS32(&pVM->vmm.s.i32RendezvousStatus)) )
    {
        VBOXSTRICTRC rcStrict = pfnRendezvous(pVM, pVCpu, pvUser);
        if (rcStrict != VINF_SUCCESS)
        {
            AssertLogRelMsg(   rcStrict <= VINF_SUCCESS
                            || (rcStrict >= VINF_EM_FIRST && rcStrict <= VINF_EM_LAST),
                            ("%Rrc\n", VBOXSTRICTRC_VAL(rcStrict)));
            int32_t i32RendezvousStatus;
            do
            {
                i32RendezvousStatus = ASMAtomicUoReadS32(&pVM->vmm.s.i32RendezvousStatus);
                if (    rcStrict == i32RendezvousStatus
                    ||  RT_FAILURE(i32RendezvousStatus)
                    ||  (   i32RendezvousStatus != VINF_SUCCESS
                         && rcStrict > i32RendezvousStatus))
                    break;
            } while (!ASMAtomicCmpXchgS32(&pVM->vmm.s.i32RendezvousStatus, VBOXSTRICTRC_VAL(rcStrict), i32RendezvousStatus));
        }
    }

    /*
     * Increment the done counter and take action depending on whether we're
     * the last to finish callback execution.
     */
    uint32_t cDone = ASMAtomicIncU32(&pVM->vmm.s.cRendezvousEmtsDone);
    if (    cDone != pVM->cCpus
        &&  (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) != VMMEMTRENDEZVOUS_FLAGS_TYPE_ONCE)
    {
        /* Signal the next EMT? */
        if ((fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ONE_BY_ONE)
        {
            rc = RTSemEventSignal(pVM->vmm.s.hEvtRendezvousEnterOneByOne);
            AssertLogRelRC(rc);
        }
        else if ((fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_ASCENDING)
        {
            Assert(cDone == pVCpu->idCpu + 1U);
            rc = RTSemEventSignal(pVM->vmm.s.pahEvtRendezvousEnterOrdered[pVCpu->idCpu + 1U]);
            AssertLogRelRC(rc);
        }
        else if ((fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) == VMMEMTRENDEZVOUS_FLAGS_TYPE_DESCENDING)
        {
            Assert(pVM->cCpus - cDone == pVCpu->idCpu);
            rc = RTSemEventSignal(pVM->vmm.s.pahEvtRendezvousEnterOrdered[pVM->cCpus - cDone - 1U]);
            AssertLogRelRC(rc);
        }

        /* Wait for the rest to finish (the caller waits on hEvtRendezvousDoneCaller). */
        if (!fIsCaller)
        {
            rc = RTSemEventMultiWait(pVM->vmm.s.hEvtMulRendezvousDone, RT_INDEFINITE_WAIT);
            AssertLogRelRC(rc);
        }
    }
    else
    {
        /* Callback execution is all done, tell the rest to return. */
        rc = RTSemEventMultiSignal(pVM->vmm.s.hEvtMulRendezvousDone);
        AssertLogRelRC(rc);
    }

    if (!fIsCaller)
        return vmmR3EmtRendezvousNonCallerReturn(pVM);
    return VINF_SUCCESS;
}


/**
 * Called in response to VM_FF_EMT_RENDEZVOUS.
 *
 * @returns VBox strict status code - EM scheduling.  No errors will be returned
 *          here, nor will any non-EM scheduling status codes be returned.
 *
 * @param   pVM         The VM handle
 * @param   pVCpu       The handle of the calling EMT.
 *
 * @thread  EMT
 */
VMMR3_INT_DECL(int) VMMR3EmtRendezvousFF(PVM pVM, PVMCPU pVCpu)
{
    return vmmR3EmtRendezvousCommon(pVM, pVCpu, false /* fIsCaller */, pVM->vmm.s.fRendezvousFlags,
                                    pVM->vmm.s.pfnRendezvous, pVM->vmm.s.pvRendezvousUser);
}


/**
 * EMT rendezvous.
 *
 * Gathers all the EMTs and execute some code on each of them, either in a one
 * by one fashion or all at once.
 *
 * @returns VBox strict status code.  This will be the the first error,
 *          VINF_SUCCESS, or an EM scheduling status code.
 *
 * @param   pVM             The VM handle.
 * @param   fFlags          Flags indicating execution methods. See
 *                          grp_VMMR3EmtRendezvous_fFlags.
 * @param   pfnRendezvous   The callback.
 * @param   pvUser          User argument for the callback.
 *
 * @thread  Any.
 */
VMMR3DECL(int) VMMR3EmtRendezvous(PVM pVM, uint32_t fFlags, PFNVMMEMTRENDEZVOUS pfnRendezvous, void *pvUser)
{
    /*
     * Validate input.
     */
    AssertMsg(   (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) != VMMEMTRENDEZVOUS_FLAGS_TYPE_INVALID
              && (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) <= VMMEMTRENDEZVOUS_FLAGS_TYPE_DESCENDING
              && !(fFlags & ~VMMEMTRENDEZVOUS_FLAGS_VALID_MASK), ("%#x\n", fFlags));
    AssertMsg(   !(fFlags & VMMEMTRENDEZVOUS_FLAGS_STOP_ON_ERROR)
              || (   (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) != VMMEMTRENDEZVOUS_FLAGS_TYPE_ALL_AT_ONCE
                  && (fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK) != VMMEMTRENDEZVOUS_FLAGS_TYPE_ONCE),
              ("type %u\n", fFlags & VMMEMTRENDEZVOUS_FLAGS_TYPE_MASK));

    VBOXSTRICTRC rcStrict;
    PVMCPU pVCpu = VMMGetCpu(pVM);
    if (!pVCpu)
        /*
         * Forward the request to an EMT thread.
         */
        rcStrict = VMR3ReqCallWait(pVM, VMCPUID_ANY,
                                   (PFNRT)VMMR3EmtRendezvous, 4, pVM, fFlags, pfnRendezvous, pvUser);
    else if (pVM->cCpus == 1)
        /*
         * Shortcut for the single EMT case.
         */
        rcStrict = pfnRendezvous(pVM, pVCpu, pvUser);
    else
    {
        /*
         * Spin lock. If busy, wait for the other EMT to finish while keeping a
         * lookout of the RENDEZVOUS FF.
         */
        int rc;
        rcStrict = VINF_SUCCESS;
        if (RT_UNLIKELY(!ASMAtomicCmpXchgU32(&pVM->vmm.s.u32RendezvousLock, 0x77778888, 0)))
        {
            while (!ASMAtomicCmpXchgU32(&pVM->vmm.s.u32RendezvousLock, 0x77778888, 0))
            {
                if (VM_FF_ISPENDING(pVM, VM_FF_EMT_RENDEZVOUS))
                {
                    rc = VMMR3EmtRendezvousFF(pVM, pVCpu);
                    if (    rc != VINF_SUCCESS
                        &&  (   rcStrict == VINF_SUCCESS
                             || rcStrict > rc))
                        rcStrict = rc;
                    /** @todo Perhaps deal with termination here?  */
                }
                ASMNopPause();
            }
        }
        Assert(!VM_FF_ISPENDING(pVM, VM_FF_EMT_RENDEZVOUS));

        /*
         * Clear the slate. This is a semaphore ping-pong orgy. :-)
         */
        for (VMCPUID i = 0; i < pVM->cCpus; i++)
        {
            rc = RTSemEventWait(pVM->vmm.s.pahEvtRendezvousEnterOrdered[i], 0);
            AssertLogRelMsg(rc == VERR_TIMEOUT || rc == VINF_SUCCESS, ("%Rrc\n", rc));
        }
        rc = RTSemEventWait(pVM->vmm.s.hEvtRendezvousEnterOneByOne, 0);         AssertLogRelMsg(rc == VERR_TIMEOUT || rc == VINF_SUCCESS, ("%Rrc\n", rc));
        rc = RTSemEventMultiReset(pVM->vmm.s.hEvtMulRendezvousEnterAllAtOnce);  AssertLogRelRC(rc);
        rc = RTSemEventMultiReset(pVM->vmm.s.hEvtMulRendezvousDone);            AssertLogRelRC(rc);
        rc = RTSemEventWait(pVM->vmm.s.hEvtRendezvousDoneCaller, 0);            AssertLogRelMsg(rc == VERR_TIMEOUT || rc == VINF_SUCCESS, ("%Rrc\n", rc));
        ASMAtomicWriteU32(&pVM->vmm.s.cRendezvousEmtsEntered, 0);
        ASMAtomicWriteU32(&pVM->vmm.s.cRendezvousEmtsDone, 0);
        ASMAtomicWriteU32(&pVM->vmm.s.cRendezvousEmtsReturned, 0);
        ASMAtomicWriteS32(&pVM->vmm.s.i32RendezvousStatus, VINF_SUCCESS);
        ASMAtomicWritePtr((void * volatile *)&pVM->vmm.s.pfnRendezvous, (void *)(uintptr_t)pfnRendezvous);
        ASMAtomicWritePtr(&pVM->vmm.s.pvRendezvousUser, pvUser);
        ASMAtomicWriteU32(&pVM->vmm.s.fRendezvousFlags, fFlags);

        /*
         * Set the FF and poke the other EMTs.
         */
        VM_FF_SET(pVM, VM_FF_EMT_RENDEZVOUS);
        VMR3NotifyGlobalFFU(pVM->pUVM, VMNOTIFYFF_FLAGS_POKE);

        /*
         * Do the same ourselves.
         */
        vmmR3EmtRendezvousCommon(pVM, pVCpu, true /* fIsCaller */, fFlags, pfnRendezvous, pvUser);

        /*
         * The caller waits for the other EMTs to be done and return before doing
         * the cleanup. This makes away with wakeup / reset races we would otherwise
         * risk in the multiple release event semaphore code (hEvtRendezvousDoneCaller).
         */
        rc = RTSemEventWait(pVM->vmm.s.hEvtRendezvousDoneCaller, RT_INDEFINITE_WAIT);
        AssertLogRelRC(rc);

        /*
         * Get the return code and clean up a little bit.
         */
        int rcMy = pVM->vmm.s.i32RendezvousStatus;
        ASMAtomicWriteNullPtr((void * volatile *)&pVM->vmm.s.pfnRendezvous);

        ASMAtomicWriteU32(&pVM->vmm.s.u32RendezvousLock, 0);

        /*
         * Merge rcStrict and rcMy.
         */
        AssertRC(VBOXSTRICTRC_VAL(rcStrict));
        if (    rcMy != VINF_SUCCESS
            &&  (   rcStrict == VINF_SUCCESS
                 || rcStrict > rcMy))
            rcStrict = rcMy;
    }

    AssertLogRelMsgReturn(   rcStrict <= VINF_SUCCESS
                          || (rcStrict >= VINF_EM_FIRST && rcStrict <= VINF_EM_LAST),
                          ("%Rrc\n", VBOXSTRICTRC_VAL(rcStrict)),
                          VERR_IPE_UNEXPECTED_INFO_STATUS);
    return VBOXSTRICTRC_VAL(rcStrict);
}


/**
 * Read from the ring 0 jump buffer stack
 *
 * @returns VBox status code.
 *
 * @param   pVM             Pointer to the shared VM structure.
 * @param   idCpu           The ID of the source CPU context (for the address).
 * @param   R0Addr          Where to start reading.
 * @param   pvBuf           Where to store the data we've read.
 * @param   cbRead          The number of bytes to read.
 */
VMMR3_INT_DECL(int) VMMR3ReadR0Stack(PVM pVM, VMCPUID idCpu, RTHCUINTPTR R0Addr, void *pvBuf, size_t cbRead)
{
    PVMCPU pVCpu = VMMGetCpuById(pVM, idCpu);
    AssertReturn(pVCpu, VERR_INVALID_PARAMETER);

#ifdef VMM_R0_SWITCH_STACK
    RTHCUINTPTR off = R0Addr - MMHyperCCToR0(pVM, pVCpu->vmm.s.pbEMTStackR3);
#else
    RTHCUINTPTR off = pVCpu->vmm.s.CallRing3JmpBufR0.cbSavedStack - (pVCpu->vmm.s.CallRing3JmpBufR0.SpCheck - R0Addr);
#endif
    if (   off          >  VMM_STACK_SIZE
        || off + cbRead >= VMM_STACK_SIZE)
        return VERR_INVALID_POINTER;

    memcpy(pvBuf, &pVCpu->vmm.s.pbEMTStackR3[off], cbRead);
    return VINF_SUCCESS;
}


/**
 * Calls a RC function.
 *
 * @param   pVM         The VM handle.
 * @param   RCPtrEntry  The address of the RC function.
 * @param   cArgs       The number of arguments in the ....
 * @param   ...         Arguments to the function.
 */
VMMR3DECL(int) VMMR3CallRC(PVM pVM, RTRCPTR RCPtrEntry, unsigned cArgs, ...)
{
    va_list args;
    va_start(args, cArgs);
    int rc = VMMR3CallRCV(pVM, RCPtrEntry, cArgs, args);
    va_end(args);
    return rc;
}


/**
 * Calls a RC function.
 *
 * @param   pVM         The VM handle.
 * @param   RCPtrEntry  The address of the RC function.
 * @param   cArgs       The number of arguments in the ....
 * @param   args        Arguments to the function.
 */
VMMR3DECL(int) VMMR3CallRCV(PVM pVM, RTRCPTR RCPtrEntry, unsigned cArgs, va_list args)
{
    /* Raw mode implies 1 VCPU. */
    AssertReturn(pVM->cCpus == 1, VERR_RAW_MODE_INVALID_SMP);
    PVMCPU pVCpu = &pVM->aCpus[0];

    Log2(("VMMR3CallGCV: RCPtrEntry=%RRv cArgs=%d\n", RCPtrEntry, cArgs));

    /*
     * Setup the call frame using the trampoline.
     */
    CPUMHyperSetCtxCore(pVCpu, NULL);
    memset(pVCpu->vmm.s.pbEMTStackR3, 0xaa, VMM_STACK_SIZE); /* Clear the stack. */
    CPUMSetHyperESP(pVCpu, pVCpu->vmm.s.pbEMTStackBottomRC - cArgs * sizeof(RTGCUINTPTR32));
    PRTGCUINTPTR32 pFrame = (PRTGCUINTPTR32)(pVCpu->vmm.s.pbEMTStackR3 + VMM_STACK_SIZE) - cArgs;
    int i = cArgs;
    while (i-- > 0)
        *pFrame++ = va_arg(args, RTGCUINTPTR32);

    CPUMPushHyper(pVCpu, cArgs * sizeof(RTGCUINTPTR32));                          /* stack frame size */
    CPUMPushHyper(pVCpu, RCPtrEntry);                                             /* what to call */
    CPUMSetHyperEIP(pVCpu, pVM->vmm.s.pfnCallTrampolineRC);

    /*
     * We hide log flushes (outer) and hypervisor interrupts (inner).
     */
    for (;;)
    {
        int rc;
        Assert(CPUMGetHyperCR3(pVCpu) && CPUMGetHyperCR3(pVCpu) == PGMGetHyperCR3(pVCpu));
        do
        {
#ifdef NO_SUPCALLR0VMM
            rc = VERR_GENERAL_FAILURE;
#else
            rc = SUPR3CallVMMR0Fast(pVM->pVMR0, VMMR0_DO_RAW_RUN, 0);
            if (RT_LIKELY(rc == VINF_SUCCESS))
                rc = pVCpu->vmm.s.iLastGZRc;
#endif
        } while (rc == VINF_EM_RAW_INTERRUPT_HYPER);

        /*
         * Flush the logs.
         */
#ifdef LOG_ENABLED
        PRTLOGGERRC pLogger = pVM->vmm.s.pRCLoggerR3;
        if (    pLogger
            &&  pLogger->offScratch > 0)
            RTLogFlushRC(NULL, pLogger);
#endif
#ifdef VBOX_WITH_RC_RELEASE_LOGGING
        PRTLOGGERRC pRelLogger = pVM->vmm.s.pRCRelLoggerR3;
        if (RT_UNLIKELY(pRelLogger && pRelLogger->offScratch > 0))
            RTLogFlushRC(RTLogRelDefaultInstance(), pRelLogger);
#endif
        if (rc == VERR_TRPM_PANIC || rc == VERR_TRPM_DONT_PANIC)
            VMMR3FatalDump(pVM, pVCpu, rc);
        if (rc != VINF_VMM_CALL_HOST)
        {
            Log2(("VMMR3CallGCV: returns %Rrc (cs:eip=%04x:%08x)\n", rc, CPUMGetGuestCS(pVCpu), CPUMGetGuestEIP(pVCpu)));
            return rc;
        }
        rc = vmmR3ServiceCallRing3Request(pVM, pVCpu);
        if (RT_FAILURE(rc))
            return rc;
    }
}


/**
 * Wrapper for SUPR3CallVMMR0Ex which will deal with VINF_VMM_CALL_HOST returns.
 *
 * @returns VBox status code.
 * @param   pVM         The VM to operate on.
 * @param   uOperation  Operation to execute.
 * @param   u64Arg      Constant argument.
 * @param   pReqHdr     Pointer to a request header. See SUPR3CallVMMR0Ex for
 *                      details.
 */
VMMR3DECL(int) VMMR3CallR0(PVM pVM, uint32_t uOperation, uint64_t u64Arg, PSUPVMMR0REQHDR pReqHdr)
{
    PVMCPU pVCpu = VMMGetCpu(pVM);
    AssertReturn(pVCpu, VERR_VM_THREAD_NOT_EMT);

    /*
     * Call Ring-0 entry with init code.
     */
    int rc;
    for (;;)
    {
#ifdef NO_SUPCALLR0VMM
        rc = VERR_GENERAL_FAILURE;
#else
        rc = SUPR3CallVMMR0Ex(pVM->pVMR0, pVCpu->idCpu, uOperation, u64Arg, pReqHdr);
#endif
        /*
         * Flush the logs.
         */
#ifdef LOG_ENABLED
        if (    pVCpu->vmm.s.pR0LoggerR3
            &&  pVCpu->vmm.s.pR0LoggerR3->Logger.offScratch > 0)
            RTLogFlushToLogger(&pVCpu->vmm.s.pR0LoggerR3->Logger, NULL);
#endif
        if (rc != VINF_VMM_CALL_HOST)
            break;
        rc = vmmR3ServiceCallRing3Request(pVM, pVCpu);
        if (RT_FAILURE(rc) || (rc >= VINF_EM_FIRST && rc <= VINF_EM_LAST))
            break;
        /* Resume R0 */
    }

    AssertLogRelMsgReturn(rc == VINF_SUCCESS || RT_FAILURE(rc),
                          ("uOperation=%u rc=%Rrc\n", uOperation, rc),
                          VERR_INTERNAL_ERROR);
    return rc;
}


/**
 * Resumes executing hypervisor code when interrupted by a queue flush or a
 * debug event.
 *
 * @returns VBox status code.
 * @param   pVM         VM handle.
 * @param   pVCpu       VMCPU handle.
 */
VMMR3DECL(int) VMMR3ResumeHyper(PVM pVM, PVMCPU pVCpu)
{
    Log(("VMMR3ResumeHyper: eip=%RRv esp=%RRv\n", CPUMGetHyperEIP(pVCpu), CPUMGetHyperESP(pVCpu)));
    AssertReturn(pVM->cCpus == 1, VERR_RAW_MODE_INVALID_SMP);

    /*
     * We hide log flushes (outer) and hypervisor interrupts (inner).
     */
    for (;;)
    {
        int rc;
        Assert(CPUMGetHyperCR3(pVCpu) && CPUMGetHyperCR3(pVCpu) == PGMGetHyperCR3(pVCpu));
        do
        {
#ifdef NO_SUPCALLR0VMM
            rc = VERR_GENERAL_FAILURE;
#else
            rc = SUPR3CallVMMR0Fast(pVM->pVMR0, VMMR0_DO_RAW_RUN, 0);
            if (RT_LIKELY(rc == VINF_SUCCESS))
                rc = pVCpu->vmm.s.iLastGZRc;
#endif
        } while (rc == VINF_EM_RAW_INTERRUPT_HYPER);

        /*
         * Flush the loggers,
         */
#ifdef LOG_ENABLED
        PRTLOGGERRC pLogger = pVM->vmm.s.pRCLoggerR3;
        if (    pLogger
            &&  pLogger->offScratch > 0)
            RTLogFlushRC(NULL, pLogger);
#endif
#ifdef VBOX_WITH_RC_RELEASE_LOGGING
        PRTLOGGERRC pRelLogger = pVM->vmm.s.pRCRelLoggerR3;
        if (RT_UNLIKELY(pRelLogger && pRelLogger->offScratch > 0))
            RTLogFlushRC(RTLogRelDefaultInstance(), pRelLogger);
#endif
        if (rc == VERR_TRPM_PANIC || rc == VERR_TRPM_DONT_PANIC)
            VMMR3FatalDump(pVM, pVCpu, rc);
        if (rc != VINF_VMM_CALL_HOST)
        {
            Log(("VMMR3ResumeHyper: returns %Rrc\n", rc));
            return rc;
        }
        rc = vmmR3ServiceCallRing3Request(pVM, pVCpu);
        if (RT_FAILURE(rc))
            return rc;
    }
}


/**
 * Service a call to the ring-3 host code.
 *
 * @returns VBox status code.
 * @param   pVM     VM handle.
 * @param   pVCpu   VMCPU handle
 * @remark  Careful with critsects.
 */
static int vmmR3ServiceCallRing3Request(PVM pVM, PVMCPU pVCpu)
{
    /*
     * We must also check for pending critsect exits or else we can deadlock
     * when entering other critsects here.
     */
    if (VMCPU_FF_ISPENDING(pVCpu, VMCPU_FF_PDM_CRITSECT))
        PDMCritSectFF(pVCpu);

    switch (pVCpu->vmm.s.enmCallRing3Operation)
    {
        /*
         * Acquire the PDM lock.
         */
        case VMMCALLRING3_PDM_LOCK:
        {
            pVCpu->vmm.s.rcCallRing3 = PDMR3LockCall(pVM);
            break;
        }

        /*
         * Grow the PGM pool.
         */
        case VMMCALLRING3_PGM_POOL_GROW:
        {
            pVCpu->vmm.s.rcCallRing3 = PGMR3PoolGrow(pVM);
            break;
        }

        /*
         * Maps an page allocation chunk into ring-3 so ring-0 can use it.
         */
        case VMMCALLRING3_PGM_MAP_CHUNK:
        {
            pVCpu->vmm.s.rcCallRing3 = PGMR3PhysChunkMap(pVM, pVCpu->vmm.s.u64CallRing3Arg);
            break;
        }

        /*
         * Allocates more handy pages.
         */
        case VMMCALLRING3_PGM_ALLOCATE_HANDY_PAGES:
        {
            pVCpu->vmm.s.rcCallRing3 = PGMR3PhysAllocateHandyPages(pVM);
            break;
        }

        /*
         * Allocates a large page.
         */
        case VMMCALLRING3_PGM_ALLOCATE_LARGE_HANDY_PAGE:
        {
            pVCpu->vmm.s.rcCallRing3 = PGMR3PhysAllocateLargeHandyPage(pVM, pVCpu->vmm.s.u64CallRing3Arg);
            break;
        }

        /*
         * Acquire the PGM lock.
         */
        case VMMCALLRING3_PGM_LOCK:
        {
            pVCpu->vmm.s.rcCallRing3 = PGMR3LockCall(pVM);
            break;
        }

        /*
         * Acquire the MM hypervisor heap lock.
         */
        case VMMCALLRING3_MMHYPER_LOCK:
        {
            pVCpu->vmm.s.rcCallRing3 = MMR3LockCall(pVM);
            break;
        }

        /*
         * Flush REM handler notifications.
         */
        case VMMCALLRING3_REM_REPLAY_HANDLER_NOTIFICATIONS:
        {
            REMR3ReplayHandlerNotifications(pVM);
            pVCpu->vmm.s.rcCallRing3 = VINF_SUCCESS;
            break;
        }

        /*
         * This is a noop. We just take this route to avoid unnecessary
         * tests in the loops.
         */
        case VMMCALLRING3_VMM_LOGGER_FLUSH:
            pVCpu->vmm.s.rcCallRing3 = VINF_SUCCESS;
            LogAlways(("*FLUSH*\n"));
            break;

        /*
         * Set the VM error message.
         */
        case VMMCALLRING3_VM_SET_ERROR:
            VMR3SetErrorWorker(pVM);
            pVCpu->vmm.s.rcCallRing3 = VINF_SUCCESS;
            break;

        /*
         * Set the VM runtime error message.
         */
        case VMMCALLRING3_VM_SET_RUNTIME_ERROR:
            pVCpu->vmm.s.rcCallRing3 = VMR3SetRuntimeErrorWorker(pVM);
            break;

        /*
         * Signal a ring 0 hypervisor assertion.
         * Cancel the longjmp operation that's in progress.
         */
        case VMMCALLRING3_VM_R0_ASSERTION:
            pVCpu->vmm.s.enmCallRing3Operation = VMMCALLRING3_INVALID;
            pVCpu->vmm.s.CallRing3JmpBufR0.fInRing3Call = false;
#ifdef RT_ARCH_X86
            pVCpu->vmm.s.CallRing3JmpBufR0.eip = 0;
#else
            pVCpu->vmm.s.CallRing3JmpBufR0.rip = 0;
#endif
#ifdef VMM_R0_SWITCH_STACK
            *(uint64_t *)pVCpu->vmm.s.pbEMTStackR3 = 0; /* clear marker  */
#endif
            LogRel((pVM->vmm.s.szRing0AssertMsg1));
            LogRel((pVM->vmm.s.szRing0AssertMsg2));
            return VERR_VMM_RING0_ASSERTION;

        /*
         * A forced switch to ring 0 for preemption purposes.
         */
        case VMMCALLRING3_VM_R0_PREEMPT:
            pVCpu->vmm.s.rcCallRing3 = VINF_SUCCESS;
            break;

        case VMMCALLRING3_FTM_SET_CHECKPOINT:
            pVCpu->vmm.s.rcCallRing3 = FTMR3SetCheckpoint(pVM, (FTMCHECKPOINTTYPE)pVCpu->vmm.s.u64CallRing3Arg);
            break;

        default:
            AssertMsgFailed(("enmCallRing3Operation=%d\n", pVCpu->vmm.s.enmCallRing3Operation));
            return VERR_INTERNAL_ERROR;
    }

    pVCpu->vmm.s.enmCallRing3Operation = VMMCALLRING3_INVALID;
    return VINF_SUCCESS;
}


/**
 * Displays the Force action Flags.
 *
 * @param   pVM         The VM handle.
 * @param   pHlp        The output helpers.
 * @param   pszArgs     The additional arguments (ignored).
 */
static DECLCALLBACK(void) vmmR3InfoFF(PVM pVM, PCDBGFINFOHLP pHlp, const char *pszArgs)
{
    int         c;
    uint32_t    f;
#define PRINT_FLAG(prf,flag) do { \
        if (f & (prf##flag)) \
        { \
            static const char *s_psz = #flag; \
            if (!(c % 6)) \
                pHlp->pfnPrintf(pHlp, "%s\n    %s", c ? "," : "", s_psz); \
            else \
                pHlp->pfnPrintf(pHlp, ", %s", s_psz); \
            c++; \
            f &= ~(prf##flag); \
        } \
    } while (0)

#define PRINT_GROUP(prf,grp,sfx) do { \
        if (f & (prf##grp##sfx)) \
        { \
            static const char *s_psz = #grp; \
            if (!(c % 5)) \
                pHlp->pfnPrintf(pHlp, "%s    %s", c ? ",\n" : "  Groups:\n", s_psz); \
            else \
                pHlp->pfnPrintf(pHlp, ", %s", s_psz); \
            c++; \
        } \
    } while (0)

    /*
     * The global flags.
     */
    const uint32_t fGlobalForcedActions = pVM->fGlobalForcedActions;
    pHlp->pfnPrintf(pHlp, "Global FFs: %#RX32", fGlobalForcedActions);

    /* show the flag mnemonics  */
    c = 0;
    f = fGlobalForcedActions;
    PRINT_FLAG(VM_FF_,TM_VIRTUAL_SYNC);
    PRINT_FLAG(VM_FF_,PDM_QUEUES);
    PRINT_FLAG(VM_FF_,PDM_DMA);
    PRINT_FLAG(VM_FF_,DBGF);
    PRINT_FLAG(VM_FF_,REQUEST);
    PRINT_FLAG(VM_FF_,CHECK_VM_STATE);
    PRINT_FLAG(VM_FF_,RESET);
    PRINT_FLAG(VM_FF_,EMT_RENDEZVOUS);
    PRINT_FLAG(VM_FF_,PGM_NEED_HANDY_PAGES);
    PRINT_FLAG(VM_FF_,PGM_NO_MEMORY);
    PRINT_FLAG(VM_FF_,PGM_POOL_FLUSH_PENDING);
    PRINT_FLAG(VM_FF_,REM_HANDLER_NOTIFY);
    PRINT_FLAG(VM_FF_,DEBUG_SUSPEND);
    if (f)
        pHlp->pfnPrintf(pHlp, "%s\n    Unknown bits: %#RX32\n", c ? "," : "", f);
    else
        pHlp->pfnPrintf(pHlp, "\n");

    /* the groups */
    c = 0;
    f = fGlobalForcedActions;
    PRINT_GROUP(VM_FF_,EXTERNAL_SUSPENDED,_MASK);
    PRINT_GROUP(VM_FF_,EXTERNAL_HALTED,_MASK);
    PRINT_GROUP(VM_FF_,HIGH_PRIORITY_PRE,_MASK);
    PRINT_GROUP(VM_FF_,HIGH_PRIORITY_PRE_RAW,_MASK);
    PRINT_GROUP(VM_FF_,HIGH_PRIORITY_POST,_MASK);
    PRINT_GROUP(VM_FF_,NORMAL_PRIORITY_POST,_MASK);
    PRINT_GROUP(VM_FF_,NORMAL_PRIORITY,_MASK);
    PRINT_GROUP(VM_FF_,ALL_BUT_RAW,_MASK);
    if (c)
        pHlp->pfnPrintf(pHlp, "\n");

    /*
     * Per CPU flags.
     */
    for (VMCPUID i = 0; i < pVM->cCpus; i++)
    {
        const uint32_t fLocalForcedActions = pVM->aCpus[i].fLocalForcedActions;
        pHlp->pfnPrintf(pHlp, "CPU %u FFs: %#RX32", i, fLocalForcedActions);

        /* show the flag mnemonics */
        c = 0;
        f = fLocalForcedActions;
        PRINT_FLAG(VMCPU_FF_,INTERRUPT_APIC);
        PRINT_FLAG(VMCPU_FF_,INTERRUPT_PIC);
        PRINT_FLAG(VMCPU_FF_,TIMER);
        PRINT_FLAG(VMCPU_FF_,PDM_CRITSECT);
        PRINT_FLAG(VMCPU_FF_,PGM_SYNC_CR3);
        PRINT_FLAG(VMCPU_FF_,PGM_SYNC_CR3_NON_GLOBAL);
        PRINT_FLAG(VMCPU_FF_,TLB_FLUSH);
        PRINT_FLAG(VMCPU_FF_,TRPM_SYNC_IDT);
        PRINT_FLAG(VMCPU_FF_,SELM_SYNC_TSS);
        PRINT_FLAG(VMCPU_FF_,SELM_SYNC_GDT);
        PRINT_FLAG(VMCPU_FF_,SELM_SYNC_LDT);
        PRINT_FLAG(VMCPU_FF_,INHIBIT_INTERRUPTS);
        PRINT_FLAG(VMCPU_FF_,CSAM_SCAN_PAGE);
        PRINT_FLAG(VMCPU_FF_,CSAM_PENDING_ACTION);
        PRINT_FLAG(VMCPU_FF_,TO_R3);
        if (f)
            pHlp->pfnPrintf(pHlp, "%s\n    Unknown bits: %#RX32\n", c ? "," : "", f);
        else
            pHlp->pfnPrintf(pHlp, "\n");

        /* the groups */
        c = 0;
        f = fLocalForcedActions;
        PRINT_GROUP(VMCPU_FF_,EXTERNAL_SUSPENDED,_MASK);
        PRINT_GROUP(VMCPU_FF_,EXTERNAL_HALTED,_MASK);
        PRINT_GROUP(VMCPU_FF_,HIGH_PRIORITY_PRE,_MASK);
        PRINT_GROUP(VMCPU_FF_,HIGH_PRIORITY_PRE_RAW,_MASK);
        PRINT_GROUP(VMCPU_FF_,HIGH_PRIORITY_POST,_MASK);
        PRINT_GROUP(VMCPU_FF_,NORMAL_PRIORITY_POST,_MASK);
        PRINT_GROUP(VMCPU_FF_,NORMAL_PRIORITY,_MASK);
        PRINT_GROUP(VMCPU_FF_,RESUME_GUEST,_MASK);
        PRINT_GROUP(VMCPU_FF_,HWACCM_TO_R3,_MASK);
        PRINT_GROUP(VMCPU_FF_,ALL_BUT_RAW,_MASK);
        if (c)
            pHlp->pfnPrintf(pHlp, "\n");
    }

#undef PRINT_FLAG
#undef PRINT_GROUP
}

