/* Definitions for partial image regions.
 *
 * J.Cupitt, 8/4/93
 */

/*

    This file is part of VIPS.
    
    VIPS is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#ifndef IM_REGION_H
#define IM_REGION_H

#ifdef __cplusplus
extern "C" {
#endif /*__cplusplus*/

#ifdef TIME_THREAD
#include <sys/time.h>
#endif /*TIME_THREAD*/

#include "rect.h"

/* Region types. 
 */
typedef enum region_type {
	IM_REGION_NONE,
	IM_REGION_LOCAL,	/* malloc() buffer local to region */
	IM_REGION_OTHER_REGION, /* memory on another region */
	IM_REGION_OTHER_IMAGE,	/* memory on another image */
	IM_REGION_MMAP_WINDOW	/* mmap() buffer on fd on another image */
} RegionType;

/* Sub-area of image.
 */
typedef struct region_struct {
	/* Users may read these two fields.
	 */
	IMAGE *im;		/* Link back to parent image */
	Rect valid;		/* Area of parent we can see */

	/* The rest of REGION is private.
	 */
	RegionType type;	/* What kind of attachment */
	char *data;		/* Off here to get data */
	int bpl;		/* Bytes-per-line for data */
	void *seq;		/* Sequence we are using to fill region */
	char *buf;		/* Private malloc() area */
	int bsize;		/* Size of private malloc() */
	PEL *baseaddr;		/* mmap() window, if any */
	size_t length;		/* Size of mmap window */
} REGION;

/* Private to iofuncs: the size of the `tiles' requested by im_generate() 
 * when acting as a data sink.
 */
#define IM__TILE_WIDTH (30)
#define IM__TILE_HEIGHT (30)

/* The height of the strips for the other two request styles.
 */
#define IM__THINSTRIP_HEIGHT (1)
#define IM__FATSTRIP_HEIGHT (10)

/* Functions on regions.
 */
REGION *im_region_create( IMAGE *im );
void im_region_free( REGION *reg );
int im_region_local( REGION *reg, Rect *r );
int im_region_image( REGION *reg, Rect *r );
int im_region_region( REGION *reg, REGION *to, Rect *r, int x, int y );
int im_region_mmap_window( REGION *reg, Rect *r );
int im_region_equalsregion( REGION *reg1, REGION *reg2 );
int im_region_position( REGION *reg1, int x, int y );

/* IMAGE functions which use regions. We do not strictly type the function
 * arguments to avoid hassle.
 */
int im_prepare( REGION *reg, Rect *r );
int im_prepare_to( REGION *reg, REGION *dest, Rect *r, int x, int y );
int im_generate( IMAGE *im, 
	void *(*start_fn)(), int (*gen_fn)(), int (*stop_fn)(), 
	void *a, void *b 
);
int im_iterate( IMAGE *im, 
	void *(*start_fn)(), int (*scan_fn)(), int (*stop_fn)(), 
	void *a, void *b 
);
void im__copy_region( REGION *reg, REGION *dest, Rect *r, int x, int y );

/* Convenience functions for im_generate()/im_iterate().
 */
void *im_start_one( IMAGE *out, IMAGE *in, void *dummy );
int im_stop_one( REGION *reg, void *dummy1, void *dummy2 );
void *im_start_many( IMAGE *out, IMAGE **in, void *dummy );
int im_stop_many( REGION **out, void *dummy1, void *dummy2 );
IMAGE **im_allocate_input_array( IMAGE *out, ... );
int im_demand_hint( IMAGE *im, im_demand_type hint, ... );
int im_demand_hint_array( IMAGE *im, im_demand_type hint, IMAGE **in );
void im_free_region_array( REGION **regs );
REGION **im_allocate_region_array( IMAGE *im, int count );
void im__find_demand_size( IMAGE *im, int *pw, int *ph );

/* Buffer processing.
 */
typedef void (*im_wrapmany_fn)( void **in, void *out, int width, 
	void *a, void *b );
int im_wrapmany( IMAGE **in, IMAGE *out, 
	im_wrapmany_fn fn, void *a, void *b );
typedef void (*im_wrapone_fn)( void *in, void *out, int width, 
	void *a, void *b );
int im_wrapone( IMAGE *in, IMAGE *out, 
	im_wrapone_fn fn, void *a, void *b );

/* Internal VIPS functions shared by partials.
 */
int im__call_start( REGION *reg );
void im__call_stop( REGION *reg );

/* Don't define if IM_NO_VIPS7_COMPAT is set.
 */
#ifndef IM_NO_VIPS7_COMPAT

/* Compatibilty macros ... delete soon. See below for the new names.
 */

/* Macros on REGIONs.
 *	lskip()		add to move down line
 *	nele()		number of elements across region
 *	rsize()		sizeof width of region
 *	addr()		address of pixel in region
 */
#define lskip(B) ((B)->bpl)
#define nele(B) ((B)->valid.width*(B)->im->Bands)
#define rsize(B) ((B)->valid.width*psize((B)->im))

/* addr() is special: if DEBUG is defined, make an addr() with bounds checking.
 */
#ifdef DEBUG
#define addr(B,X,Y) \
	( (im_rect_includespoint( &(B)->valid, (X), (Y) ))? \
	  ((B)->data + ((Y) - (B)->valid.top)*lskip(B) + \
	  ((X) - (B)->valid.left)*psize((B)->im)): \
	  (fprintf( stderr, \
		"addr: point out of bounds, file \"%s\", line %d\n" \
		"(point x=%d, y=%d\n" \
		" should have been within Rect left=%d, top=%d, " \
		"width=%d, height=%d)\n", \
		__FILE__, __LINE__, \
		(X), (Y), \
		(B)->valid.left, \
		(B)->valid.top, \
		(B)->valid.width, \
		(B)->valid.height ), abort(), (char *) NULL) \
	)
#else /*DEBUG*/
#define addr(B,X,Y) ((B)->data + ((Y)-(B)->valid.top)*lskip(B) + \
	((X)-(B)->valid.left)*psize((B)->im))
#endif /*DEBUG*/

#endif /*IM_NO_VIPS7_COMPAT*/

/* Macros on REGIONs.
 *	IM_REGION_LSKIP()		add to move down line
 *	IM_REGION_N_ELEMENTS()		number of elements across region
 *	IM_REGION_SIZEOF_LINE()		sizeof width of region
 *	IM_REGION_ADDR()		address of pixel in region
 */
#define IM_REGION_LSKIP(R) ((R)->bpl)
#define IM_REGION_N_ELEMENTS(R) ((R)->valid.width*(R)->im->Bands)
#define IM_REGION_SIZEOF_LINE(R) \
	((R)->valid.width * IM_IMAGE_SIZEOF_PEL((R)->im))

/* If DEBUG is defined, add bounds checking.
 */
#ifdef DEBUG
#define IM_REGION_ADDR(B,X,Y) \
	( (im_rect_includespoint( &(B)->valid, (X), (Y) ))? \
	  ((B)->data + ((Y) - (B)->valid.top)*IM_REGION_LSKIP(B) + \
	  ((X) - (B)->valid.left)*IM_IMAGE_SIZEOF_PEL((B)->im)): \
	  (fprintf( stderr, \
		"IM_REGION_ADDR: point out of bounds, " \
		"file \"%s\", line %d\n" \
		"(point x=%d, y=%d\n" \
		" should have been within Rect left=%d, top=%d, " \
		"width=%d, height=%d)\n", \
		__FILE__, __LINE__, \
		(X), (Y), \
		(B)->valid.left, \
		(B)->valid.top, \
		(B)->valid.width, \
		(B)->valid.height ), abort(), (char *) NULL) \
	)
#else /*DEBUG*/
#define IM_REGION_ADDR(B,X,Y) \
	((B)->data + \
	((Y)-(B)->valid.top) * IM_REGION_LSKIP(B) + \
	((X)-(B)->valid.left) * IM_IMAGE_SIZEOF_PEL((B)->im))
#endif /*DEBUG*/

#ifdef __cplusplus
}
#endif /*__cplusplus*/

#endif /*IM_REGION_H*/
