
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL

#include "ExtendedScrollView.h"

#include <Foundation/NSString.h>
#include <AppKit/NSButton.h>
#include <AppKit/NSImage.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSScroller.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSView.h>

/*
 * Non-Public methods.
 */
@interface ExtendedScrollView(Private)
- (void) createPageField;
- (void) createControlsView;
- (NSButton*) makeButtonAt: (float)posX
                     image: (NSString*)imageName
                     width: (int)width;
@end

/*
 * Provides a custom ScrollView with controls to 
 * display / manipulate the current page etc.
 *
 * Note: The additional controls will only be displayed
 *       correctly if the horizontal scroller is visible.
 */
@implementation ExtendedScrollView

/*
 * Designated initializer.
 */
- (id) initWithFrame: (NSRect)frame
{
   if (self = [super initWithFrame: frame])
   {
      [self createControlsView];
      [self createPageField];
      [self setDisplayPageControls: NO];
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc ExtendedScrollView, retain count is %d", [self retainCount]);

   [pageField release];
   [controlsView release];

   [super dealloc];
}


/* Outlets */

- (NSButton*) firstPageButton
{
   return firstPageButton;
}


- (NSButton*) nextPageButton
{
   return nextPageButton;
}


- (NSButton*) prevPageButton
{
   return prevPageButton;
}


- (NSButton*) lastPageButton
{
   return lastPageButton;
}


- (NSTextField*) pageField
{
   return pageField;
}


/*
 * Set whether the page control elements should be
 * display at the left of the horizontal scroller.
 */
- (void) setDisplayPageControls: (BOOL)display
{
   displayPageControls = display;
}


/*
 * Determine whether the page control elements
 * are displayed.
 */
- (BOOL) displayPageControls
{
   return displayPageControls;
}


/*
 * Consider additional components when the ScrollView is
 * layoutet.
 */ 
- (void) tile
{
   [super tile];

   if ([self displayPageControls])
   {
      NSRect scrollerRect;
      NSRect controlsRect;

      scrollerRect = [[self horizontalScroller] frame];
      NSDivideRect(scrollerRect,
                   &controlsRect,
                   &scrollerRect,
                   160.0,
                   NSMinXEdge);
   
      [[self horizontalScroller] setFrame: scrollerRect];
      [controlsView setFrame: NSInsetRect(controlsRect, 1, 1)];
   }
}


/*
 * The controlsView takes the additional components to maniplute
 * some settings. It is displayed along with the horizontal scroller
 * on the left side.
 */
- (void) createControlsView
{
   NSRect frame;

   // the frame doesn't matter much here because it will
   // be re-calcualted when the scrollview is layoutet.
   frame = NSMakeRect(0, 0, 10, 10);

   controlsView = [[NSView alloc] initWithFrame: frame];
   [controlsView setAutoresizesSubviews: YES];

   [self addSubview: controlsView];
}


- (void) createPageField
{
   NSRect     frame;
   id         pageLabel;
   NSString*  labelText = @"Page";
   int        height;

   // the height for all components
   height = [NSScroller scrollerWidthForControlSize: NSRegularControlSize];

   // the label
   frame = NSMakeRect(0, 0, 35, height);
   pageLabel = [[[NSTextField alloc] initWithFrame: frame] autorelease];

   [pageLabel setFont: [NSFont labelFontOfSize: 10.0]];
   [pageLabel setBordered: NO];
   [pageLabel setBezeled: NO];
   [pageLabel setSelectable: NO];
   [pageLabel setDrawsBackground: NO];
   [pageLabel setEditable: NO];
   [pageLabel setAutoresizingMask: NSViewNotSizable];
   [pageLabel setStringValue: labelText];
   [pageLabel setAlignment: NSCenterTextAlignment];

   [controlsView addSubview: pageLabel];

   // the page-of field
   frame = NSMakeRect(35, 0, 60, height);
   pageField = [[NSTextField alloc] initWithFrame: frame];
   [pageField setAutoresizingMask: NSViewNotSizable];
   [pageField setFont: [NSFont labelFontOfSize: 10.0]];
   [pageField setBordered: NO];
   [pageField setStringValue: @""];
   
   [controlsView addSubview: pageField];

   // Buttons
   firstPageButton = [self makeButtonAt: 100
                           image: @"first_page.tiff"
                           width: -1];
   [controlsView addSubview: firstPageButton];

   prevPageButton = [self makeButtonAt: [firstPageButton frame].origin.x
                                      + [firstPageButton frame].size.width
                          image: @"prev_page.tiff"
                          width: 15];
   [controlsView addSubview: prevPageButton];

   nextPageButton = [self makeButtonAt: [prevPageButton frame].origin.x
                                      + [prevPageButton frame].size.width
                          image: @"next_page.tiff"
                          width: 15];
   [controlsView addSubview: nextPageButton];

   lastPageButton = [self makeButtonAt: [nextPageButton frame].origin.x
                                      + [prevPageButton frame].size.width
                          image :@"last_page.tiff"
                          width: -1];
   [controlsView addSubview: lastPageButton];
}


/*
 * Create a button that is shown on the vertical
 * scroller.
 */
- (NSButton*) makeButtonAt: (float)posX
                     image: (NSString*)imageName
                     width: (int)width
{
   NSButton* theButton;
   NSRect    frame;

   frame = NSMakeRect(posX, 0, 0, 0);
   theButton = [[[NSButton alloc] initWithFrame: frame] autorelease];
   [theButton setButtonType: NSMomentaryLight];
   [theButton setImagePosition: NSImageOnly];
   [theButton setImage: [NSImage imageNamed:imageName]];
   [theButton setBordered: NO];
   [theButton sizeToFit];
   [theButton setEnabled: YES];
   if (width > 0)
   {
      [theButton setFrameSize: NSMakeSize(width,
                                          [theButton frame].size.height)];
   }

   return theButton;
}



@end
