#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <gtk/gtk.h>

#include "../include/string.h"
#include "../include/fio.h"
#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"

#include "v3dmp.h"
#include "v3dmodel.h"

#include "editor.h"

#include "vma.h"
#include "vmautils.h"
#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


const gchar *VMADefaultDataGlobalDir(void);
const gchar *VMATempName(
	const gchar *tmp_dir, const gchar *name_prefix
);
FILE *VMAMakeOpenTmpFile(
	gpointer core_ptr,
	const gchar *tmp_dir,
	const gchar *name_prefix,
	gchar **new_path_rtn
);

static glong VMAModelGetStatsPrimitives(gpointer p);

gint VMATextFileToCList(
	GtkCList *clist, const gchar *path, GtkStyle *style
);
gint VMAModelGetStats(
	v3d_model_struct *model_ptr, vma_model_stats_struct *buf
);
void VMAReportGLError(gpointer core_ptr, gint error_code);
gint VMAGetGLVersion(
	gpointer core_ptr,
	gint *gl_version_major, gint *gl_version_minor,
	const gchar **renderer,
	const gchar **vendor,
	const gchar **extensions
);

gbool VMAWriteProtectCheck(gpointer editor_ptr);
void VMAFunctionNotAvailable(gpointer editor_ptr, const gchar *name);



/*
 *	Returns a statically allocated string containing the default
 *	global data directory. Note this value is not from the global
 *	configuration options list.
 *
 *	The value defined in the environment variable will be checked
 *	first, if it is NULL then VMA_DEF_DATA_GLOBAL_DIR is returned.
 */
const gchar *VMADefaultDataGlobalDir(void)
{
	const gchar *cstrptr;
	static gchar rtn_path[PATH_MAX];

	cstrptr = (const gchar *)getenv(VMA_ENVNAME_DATA_GLOBAL_DIR);
	if(cstrptr == NULL)
	    strncpy(rtn_path, VMA_DEF_DATA_GLOBAL_DIR, PATH_MAX);
	else
	    strncpy(rtn_path, cstrptr, PATH_MAX);
	rtn_path[PATH_MAX - 1] = '\0';

	return((const gchar *)rtn_path);
}

/*
 *	Works similar to tempnam(), but returns a statically allocated
 *	string that must not be deallocated or modified.
 *
 *	Can return NULL if a tempory file name cannot be generated.
 */
const gchar *VMATempName(
	const gchar *tmp_dir, const gchar *name_prefix
)
{
	gint len;
	guint count;
	const gchar *cstrptr;
	struct stat stat_buf;
	static gchar tmp_path_rtn[PATH_MAX + NAME_MAX];


	/* If given name prefix is NULL then juts tack on an arbitary one. */
	if(name_prefix == NULL)
	    name_prefix = "tmp";

	/* Begin getting tempory directory. */
	/* Environment variable TMPDIR. */
	cstrptr = (const gchar *)getenv("TMPDIR");
	if(cstrptr != NULL)
	{
	    /* Must exist and be writeable. */
	    if(!stat(cstrptr, &stat_buf))
	    {
		tmp_dir = cstrptr;
	    }
	}

	/* If the given tmp_dir is not NULL but does not exist, then
	 * we need to fail.
	 */
	if(tmp_dir != NULL)
	{
	    /* Dosen't exist? */
	    if(stat(tmp_dir, &stat_buf))
		return(NULL);
	}

	/* If the given tmp_dir is NULL then go through some fallbacks. */
	if(tmp_dir == NULL)
	{
#ifdef P_tmpdir
	    tmp_dir = P_tmpdir;
#else
	    tmp_dir = "/tmp";
#endif
	}

	/* Still not able to get tempory directory? */
	if(tmp_dir == NULL)
	    return(NULL);

	/* Get maximum possible length of return path. */
	len = strlen(tmp_dir) + strlen(name_prefix) + 80;
	/* Would the maximum possible length of return path exceed
	 * the allocation of our return path?
	 */
	if(len >= (PATH_MAX + NAME_MAX))
	    return(NULL);

	/* Begin formulating tempory file path. */
	(*tmp_path_rtn) = '\0';

	for(count = 0; count < ((guint16)-1); count++)
	{
	    sprintf(
		tmp_path_rtn,
		"%s/%s%i",
		tmp_dir, name_prefix, count
	    );
	    /* Check if file does NOT exist. */
	    if(stat(tmp_path_rtn, &stat_buf))
		break;
	}
	if(count >= ((guint16)-1))
	    return(NULL);
	else
	    return(tmp_path_rtn);
}


/*
 *	Attempts to create a tempory file with up to five characters from
 *	name_prefix if it is not NULL.
 *
 *	The file will be opened as "wb" and it's FILE pointer returned
 *	or NULL on error.
 *
 *	If the directory tmp_dir does not exist, then it will be created.
 *	If the object specified by tmp_dir is not a directory, then NULL
 *	will be returned. Note that tmp_dir may not be NULL.
 *
 *	If (and only if) new_path_rtn is not NULL then a dynamically
 *	allocated string will be set to it. The calling function is
 *	responsible for deallocating it. If new_path_rtn is NULL then
 *	there will be no allocated string.
 */
FILE *VMAMakeOpenTmpFile(
	gpointer core_ptr,
	const gchar *tmp_dir,
	const gchar *name_prefix,
	gchar **new_path_rtn
)
{
	const gchar *name;
	struct stat stat_buf;


	if(new_path_rtn != NULL)
	    (*new_path_rtn) = NULL;

	/* Cannot generate tmp file without dir name. */
	if(tmp_dir == NULL)
	    return(NULL);

	/* Check if given directory exists. */
	if(stat(tmp_dir, &stat_buf))
	{
	    /* Does not exist, create it. */
	    mode_t m = (S_IRUSR | S_IWUSR | S_IXUSR |
			S_IRGRP | S_IXGRP
	    );
	    if(rmkdir(tmp_dir, m))
	    {
		/* Could not create directory. */
		return(NULL);
	    }
	}
	else
	{
	    /* Exists, but check if it is not a directory. */
	    if(!S_ISDIR(stat_buf.st_mode))
	    {
		/* Not a directory, we can't create objects in it.
		 * At this point we need to give up. 
		 */
		return(NULL);
	    }
	}

	/* At this point we are sure that tmp_dir exists and is a
	 * directory.
	 */
	name = VMATempName(tmp_dir, name_prefix);
	if(name != NULL)
	{
	    FILE *fp = FOpen(name, "wb");

	    /* If new_path_rtn is not NULL then pass name to it, otherwise
	     * dealloate the local name.
	     */
	    if(new_path_rtn != NULL)
		(*new_path_rtn) = g_strdup(name);

	    return(fp);
	}
	else
	{
	    return(NULL);
	}
}

/*
 *	Loads the contents of a text file into the given clist.
 *
 *	The calling function needs to clear the clist.
 *
 *	The style specifies the style for each row.
 *
 *	Returns non-zero on error.
 */
gint VMATextFileToCList(
	GtkCList *clist, const gchar *path, GtkStyle *style
)
{
	gint columns;
	gchar *buf, *strptr;
	gchar **val;
	FILE *fp;

	if((clist == NULL) || (path == NULL))
	    return(-1);

	columns = clist->columns;
	if(columns < 1)
	    return(-1);

	/* Open file. */
	fp = FOpen(path, "rb");
	if(fp == NULL)
	    return(-1);

	/* Allocate clist row value array. */
	val = (gchar **)calloc(
	    columns, sizeof(gchar *)
	);
	if(val == NULL)
	{
	    FClose(fp);			/* Close file. */
	    return(-1);
	}

	/* Begin reading file. */
	buf = FGetStringLiteral(fp);
	while(buf != NULL)
	{
	    strptr = strchr(buf, '\n');
	    if(strptr != NULL)
		(*strptr) = '\0';

	    if((columns > 0) && (val != NULL))
	    {
		gint n;

		val[0] = buf;
		n = gtk_clist_append(clist, val);
		if((n > -1) && (style != NULL))
		    gtk_clist_set_row_style(
			clist, n, style
		    );
	    }

	    free(buf);
	    memset(val, 0x00, columns * sizeof(gchar *));
	    buf = FGetStringLiteral(fp);
	}

	FClose(fp);		/* Close file. */
	g_free(val);		/* Deallocate clist row value array. */

	return(0);
}

/*
 *	Called by VMAModelGetStats() to calculate memory used by the
 *	given primitive.
 */
static glong VMAModelGetStatsPrimitives(gpointer p)
{
	gint i;
	glong total = 0;
	gint ptype = (*(gint *)p);
	const gchar *line_ptr;

	mp_comment_struct *mp_comment;
	mp_translate_struct *mp_translate;
	mp_untranslate_struct *mp_untranslate;
	mp_rotate_struct *mp_rotate;
	mp_unrotate_struct *mp_unrotate;
	mp_point_struct *mp_point;
	mp_line_struct *mp_line;
	mp_line_strip_struct *mp_line_strip;
	mp_line_loop_struct *mp_line_loop;
	mp_triangle_struct *mp_triangle;
	mp_triangle_strip_struct *mp_triangle_strip;
	mp_triangle_fan_struct *mp_triangle_fan;
	mp_quad_struct *mp_quad;
	mp_quad_strip_struct *mp_quad_strip;
	mp_polygon_struct *mp_polygon;
	mp_color_struct *mp_color;
	mp_texture_select_struct *mp_texture_select;
	mp_texture_orient_xy_struct *mp_texture_xy;
	mp_texture_orient_yz_struct *mp_texture_yz;
	mp_texture_orient_xz_struct *mp_texture_xz;
	mp_texture_off_struct *mp_texture_off;
	mp_heightfield_load_struct *mp_heightfield_load;


	switch(ptype)
	{
	  case V3DMP_TYPE_COMMENT:
#define PTR	mp_comment
#define PTYPE	mp_comment_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
	    for(i = 0; i < PTR->total_lines; i++)
	    {
		line_ptr = (const char *)PTR->line[i];
		if(line_ptr != NULL)
		    total += (long)(strlen(line_ptr) + 1);
	    }
	    total += PTR->total_lines * sizeof(char *);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TRANSLATE:
#define PTR     mp_translate
#define PTYPE   mp_translate_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);            
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_UNTRANSLATE:
#define PTR     mp_untranslate
#define PTYPE   mp_untranslate_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_ROTATE:
#define PTR     mp_rotate
#define PTYPE   mp_rotate_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;   
	
	  case V3DMP_TYPE_UNROTATE:
#define PTR     mp_unrotate
#define PTYPE   mp_unrotate_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_POINT:
#define PTR     mp_point
#define PTYPE   mp_point_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_LINE:
#define PTR     mp_line
#define PTYPE   mp_line_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_LINE_STRIP:
#define PTR     mp_line_strip
#define PTYPE   mp_line_strip_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct *);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_LINE_LOOP:
#define PTR     mp_line_loop
#define PTYPE   mp_line_loop_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct);  
	    total += PTR->total * 3 * sizeof(mp_vertex_struct *);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TRIANGLE:
#define PTR     mp_triangle
#define PTYPE   mp_triangle_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TRIANGLE_STRIP:
#define PTR     mp_triangle_strip
#define PTYPE   mp_triangle_strip_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct *);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TRIANGLE_FAN:
#define PTR     mp_triangle_fan
#define PTYPE   mp_triangle_fan_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct *);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_QUAD:
#define PTR     mp_quad
#define PTYPE   mp_quad_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_QUAD_STRIP:
#define PTR     mp_quad_strip
#define PTYPE   mp_quad_strip_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct *);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_POLYGON:
#define PTR     mp_polygon
#define PTYPE   mp_polygon_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct);
	    total += PTR->total * 3 * sizeof(mp_vertex_struct *);
#undef PTR  
#undef PTYPE
	    break;

	  case V3DMP_TYPE_COLOR:
#define PTR     mp_color
#define PTYPE   mp_color_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TEXTURE_SELECT:
#define PTR     mp_texture_select
#define PTYPE   mp_texture_select_struct
	    PTR = (PTYPE *)p;  
	    total += sizeof(PTYPE);
	    if(PTR->name != NULL)
		total += (long)(strlen(PTR->name) + 1);
	    /* Skip client data. */
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TEXTURE_ORIENT_XY:
#define PTR     mp_texture_xy
#define PTYPE   mp_texture_orient_xy_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TEXTURE_ORIENT_YZ:
#define PTR     mp_texture_yz
#define PTYPE   mp_texture_orient_yz_struct
	    PTR = (PTYPE *)p;   
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TEXTURE_ORIENT_XZ:
#define PTR     mp_texture_xz
#define PTYPE   mp_texture_orient_xz_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_TEXTURE_OFF:
#define PTR     mp_texture_off
#define PTYPE   mp_texture_off_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
#undef PTR
#undef PTYPE
	    break;

	  case V3DMP_TYPE_HEIGHTFIELD_LOAD:
#define PTR     mp_heightfield_load
#define PTYPE   mp_heightfield_load_struct
	    PTR = (PTYPE *)p;
	    total += sizeof(PTYPE);
	    if(PTR->path != NULL)
		total += (long)(strlen(PTR->path) + 1);
#undef PTR
#undef PTYPE
	    break;
 

/* Add support for other primitive types here. */



	  default:
	    break;
	}

	return(total);
}

/*
 *	Gets statistics from the given model and stores them on the given
 *	buf.
 *
 *	Can return non-zero on error.
 */
gint VMAModelGetStats(
	v3d_model_struct *model_ptr, vma_model_stats_struct *buf
)
{
	gint i;
	gpointer p;
	const gchar *line_ptr;


	if((model_ptr == NULL) || (buf == NULL))
	    return(-1);

	/* Reset buf. */
	memset(buf, 0x00, sizeof(vma_model_stats_struct));


	/* Begin calculating. */
	buf->type = model_ptr->type;

	/* Name. */
	if(model_ptr->name != NULL)
	    buf->size_misc += (long)(strlen(model_ptr->name) + 1);

	/* Primitives. */
	buf->primitives = model_ptr->total_primitives;
	for(i = 0; i < model_ptr->total_primitives; i++)
	{
	    p = model_ptr->primitive[i];
	    if(p == NULL)
		continue;

	    buf->size_primitives += VMAModelGetStatsPrimitives(p);
	}
	buf->size_primitives += model_ptr->total_primitives * sizeof(void *);

	/* Other data lines. */
	buf->lines = model_ptr->total_other_data_lines;
	for(i = 0; i < model_ptr->total_other_data_lines; i++)
	{   
	    line_ptr = (const char *)model_ptr->other_data_line[i];
	    if(line_ptr == NULL)
		continue;
	
	    buf->size_lines += (long)(strlen(line_ptr) + 1);
	}
	buf->size_lines += model_ptr->total_other_data_lines * sizeof(char *);


	/* Calculate total. */
	buf->size = buf->size_primitives + buf->size_lines + 
	    buf->size_misc + sizeof(v3d_model_struct);

	return(0);
}


/*
 *	Reports the given GL error (even if it is success).
 */
void VMAReportGLError(gpointer core_ptr, gint error_code)
{
	const gchar *error_mesg = NULL, *error_type = NULL;

	switch((GLenum)error_code)
	{
	  case GL_INVALID_ENUM:  
	    error_type = "GL_INVALID_ENUM";  
	    error_mesg =
		"Invalid GLenum argument (possibly out of range)";
	    break;
 
	  case GL_INVALID_VALUE:  
	    error_type = "GL_INVALID_VALUE";  
	    error_mesg =
		"Numeric argument out of range";
	    break;
  
	  case GL_INVALID_OPERATION:
	    error_type = "GL_INVALID_OPERATION";
	    error_mesg =
		"Operation illegal in current state";
	    break;
	    
	  case GL_STACK_OVERFLOW:
	    error_type = "GL_STACK_OVERFLOW";
	    error_mesg = 
		"Command would cause stack overflow";
	    break;

	  case GL_STACK_UNDERFLOW:
	    error_type = "GL_STACK_UNDERFLOW";
	    error_mesg =
		"Command would cause a stack underflow";
	    break;

	  case GL_OUT_OF_MEMORY:
	    error_type = "GL_OUT_OF_MEMORY";
	    error_mesg =
		"Not enough memory left to execute command";
	    break;

	  default:
	    error_type = "*UNKNOWN*";
	    error_mesg = "Undefined GL error";
	    break;
	}

	fprintf(
	    stderr,
	    "GL Error code %i: %s: %s\n",
	    error_code, error_type, error_mesg
	);
}
	    
/*
 *	Returns the GL version of the binded context from the
 *	first view on the first initialized editor.
 *
 *	Returns non-zero on error.
 */
gint VMAGetGLVersion(
	gpointer core_ptr,
	gint *gl_version_major, gint *gl_version_minor,
	const gchar **renderer,
	const gchar **vendor,
	const gchar **extensions
)
{
	gint i;
	ma_editor_struct *editor;
	static gchar s_renderer[256];
	static gchar s_vendor[256];
	static gchar s_extensions[1024];
	vma_core_struct *core = (vma_core_struct *)core_ptr;


	if(gl_version_major != NULL)
	    (*gl_version_major) = 0;
	if(gl_version_minor != NULL)
	    (*gl_version_minor) = 0;
	if(renderer != NULL)
	    (*renderer) = NULL;
	if(vendor != NULL)
	    (*vendor) = NULL;
	if(extensions != NULL)
	    (*extensions) = NULL;


	if(core == NULL)
	    return(-1);

	for(i = 0; i < core->total_editors; i++)
	{
	    editor = core->editor[i];
	    if(editor == NULL)
		continue;

	    if(!editor->initialized || editor->processing)
		continue;

	    if(VMA_MAX_2D_VIEWS_PER_EDITOR > 0)
	    {
		const GLubyte *cstrptr;


		if(View2DGLEnableContext(editor->view2d[0]))
		    continue;

		/* Get version. */
		cstrptr = glGetString(GL_VERSION);
		if(cstrptr != NULL)
		{
		    if(gl_version_major != NULL)
			(*gl_version_major) = atoi(cstrptr);
		    cstrptr = strchr(cstrptr, '.');
		    if((cstrptr != NULL) && (gl_version_minor != NULL))
			(*gl_version_minor) = atoi(cstrptr + 1);
		}

		/* Get renderer. */
		cstrptr = glGetString(GL_RENDERER);
		if(cstrptr != NULL)
		{
		    strncpy(s_renderer, cstrptr, 256);
		    s_renderer[256 - 1] = '\0';
		    if(renderer != NULL)
			(*renderer) = (const char *)s_renderer;
		}

		/* Get vendor. */
		cstrptr = glGetString(GL_VENDOR);
		if(cstrptr != NULL)
		{
		    strncpy(s_vendor, cstrptr, 256);
		    s_vendor[256 - 1] = '\0';
		    if(vendor != NULL)
			(*vendor) = (const char *)s_vendor;
		}

		/* Get extensions. */
		cstrptr = glGetString(GL_EXTENSIONS);
		if(cstrptr != NULL)
		{
		    strncpy(s_extensions, cstrptr, 256);
		    s_extensions[256 - 1] = '\0';
		    if(extensions != NULL)
			(*extensions) = (const char *)s_extensions;
		}

		return(0);
	    }
	}

	return(-1);
}


/*
 *	Checks if write protect is on the given editor, if it is
 *	on then the comfirmation dialog will be mapped and show
 *	the appropriate message and return TRUE.
 *
 *	If write protect is not enabled or there was an error, then
 *	FALSE will be returned.
 *
 *	This function does not check if the editor is currently
 *	processing.
 */
gbool VMAWriteProtectCheck(gpointer editor_ptr)
{
	ma_editor_struct *editor = (ma_editor_struct *)editor_ptr;
	if(editor == NULL)
	    return(FALSE);

	if(!editor->initialized)
	    return(FALSE);

	/* Is write protect enabled? */
	if(editor->write_protect)
	{
	    GtkWidget *w = editor->toplevel;

	    CDialogSetTransientFor(w);
	    CDialogGetResponse(
"Write Protect Enabled",
"Changes cannot be made because the editor's\n\
write protect is enabled. If you wish to make\n\
changes then you must first click on `OK' on this\n\
dialog and then disable write protect by going\n\
to Edit->Write Protect.",
"You have requested an operation that would modify\n\
the loaded data on this editor, however this editor's\n\
write protect is enabled and thus the editor is in\n\
read-only mode. If you wish to make changes, then you\n\
need to disable write protect. First click on `OK' and\n\
then go to Edit->Write Protect.",
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	}

	return(editor->write_protect);
}


/*
 *	Just displays a message stating that the given function is not
 *	available. The name of the feature is described by name, if
 *	name is NULL then a more generic message will be used.
 */
void VMAFunctionNotAvailable(gpointer editor_ptr, const gchar *name)
{
	gchar *buf;
	gint buf_len;
	GtkWidget *w = NULL;
	ma_editor_struct *editor = (ma_editor_struct *)editor_ptr;


	if(editor != NULL)
	    w = editor->toplevel;

	buf_len = ((name == NULL) ? 0 : strlen(name)) + 2048;
	buf = (gchar *)g_malloc((buf_len + 1) * sizeof(gchar));
	if(buf != NULL)
	{
	    sprintf(
		buf,
"Sorry, the function `%s' is either not available or\n\
under construction.\n\
You may want to check if a newer version of this\n\
application has this function completed.",
		name
	    );

	    CDialogSetTransientFor(w);
	    CDialogGetResponse(
"Function Not Available",
		buf,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);

	    g_free(buf);
	}
}
