#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <gtk/gtk.h>

#include "vmacfg.h"
#include "vmacfgfio.h"
#include "vmacfglist.h"
#include "vmastyles.h"
#include "config.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


static gbool load_font_to_style(GtkStyle *style_ptr, const gchar *font_name);

void VMAStylesListCreate(vma_styles_list_struct *buf);
void VMAStylesListUpdate(vma_styles_list_struct *buf);
void VMAStylesListDestroy(vma_styles_list_struct *buf);



/*
 *      Macro to load font to given style structure.
 *
 *      Returns TRUE if the font was not able to be loaded, otherwise
 *      FALSE on success.
 */
static gbool load_font_to_style(GtkStyle *style_ptr, const gchar *font_name)
{
	GdkFont *new_font;


	/* Invalid font name? */
	if((font_name != NULL) ? ((*font_name) == '\0') : 1)
	    return(TRUE);

	/* Load new font. */
	new_font = gdk_font_load(font_name);
	if(new_font == NULL)
	    return(TRUE);
	
	/* Unref current font on style structure. */
	if(style_ptr->font != NULL)
	    gdk_font_unref(style_ptr->font);

	/* Set newly loaded font to style. */
	style_ptr->font = new_font;

	return(FALSE);
}


/*
 *	Creates a GTK+ styles list on the given buffer.
 */
void VMAStylesListCreate(vma_styles_list_struct *buf)
{
	GtkStyle *style_standard;


	if(buf == NULL)
	    return;

	memset(buf, 0x00, sizeof(vma_styles_list_struct));

	/* Get standard style from GTK+ internal. */
	buf->standard = style_standard = gtk_widget_get_default_style();
	if(style_standard == NULL)
	    return;

	/* Create styles. */
	buf->pref_banner = gtk_style_copy(style_standard);
	buf->text_editable = gtk_style_copy(style_standard);
	buf->text_terminal = gtk_style_copy(style_standard);
	buf->text_fine_print = gtk_style_copy(style_standard);
	buf->text_heading1 = gtk_style_copy(style_standard);
	buf->text_heading2 = gtk_style_copy(style_standard);
	buf->text_heading3 = gtk_style_copy(style_standard);
	buf->text_heading4 = gtk_style_copy(style_standard);
	buf->text_heading5 = gtk_style_copy(style_standard);
	buf->text_heading6 = gtk_style_copy(style_standard);

	VMAStylesListUpdate(buf);
}


/*
 *	Updates the given GTK+ styles list with the global configuration
 *	values.
 *
 *	This function should be called whenever the configuration has
 *	changed.
 */
void VMAStylesListUpdate(vma_styles_list_struct *buf)
{
	GdkColormap *colormap;
	GtkStyle *style, *style_standard;
	GdkColor *c;
	const gchar *cstrptr;


	/* Get colormap. */
	colormap = gdk_colormap_get_system();
	if(colormap == NULL)
	    return;

	/* Get standard GTK+ style. */
	style_standard = buf->standard;
	if(style_standard == NULL)
	    return;

	/* Preferences banner. */
	style = buf->pref_banner;
	if(style != NULL)
	{
	    /* Set `inverted colors' for foreground and background. */
	    c = &style->fg[GTK_STATE_NORMAL];
	    gdk_colormap_free_colors(colormap, c, 1);
	    (*c) = style_standard->bg[GTK_STATE_NORMAL];
	    gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);

	    c = &style->bg[GTK_STATE_NORMAL];
	    gdk_colormap_free_colors(colormap, c, 1);
	    (*c) = style_standard->fg[GTK_STATE_NORMAL];
	    gdk_colormap_alloc_color(colormap, c, TRUE, TRUE);

	    load_font_to_style(style, VMA_GTK_FONT_NAME_PREF_BANNER);
	}

	/* Text editable. */
	style = buf->text_editable;
	if(style != NULL)
	{
	    cstrptr = VMACFGItemListGetValueS(
		option, VMA_CFG_PARM_FONT_NAME_EDITABLE
	    );
	    load_font_to_style(style, cstrptr);
	}

	/* Text terminal. */
	style = buf->text_terminal;
	if(style != NULL)
	{
	    cstrptr = VMACFGItemListGetValueS(
		option, VMA_CFG_PARM_FONT_NAME_TERMINAL
	    );
	    load_font_to_style(style, cstrptr);
	}

	/* Text fine print. */
	style = buf->text_fine_print;
	{
	    cstrptr = VMA_GTK_FONT_NAME_TEXT_LICENSE;
	    load_font_to_style(style, cstrptr);
	}

	/* Text heading1. */
	style = buf->text_heading1;
	{
	    cstrptr = VMA_GTK_FONT_NAME_TEXT_HEADING1;
	    load_font_to_style(style, cstrptr);
	}
	/* Text heading2. */
	style = buf->text_heading2;
	{
	    cstrptr = VMA_GTK_FONT_NAME_TEXT_HEADING2;
	    load_font_to_style(style, cstrptr);
	}
	/* Text heading3. */
	style = buf->text_heading3;
	{
	    cstrptr = VMA_GTK_FONT_NAME_TEXT_HEADING3;
	    load_font_to_style(style, cstrptr);
	}
	/* Text heading4. */
	style = buf->text_heading4;
	{
	    cstrptr = VMA_GTK_FONT_NAME_TEXT_HEADING4;
	    load_font_to_style(style, cstrptr);
	}
	/* Text heading5. */
	style = buf->text_heading5;
	{
	    cstrptr = VMA_GTK_FONT_NAME_TEXT_HEADING5;
	    load_font_to_style(style, cstrptr);
	}
	/* Text heading6. */
	style = buf->text_heading6;
	{
	    cstrptr = VMA_GTK_FONT_NAME_TEXT_HEADING6;
	    load_font_to_style(style, cstrptr);
	}

}


/*
 *	Destroys a GTK+ styles list.
 */
void VMAStylesListDestroy(vma_styles_list_struct *buf)
{
	GtkStyle **style;


	if(buf == NULL)
	    return;

#define DO_DETACH_STYLE		\
{ \
 if((*style) != NULL) \
 { \
  GtkStyle *ts = *style; \
  (*style) = NULL; \
  gtk_style_detach(ts); \
 } \
}

	/* The standard style came from the GTK+ internal (do not detach
	 * it).
	 */
	buf->standard = NULL;

	/* Begin detaching GTK+ styles. */
	style = &buf->pref_banner;
	DO_DETACH_STYLE
	style = &buf->text_editable;
	DO_DETACH_STYLE
	style = &buf->text_terminal;
	DO_DETACH_STYLE
	style = &buf->text_fine_print;
	DO_DETACH_STYLE
	style = &buf->text_heading1;
	DO_DETACH_STYLE
	style = &buf->text_heading2;
	DO_DETACH_STYLE
	style = &buf->text_heading3;
	DO_DETACH_STYLE
	style = &buf->text_heading4;
	DO_DETACH_STYLE
	style = &buf->text_heading5;
	DO_DETACH_STYLE
	style = &buf->text_heading6;
	DO_DETACH_STYLE

#undef DO_DETACH_STYLE
}
