/*
   NCURSES interface
   --------------------------------------------------------------------
   VCHE - Virtual Console Hex Editor

   Copyright (C) 1998, 1999 Diego Javier Grigna <diego@grigna.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "common.h"

/*
 * Creates a window at 'x', 'y' with 'xlen' width and 'ylen' height
 * and draws a border.
 */
WINDOW *nc_box( int x, int y, int xlen, int ylen, chtype attr)
{
 WINDOW *w;

 w = newwin( ylen, xlen, y, x);

 wbkgd( w, ' ' | attr);
 box( w, 0, 0);

 wrefresh( w);
 return w;
}

void nc_color_wputs( WINDOW *w, int x, int y, int size, char *str, chtype attr)
{
 wattrset( w, attr);
 mvwaddnstr( w, y, x, str, size);
 wrefresh( w);
}

void nc_color_puts( int x, int y, int size, char *str, chtype attr)
{
 attrset( attr);
 mvaddnstr( y, x, str, size);
 refresh();
}

void nc_color_wputc( WINDOW *w, int x, int y, chtype byte, chtype attr)
{
 wattrset( w, attr);
 mvwaddch( w, y, x, byte);
 wrefresh( w);
}

void nc_color_putc( int x, int y, chtype byte, chtype attr)
{
 attrset( attr);
 mvaddch( y, x, byte);
 refresh();
}

/*
 * Write a string to the output buffer
 */
void nc_color_buffer_puts( int x, int y, int size, char *str, int what_attr)
{
 int pos = 2 *( y * COLS + x);
 int i   = 0;

 while( i < size * 2) {
      obuffer[ pos + i++] = *str++;
      obuffer[ pos + i++] = what_attr;
 }
}

/*
 * Draws a dialog box.
 */
WINDOW *nc_dialog_box( char *titlestr, char *valuestr, int size, chtype tattr, chtype vattr)
{
 WINDOW *w;
 char buf[ 2048];
 int titlelen, len;
 int x, y;

 titlelen = strlen( titlestr);
 len = titlelen > size ? titlelen : size;

 y = LINES / 2;

 sprintf( buf, "%-250.250s", valuestr);

 x = ( COLS - len - 4) / 2;
 w = nc_box( x, y - 2, len + 4, 4, tattr);

 nc_color_wputs( w, 2, 1, titlelen, titlestr, tattr);
 nc_color_wputs( w, 2, 2, size, buf, vattr);

 return w;
}

/*
 * Fill the screen with attribute 'attr'
 */
void nc_fill_screen( chtype attr)
{
 bkgd( ' ' | attr);
 refresh();
}

/*
 * Shows an error (or attention) message box and wait for a key.
 */
void nc_show_alert_box( char *str, int addy)
{
 WINDOW *w;
 char *buf;
 int len;
 int x, y;

 len = strlen( str);
 buf = ( char *) allocate_mem( len + 64);

 strcpy( buf, str);

 if( len < 16) len = 16;
 if( len > COLS - 4) {
     len = COLS - 4;
     buf[ COLS - 5] = ( char) ACS_RARROW;
 }

 y = LINES / 2;
 y+= addy;

 x = ( COLS - len - 4) / 2;

 term_beep();

 w = nc_box(  x, y - 2, len + 4, 4, VC_COLOR_ATTENTION);

 nc_color_wputs( w, 2, 1, len, buf               , VC_COLOR_ATTENTION);

 x = (len - 12) / 2;
 nc_color_wputs( w, x, 2, 16 , "Press any key...", VC_COLOR_ATTENTION);

 wgetch( w);
 delwin( w);

 refresh();
 free( buf);
}

/*
 * Do a box, fill it with '\n' separated lines of text and wait for a key
 */
void nc_show_screentext( char *text, char *title)
{
 WINDOW *w;
 char *buffer;
 char *p = text;
 int y = 2, i = 0;

 buffer = ( char *) allocate_mem( COLS * 2);

 w = nc_box( 0, 0, COLS, LINES, VC_COLOR_BACKCYAN);

 nc_do_title( w, title);

 while( *p && y < (LINES - 3)) {

        /* 
         * I love C language :) 
         * Print the line if we reach an '\n' or the end of the screen
         */

        if( ((*p == '\n') ? *p++: 0) || i == COLS - 4) {
            nc_color_wputs( w, 2, y++, i, buffer, VC_COLOR_BACKCYAN);
            i = 0;
            continue;
        }
        buffer[ i++] = *p++;
 }

 nc_color_wputs( w, ( COLS - 18) / 2, LINES - 2, 18, " Press any key... ", VC_COLOR_BACKGRAY);

 wgetch( w);

 delwin( w);
 free( buffer);
 refresh();
}

/*
 * Write a title
 */
void nc_do_title( WINDOW *w, char *title)
{
 int len;
 int x;

 len = strlen( title);
 x = ( COLS - len) / 2;

 nc_color_wputc( w, x -   1,   0, ACS_RTEE   , VC_COLOR_BACKCYAN);
 nc_color_wputc( w, x + len,   0, ACS_LTEE   , VC_COLOR_BACKCYAN);
 nc_color_wputs( w, x      ,   0, len, title , VC_COLOR_BACKGRAY);
}

/*
 * Write a centered bottom line
 */
void nc_put_bottom_line( WINDOW *w, char *text)
{
 int len;
 int x;
 int y;

 len = strlen( text);
 x = ( COLS - len) / 2;
 y = LINES - 1;

 nc_color_wputs( w, 0, y, COLS, space_buffer, VC_COLOR_BACKGRAY);
 nc_color_wputs( w, x, y, len, text, VC_COLOR_BACKGRAY);
}

/*
 * Flush the output buffer to the screen
 * (It's optimized for speed using ncurses)
 */
void nc_flush( void)
{
 attr_t fattr = 0;
 short fcolor = 0;
 char *buffer;
 int cfbuffer;
 int i, j;
 int obuffer_size = COLS * 2 * ( LINES - 2);
 int buffer_size  = COLS *     ( LINES - 2);

 i = 0;

 buffer = ( char *) allocate_mem( buffer_size);

 term_goto( 0, 1);

 cfbuffer = obuffer[ 1];
 translate_buffer_colors( cfbuffer, &fattr, &fcolor);

 while( i < obuffer_size) {

        j = 0;

        attrset( COLOR_PAIR( fcolor) | fattr);

        while( obuffer[ i + 1] == cfbuffer) {
            buffer[ j] = obuffer[ i];
            i += 2;
            j++;
        }

        addnstr( buffer, j);

        cfbuffer = obuffer[ i + 1];
        translate_buffer_colors( cfbuffer, &fattr, &fcolor);
 }

 refresh();
 free( buffer);
}

/*
 * Shows a question message, and let navigate options.
 */
int nc_ask( char *msg, char *q1, char *q2, char *q3, int posit)
{
 WINDOW *w;
 int lenmsg;
 int lenqs;
 int lq1,lq2,lq3;
 int len;
 int xq1, xq2, xq3;
 int x, y;
 int pos = posit;
 int c;

 term_hide_cursor();

 lq1 = strlen( q1);
 lq2 = strlen( q2);
 lq3 = strlen( q3);

 lenmsg = strlen( msg);
 lenqs  = lq1 + lq2 + lq3;

 len = lenmsg > lenqs ? lenmsg : lenqs;
 y = LINES / 2;

 x = ( COLS - len - 4) / 2;
 w = nc_box(  x, y - 2, len + 4, 4, VC_COLOR_ATTENTION);

 wbkgdset( w, VC_COLOR_RED);

 nc_color_wputs( w, 2, 1, lenmsg, msg, VC_COLOR_ATTENTION);

 xq1 = ( len + 4 - lenqs) / 2;
 xq2 = xq1 + lq1 + 1;
 xq3 = xq2 + lq2 + 1;

 while( 1) {
        if( pos > 2) pos = 0;
        if( pos < 0) pos = 2;

        switch( pos) {
                case 0: nc_color_wputs( w, xq1, 2, lq1, q1, VC_COLOR_BACKGRAY  );
                        nc_color_wputs( w, xq2, 2, lq2, q2, VC_COLOR_ATTENTION );
                        nc_color_wputs( w, xq3, 2, lq3, q3, VC_COLOR_ATTENTION );
                        break;
                case 1: nc_color_wputs( w, xq1, 2, lq1, q1, VC_COLOR_ATTENTION );
                        nc_color_wputs( w, xq2, 2, lq2, q2, VC_COLOR_BACKGRAY  );
                        nc_color_wputs( w, xq3, 2, lq3, q3, VC_COLOR_ATTENTION );
                        break;
                case 2: nc_color_wputs( w, xq1, 2, lq1, q1, VC_COLOR_ATTENTION );
                        nc_color_wputs( w, xq2, 2, lq2, q2, VC_COLOR_ATTENTION );
                        nc_color_wputs( w, xq3, 2, lq3, q3, VC_COLOR_BACKGRAY  );
                        break;
        }

        c = getch();
   
        switch( c) {
                case '\n'     : delwin( w);
                                term_unhide_cursor();
                                refresh();
                                return pos;
                                break;
                case KEY_RIGHT: pos++;
                                break;
                case KEY_LEFT : pos--;
                                break;
                case KEY_HOME : pos = 0;
                                break;
                case KEY_END  : pos = 2;
                                break;
        }
 }

}

/*
 * Shows a message without waiting for a key
 */
WINDOW *nc_show_message( char *msg, int addy)
{
 WINDOW *w;
 int len;
 int x, y;

 len = strlen( msg);

 y = ( LINES / 2) - 1;
 y += addy;

 x = ( COLS - len - 4) / 2;
 w = nc_box(  x, y - 1, len + 4, 3, VC_COLOR_ATTENTION);

 nc_color_wputs( w, 2, 1, len, msg, VC_COLOR_ATTENTION);

 return w;
}

void nc_fill_obuffer( void)
{
 int i = 0;
 int obuffer_size = COLS * 2 * ( LINES - 2);

 while( i < obuffer_size) {
        obuffer[ i++] = ' ';
        obuffer[ i++] = 0;
 }
}

