/* valaintegerliteral.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valaintegerliteral.h"
#include <vala/valasourcereference.h>
#include <vala/valaintegerliteral.h>
#include <vala/valacodevisitor.h>

struct _ValaIntegerLiteralPrivate {
	char* _value;
};
#define VALA_INTEGER_LITERAL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTEGER_LITERAL, ValaIntegerLiteralPrivate))
enum  {
	VALA_INTEGER_LITERAL_DUMMY_PROPERTY,
	VALA_INTEGER_LITERAL_VALUE
};
static void vala_integer_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_integer_literal_real_to_string (ValaCodeNode* base);
static gpointer vala_integer_literal_parent_class = NULL;
static void vala_integer_literal_dispose (GObject * obj);


/**
 * Creates a new integer literal.
 *
 * @param i      literal value
 * @param source reference to source code
 * @return       newly created integer literal
 */
ValaIntegerLiteral* vala_integer_literal_new (const char* i, ValaSourceReference* source)
{
	ValaIntegerLiteral * self;
	g_return_val_if_fail (i != NULL, NULL);
	g_return_val_if_fail (source == NULL || VALA_IS_SOURCE_REFERENCE (source), NULL);
	self = g_object_newv (VALA_TYPE_INTEGER_LITERAL, 0, NULL);
	vala_integer_literal_set_value (self, i);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source);
	return self;
}


static void vala_integer_literal_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaIntegerLiteral * self;
	self = ((ValaIntegerLiteral*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_integer_literal (visitor, self);
}


static char* vala_integer_literal_real_to_string (ValaCodeNode* base)
{
	ValaIntegerLiteral * self;
	self = ((ValaIntegerLiteral*) base);
	return g_strdup (vala_integer_literal_get_value (self));
}


/**
 * Returns the type name of the value this literal represents.
 *
 * @return the name of literal type
 */
char* vala_integer_literal_get_type_name (ValaIntegerLiteral* self)
{
	char* number;
	gint l;
	gboolean u;
	gint64 n;
	g_return_val_if_fail (VALA_IS_INTEGER_LITERAL (self), NULL);
	number = g_strdup (vala_integer_literal_get_value (self));
	l = 0;
	while (g_str_has_suffix (number, "L")) {
		char* __temp1;
		l++;
		__temp1 = NULL;
		number = (__temp1 = g_strndup (number, strlen (number) - 1), (number = (g_free (number), NULL)), __temp1);
	}
	u = FALSE;
	if (g_str_has_suffix (number, "U")) {
		char* __temp2;
		u = TRUE;
		__temp2 = NULL;
		number = (__temp2 = g_strndup (number, strlen (number) - 1), (number = (g_free (number), NULL)), __temp2);
	}
	n = g_ascii_strtoll (number, NULL, 0);
	if (!u && n > 0x7fffffff) {
		/* value doesn't fit into signed 32-bit*/
		l = 2;
	} else {
		if (u && n > 0xffffffff) {
			/* value doesn't fit into unsigned 32-bit*/
			l = 2;
		}
	}
	if (l == 0) {
		if (u) {
			char* __temp3;
			__temp3 = NULL;
			return (__temp3 = g_strdup ("uint"), (number = (g_free (number), NULL)), __temp3);
		} else {
			char* __temp4;
			__temp4 = NULL;
			return (__temp4 = g_strdup ("int"), (number = (g_free (number), NULL)), __temp4);
		}
	} else {
		if (l == 1) {
			if (u) {
				char* __temp5;
				__temp5 = NULL;
				return (__temp5 = g_strdup ("ulong"), (number = (g_free (number), NULL)), __temp5);
			} else {
				char* __temp6;
				__temp6 = NULL;
				return (__temp6 = g_strdup ("long"), (number = (g_free (number), NULL)), __temp6);
			}
		} else {
			if (u) {
				char* __temp7;
				__temp7 = NULL;
				return (__temp7 = g_strdup ("uint64"), (number = (g_free (number), NULL)), __temp7);
			} else {
				char* __temp8;
				__temp8 = NULL;
				return (__temp8 = g_strdup ("int64"), (number = (g_free (number), NULL)), __temp8);
			}
		}
	}
	(number = (g_free (number), NULL));
}


char* vala_integer_literal_get_value (ValaIntegerLiteral* self)
{
	g_return_val_if_fail (VALA_IS_INTEGER_LITERAL (self), NULL);
	return self->priv->_value;
}


void vala_integer_literal_set_value (ValaIntegerLiteral* self, const char* value)
{
	char* __temp10;
	g_return_if_fail (VALA_IS_INTEGER_LITERAL (self));
	__temp10 = NULL;
	self->priv->_value = (__temp10 = g_strdup (value), (self->priv->_value = (g_free (self->priv->_value), NULL)), __temp10);
}


static void vala_integer_literal_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaIntegerLiteral * self;
	self = VALA_INTEGER_LITERAL (object);
	switch (property_id) {
		case VALA_INTEGER_LITERAL_VALUE:
		g_value_set_string (value, vala_integer_literal_get_value (self));
		break;
	}
}


static void vala_integer_literal_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaIntegerLiteral * self;
	self = VALA_INTEGER_LITERAL (object);
	switch (property_id) {
		case VALA_INTEGER_LITERAL_VALUE:
		vala_integer_literal_set_value (self, g_value_get_string (value));
		break;
	}
}


static void vala_integer_literal_class_init (ValaIntegerLiteralClass * klass)
{
	vala_integer_literal_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaIntegerLiteralPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_integer_literal_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_integer_literal_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_integer_literal_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_integer_literal_real_accept;
	VALA_CODE_NODE_CLASS (klass)->to_string = vala_integer_literal_real_to_string;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_INTEGER_LITERAL_VALUE, g_param_spec_string ("value", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_integer_literal_init (ValaIntegerLiteral * self)
{
	self->priv = VALA_INTEGER_LITERAL_GET_PRIVATE (self);
}


static void vala_integer_literal_dispose (GObject * obj)
{
	ValaIntegerLiteral * self;
	ValaIntegerLiteralClass * klass;
	GObjectClass * parent_class;
	self = VALA_INTEGER_LITERAL (obj);
	(self->priv->_value = (g_free (self->priv->_value), NULL));
	klass = VALA_INTEGER_LITERAL_CLASS (g_type_class_peek (VALA_TYPE_INTEGER_LITERAL));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_integer_literal_get_type (void)
{
	static GType vala_integer_literal_type_id = 0;
	if (G_UNLIKELY (vala_integer_literal_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaIntegerLiteralClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_integer_literal_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaIntegerLiteral), 0, (GInstanceInitFunc) vala_integer_literal_init };
		vala_integer_literal_type_id = g_type_register_static (VALA_TYPE_LITERAL, "ValaIntegerLiteral", &g_define_type_info, 0);
	}
	return vala_integer_literal_type_id;
}




