/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include <gtk/gtk.h>
#ifdef HAVE_LIBGTKGLEXT_X11_1_0
#include <gtk/gtkgl.h>
#endif
#include <unistd.h> /* For the access markers R_OK, W_OK ... */

#include "opengl.h"
#include "visu_gtk.h"
#include "visu_object.h"
#include "visu_basic.h"
#include "visu_commandLine.h"
#include "visu_configFile.h"
#include "visu_rendering.h"
#include "visu_extension.h"
#include "renderingMethods/renderingAtomic.h"
#include "renderingMethods/renderingSpin.h"
#include "coreTools/toolShade.h"
#include "coreTools/toolConfigFile.h"
#include "extraFunctions/plane.h"
#include "extraFunctions/dataFile.h"
#include "panelModules/panelSurfaces.h"
#include "gtk_renderingWindowWidget.h"
#include "extraGtkFunctions/gtk_colorComboBoxWidget.h"
#include "OSOpenGL/visu_openGL.h"
#include "extensions/fogAndBGColor.h"
#include "extensions/box.h"
#include "extensions/scale.h"
#include "renderingBackend/visu_windowInterface.h"

/* Parameters. */
#define FLAG_PARAMETER_PREVIEW "main_usePreview"
#define DESC_PARAMETER_PREVIEW "Automatically compute preview in filechooser ; boolean"
static gboolean usePreview;

/* Static variables. */
static gint gtkFileChooserWidth = -1, gtkFileChooserHeight = -1;

/* This hashtable associate a #RenderingMethod with a Gtk dialog to
   choose files. */
static GHashTable *visuGtkLoadMethods;
/* Store the last open directory. It is initialised
   to current working directory. */
static gchar *visuGtkLastDir;

static GtkWindow *visuGtkPanel;
static GtkWindow *visuGtkRender;

/* Local routines. */
static void exportParameters(GString *data, VisuData *dataObj);
static gboolean readUsePreview(gchar **lines, int nbLines,
			       int position, VisuData *dataObj, GError **error);
static void onPreviewToggled(GtkToggleButton *button, gpointer data);


/* Draw the alert window with the specified message. */
void visuGtkRaise_warning(gchar *action, gchar *message, GtkWindow *window)
{
  GtkWidget *alert;
  gchar *str;
  
  if (!window)
    window = visuGtkRender;

  DBG_fprintf(stderr, "Visu Gtk: raise the error dialog (parent %p).\n",
	      (gpointer)window);

#if GTK_MINOR_VERSION > 5
  str = action;
#else
  str = message;
#endif
  alert = gtk_message_dialog_new(GTK_WINDOW(window),
				 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				 GTK_MESSAGE_WARNING, GTK_BUTTONS_OK, str);
  gtk_window_set_title(GTK_WINDOW(alert), _("V_Sim error message"));
#if GTK_MINOR_VERSION > 5
  gtk_widget_set_name(alert, "error");
#else
  gtk_widget_set_name(alert, action);
#endif
#if GTK_MINOR_VERSION > 5
  gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(alert), message);
#endif
  gtk_widget_show_all(alert);

  /* block in a loop waiting for reply. */
  gtk_dialog_run(GTK_DIALOG(alert));
  gtk_widget_destroy(alert);
}
/* Draw the alert window with the specified message. */
void visuGtkRaise_warningLong(gchar *action, gchar *message, GtkWindow *window)
{
  GtkWidget *alert;
  GtkWidget *text, *scroll;
  GtkTextBuffer *buf;

  if (!window)
    window = visuGtkRender;

  alert = gtk_message_dialog_new(GTK_WINDOW(window),
				 GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				 GTK_MESSAGE_WARNING,
				 GTK_BUTTONS_OK,
				 action);
  gtk_window_set_resizable(GTK_WINDOW(alert), TRUE);
  gtk_widget_set_name(alert, "error");
#if GTK_MINOR_VERSION > 5
  gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(alert),
					   _("Output errors:"));
#endif
  scroll = gtk_scrolled_window_new((GtkAdjustment*)0, (GtkAdjustment*)0);
  gtk_widget_set_size_request(scroll, 300, 200);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scroll),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scroll),
				      GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(alert)->vbox), scroll, TRUE, TRUE, 2);
  text = gtk_text_view_new();
  gtk_text_view_set_editable(GTK_TEXT_VIEW(text), FALSE);
  gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(text), FALSE);
  gtk_container_add(GTK_CONTAINER(scroll), text);
  buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW(text));
  gtk_text_buffer_set_text(GTK_TEXT_BUFFER(buf), message, -1);
  gtk_widget_show_all(alert);

  /* block in a loop waiting for reply. */
  gtk_dialog_run (GTK_DIALOG (alert));
  gtk_widget_destroy(alert);
}
/* This function must be called in a blocking loop to update
   different things like the gtk for example. */
void visuGtkWait()
{
  while(gtk_events_pending())
    gtk_main_iteration();
}

struct _VisuGtkSetFilesFunc
{
  VisuGtkSetFilesFunc load;
};
void visuGtkSet_renderingSpecificMethods(VisuRendering *method,
					 VisuGtkSetFilesFunc methodLoad)
{
  struct _VisuGtkSetFilesFunc *ct;

  g_return_if_fail(method && visuGtkLoadMethods);

  DBG_fprintf(stderr, "Visu Gtk: set rendering specific for method '%s'.\n",
	      visuRenderingGet_name(method, FALSE));
  ct = g_malloc(sizeof(struct _VisuGtkSetFilesFunc));
  ct->load = methodLoad;
  g_hash_table_insert(visuGtkLoadMethods, method, ct);
}
VisuGtkSetFilesFunc visuGtkGet_renderingSpecificOpen(VisuRendering *method)
{
  struct _VisuGtkSetFilesFunc *ct;

  g_return_val_if_fail(method && visuGtkLoadMethods, (VisuGtkSetFilesFunc)0);
  
  DBG_fprintf(stderr, "Visu Gtk: looking for a specific load interface for rendering"
	      " method '%s'...\n", visuRenderingGet_name(method, FALSE));
  
  ct = (struct _VisuGtkSetFilesFunc *)g_hash_table_lookup(visuGtkLoadMethods, method);
  return (ct && ct->load)?ct->load:visuGtkGet_fileFromDefaultFileChooser;
}

gchar** visuGtkGet_selectedDirectory(GtkWindow *parent, gboolean multiple,
				     const gchar *dir)
{
  GtkWidget *file_selector, *hbox, *wd;
  gchar **dirnames;
  char *directory;
  GSList* names, *tmpLst;
  int i;

  /* Create the selector */
  if (!parent)
    parent = visuGtkRender;

  file_selector = gtk_file_chooser_dialog_new(_("Choose a directory"), parent,
					      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_OK,
					      NULL);
  if (gtkFileChooserWidth > 0 || gtkFileChooserHeight > 0)
    gtk_window_set_default_size(GTK_WINDOW(file_selector), gtkFileChooserWidth,
				gtkFileChooserHeight);
  if (multiple)
    {
      hbox = gtk_hbox_new(FALSE, 0);
      gtk_file_chooser_set_extra_widget(GTK_FILE_CHOOSER(file_selector), hbox);
      wd = gtk_image_new_from_stock(GTK_STOCK_HELP, GTK_ICON_SIZE_MENU);
      gtk_box_pack_start(GTK_BOX(hbox), wd, FALSE, FALSE, 0);
      wd = gtk_label_new("");
      gtk_box_pack_start(GTK_BOX(hbox), wd, TRUE, TRUE, 5);
      gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
      gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
      gtk_label_set_markup(GTK_LABEL(wd), _("<span size=\"smaller\">Choose several"
					    " directories using the"
					    " <span font_desc=\"courier\">"
					    "Control</span> key.</span>"));
      gtk_widget_show_all(hbox);
    }
  if (!dir)
    directory = visuGtkGet_lastOpenDirectory();
  else
    directory = (char*)dir;
  if (directory)
    {
      DBG_fprintf(stderr, "Visu Gtk: open a directory chooser, set on '%s'.\n",
		  directory);
      gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(file_selector),
					  directory);
    }
  gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(file_selector), multiple);

  gtk_widget_set_name(file_selector, "filesel");
  gtk_window_set_position(GTK_WINDOW(file_selector), GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_modal(GTK_WINDOW (file_selector), TRUE);

  if (gtk_dialog_run (GTK_DIALOG (file_selector)) == GTK_RESPONSE_OK)
    {
      names = gtk_file_chooser_get_filenames(GTK_FILE_CHOOSER(file_selector));
      dirnames = g_malloc(sizeof(gchar*) * (g_slist_length(names) + 1));
      tmpLst = names;
      i = 0;
      while(tmpLst)
	{
	  dirnames[i] = (gchar*)tmpLst->data;
	  i += 1;
	  tmpLst = g_slist_next(tmpLst);
	}
      dirnames[i] = (gchar*)0;
      g_slist_free(names);
    }
  else
    dirnames = (gchar**)0;

  gtk_window_get_size(GTK_WINDOW(file_selector), &gtkFileChooserWidth,
		      &gtkFileChooserHeight);

  gtk_widget_destroy (file_selector);

  return dirnames;
}

static void free_image(guchar *image, gpointer data _U_)
{
  DBG_fprintf(stderr, "Visu Gtk: free the preview image data.\n");
  g_free(image);
}
void visuGtkPreview_update(VisuGtkPreview *preview, const char *filenames[])
{
  gchar *text, *comment;
  VisuData *data;
  gboolean valid;
  GError *error;
  DumpImage *dumpData;
  guchar* image;
  GdkPixbuf *pixbuf;
  visuRenderingLoadFunc loadMethod;
  gint width, height, nRows, nFiles, nb;
  VisuDataIter iter;
  GtkWidget *wd;
  RenderingWindow *currentWindow;
  OpenGLView *view;

  DBG_fprintf(stderr, "Visu Gtk: update preview with given filenames.\n");

  if (preview->table)
    {
      gtk_widget_destroy(preview->table);
      preview->table = (GtkWidget*)0;
    }

  loadMethod = visuRenderingClassGet_currentLoadFunc();
  g_return_if_fail(loadMethod);

  nb = visuRenderingGet_nbFileType(visuRenderingClassGet_current());
  for (nFiles = 0; filenames[nFiles]; nFiles++);

  if (nFiles < nb)
    gtk_image_set_from_pixbuf(GTK_IMAGE(preview->image), (GdkPixbuf*)0);
  else
    {
      if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(preview->check)))
	{
	  gtk_image_set_from_pixbuf(GTK_IMAGE(preview->image), (GdkPixbuf*)0);
	  return;
	}

      data = visuDataNew();
      for (nFiles = 0; filenames[nFiles]; nFiles++)
	visuDataAdd_file(data, (char*)filenames[nFiles], nFiles, (FileFormat*)0);

      /* We save the current rendering context. */
      currentWindow = visuRenderingWindowGet_current();

      /* We change the context since loading a new data will
	 generate some changes on the OpenGL rendering. */
      width = 150;
      height = 150;
      dumpData = visuOpenGLNew_pixmapContext((guint)width, (guint)height);
      if (!dumpData)
	{
	  /* We free the VisuData. */
	  g_object_unref(data);

	  gtk_image_set_from_stock(GTK_IMAGE(preview->image),
				   GTK_STOCK_DIALOG_ERROR,
				   GTK_ICON_SIZE_DIALOG);
	  wd = gtk_label_new(_("Internal error,\nno preview available"));
	  gtk_box_pack_start(GTK_BOX(preview->vbox), wd,
			     FALSE, FALSE, 0);
	  gtk_widget_show_all(wd);
	  preview->table = wd;

          /* Set the rendering window current for OpenGL. */
          renderingWindowSet_current(currentWindow, TRUE);

	  return;
	}

      error = (GError*)0;
      valid = loadMethod(data, (FileFormat*)0, 0, &error);
      if (!valid)
	{
	  gtk_image_set_from_stock(GTK_IMAGE(preview->image),
				   GTK_STOCK_DIALOG_QUESTION,
				   GTK_ICON_SIZE_DIALOG);
	  wd = gtk_label_new(_("Not a V_Sim file"));
	  gtk_box_pack_start(GTK_BOX(preview->vbox), wd,
			     FALSE, FALSE, 0);
	  gtk_widget_show_all(wd);
	  preview->table = wd;
	}
      else
	{
	  if (error)
	    {
	      gtk_image_set_from_stock(GTK_IMAGE(preview->image),
				       GTK_STOCK_DIALOG_ERROR,
				       GTK_ICON_SIZE_DIALOG);
	      wd = gtk_label_new(_("This file has errors"));
	      gtk_box_pack_start(GTK_BOX(preview->vbox), wd,
				 FALSE, FALSE, 0);
	      gtk_widget_show_all(wd);
	      preview->table = wd;
	      g_error_free(error);
	    }
	  else
	    {
	      view = visuDataGet_openGLView(data);
	      /* We set the glViewport of this new context. */
	      glViewport(0, 0, width, height);
	      /* We call the given draw method. */
	      openGLInit_context();
	      visuDataSet_sizeOfView(data, (guint)width, (guint)height);
	      openGLViewCompute_matrixAndView(view);
	      glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
	      OpenGLExtensionRebuild_list(data, EXT_FOG_AND_BG_ID);
	      OpenGLExtensionRebuild_list(data, EXT_VISU_DATA_ID);
	      OpenGLExtensionRebuild_list(data, EXT_BOX_ID);
	      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	      OpenGLExtensionCall_list(EXT_FOG_AND_BG_ID, FALSE);
	      OpenGLExtensionCall_list(EXT_VISU_DATA_ID, FALSE);
	      OpenGLExtensionCall_list(EXT_BOX_ID, FALSE);
	      /* We copy the pixmap into generic data. */
	      image = visuOpenGLGet_pixmapData((guint)width,
					       (guint)height, FALSE);
	      pixbuf =
		gdk_pixbuf_new_from_data((guchar*)image, GDK_COLORSPACE_RGB,
					 FALSE, 8, width, height, 3 * width,
					 free_image, (gpointer)0);
	      gtk_image_set_from_pixbuf(GTK_IMAGE(preview->image), pixbuf);
	      gdk_pixbuf_unref(pixbuf);
	      /* We reset the flag of material. */
	      comment = visuDataGet_fileCommentary(data, 0);
	      visuDataIter_new(data, &iter);
	      nRows = (comment && comment[0])?
		iter.nElements + 3:iter.nElements + 1;
	      preview->table = gtk_table_new(iter.nElements + 1, 2, FALSE);
	      wd = gtk_label_new(_("<i>Box composition:</i>"));
	      gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
	      gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
	      gtk_table_attach(GTK_TABLE(preview->table), wd, 0, 2, 0, 1,
			       GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 5);
	      for (visuDataIter_start(data, &iter); iter.element;
		   visuDataIter_nextElement(data, &iter))
		{
		  iter.element->materialIsUpToDate = FALSE;
		  wd = gtk_label_new("");
		  text = g_markup_printf_escaped
		    (_("<span size=\"small\"><b>%s:</b></span>"),
		     iter.element->name);
		  gtk_label_set_markup(GTK_LABEL(wd), text);
		  gtk_misc_set_alignment(GTK_MISC(wd), 1., 0.5);
		  g_free(text);
		  gtk_table_attach(GTK_TABLE(preview->table), wd, 0, 1,
				   iter.iElement + 1, iter.iElement + 2,
				   GTK_FILL, GTK_SHRINK, 2, 0);
		  wd = gtk_label_new("");
		  if (iter.nStoredNodes[iter.iElement] > 1)
		    text = g_markup_printf_escaped
		      (_("<span size=\"small\">%d nodes</span>"),
		       iter.nStoredNodes[iter.iElement]);
		  else
		    text = g_markup_printf_escaped
		      (_("<span size=\"small\">%d node</span>"),
		       iter.nStoredNodes[iter.iElement]);
		  gtk_label_set_markup(GTK_LABEL(wd), text);
		  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
		  g_free(text);
		  gtk_table_attach(GTK_TABLE(preview->table), wd, 1, 2,
				   iter.iElement + 1, iter.iElement + 2,
				   GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 0);
		}
	      if (comment && comment[0])
		{
		  wd = gtk_label_new(_("<i>Description:</i>"));
		  gtk_label_set_use_markup(GTK_LABEL(wd), TRUE);
		  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
		  gtk_table_attach(GTK_TABLE(preview->table), wd, 0, 2,
				   iter.nElements + 2, iter.nElements + 3,
				   GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 5);
		  wd = gtk_label_new("");
		  text = g_markup_printf_escaped("<span size=\"small\">%s</span>",
						 comment);
		  gtk_label_set_markup(GTK_LABEL(wd), text);
		  g_free(text);
		  gtk_misc_set_alignment(GTK_MISC(wd), 0., 0.5);
		  gtk_label_set_justify(GTK_LABEL(wd), GTK_JUSTIFY_FILL);
		  gtk_label_set_line_wrap(GTK_LABEL(wd), TRUE);
		  gtk_widget_set_size_request(wd, 150, -1);
		  gtk_table_attach(GTK_TABLE(preview->table), wd, 0, 2,
				   iter.nElements + 3, iter.nElements + 4,
				   GTK_FILL | GTK_EXPAND, GTK_SHRINK, 2, 0);
		}
	      gtk_box_pack_start(GTK_BOX(preview->vbox), preview->table,
				 FALSE, FALSE, 0);
	      gtk_widget_show_all(preview->table);
	    }
	}
      /* Set the rendering window current for OpenGL. */
      renderingWindowSet_current(currentWindow, TRUE);

      /* We free the pixmap context. */
      visuOpenGLFree_pixmapContext(dumpData);

      /* We free the VisuData. */
      g_object_unref(data);
    }
  DBG_fprintf(stderr, "Visu Gtk: end of preview creation.\n");
}
static void onPreviewToggled(GtkToggleButton *button, gpointer data _U_)
{
  usePreview = gtk_toggle_button_get_active(button);
}

VisuGtkPreview* visuGtkPreview_add(VisuGtkPreview *preview, GtkFileChooser *chooser)
{
  GtkWidget *wd, *frame;

  g_return_val_if_fail(preview, (VisuGtkPreview*)0);

  preview->vbox = gtk_vbox_new(FALSE, 0);
  preview->check = gtk_check_button_new_with_mnemonic(_("_Preview:"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(preview->check), usePreview);
  g_signal_connect(G_OBJECT(preview->check), "toggled",
		   G_CALLBACK(onPreviewToggled), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(preview->vbox), preview->check, FALSE, FALSE, 5);
  wd = gtk_alignment_new(0.5, 0.5, 0., 0.);
  gtk_box_pack_start(GTK_BOX(preview->vbox), wd, TRUE, TRUE, 0);
  frame = gtk_frame_new(NULL);
  gtk_widget_set_size_request(frame, 150, 150);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);
  gtk_container_add(GTK_CONTAINER(wd), frame);
  preview->image = gtk_image_new();
  gtk_container_add(GTK_CONTAINER(frame), preview->image);
  preview->table = NULL;
  gtk_widget_show_all(preview->vbox);
  
  gtk_file_chooser_set_preview_widget(chooser, preview->vbox);
  gtk_file_chooser_set_use_preview_label(chooser, FALSE);
  gtk_file_chooser_set_preview_widget_active(chooser, TRUE);

  return preview;
}

static void update_preview(GtkFileChooser *chooser, VisuGtkPreview *preview)
{
  const char *filenames[2];

  filenames[0] = gtk_file_chooser_get_preview_filename(chooser);
  filenames[1] = (char*)0;

  /* We test if the selected filename is a directory or not. */
  if (filenames[0] && !g_file_test(filenames[0], G_FILE_TEST_IS_DIR))
    visuGtkPreview_update(preview, filenames);
  if (filenames[0])
    g_free((char*)filenames[0]);
}

gboolean visuGtkGet_fileFromDefaultFileChooser(VisuData *data, GtkWindow *parent)
{
  GtkWidget *fileSelection;
  GList *filters, *tmpLst;
  VisuRendering *method;
  gchar* directory, *filename;
  FileFormat *selectedFormat;
  GtkFileFilter *filterDefault;
  gboolean res;
  VisuGtkPreview preview;

  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Visu Gtk: default filechooser for file opening.\n");

  method = visuRenderingClassGet_current();
  g_return_val_if_fail(method, FALSE);

  fileSelection = gtk_file_chooser_dialog_new(_("Load session"),
					      GTK_WINDOW(parent),
					      GTK_FILE_CHOOSER_ACTION_OPEN,
					      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					      GTK_STOCK_OPEN, GTK_RESPONSE_OK,
					      NULL);
  if (gtkFileChooserWidth > 0 || gtkFileChooserHeight > 0)
    gtk_window_set_default_size(GTK_WINDOW(fileSelection), gtkFileChooserWidth,
				gtkFileChooserHeight);
  directory = visuGtkGet_lastOpenDirectory();
  if (directory)
    gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(fileSelection), directory);

  gtk_widget_set_name(fileSelection, "filesel");
  gtk_window_set_position(GTK_WINDOW(fileSelection), GTK_WIN_POS_CENTER_ON_PARENT);
  gtk_window_set_modal(GTK_WINDOW(fileSelection), TRUE);

  /* Create and add the filters. */
  filters = visuGtkCreate_fileChooserFilter
    (visuRenderingGet_fileType(visuRenderingClassGet_current(), 0), fileSelection);

  /* Create and add the preview. */
  visuGtkPreview_add(&preview, GTK_FILE_CHOOSER(fileSelection));
  g_signal_connect(GTK_FILE_CHOOSER(fileSelection), "update-preview",
		   G_CALLBACK(update_preview), &preview);

  if (gtk_dialog_run(GTK_DIALOG(fileSelection)) == GTK_RESPONSE_OK)
    {
      /* Get the selected filter. */
      selectedFormat = (FileFormat*)0;
      filterDefault = gtk_file_chooser_get_filter(GTK_FILE_CHOOSER(fileSelection));
      tmpLst = filters;
      while(tmpLst)
	{
	  if (filterDefault == ((FileFilterCustom*)tmpLst->data)->gtkFilter)
	    selectedFormat = ((FileFilterCustom*)tmpLst->data)->visuFilter;
	  tmpLst = g_list_next(tmpLst);
	}
      filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(fileSelection));
      visuDataAdd_file(data, filename, 0, selectedFormat);
      g_free(filename);
      res = TRUE;
    }
  else
    res = FALSE;

  directory =
    gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(fileSelection));
  visuGtkSet_lastOpenDirectory((char*)directory, VISU_DIR_FILE);
  g_free(directory);

  gtk_window_get_size(GTK_WINDOW(fileSelection), &gtkFileChooserWidth,
		      &gtkFileChooserHeight);

  gtk_widget_destroy(fileSelection);

  /* Free the filters list. */
  tmpLst = filters;
  while(tmpLst)
    {
      g_free(tmpLst->data);
      tmpLst = g_list_next(tmpLst);
    }
  g_list_free(filters);
  DBG_fprintf(stderr, "Visu Gtk: free load dialog OK.\n");

  return res;
}

void visuGtkCreate_main(GtkWindow **panel, GtkWindow **render)
{
  int width, height;
  GtkWidget *renderingWindow;

  /* Force the creation of the Scale class. */
  commandLineGet_XWindowGeometry(&width, &height);
  DBG_fprintf(stderr,"Visu Gtk: create a rendering window (%dx%d).\n",
	      width, height);
  renderingWindow =
    renderingWindow_newEmbedded(width, height);
  *render = GTK_WINDOW(g_object_get_data(G_OBJECT(renderingWindow),
					 RENDERING_WINDOW_ID));
  g_signal_connect(G_OBJECT(*render), "delete-event",
		   G_CALLBACK(gtk_main_quit), (gpointer)0);
  g_signal_connect(G_OBJECT(*render), "destroy-event",
		   G_CALLBACK(gtk_main_quit), (gpointer)0);
  gtk_widget_show(GTK_WIDGET(*render));
  visuRenderingWindowSet_default(renderingWindow);
  *panel = (GtkWindow*)0;

  return;
}

void visuGtkMain(VisuGtkInitWidgetFunc panelFunc)
{
  VisuConfigFileEntry *entry;
  GString *message;

  g_return_if_fail(panelFunc);

  /* Retrieve the current working directory. */
  visuGtkLastDir = g_get_current_dir();

  /* Set private variables. */
  usePreview = TRUE;
  entry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
				  FLAG_PARAMETER_PREVIEW,
				  DESC_PARAMETER_PREVIEW,
				  1, readUsePreview);
  visuConfigFileSet_version(entry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParameters);
  visuGtkLoadMethods = g_hash_table_new_full(g_direct_hash, g_direct_equal,
					     NULL, g_free);
  panelFunc(&visuGtkPanel, &visuGtkRender);
  g_return_if_fail(visuGtkRender);

  /* Add the gtk tag as a known tag to allow to read such parameters. */
  message = visuBasicParse_configFiles();
  if (message)
    {
      visuGtkRaise_warningLong(_("Reading the configuration files"), message->str,
			       visuGtkRender);
      g_string_free(message, TRUE);
    }

  return;
}

gboolean visuGtkRun_commandLine(gpointer data _U_)
{
  RenderingWindow *window;
  VisuData *obj;
  VisuOptionSet *set;
  GError *error;

  window = RENDERING_WINDOW(visuRenderingWindowGet_current());
  g_return_val_if_fail(window, FALSE);

  obj = renderingWindowGet_visuData(window);
  if (!obj)
    return FALSE;

  set = g_malloc(sizeof(VisuOptionSet));
  error = (GError*)0;
  if (!visuBasicApply_commandLine(obj, set, &error))
    {
      visuGtkRaise_warning(_("Parsing command line"), error->message, (GtkWindow*)0);
      g_error_free(error);
    }
  else
    visuBasicCreate_extensions(obj, set, TRUE);

  visuData_createAllNodes(obj);

  g_idle_add(visuObjectRedraw, (gpointer)0);
  
  return FALSE;
}

struct _load_struct
{
  VisuData *data;
  guint iSet;
};
static gboolean _visuGtkLoad_file(gpointer data)
{
  VisuData *obj;
  GError *error;
  VisuData *prevData;
  gboolean changeElement, res;
  RenderingWindow *window;
  struct _load_struct *pt;
  int iSet;

  /* obj is the new object and main the panel that handle the
     loading. */
  pt = (struct _load_struct*)data;
  obj = pt->data;
  iSet = (int)pt->iSet;
  g_free(pt);
  window = visuRenderingWindowGet_current();

  prevData = renderingWindowGet_visuData(RENDERING_WINDOW(window));
  DBG_fprintf(stderr, "Visu Gtk: loading process ... %p points to"
	      " previous VisuData.\n", (gpointer)prevData);

  error = (GError*)0;
  res = visuBasicLoad_dataFromFile(obj, (FileFormat*)0, iSet, &error);
  DBG_fprintf(stderr, "Visu Gtk: basic load OK, continue with Gtk loading parts.\n");

  if (!res)
    {
      g_object_unref(obj);
      obj = (VisuData*)0;
      if (error)
	{
	  visuGtkRaise_warning(_("Loading a file"), error->message, NULL);
	  g_error_free(error);
	}
      else
	g_warning("No error message.");
    }
  else
    if (prevData && obj)
      {
	changeElement = visuData_compareElements(prevData, obj);
	visuDataSet_changeElementFlag(obj, changeElement);
      }

  DBG_fprintf(stderr, "Visu Gtk: loading process ... try to load %p.\n",
	      (gpointer)obj);
  renderingWindowSet_visuData(RENDERING_WINDOW(window), obj);

  if (!obj)
    return FALSE;

  /* We release a ref on obj, since
     renderingWindowSet_visuData has increased it. */
  g_object_unref(G_OBJECT(obj));
	 
  visuData_createAllNodes(obj);

  return FALSE;
}
void visuGtkLoad_file(VisuData *data, guint iSet)
{
  struct _load_struct *pt;

  pt = g_malloc(sizeof(struct _load_struct));
  pt->data = data;
  pt->iSet = iSet;
  g_idle_add(_visuGtkLoad_file, pt);
}


GList* visuGtkCreate_fileChooserFilter(GList *list, GtkWidget *fileChooser)
{
  GtkFileFilter *filter, *filterAll;
  GList *tmpLst, *tmpLst2;
  char *name;
  FileFilterCustom *data;
  GList *returnedFilters;

  DBG_fprintf(stderr, "Visu Gtk: creating list of filters.\n");
  returnedFilters = (GList*)0;
  filterAll = gtk_file_filter_new ();
  gtk_file_filter_set_name(filterAll, _("All supported formats"));
  tmpLst = list;
  while(tmpLst)
    {
      filter = gtk_file_filter_new ();
      name = fileFormatGet_label((FileFormat*)tmpLst->data);
      if (name)
	gtk_file_filter_set_name(filter, name);
      else
	gtk_file_filter_set_name(filter, _("No description"));
      tmpLst2 = ((FileFormat*)tmpLst->data)->fileType;
      while (tmpLst2)
	{
	  gtk_file_filter_add_pattern (filter, (char*)tmpLst2->data);
	  gtk_file_filter_add_pattern (filterAll, (char*)tmpLst2->data);
	  tmpLst2 = g_list_next(tmpLst2);
	}
      data = g_malloc(sizeof(FileFilterCustom));
      data->gtkFilter = filter;
      data->visuFilter = (FileFormat*)tmpLst->data;
      returnedFilters = g_list_append(returnedFilters, (gpointer)data);
      tmpLst = g_list_next(tmpLst);
    }
  data = g_malloc(sizeof(FileFilterCustom));
  data->gtkFilter = filterAll;
  data->visuFilter = (FileFormat*)0;
  returnedFilters = g_list_append(returnedFilters, (gpointer)data);
  filter = gtk_file_filter_new ();
  gtk_file_filter_set_name(filter, _("All files"));
  gtk_file_filter_add_pattern (filter, "*");
  data = g_malloc(sizeof(FileFilterCustom));
  data->gtkFilter = filter;
  data->visuFilter = (FileFormat*)0;
  returnedFilters = g_list_append(returnedFilters, (gpointer)data);
  
  DBG_fprintf(stderr, "Gtk Main : attach list to the given filechooser.\n");
  tmpLst = returnedFilters;
  while(tmpLst)
    {
      gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(fileChooser),
				  ((FileFilterCustom*)tmpLst->data)->gtkFilter);
      tmpLst = g_list_next(tmpLst);
    }
  gtk_file_chooser_set_filter(GTK_FILE_CHOOSER(fileChooser), filterAll);

  return returnedFilters;
}

static void exportParameters(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_PREVIEW);
  g_string_append_printf(data, "%s[gtk]: %d\n\n", FLAG_PARAMETER_PREVIEW,
			 usePreview);
}
static gboolean readUsePreview(gchar **lines, int nbLines,
			       int position, VisuData *dataObj _U_, GError **error)
{
  int res;
  int val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  res = sscanf(lines[0],"%d", &val);
  if (res != 1)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_MISSING,
			   _("Parse error at line %d,"
			     " 1 boolean value must appear"
			     " after the %s markup.\n"),
			   position, FLAG_PARAMETER_PREVIEW);
      return FALSE;
    }
  else
    usePreview = (gboolean)val;

  return TRUE;
}

char* visuGtkGet_lastOpenDirectory()
{
  DBG_fprintf(stderr, "Visu Gtk: get the last open directory : '%s'.\n",
	      visuGtkLastDir);
  return visuGtkLastDir;
}
void visuGtkSet_lastOpenDirectory(char* directory, VisuGtkDirType type)
{
  if (visuGtkLastDir)
    g_free(visuGtkLastDir);

  if (!g_path_is_absolute(directory))
    visuGtkLastDir = g_build_filename(g_get_current_dir(),
					 directory, NULL);
  else
    visuGtkLastDir = g_build_filename(directory, NULL);
  DBG_fprintf(stderr, "Visu Gtk: set the last open directory to '%s', emit signal.\n",
	      visuGtkLastDir);

  g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DIR_SIGNAL],
		0 /* details */, type, NULL);
  DBG_fprintf(stderr, "Visu Gtk: emission done (DirectoryChanged).\n");
}

GdkPixbuf* visuGtkCreate_pixbuf(const gchar *filename)
{
  gchar *pathname = NULL;
  GdkPixbuf *pixbuf;
  GError *error = NULL;

  g_return_val_if_fail(filename && filename[0], (GdkPixbuf*)0);

  pathname = g_build_filename(V_SIM_PIXMAPS_DIR, filename, NULL);

  pixbuf = gdk_pixbuf_new_from_file(pathname, &error);
  if (!pixbuf)
    {
      g_warning(_("failed to load pixbuf file '%s': %s\n"),
		pathname, error->message);
      g_error_free(error);
    }
  g_free(pathname);

  return pixbuf;
}

GtkWindow* visuGtkGet_render()
{
  return visuGtkRender;
}
GtkWindow* visuGtkGet_panel()
{
  return visuGtkPanel;
}
