/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include <gtk/gtksignal.h>

#include "gtk_numericalEntryWidget.h"

/**
 * SECTION:numericalentry
 * @short_description: Defines a widget to enter numerical values without
 * any boundary or precision constrains.
 * 
 * <para>This widget is based on the #GtkEntry widget but behaves more
 * like a #GtkSpinButton is fact. It is designed to enter numerical
 * values, but without any boundary or precision constrains. One can
 * use either plain or scientific notations.</para>
 */

enum {
  VALUE_CHANGED_SIGNAL,
  LAST_SIGNAL
};

static guint numericalEntry_signals[LAST_SIGNAL] = { 0 };

#define NUMERICAL_ENTRY_FORMAT_DEFAULT "%g"

struct _NumericalEntry
{
  GtkEntry entry;

  double value;
  gchar *format;
/*   NumericalEntryPrivate *private; */
};

struct _NumericalEntryClass
{
  GtkEntryClass parent_class;

  void (*numericalEntry) (NumericalEntry *numEntry);

  gulong valueChangedSignalId;
};

G_DEFINE_TYPE(NumericalEntry, numericalEntry, GTK_TYPE_ENTRY)

/* Local methods. */
void printStoredValue(NumericalEntry* numericalEntry);
gboolean parsePrintedValue(NumericalEntry *numericalEntry, double *value);
void onEntryValueChange(NumericalEntry *numericalEntry, gpointer data);
gboolean onEntryLeaveFocus(NumericalEntry *numericalEntry,
			   GdkEventFocus *event, gpointer data);
gboolean onEntryGetFocus(NumericalEntry *numericalEntry,
			 GdkEventFocus *event, gpointer data);

static void numericalEntry_class_init(NumericalEntryClass *klass)
{
  DBG_fprintf(stderr, "Gtk NumericalEntry : creating the class of the widget.\n");

  DBG_fprintf(stderr, "                     - adding new signals ;\n");
  /**
   * NumericalEntry::value-changed:
   * @entry: the #NumericalEntry that emits the signal ;
   * @oldValue: the previous value.
   *
   * This signal is emitted when a new valid numerical value is entered.
   *
   * Since: 3.1
   */
  numericalEntry_signals[VALUE_CHANGED_SIGNAL] =
    g_signal_new ("value-changed",
		  G_TYPE_FROM_CLASS (klass),
		  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		  G_STRUCT_OFFSET (NumericalEntryClass, numericalEntry),
		  NULL, 
		  NULL,                
		  g_cclosure_marshal_VOID__DOUBLE,
		  G_TYPE_NONE, 1, G_TYPE_DOUBLE);
}

static void numericalEntry_init(NumericalEntry *numericalEntry)
{
  DBG_fprintf(stderr, "Gtk NumericalEntry : initializing new object (%p).\n",
	      (gpointer)numericalEntry);

  numericalEntry->format = g_strdup(NUMERICAL_ENTRY_FORMAT_DEFAULT);

  g_signal_connect_swapped(G_OBJECT(numericalEntry), "activate",
			   G_CALLBACK(onEntryValueChange), (gpointer)numericalEntry);
  g_signal_connect_swapped(G_OBJECT(numericalEntry), "focus-out-event",
			   G_CALLBACK(onEntryLeaveFocus), (gpointer)numericalEntry);
  g_signal_connect_swapped(G_OBJECT(numericalEntry), "focus-in-event",
			   G_CALLBACK(onEntryGetFocus), (gpointer)numericalEntry);
}

GtkWidget* numericalEntry_new(double value)
{
  NumericalEntry *numericalEntry;

  DBG_fprintf(stderr, "Gtk NumericalEntry : creating new object with value : %g.\n",
	      value);

  numericalEntry = NUMERICAL_ENTRY(g_object_new(numericalEntry_get_type(), NULL));
  numericalEntry->value = value;
  
  printStoredValue(numericalEntry);

  return GTK_WIDGET(numericalEntry);
}

void printStoredValue(NumericalEntry* numericalEntry)
{
  GString *str;

  g_return_if_fail(IS_NUMERICAL_ENTRY(numericalEntry));

  str = g_string_new("");
  g_string_printf(str, numericalEntry->format, numericalEntry->value);
  gtk_entry_set_text(GTK_ENTRY(numericalEntry), str->str);
  g_string_free(str, TRUE);
}
gboolean parsePrintedValue(NumericalEntry *numericalEntry, double *value)
{
  double valueDouble;
  gchar *last;

  g_return_val_if_fail(IS_NUMERICAL_ENTRY(numericalEntry) && value, FALSE);
  
  valueDouble = g_ascii_strtod(gtk_entry_get_text(GTK_ENTRY(numericalEntry)),
			  &last);
  if (*last != '\0')
    {
      /* Wrong number. */
      g_warning("Can't read a number from '%s'.\n",
		gtk_entry_get_text(GTK_ENTRY(numericalEntry)));
      return FALSE;
    }

  *value = valueDouble;
  return TRUE;
}

void numericalEntrySet_value(NumericalEntry* numericalEntry, double value)
{
  double valueOld;

  g_return_if_fail(IS_NUMERICAL_ENTRY(numericalEntry));

  if (value == numericalEntry->value)
    return;

  valueOld = numericalEntry->value;
  numericalEntry->value = value;

  printStoredValue(numericalEntry);

  DBG_fprintf(stderr, "Gtk NumericalEntry : emitting 'value-changed' signal.\n");
  g_signal_emit(G_OBJECT(numericalEntry),
		numericalEntry_signals[VALUE_CHANGED_SIGNAL], 0, valueOld, NULL);
}
double numericalEntryGet_value(NumericalEntry *numericalEntry)
{
  g_return_val_if_fail(IS_NUMERICAL_ENTRY(numericalEntry), 0.);

  return numericalEntry->value;
}

void onEntryValueChange(NumericalEntry *numericalEntry, gpointer data _U_)
{
  gboolean valid;
  double valueDouble;

  valid = parsePrintedValue(numericalEntry, &valueDouble);

  if (!valid)
    printStoredValue(numericalEntry);
  else
    numericalEntrySet_value(numericalEntry, valueDouble);
}
gboolean onEntryLeaveFocus(NumericalEntry *numericalEntry,
			   GdkEventFocus *event _U_, gpointer data _U_)
{
  onEntryValueChange(numericalEntry, (gpointer)0);
  return FALSE;
}
gboolean onEntryGetFocus(NumericalEntry *numericalEntry,
			 GdkEventFocus *event _U_, gpointer data _U_)
{
  gtk_editable_select_region(GTK_EDITABLE(numericalEntry), 0, -1);
  return FALSE;
}
