/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "visu_configFile.h"

#include "visu_tools.h"
#include "visu_basic.h"
#include "visu_object.h"
#include "coreTools/toolConfigFile.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h> /* For the access markers R_OK, W_OK ... */

/**
 * SECTION:visu_configFile
 * @short_description: Defines methods to access (read/write) to
 * config files and to create different entries.
 *
 * <para>V_Sim uses two different configuration files. The first
 * stores the configuration of the program itself and is called
 * parameters file. The second stores differents values that control
 * the way files are rendered. It is called resources file. For
 * example, their is an entry in the parameters file that controls
 * your favorite rendering method ; and there is an entry in the
 * resources file that codes that vacancy elements are rendered by
 * cube in the atomic rendering method. Most methods of this part uses
 * a first argument usually called 'kind', that control if the method
 * will apply on the parameters file or on the resources
 * file. #VISU_CONFIGFILE_PARAMETER and #VISU_CONFIGFILE_RESOURCE are
 * the two flags that should be used for the 'kind' argument.</para>
 *
 * <para>There are different paths where these files can be
 * stored. These paths are stored in V_Sim with an order : for example
 * parameters file are first looked for in the current working
 * directory, then in the $HOME/.v_sim directory and finally in the
 * installation directory. This is transparent for the user and
 * visuConfigFileGet_validPath() is the right method to access to the
 * best readable configuration file.</para>
 *
 * <para>Different part of V_Sim can add entries in these files. The
 * method visuConfigFileAdd_entry() is designed to this purpose. The
 * entries are defined by their name and they appear in the
 * configuration file as 'name:' followed by the data associated to
 * this entry. In the parameters file, the data are on the same
 * line. In the resources file, the data begin the line after and can
 * be longer that one line. When a configuration file is read, the
 * method associated to each entry (visuConfigFileReadFunc()) is
 * called with a copy of their data lines. The method
 * visuConfigFileAdd_exportFunction() should be used to add a callback
 * when the configurations files are written, then each part of V_Sim
 * that have entries can put some lines in the configuration
 * files.</para>
 */

#define PARAMETER_HEADER     "#V_Sim parameters file v"
#define RESOURCE_HEADER      "#V_Sim resources file v"
#define VERSION_HEADER       "3.0"
#define RESOURCES_FILE_NAME  "v_sim.res"
#define PARAMETERS_FILE_NAME "v_sim.par"

#define FLAG_RESOURCES_PATH "main_resourcesPath"
#define DESC_RESOURCES_PATH "Favorite paths to find and save the resources file ; chain[:chain]"
#define DEFAULT_RESOURCES_PATH ""
static gboolean readResourcesPaths(gchar **lines, int nbLines, int position _U_,
				   VisuData *dataObj _U_, GError **error _U_);
static gboolean exportResourcesPaths(GString *data, int *nbLinesWritten,
				     VisuData *dataObj);


struct _VisuConfigFileEntry
{
  /* Name of the key. */
  gchar *key;
  gchar *description;
  /* Version, default is 3.0. */
  float version;
  /* If set, entry is obsolete and newKey should replaces it. */
  gchar *newKey;

  /* A parameter or a resource */
  int kind;

  /* Number of line used by this resources.
     This is not used if the entry is a parameter
     since, parameters are on the same line than 
     the key and are one line. */
  int nbLines;

  /* This method is called when a file is read
     and the entry is found. */
  visuConfigFileReadFunc read;

  /* Tag, tags are used to ignore or not some
     entries when a file is read. */
  gchar *tag;
};

struct writeFunc_struct
{
  visuConfigFileExportFunc writeFunc;
};

/* This hashtable stores all the known entries.
   The keys are the name of the entry (its key), and
   the value is a pointer to a VisuConfigFileEntry. */
static GHashTable *visuConfigFile_entryList;
static GList *registeredResources, *registeredParameters;
static GList *exportResourcesList, *exportParametersList;
static GHashTable *knownTags;

/* Store the paths to where it is possible to store
   and/or read resource files. This list is ordered
   and first element is the most prefered path. */
static GList *resourcesPath;
/* Store the paths to where it is possible to store
   parameters files. This list is ordered and first element is
   the most prefered path.*/
static GList *parametersPath;


static gint compareStringsInGList(gconstpointer a, gconstpointer b)
{
  return strcmp((char*)a, (char*)b);
}

VisuConfigFileEntry* visuConfigFileAdd_entry(int kind, const gchar *key,
					     const gchar* description, int nbLines,
					     visuConfigFileReadFunc readFunc)
{
  VisuConfigFileEntry *entry;

  g_return_val_if_fail(key && *key, (VisuConfigFileEntry*)0);
  g_return_val_if_fail(description && *description, (VisuConfigFileEntry*)0);
  g_return_val_if_fail(nbLines > 0 && (kind == VISU_CONFIGFILE_PARAMETER ||
				       kind == VISU_CONFIGFILE_RESOURCE),
		       (VisuConfigFileEntry*)0);
  g_return_val_if_fail(!g_hash_table_lookup(visuConfigFile_entryList, (gpointer)key),
		       (VisuConfigFileEntry*)0);

  entry = g_malloc(sizeof(VisuConfigFileEntry));
  entry->key = g_strdup(key);
  entry->description = g_strdup(description);
  entry->kind = kind;
  if (kind == VISU_CONFIGFILE_PARAMETER)
    entry->nbLines = 1;
  else
    entry->nbLines = nbLines;
  entry->read = readFunc;
  entry->tag = (gchar*)0;
  entry->newKey = (gchar*)0;
  entry->version = 3.0f;

  g_hash_table_insert(visuConfigFile_entryList,
		      (gpointer)entry->key, (gpointer)entry);

  if (kind == VISU_CONFIGFILE_RESOURCE)
    registeredResources = g_list_append(registeredResources, (gpointer)entry);
  else if (kind == VISU_CONFIGFILE_PARAMETER)
    registeredParameters = g_list_append(registeredParameters, (gpointer)entry);
  return entry;
}

GList* visuConfigFileGet_entries(int kind)
{
  g_return_val_if_fail(kind == VISU_CONFIGFILE_PARAMETER ||
		       kind == VISU_CONFIGFILE_RESOURCE,
		       (GList*)0);
  if (kind == VISU_CONFIGFILE_PARAMETER)
    return registeredParameters;
  else
    return registeredResources;
}

/* Method to add a new known tag. */
void visuConfigFileAdd_knownTag(gchar* tag)
{
  g_return_if_fail(tag && *tag);

  g_hash_table_insert(knownTags, (gpointer)tag, GINT_TO_POINTER(1));
}

void visuConfigFileAdd_exportFunction(int kind, visuConfigFileExportFunc writeFunc)
{
  struct writeFunc_struct *str;

  if (!writeFunc)
    return;
  g_return_if_fail(kind == VISU_CONFIGFILE_PARAMETER ||
		   kind == VISU_CONFIGFILE_RESOURCE);

  str = g_malloc(sizeof(struct writeFunc_struct));
  str->writeFunc = writeFunc;
  if (kind == VISU_CONFIGFILE_RESOURCE)
    exportResourcesList =
      g_list_append(exportResourcesList, (gpointer)str);
  else if (kind == VISU_CONFIGFILE_PARAMETER)
    exportParametersList =
      g_list_append(exportParametersList, (gpointer)str);
}

static void freeConfigEntry(gpointer data)
{
  VisuConfigFileEntry *entry;

  if (!data)
    return;

  entry = (VisuConfigFileEntry*)data;
  g_free(entry->key);
  g_free(entry->description);
  if (entry->tag)
    g_free(entry->tag);
  if (entry->newKey)
    g_free(entry->newKey);
  g_free(entry);
}

void visuConfigFileSet_tag(VisuConfigFileEntry *entry, gchar *tag)
{
  g_return_if_fail(entry);

  if (entry->tag)
    g_free(entry->tag);
  entry->tag = g_strdup(tag);
}

void visuConfigFileSet_version(VisuConfigFileEntry *entry, float version)
{
  g_return_if_fail(entry && version > 3.0f);

  entry->version = version;
}

void visuConfigFileSet_replace(VisuConfigFileEntry *newEntry,
			       VisuConfigFileEntry *oldEntry)
{
  g_return_if_fail(newEntry && oldEntry);

  if (oldEntry->newKey)
    g_free(oldEntry->newKey);
  oldEntry->newKey = g_strdup(newEntry->key);
}

gboolean visuConfigFileLoad(int kind, const char* fileName,
			    VisuData *dataObj, GError **error)
{
  GIOChannel *ioFile;
  GString *line;
  GIOStatus status; 
  int nbLine, i;
  char *deuxPoints;
  gchar **tokens;
  gchar *key, *tag, *finTag, *finKey;
  VisuConfigFileEntry *entry;
  int withErrors;
  GString *message;

  DBG_fprintf(stderr, "Visu ConfigFile : parsing '%s' file for"
	      " resources/parameters...\n", fileName);
  
  g_return_val_if_fail(kind == VISU_CONFIGFILE_RESOURCE ||
		       kind == VISU_CONFIGFILE_PARAMETER, FALSE);

  ioFile = g_io_channel_new_file(fileName, "r", error);
  if (*error)
    return FALSE;

  line = g_string_new("");
  nbLine = 0;
  status = g_io_channel_read_line_string(ioFile, line, NULL, error);
  if (*error)
    return FALSE;
  nbLine += 1;
  withErrors = 0;
  message = (GString*)0;
  while (status == G_IO_STATUS_NORMAL)
    {
      if (line->str[0] != '#' && line->str[0] != '\n')
	{
	  deuxPoints = strchr(line->str, ':');
	  if (deuxPoints)
	    {
	      tokens = g_strsplit_set(line->str, ":", 2);
	      key = g_strdup(tokens[0]);
	      key = g_strstrip(key);
	      finKey = g_strdup(tokens[1]);
	      g_strfreev(tokens);
	      /* Look for the tag */
	      tag = strchr(key, '[');
	      if (tag)
		{
		  *tag = '\0';
		  tag = tag + 1;
		  finTag = strchr(tag, ']');
		  if (finTag)
		    *finTag = '\0';
		  else
		    {
		      g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_TAG,
				  _("Parse error at line %d,"
				    " the tag '%s' is not closed.\n"),
				  nbLine, tag);
		      withErrors = 1;
		    }
		  DBG_fprintf(stderr,"Visu ConfigFile: read a flag (tag): '%s' (%s).\n", key, tag);
		}
	      else
		  DBG_fprintf(stderr,"Visu ConfigFile: read a flag (tag): '%s' (none).\n", key);

	      if (tag && !g_hash_table_lookup(knownTags, (gpointer)tag))
		{
		  DBG_fprintf(stderr, "Visu ConfigFile: the entry '%s' has an unknown tag (%s),"
			      " it will be dismissed.\n", key, tag);
		}
	      else
		{
		  entry = (VisuConfigFileEntry*)
		    g_hash_table_lookup(visuConfigFile_entryList, (gpointer)key);
		  if (entry)
		    {
		      tokens = g_malloc(sizeof(gchar*) * (entry->nbLines + 1));
		      if (kind == VISU_CONFIGFILE_RESOURCE)
			for (i = 0; i < entry->nbLines; i++)
			  {
			    status = g_io_channel_read_line_string(ioFile, line,
								   NULL, error);
			    nbLine += 1;
			    if (status != G_IO_STATUS_NORMAL)
			      {
				tokens[i] = (gchar*)0;
				g_strfreev(tokens);
				tokens = (gchar**)0;
				*error = g_error_new(CONFIG_FILE_ERROR,
						     CONFIG_FILE_ERROR_MISSING,
						     _("Parse error at line %d,"
						       " '%s' needs %d lines but only %d were read.\n"),
						     nbLine, key, entry->nbLines, nbLine);
				withErrors = 1;
			      }
			    tokens[i] = g_strdup(line->str);
			  }
		      else
			tokens[0] = finKey;
		      if (tokens)
			{
			  tokens[entry->nbLines] = (gchar*)0;
			  if (entry->read && !entry->read(tokens, entry->nbLines,
							  nbLine, dataObj, error))
			    {
			      g_return_val_if_fail(*error, FALSE);
			      withErrors = 1;
			    }
			  g_strfreev(tokens);
			}
		      if (entry->newKey)
			g_warning(_("Parsing resource file, markup '%s' is"
				    " obsolete, replaced by '%s'."),
				  key, entry->newKey);
		    }
		  else
		    {
		      *error = g_error_new(CONFIG_FILE_ERROR,
					   CONFIG_FILE_ERROR_MARKUP,
					   _("Parse error at line %d,"
					     " '%s' is an unknown markup.\n"),
					   nbLine, key);
		      withErrors = 1;
		    }
		}
	      if (withErrors && *error)
		{
		  if (!message)
		    message = g_string_new("");
		  g_string_append_printf(message, "[%s]: %s\n",
					 key, (*error)->message);
		  g_error_free(*error);
		  *error = (GError*)0;
		}
	      g_free(key);
	    }
	}

      status = g_io_channel_read_line_string(ioFile, line, NULL, error);
      nbLine += 1;
    }
  g_string_free(line, TRUE);
  if (status == G_IO_STATUS_ERROR)
    {
      g_io_channel_shutdown(ioFile, FALSE, NULL);
      g_io_channel_unref(ioFile);
      return FALSE;
    }

  status = g_io_channel_shutdown(ioFile, FALSE, error);
  g_io_channel_unref(ioFile);
  if (status != G_IO_STATUS_NORMAL)
    return FALSE;
  DBG_fprintf(stderr, "Visu ConfigFile: read OK.\n");

  if (withErrors)
    {
      g_return_val_if_fail(message, FALSE);
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_READ, message->str);
      g_string_free(message, TRUE);
      return FALSE;
    }
  else
    return TRUE;
}

/* Routine to export the resources (color and
   material of elements ...) to file. */
gboolean visuConfigFileSave(int kind, const char* fileName, int *lines,
			    VisuData *dataObj, GError **error)
{
  gchar *fileUTF8;
  GString *exportString, *data, *message;
  int nbLine, nb;
  GList *pos;
  gboolean ok;
  int withErrors;

  g_return_val_if_fail(error && !*error, FALSE);

  DBG_fprintf(stderr, "Visu ConfigFile : exporting '%s' file for"
	      " resources/parameters...\n", fileName);
  
  g_return_val_if_fail(kind == VISU_CONFIGFILE_RESOURCE ||
		       kind == VISU_CONFIGFILE_PARAMETER, FALSE);

  fileUTF8 = g_filename_from_utf8(fileName, -1, NULL, NULL, NULL);
  g_return_val_if_fail(fileUTF8, FALSE);

  message = g_string_new("");
  if (kind == VISU_CONFIGFILE_RESOURCE)
    g_string_append_printf(message, _("Writing '%s' file for"
				      " resources...\n\n"), fileUTF8);
  else if (kind == VISU_CONFIGFILE_PARAMETER)
    g_string_append_printf(message, _("Writing '%s' file for"
				      " parameters...\n\n"), fileUTF8);
  g_free(fileUTF8);

  withErrors = 0;
  nbLine = 0;
  exportString = g_string_new("");
  if (kind == VISU_CONFIGFILE_RESOURCE)
    g_string_append_printf(exportString, 
			   RESOURCE_HEADER);
  else if (kind == VISU_CONFIGFILE_PARAMETER)
    g_string_append_printf(exportString, 
			   PARAMETER_HEADER);
  g_string_append_printf(exportString, 
	  VERSION_HEADER
	  "\n"
	  "#====================\n"
	  "\n"
	  "#WARNING: this file format is DIFFERENT from that for\n"
	  "#standard v_sim version <= 2.x\n"
	  "\n");
  nbLine += 6;
  g_string_append_printf(exportString, 
	  "#Line beginning with a # are not parsed.\n"
	  "\n");
  nbLine += 2;
  pos = (GList*)0;
  if (kind == VISU_CONFIGFILE_RESOURCE)
    {
      g_string_append_printf(exportString, 
			     "#The only \"useful\" lines must have the following contents\n"
			     "#several two or more lines patterns:\n"
			     "#resource_name:\n"
			     "#values separeted by blank characters\n"
			     "\n");
      nbLine += 5;
      g_string_append_printf(exportString, 
			     "#The following resource names are valid :\n");
      nbLine += 1;
      pos = registeredResources;
    }
  else if (kind == VISU_CONFIGFILE_PARAMETER)
    {
      g_string_append_printf(exportString, 
			     "#The only \"useful\" lines must have the following pattern:\n"
			     "#parameter_name: value\n"
			     "\n");
      nbLine += 3;
      g_string_append_printf(exportString, 
			     "#The following parameter names are valid :\n");
      nbLine += 1;
      pos = registeredParameters;
    }
  while(pos)
    {
      g_string_append_printf(exportString, "# %s\n",
			     ((VisuConfigFileEntry*)(pos->data))->key);
      nbLine += 1;
      pos = g_list_next(pos);
    }
  g_string_append_printf(exportString, "\n");
  nbLine += 1;
  pos = (GList*)0;
  if (kind == VISU_CONFIGFILE_RESOURCE)
    pos = exportResourcesList;
  else if (kind == VISU_CONFIGFILE_PARAMETER)
    pos = exportParametersList;
  while(pos)
    {
      data = g_string_new("");
      ok = ((struct writeFunc_struct*)(pos->data))->writeFunc(data, &nb, dataObj);
      if (ok)
	{
	  g_string_append_printf(exportString, "%s", data->str);
	  nbLine += nb;
	}
      else
	{
	  g_string_append_printf(message, _("Error on line %d.\n"), nbLine);
	  withErrors = 1;
	}
      g_string_free(data, TRUE);
      pos = g_list_next(pos);
    }

  if (!g_file_set_contents(fileName, exportString->str, -1, error))
    {
      g_string_free(exportString, TRUE);
      g_string_free(message, TRUE);
      return FALSE;
    }
  g_string_free(exportString, TRUE);

  if (lines)
    *lines = nbLine;
  
  if (withErrors)
    {
      g_error_new(G_FILE_ERROR, 0, message->str);
      g_string_free(message, TRUE);
      return FALSE;
    }
  else
    {
      g_string_free(message, TRUE);
      return TRUE;
    }
}

gchar* getValidFileWithHeader(int mode, char* filename, char* header, GList **list)
{
  gchar *res;
  FILE *file;
  char *msg;
  char line[MAX_LINE_LENGTH];
  float version;
  
  /* Look for a valid file.
     If it is for writing, a valid file is just given by a valid path.
     If it is for reading, a valid file is a valid path AND has a valid header. */
  while (*list)
    {
      /* We get the next valid path. */
      res = getValidPath(list, filename, mode);
      if (!res)
	{
	  DBG_fprintf(stderr, "Visu ConfigFile : no file available.\n");
	  return (gchar*)0;
	}

      /* if we are in reading mode, we test the header. */
      if (mode & R_OK)
	{
	  DBG_fprintf(stderr, "Visu ConfigFile : looking for header of \n  '%s' ... ", res);
	  file = fopen(res, "r");
	  if (!file)
	    {
	      g_warning("The file '%s' should be readable but something goes"
			" nasty when one wants to open it.\n", res);
	      g_free(res);
	      return (gchar*)0;
	    }
	  version = 0.;
	  msg = fgets(line, MAX_LINE_LENGTH, file);
	  fclose(file);
	  if (msg && !strncmp(line, header, strlen(header))
	      && sscanf(line + strlen(header), "%f", &version))
	    if (version >= 3.)
	      {
		DBG_fprintf(stderr, "ok.\n");
		return res;
	      }
	  DBG_fprintf(stderr, "wrong.\n");
	}
      /* We are in writing mode so the valid path is ok. */
      else
	return res;
      *list = g_list_next(*list);
    }
  DBG_fprintf(stderr, "Visu ConfigFile : no file available.\n");
  return (gchar*)0;
}

gchar* visuConfigFileGet_validPath(int kind, int mode, int utf8)
{
  GList *list;
  gchar* file;
  gchar* fileUTF8;

  g_return_val_if_fail(kind == VISU_CONFIGFILE_RESOURCE ||
		       kind == VISU_CONFIGFILE_PARAMETER, (gchar*)0);

  if (kind == VISU_CONFIGFILE_RESOURCE)
    {
      list = resourcesPath;
      file = getValidFileWithHeader(mode, RESOURCES_FILE_NAME,
				    RESOURCE_HEADER, &list);
    }
  else
    {
      list = parametersPath;
      file = getValidFileWithHeader(mode, PARAMETERS_FILE_NAME,
				    PARAMETER_HEADER, &list);
    }

  if (!file)
    return file;

  if (utf8)
    {
      fileUTF8 = g_filename_from_utf8(file, -1, NULL, NULL, NULL);
      g_free(file);
      return fileUTF8;
    }
  else
    return file;
}
gchar* visuConfigFileGet_nextValidPath(int kind, int accessMode, GList **list, int utf8)
{
  gchar* file;
  gchar* fileUTF8;

  g_return_val_if_fail(kind == VISU_CONFIGFILE_RESOURCE ||
		       kind == VISU_CONFIGFILE_PARAMETER, (gchar*)0);
  g_return_val_if_fail(list, (gchar*)0);

  if (!*list)
    return (gchar*)0;

  if (kind == VISU_CONFIGFILE_RESOURCE)
    file = getValidFileWithHeader(accessMode, RESOURCES_FILE_NAME,
				  RESOURCE_HEADER, list);
  else
    file = getValidFileWithHeader(accessMode, PARAMETERS_FILE_NAME,
				  PARAMETER_HEADER, list);

  if (*list)
    *list = g_list_next(*list);

  if (!file)
    return file;

  if (utf8)
    {
      fileUTF8 = g_filename_from_utf8(file, -1, NULL, NULL, NULL);
      g_free(file);
      return fileUTF8;
    }
  else
    return file;
}
gchar* visuConfigFileGet_defaultFileName(int kind)
{
  g_return_val_if_fail(kind == VISU_CONFIGFILE_RESOURCE ||
		       kind == VISU_CONFIGFILE_PARAMETER, (gchar*)0);

  if (kind == VISU_CONFIGFILE_RESOURCE)
    return (gchar*)RESOURCES_FILE_NAME;
  else
    return (gchar*)PARAMETERS_FILE_NAME;
}

GList* visuConfigFileGet_pathList(int kind)
{
  g_return_val_if_fail(kind == VISU_CONFIGFILE_RESOURCE ||
		       kind == VISU_CONFIGFILE_PARAMETER, (GList*)0);

  if (kind == VISU_CONFIGFILE_RESOURCE)
    return resourcesPath;
  else
    return parametersPath;
}
GList* visuConfigFileAdd_resourcesPath(char* dir)
{
  GList *element;

  if (!dir || !dir[0])
    return (GList*)0;

  element = g_list_find_custom(resourcesPath, (gconstpointer)dir,
			       compareStringsInGList);
  if (!element)
    {
      DBG_fprintf(stderr, "Visu ConfigFile : add a new resource directory"
		  " to the path :\n '%s'\n", dir);
      resourcesPath = g_list_insert(resourcesPath, (gpointer)dir, 1);
      return resourcesPath->next;
    }
  else
    return (GList*)element;
}

gboolean visuConfigFileExport_toXML(const gchar *filename, int kind, GError **error)
{
  GString *str;
  GList *tmpLst;
  VisuConfigFileEntry *entry;
  gboolean status;
  gchar *desc;

  g_return_val_if_fail(filename && *filename, FALSE);
  g_return_val_if_fail(kind == VISU_CONFIGFILE_RESOURCE ||
		       kind == VISU_CONFIGFILE_PARAMETER, FALSE);

  str = g_string_new("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
  if (kind == VISU_CONFIGFILE_PARAMETER)
    {
      tmpLst = registeredParameters;
      g_string_append_printf(str, "<configFile kind=\"parameters\">\n");
    }
  else
    {
      tmpLst = registeredResources;
      g_string_append_printf(str, "<configFile kind=\"resources\">\n");
    }
  while (tmpLst)
    {
      entry = (VisuConfigFileEntry*)tmpLst->data;
      if (entry->tag)
	g_string_append_printf(str, "  <entry name=\"%s\" tag=\"%s\" "
			       "version=\"%f3.1\">\n",
			       entry->key, entry->tag, entry->version);
      else
	g_string_append_printf(str, "  <entry name=\"%s\" version=\"%3.1f\">\n",
			       entry->key, entry->version);
      desc = g_markup_escape_text(entry->description, -1);
      g_string_append_printf(str, "    <description>%s</description>\n", desc);
      g_free(desc);
      if (entry->newKey)
	g_string_append_printf(str, "    <obsolete replacedBy=\"%s\" />\n",
			       entry->newKey);
      g_string_append_printf(str, "  </entry>\n");
      tmpLst = g_list_next(tmpLst);
    }
  g_string_append_printf(str, "</configFile>\n");

  status = g_file_set_contents(filename, str->str, -1, error);
  g_string_free(str, TRUE);
  return status;
}


int visuConfigFile_init()
{
  gchar *currentDir;
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr, "Visu ConfigFile : initialization process ...");
  configFileInit();

  visuConfigFile_entryList = g_hash_table_new_full(g_str_hash, g_str_equal,
						   NULL, freeConfigEntry);
  g_return_val_if_fail(visuConfigFile_entryList, 0);

  registeredResources = (GList*)0;
  registeredParameters = (GList*)0;
  exportResourcesList = (GList*)0;
  exportParametersList = (GList*)0;
  knownTags = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, NULL);
  g_return_val_if_fail(knownTags, 0);

  resourcesPath = (GList*)0;
  parametersPath = (GList*)0;
  resourcesPath = g_list_prepend(resourcesPath, (gpointer)V_SIM_DATA_DIR);
  parametersPath = g_list_prepend(parametersPath, (gpointer)V_SIM_DATA_DIR);

  resourcesPath = g_list_prepend(resourcesPath, (gpointer)V_SIM_OLD_LOCAL_CONF_DIR);
  parametersPath = g_list_prepend(parametersPath, (gpointer)V_SIM_OLD_LOCAL_CONF_DIR);

  resourcesPath = g_list_prepend(resourcesPath, (gpointer)V_SIM_LOCAL_CONF_DIR);
  parametersPath = g_list_prepend(parametersPath, (gpointer)V_SIM_LOCAL_CONF_DIR);

  currentDir = g_get_current_dir();
  resourcesPath = g_list_prepend(resourcesPath, (gpointer)currentDir);
  parametersPath = g_list_prepend(parametersPath, (gpointer)currentDir);

  DBG_fprintf(stderr, " OK\n");

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_RESOURCES_PATH,
					  DESC_RESOURCES_PATH,
					  1, readResourcesPaths);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportResourcesPaths);

  return 1;
}

static gboolean readResourcesPaths(gchar **lines, int nbLines, int position _U_,
				   VisuData *dataObj _U_, GError **error _U_)
{
  int i;
  gchar **tokens;
  gchar *key;

  g_return_val_if_fail(nbLines == 1, FALSE);

  tokens = g_strsplit_set(lines[0], ":", -1);
  for (i = 0; tokens[i]; i++)
    {
      key = g_strdup(tokens[i]);
      key = g_strstrip(key);
      visuConfigFileAdd_resourcesPath(key);
    }
  g_strfreev(tokens);
  return TRUE;
}
static gboolean exportResourcesPaths(GString *data, int *nbLinesWritten,
				     VisuData *dataObj _U_)
{
  GList *pnt;
  
  g_string_append_printf(data, "# %s\n", DESC_RESOURCES_PATH);
  g_string_append_printf(data, "%s: ", FLAG_RESOURCES_PATH);
  pnt = resourcesPath;
  while(pnt)
    {
      /* We cancel the first and the last because it's the current working dir
	 and the install dir. */
      if (pnt->prev && pnt->next && pnt->next->next)
	g_string_append_printf(data, "%s", (char*)pnt->data);
      if (pnt->prev && pnt->next && pnt->next->next && pnt->next->next->next)
	g_string_append_printf(data, ":");
      pnt = g_list_next(pnt);
    }
  g_string_append_printf(data, "\n\n");
  *nbLinesWritten = 3;

  return TRUE;
}
