/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD, Damien
	CALISTE, Olivier D'Astier, laboratoire L_Sim, (2001-2005)
  
	Adresses ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD and Damien
	CALISTE and Olivier D'Astier, laboratoire L_Sim, (2001-2005)

	E-mail addresses :
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.
	D'ASTIER, dastier AT iie P cnam P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#ifndef SURFACES_H
#define SURFACES_H

#include <GL/gl.h>
#include <GL/glu.h> 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <glib.h>

#include <openGLFunctions/view.h>
#include <coreTools/toolColor.h>
#include <visu_data.h>

/**
 * SurfaceResource_struct:
 * @surfnom: the name of the surface (in UTF-8) ;
 * @color: a #Color for the surface ;
 * @material: the lighting effects of the surface ;
 * @rendered: boolean for the visibility of the surface.
 *
 * This structure defines some drawing properties of a set of #Surfaces.
 */
struct SurfaceResource_struct
{
  /* Name used to label the surface. */
  gchar *surfnom;

  /* Material used to draw a specific surface. */
  Color *color;
  float material[5];

  /* Rendered or not */
  gboolean rendered;
};
/**
 * SurfaceResource:
 *
 * Short name to adress #SurfaceResource_struct objects.
 */
typedef struct SurfaceResource_struct SurfaceResource;

struct Surfaces_struct;
/**
 * Surfaces:
 *
 * Short name to adress #Surfaces_struct objects.
 */
typedef struct Surfaces_struct Surfaces;

/**
 * SurfacesOrder_struct:
 * 
 * An opaque structure to store the order of drawn polygons when surfaces are involved.
 */
struct SurfacesOrder_struct;
/**
 * SurfacesOrder:
 *
 * Short name to adress #SurfacesOrder_struct objects.
 */
typedef struct SurfacesOrder_struct SurfacesOrder;

GQuark VISU_ERROR_ISOSURFACES;
enum
  {
    VISU_ERROR_FORMAT,
    VISU_ERROR_CHECKSUM
  };

/**
 * ISOSURFACES_PROPERTY_POTENTIAL
 *
 * Flag used in an ASCII surf file to give informations on the value
 * the surface is built from.
 */
#define ISOSURFACES_PROPERTY_POTENTIAL "potential_values"


/* Some constructors, destructors. */
/**
 * isosurfacesNew:
 *
 * Create a new (with unallocated internal arrays) structure to store surfaces.
 *
 * Returns: a newly allocated #Surfaces structure.
 */
Surfaces* isosurfacesNew();
/**
 * isosurfacesAllocate:
 * @surf: a #Surfaces structure ;
 * @nsurf: the number of surfaces to store ;
 * @npolys: the number of polygons (in total) ;
 * @npoints: the total number of vertices.
 *
 * Allocate internal arrays to store surfaces having the given description.
 */
void isosurfacesAllocate(Surfaces *surf, int nsurf, int npolys, int npoints);
/**
 * isosurfacesReallocate:
 * @surf: a #Surfaces structure ;
 * @nsurf: the number of surfaces to store ;
 * @npolys: the number of polygons (in total) ;
 * @npoints: the total number of vertices.
 *
 * Change the allocation of internal arrays to store surfaces
 * with the updated description.
 */
void isosurfacesReallocate(Surfaces **surf, int nsurf, int npolys, int npoints);
/**
 * isosurfacesFree:
 * @surf: a #Surfaces structure.
 *
 * Free all memory associated to the given surfaces.
 */
void isosurfacesFree(Surfaces *surf);

/** 
 * isosurfacesLoad_file:
 * @file: target file to load ;
 * @surf: a set of surfaces (location) ;
 * @error: a location to store errors.
 *
 * This loads a surface file and set default material properties for it.
 * See surf file specifications.
 *
 * Returns: TRUE in case of success, FALSE otherwise. Even in case of success
 *          @error may have been set. 
 *
 */
gboolean isosurfacesLoad_file(const char *file, Surfaces **surf, GError **error);

/** 
 * isoSurfacesSet_fitToBox:
 * @data: a #VisuData object ;
 * @surf: a set of surfaces.
 *
 * This method changes the position of all vertices in the structure
 * to match the box description given in @data.
 */
void isoSurfacesSet_fitToBox(VisuData *data, Surfaces *surf);

/**
 * isosurfacesOrder_new:
 *
 * Create an object to hold the order in which the surfaces must be
 * drawn. See isosurfacesOrder_polygons() to set this object.
 *
 * Returns: a newly created #SurfacesOrder object without any values.
 */
SurfacesOrder* isosurfacesOrder_new();
/**
 * isosurfacesOrder_free:
 * @order: the object to be freed.
 *
 * Free memory used by a #SurfacesOrder object.
 */
void isosurfacesOrder_free(SurfacesOrder *order);


/** 
 * isosurfacesGet_nbSurfaces:
 * @surf: the surface object.
 *
 * Retrieves th number of surfaces stired in a given @surf object.
 *
 * Returns: number of surfaces.
 */
int isosurfacesGet_nbSurfaces(Surfaces *surf);

/**
 * isosurfacesGet_surfaceName:
 * @surf: the surface object ;
 * @surf_index: the number of the surface.
 * 
 * This returns for the given @surf_index its name 
 * (1 <= surf_index <= surfaces_number) 
 *
 * Returns: the name of the surface or empty name or NULL, if @surf_index is invalid.
 */
const gchar* isosurfacesGet_surfaceName(Surfaces *surf, int surf_index);
/**
 * isosurfacesGet_surfaceRendered:
 * @surf: the surface object ;
 * @surf_index: the number of the surface.
 * 
 * This returns for the given @surf_index its visibility.
 *
 * Returns: the visibility of the surface or FALSE, if @surf_index is invalid.
 */
gboolean isosurfacesGet_surfaceRendered(Surfaces *surf, int surf_index);
/**
 * isosurfacesGet_surfaceResource:
 * @surf: the surface object ;
 * @surf_index: the number of the surface.
 * 
 * This returns for the given @surf_index its resource information.
 *
 * Returns: the resource of the surface or NULL, if @surf_index is invalid.
 */
SurfaceResource* isosurfacesGet_surfaceResource(Surfaces *surf, int surf_index);
/**
 * isosurfacesSet_surfaceResource:
 * @surf: the surface object ;
 * @surf_index: the number of the surface ;
 * @res: the new resource.
 * 
 * This method is used to change the resource of a surface.
 */
void isosurfacesSet_surfaceResource(Surfaces *surf, int surf_index, SurfaceResource *res);
/**
 * isosurfacesGet_resourceFromName:
 * @surf_name: the name of the surface ;
 * @new: a location to store a boolean value (can be NULL).
 * 
 * This returns the resource information matching the given @surf_name. If
 * the resource doesn't exist, it is created and @new is set to TRUE.
 *
 * Returns: the resource of the surface or NULL, if @surf_name is invalid.
 */
SurfaceResource* isosurfacesGet_resourceFromName(const gchar *surf_name, gboolean *new);
/**
 * isosurfacesCopy_resource:
 * @res: an allocated #SurfaceResource object to receive values ;
 * @res_old: a #SurfaceResource to read the values from.
 * 
 * This method copies all values from @res_old to @res.
 */
void isosurfacesCopy_resource(SurfaceResource *res, SurfaceResource *res_old);
/**
 * isosurfacesGet_surfaceId:
 * @surf: the surface object ;
 * @i: the number of the surface.
 * 
 * This returns for the given @i its id information.
 *
 * Returns: the id of the surface or 0, if @i is invalid.
 */
int isosurfacesGet_surfaceId(Surfaces *surf, int i);
/**
 * isosurfacesGet_surfaceSortedById:
 * @surf: the surface object.
 * 
 * This returns the surface numbers sorted using their ids.
 *
 * Returns: a newly allocated array with surface numbers.
 */
int* isosurfacesGet_surfaceSortedById(Surfaces *surf);
/**
 * isosurfacesGet_newId:
 * @surf: the surface object.
 * 
 * This returns a unique id to create a new surface.
 *
 * Returns: a value suitable to create a new surface in this set of surfaces.
 */
int isosurfacesGet_newId(Surfaces *surf);
/**
 * isosurfacesGet_surfacePosition:
 * @surf: the surface object ;
 * @id: the id of the surface.
 * 
 * This returns for the given @id its number.
 *
 * Returns: the number of the surface or 0, if @id is invalid.
 */
int isosurfacesGet_surfacePosition(Surfaces *surf, int id);

/**
 * isosurfacesOrder_polygons:
 * @order: the description of the polygons order ;
 * @surf: an array of #Surfaces object, must be NULL terminated.
 *
 * Re-orders the polygons in back to front order.
 * This function should be called everytime a redraw is needed.
 */
void isosurfacesOrder_polygons(SurfacesOrder *order, Surfaces *surf[]);

/**
 * isosurfacesDraw_surfaces:
 * @openGLId: an id for the OpenGL list ;
 * @view: a #OpenGLView object that describes the rendering box.
 * @surf: an array of #Surfaces object, must be NULL terminated.
 * @order: a #SurfacesOrder object.
 *
 * Rebuild each visible surface's list. The order in which to draw the surfaces
 * is given in the @order argument.
 *
 */
void isosurfacesDraw_surfaces(int openGLId, OpenGLView *view,
			      Surfaces *surf[], SurfacesOrder *order);

/**
 * isosurfacesGet_floatProperty:
 * @surf: a #Surfaces object ;
 * @name: the name of the property to look for.
 *
 * Some properties can be associated to the surfaces stored in @surf.
 * This method is used to retrieve floating point values properties.
 *
 * Returns: a table with the values if the property is found, NULL
 *            otherwise.
 */
float* isosurfacesGet_floatProperty(Surfaces *surf, const gchar *name);
/**
 * isosurfacesAdd_floatProperty:
 * @surf: a #Surfaces object ;
 * @name: the name of the property to add.
 *
 * Some properties can be associated to the surfaces stored in @surf.
 * This method is add a new property.
 *
 * Returns: a newly allocated array that can be populated.
 */
float* isosurfacesAdd_floatProperty(Surfaces *surf, const gchar* name);
/**
 * isosurfacesGet_floatPropertyValue:
 * @surf: a #Surfaces object ;
 * @idSurf: a surface number ;
 * @name: the name of the property to get the value from ;
 * @value: a location to store the value.
 *
 * This method retrieves a float value stored as a property called @name for
 * the surface defined by its number @idSurf.
 *
 * Returns: TRUE if a value is indeed found.
*/
gboolean isosurfacesGet_floatPropertyValue(Surfaces *surf, int idSurf,
					   const gchar *name, float *value);

/**
 * isosurfacesRemove:
 * @surf: a #Surfaces object ;
 * @idSurf: the id of the surf to remove.
 *
 * Remove from memory all polygons from the given surface.
 *
 * Returns: TRUE if the surface list is reduced to zero (and @surf
 *          to be freed).
 */
gboolean isosurfacesRemove(Surfaces *surf, int idSurf);

/**
 * isosurfacesCheck_consistency:
 * @surf: a #Surfaces object.
 *
 * Check if all arrays in the structures are consistent (without
 * overflow).
 */
void isosurfacesCheck_consistency(Surfaces* surf);
/**
 * isosurfacesInit:
 *
 * Method used to initialised the surface handling, should not be called.
 */
void isosurfacesInit();

/**
 * isosurfacesSet_showAll:
 * @surf: a #Surfaces object ;
 * @show: TRUE to show all surfaces, FALSE to hide them.
 * 
 * Shows or hides all surfaces and check their "draw" status in the panel accordingly.
 */
void isosurfacesSet_showAll(Surfaces *surf, gboolean show);

/* Private area. */

/**
 * Surfaces_struct:
 * @nsurf: number of surfaces encoded in this structure ;
 * @num_polys: number of polygoins stored in this structure ;
 * @num_points: number of vertices stored in this structure ;
 * @poly_surf_index: gives the id of the surface for each polygon ;
 * @poly_num_vertices: gives the number of vertices used by each polygons ;
 * @poly_vertices: returns the id j of the vertices of polygon i ;
 * @poly_points: vectors giving points of the vertex i ;
 * @poly_normals: vectors giving normals of the vertex i ;
 * @local_box: the description of the box where surfaces are drawn ;
 * @resources: for each surface points on a #SurfaceResource ;
 * @ids: gives a list of ids used to index surfaces ;
 * @properties: an hashtable of properties.
 *
 * This structure stores surfaces. Several surfaces are stored in a single
 * structure for improved performences.
 */
struct Surfaces_struct
{
  /* Number of different surfaces. */
  int nsurf;

  /* Number of polygons */
  int num_polys, num_points;

  /* Give the number of the surface when the number of the
     polygon is given. */
  int *poly_surf_index;

  /* Return the number of vertices when the id of
     the polygon is given. */
  int *poly_num_vertices;

  /* Return the id j of the vertice of polygon i. */
  int **poly_vertices;

  /* Vectors giving points and normal of the vertice i. */
  float **poly_points;
  float **poly_normals;

  /* The description of the box where surfaces are drawn. */
  float local_box[6];

  /* Resources for each surfaces. */
  SurfaceResource **resources;

  /* Id for each surfaces. */
  int *ids;

  /* Table to add properties to surfaces. */
  GHashTable *properties;
};

#endif
