/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "toolShade.h"
#include "toolColor.h"

#include <string.h>
#include <GL/gl.h>
#include <GL/glu.h> 

#include <visu_tools.h>
#include <openGLFunctions/text.h>

#define SHADE_LEGEND_WIDTH 20
#define SHADE_LEGEND_HEIGHT 150
#define SHADE_LEGEND_N_QUADS 20

struct Shade_struct
{
  gchar* labelUTF8;
  float vectA[3], vectB[3];
  gboolean vectX[3];
  int colorMode;
  gchar* pathToImage;
};

GList *toolShadeList = (GList*)0;

Shade* shadeNew(const gchar* labelUTF8, float vectA[3], float vectB[3],
	       gboolean vectX[3], ShadeColorMode colorMode)
{
  Shade *shade;
  int i;

  g_return_val_if_fail(labelUTF8 && vectA && vectB && vectX &&
		       colorMode < shade_nb_colorMode, (Shade*)0);

  DBG_fprintf(stderr, "Tool Shade : creating a new Shade object :");
  shade = g_malloc(sizeof(Shade));
  shade->labelUTF8 = g_strdup(labelUTF8);
  for (i = 0; i < 3; i++)
    {
      shade->vectA[i] = vectA[i];
      shade->vectB[i] = vectB[i];
      shade->vectX[i] = vectX[i];
    }
  shade->colorMode = colorMode;
  shade->pathToImage = (gchar*)0;

  DBG_fprintf(stderr, " %p.\n", (gpointer)shade);
  return shade;
}

gboolean shadeSet_pathToImage(Shade *shade, const gchar* path)
{
  gchar *absPath;

  g_return_val_if_fail(shade, FALSE);

  if (!g_path_is_absolute(path))
    absPath = g_build_filename(v_sim_pixmaps_dir, path, NULL);
  else
    absPath = g_strdup(path);

  if (shade->pathToImage && strcmp((char*)shade->pathToImage, (char*)absPath) == 0)
    {
      g_free(absPath);
      return FALSE;
    }

  if (shade->pathToImage)
    g_free(shade->pathToImage);

  shade->pathToImage = absPath;

  DBG_fprintf(stderr, "Tool Shade : set pathToImage of shade"
	      " %p to '%s'.\n", (gpointer)shade, absPath);
  return TRUE;
}

gchar* shadeGet_label(Shade *shade)
{
  g_return_val_if_fail(shade, (gchar*)0);
  return shade->labelUTF8;
}
gchar* shadeGet_pathToImage(Shade *shade)
{
  g_return_val_if_fail(shade, (gchar*)0);
  return shade->pathToImage;
}
ShadeColorMode shadeGet_colorMode(Shade *shade)
{
  g_return_val_if_fail(shade, (int)0);
  return shade->colorMode;
}
gboolean shadeGet_colorTransformation(Shade *shade, float **vectA, float **vectB, gboolean **vectX)
{
  g_return_val_if_fail(shade, FALSE);
  g_return_val_if_fail(vectA && vectB && vectX, FALSE);
  
  *vectA = shade->vectA;
  *vectB = shade->vectB;
  *vectX = shade->vectX;
  return TRUE;
}
void shadeGet_valueTransformedInRGB(Shade *shade, float rgba[4], float value)
{
  int i;

  g_return_if_fail(shade);

  for (i = 0; i < 3; i++)
    {
      rgba[i] = shade->vectA[i] * value + shade->vectB[i];
      rgba[i] = (rgba[i] < 0.)?0.:rgba[i];
      rgba[i] = (rgba[i] > 1.)?1.:rgba[i];
    }
  /* Don't use alpha channel at present time. */
  rgba[3] = 1.;
  /* Transform if required. */
  if (shade->colorMode == shade_colorModeHSV)
    color_HSVtoRGB(rgba, rgba);

}

/****************/
/* OpenGL part. */
/****************/
void shadeDraw_legend(Shade *shade, float widthVal, float heightVal,
		      float minMax[2], float *marks, int nbMarks)
{
  int viewport[4];
  guint width, height, wTotal, hTotal;
  int i;
  float yStep;
  float rgba[4];
  char value[16];

  g_return_if_fail(shade);
  g_return_if_fail(widthVal != 0. && heightVal != 0.);

  /* We get the size of the current viewport. */
  glGetIntegerv(GL_VIEWPORT, viewport);

  /* We change the viewport. */
  if (widthVal < 0.)
    width = SHADE_LEGEND_WIDTH;
  else if (widthVal < 1.)
    width = (guint)((float)viewport[2] * widthVal);
  else
    width = (guint)widthVal;
  if (heightVal < 0.)
    height = SHADE_LEGEND_HEIGHT;
  else if (heightVal < 1.)
    height = (guint)((float)viewport[3] * heightVal);
  else
    height = (guint)heightVal;
  wTotal = width + 5 * 3 + 12 * 7;
  hTotal = height + 5 * 2;
  glViewport(10, 10, wTotal, hTotal);

  glDisable(GL_FOG);
  glDisable(GL_LIGHTING);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

  /* We change the projection for a 2D one. */
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0.0, (float)wTotal, 0., (float)hTotal);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();

  /* We draw a big transparent square to do the back. */
  glColor4f(1., 1., 1., 0.5);
  glRecti(0, 0, wTotal, hTotal);

  /* We draw the colored bar. */
  shadeGet_valueTransformedInRGB(shade, rgba, 0.);
  glColor4fv(rgba);
  yStep = (float)height / (float)SHADE_LEGEND_N_QUADS;
  glBegin(GL_QUAD_STRIP);
  for (i = 0; i <= SHADE_LEGEND_N_QUADS; i++)
    {
      glVertex2f(5., 5. + (float)i * yStep);
      glVertex2f(5. + (float)width, 5. + (float)i * yStep);
      shadeGet_valueTransformedInRGB(shade, rgba, (float)i / (float)SHADE_LEGEND_N_QUADS);
      glColor4fv(rgba);
    }
  glEnd();

  /* We draw some marks. */
  if (marks)
    {
      glLineWidth(2);
      glBegin(GL_LINES);
      for (i = 0; i < nbMarks; i++)
	{
	  yStep = (marks[i] - minMax[0]) / (minMax[1] - minMax[0]);
	  shadeGet_valueTransformedInRGB(shade, rgba, yStep);
	  rgba[0] = 1. - rgba[0];
	  rgba[1] = 1. - rgba[1];
	  rgba[2] = 1. - rgba[2];
	  glColor4fv(rgba);
	  yStep *= (float)height;
	  glVertex2f(5. + 3., 5. + yStep);
	  glVertex2f(5. + (float)width - 3., 5. + yStep);
	}
      glEnd();
    }

  /* We draw the frame around. */
  glColor3f(0., 0., 0.);
  glLineWidth(1);
  glBegin(GL_LINE_STRIP);
  glVertex2i(5, 5);
  glVertex2i(5 + width, 5);
  glVertex2i(5 + width, 5 + height);
  glVertex2i(5, 5 + height);
  glVertex2i(5, 5);
  glEnd();

  /* We draw the tics. */
  glColor3f(0., 0., 0.);
  glLineWidth(1);
  glBegin(GL_LINES);
  glVertex2i(5 + width    , 5);
  glVertex2i(5 + width + 3, 5);
  glVertex2i(5 + width    , 5 + height / 3);
  glVertex2i(5 + width + 3, 5 + height / 3);
  glVertex2i(5 + width    , 5 + 2 * height / 3);
  glVertex2i(5 + width + 3, 5 + 2 * height / 3);
  glVertex2i(5 + width    , 5 + height);
  glVertex2i(5 + width + 3, 5 + height);
  glEnd();
  
  /* We print the labels. */
  sprintf(value, "%.3g", minMax[0]);
  glRasterPos2i(5 + width + 5, 5); openGLText_drawChars(value); 
  sprintf(value, "%.3g", (minMax[0] + minMax[1]) / 3.);
  glRasterPos2i(5 + width + 5, 5 + height / 3 - 6); openGLText_drawChars(value); 
  sprintf(value, "%.3g", (minMax[0] + minMax[1]) / 3. * 2.);
  glRasterPos2i(5 + width + 5, 5 + 2 * height / 3 - 6); openGLText_drawChars(value);
  sprintf(value, "%.3g", minMax[1]);
  glRasterPos2i(5 + width + 5, 5 + height - 12); openGLText_drawChars(value);

  /* We change the projection back. */
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  /* We set viewport back. */
  glViewport(0, 0,viewport[2], viewport[3]);
}


/*********************************************/
/* Methods to deal with internal shade list. */
/*********************************************/
GList* toolShadeListGet()
{
  return toolShadeList;
}
GList* toolShadeListAppend(Shade *shade)
{
  g_return_val_if_fail(shade, (GList*)0);

  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);
  return toolShadeList;
}
GList* toolShadeBuild_presetList()
{
  Shade *shade;
  float vectA[3], vectB[3];
  gboolean vectX[3];

  /* Create a blue to red color range. */
  vectA[0] = -0.66667;
  vectA[1] = 0.;
  vectA[2] = 0.;
  vectB[0] = 0.66667;
  vectB[1] = 1.;
  vectB[2] = 1.;
  vectX[0] = TRUE;
  vectX[1] = FALSE;
  vectX[2] = FALSE;
  shade = shadeNew(_("blue to red"), vectA, vectB, vectX, shade_colorModeHSV);
  shadeSet_pathToImage(shade, "stock-blue-to-red.png");
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a black to white (through yellow and red) color range. */
  vectA[0] = 2.66667;
  vectA[1] = 2.66667;
  vectA[2] = 4.;
  vectB[0] = 0.;
  vectB[1] = -1.;
  vectB[2] = -3.;
  vectX[0] = TRUE;
  vectX[1] = TRUE;
  vectX[2] = TRUE;
  shade = shadeNew(_("hot color"), vectA, vectB, vectX, shade_colorModeRGB);
  shadeSet_pathToImage(shade, "stock-hot-color.png");
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a blue to yellow (through dark purple) color range. */
  vectA[0] = 1.33333;
  vectA[1] = 2.;
  vectA[2] = -2;
  vectB[0] = 0.;
  vectB[1] = -1.;
  vectB[2] = 1.;
  vectX[0] = TRUE;
  vectX[1] = TRUE;
  vectX[2] = TRUE;
  shade = shadeNew(_("blue to yellow"), vectA, vectB, vectX, shade_colorModeRGB);
  shadeSet_pathToImage(shade, "stock-blue-to-yellow.png");
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a green to red color range. */
  vectA[0] = -.3333;
  vectA[1] = 0.;
  vectA[2] = 0.;
  vectB[0] = 0.3333;
  vectB[1] = 1.;
  vectB[2] = 1.;
  vectX[0] = TRUE;
  vectX[1] = FALSE;
  vectX[2] = FALSE;
  shade = shadeNew(_("green to red"), vectA, vectB, vectX, shade_colorModeHSV);
  shadeSet_pathToImage(shade, "stock-green-to-red.png");
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a white to red color range. */
  vectA[0] = -.3333;
  vectA[1] = 0.8;
  vectA[2] = 0.;
  vectB[0] = 0.3333;
  vectB[1] = 0.1;
  vectB[2] = 1.;
  vectX[0] = TRUE;
  vectX[1] = TRUE;
  vectX[2] = FALSE;
  shade = shadeNew(_("light green to red"), vectA, vectB, vectX, shade_colorModeHSV);
  shadeSet_pathToImage(shade, "stock-white-to-red.png");
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a black to white color range. */
  vectA[0] = 0.;
  vectA[1] = 0.;
  vectA[2] = 1.;
  vectB[0] = 0.;
  vectB[1] = 0.;
  vectB[2] = 0.;
  vectX[0] = FALSE;
  vectX[1] = FALSE;
  vectX[2] = TRUE;
  shade = shadeNew(_("black to white"), vectA, vectB, vectX, shade_colorModeHSV);
  shadeSet_pathToImage(shade, "stock-black-to-white.png");
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);

  /* Create a black to white color range. */
  vectA[0] = 0.;
  vectA[1] = 0.;
  vectA[2] = -1.;
  vectB[0] = 0.;
  vectB[1] = 0.;
  vectB[2] = 1.;
  vectX[0] = FALSE;
  vectX[1] = FALSE;
  vectX[2] = TRUE;
  shade = shadeNew(_("white to black"), vectA, vectB, vectX, shade_colorModeHSV);
  shadeSet_pathToImage(shade, "stock-white-to-black.png");
  toolShadeList = g_list_append(toolShadeList, (gpointer)shade);
  
  return toolShadeList;
}
