/* UltraPoint -- the GGI presentation program.

   Copyright (C) 2000 Daiki Ueno <ueno@unixuser.org>

   Author: Daiki Ueno <ueno@unixuser.org>
   Created: 2000-05-16

   This file is part of UltraPoint.

   This program is free software; you can redistribute it and/or modify 
   it under the terms of the GNU General Public License as published by 
   the Free Software Foundation; either version 2, or (at your option)  
   any later version.                                                   

   This program is distributed in the hope that it will be useful,      
   but WITHOUT ANY WARRANTY; without even the implied warranty of       
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the        
   GNU General Public License for more details.                         

   You should have received a copy of the GNU General Public License    
   along with GNU Emacs; see the file COPYING.  If not, write to the    
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,         
   Boston, MA 02111-1307, USA.                                          

*/

#include "slide.h"
#define _GNU_SOURCE
#include <getopt.h>

static gchar *opt_background_image = NULL;
static gchar *opt_vflibcap = "vflibcap-ultrapoint";
static gchar *opt_ggi_mode = "800x600";
static gchar *opt_ggi_target = NULL;
static gchar *opt_validate_dtd = NULL;
static gchar *opt_foreground = NULL;
static gchar *opt_background = NULL;
static gchar *opt_header_foreground = NULL;
static gchar *opt_itemize_foreground = NULL;
static gchar *opt_border_foreground = NULL;
static int opt_start_page = 0;
static gboolean opt_use_multiple_frames = FALSE;
static gboolean opt_disable_antialias = FALSE;

static gchar *strusage (gint level);
static ggi_visual_t init_ggi (void);

static const struct option long_options[] = {
  { "version", 0, 0, 'v' },
  { "help", 0, 0, 'h' },
  { "ggi-target", 1, 0, 't' },
  { "ggi-mode", 1, 0, 'm' },
  { "vflibcap", 1, 0, 'V' },
  { "background-image", 1, 0, 'I' },
  { "validate-dtd", 1, 0, 'C' },
  { "min-glue-width", 1, 0, 0 },
  { "max-glue-width", 1, 0, 0 },
  { "agglutinative-min-glue-width", 1, 0, 0 },
  { "agglutinative-max-glue-width", 1, 0, 0 },
  { "line-space", 1, 0, 0 },
  { "multiple-frames", 0, 0, 0 },
  { "no-antialias", 0, 0, 0 },
  { "foreground", 1, 0, 0 },
  { "background", 1, 0, 0 },
  { "border-foreground", 1, 0, 0 },
  { "border-thickness", 1, 0, 0 },
  { "header-foreground", 1, 0, 0 },
  { "itemize-foreground", 1, 0, 0 },
  { "start-page", 1, 0, 'P' },
  { 0, 0, 0, 0 }
};

static const gchar *option_description[] = {
  N_("show version information"),
  N_("show this help"),
  N_("set GGI target"),
  N_("set GGI mode"),
  N_("set vflibcap"),
  N_("set background image"),
  N_("validate input with DTD"),
  N_("set minimal width of a glue"),
  N_("set maximal width of a glue"),
  N_("set minimal width of a glue for agglutinative language"),
  N_("set maximal width of a glue for agglutinative language"),
  N_("set width of lines"),
  N_("use multiple frames"),
  N_("disable antialiasing"),
  N_("set foreground color"),
  N_("set background color"),
  N_("set border foreground color"),
  N_("set border thickness"),
  N_("set header foreground color"),
  N_("set itemize foreground color"),
  N_("specify starting page"),
};

static const gchar *progname;

enum {
  USAGE_VERSION,
  USAGE_COPYRIGHT,
  USAGE_OPTIONS
};

static gchar *
strusage (level)
     gint level;
{
  static gchar usage[256];

  switch (level)
    {
    case USAGE_VERSION:
      g_snprintf (usage, sizeof(usage), "%s (%s) %s\n",
		  progname, PACKAGE, VERSION);
      break;
    case USAGE_COPYRIGHT:
      g_snprintf (usage, sizeof(usage),
		  "Copyright (C) 2000 Daiki Ueno\n"
		  "This is free software; "
		  "see the source for copying conditions.  "
		  "There is NO\nwarranty; not even for MERCHANTABILITY "
		  "or FITNESS FOR A PARTICULAR PURPOSE.\n");
      break;
    default:
      g_snprintf (usage, sizeof(usage),
		  "Usage: %s [options] file\n", progname);
      break;
    }

  return usage;
}

static ggi_visual_t
init_ggi (void)
{
  UptColor *color;
  ggi_pixel fg, bg;
  ggi_visual_t vis;
  ggi_mode vis_mode;

  if (ggiInit ())
    ggiPanic (_("unable to initialize libggi.\n"));
  
  vis = ggiOpen (opt_ggi_target, NULL);
  if (!vis)
    ggiPanic (_("failed to open visual.\n"));

  ggiParseMode (opt_ggi_mode, &vis_mode);
  ggiSetFlags (vis, GGIFLAG_ASYNC);
  ggiCheckMode (vis, &vis_mode);
  if (ggiSetMode (vis, &vis_mode) < 0)
    ggiPanic (_("mode refused.\n"));

  if (GT_SCHEME (vis_mode.graphtype) == GT_PALETTE)
    ggiSetColorfulPalette (vis);

  color = NULL;
  if (opt_foreground)
    color = upt_color_by_name (opt_foreground);
  if (!color)
    color = upt_color_by_rgb (0, 0, 0);
  fg = upt_color_map (vis, color);
  upt_object_unref (color);

  color = NULL;
  if (opt_background)
    color = upt_color_by_name (opt_background);
  if (!color)
    color = upt_color_by_rgb (1, 1, 1);
  bg = upt_color_map (vis, color);
  upt_object_unref (color);

  color = NULL;
  if (opt_header_foreground)
    color = upt_color_by_name (opt_header_foreground);
  if (!color)
    color = upt_color_by_rgb (1, 0, 0);
  slide_header_foreground_pixel = upt_color_map (vis, color);
  upt_object_unref (color);

  color = NULL;
  if (opt_itemize_foreground)
    color = upt_color_by_name (opt_itemize_foreground);
  if (!color)
    color = upt_color_by_rgb (0, 1, 0);
  slide_itemize_foreground_pixel = upt_color_map (vis, color);
  upt_object_unref (color);

  color = NULL;
  if (opt_border_foreground)
    color = upt_color_by_name (opt_border_foreground);
  if (!color)
    color = upt_color_by_rgb (0.7, 0.7, 0.7);
  slide_border_foreground_pixel = upt_color_map (vis, color);
  upt_object_unref (color);

  ggiSetGCBackground (vis, bg);
  ggiSetGCForeground (vis, fg);

  return vis;
}

static void
parse_options (argc, argv)
     int *argc;
     char ***argv;
{
  gint i;

  while (1)
    {
      gint option_index = 0, c;

      c = getopt_long (*argc, *argv, "vht:m:V:I:C:P:",
		       long_options, &option_index);
      if (c == -1)
	{
	  *argv = &(*argv)[--optind];
	  *argc -= optind;
	  break;
	}
      
      switch (c)
	{
	case 0:
	  switch (option_index)
	    {
	    case 7:
	      slide_min_glue_width =
		MAX (0, g_strtod (optarg, NULL));
	      break;
	    case 8:
	      slide_max_glue_width =
		MAX (0, g_strtod (optarg, NULL));
	      break;
	    case 9:
	      slide_agglutinative_min_glue_width =
		MAX (0, g_strtod (optarg, NULL));
	      break;
	    case 10:
	      slide_agglutinative_max_glue_width =
		MAX (0, g_strtod (optarg, NULL));
	      break;
	    case 11:
	      slide_base_lineskip =
		MAX (0, g_strtod (optarg, NULL));
	      break;
	    case 12:
	      opt_use_multiple_frames = TRUE;
	      break;
	    case 13:
	      opt_disable_antialias = TRUE;
	      break;
	    case 14:
	      opt_foreground = optarg;
	      break;
	    case 15:
	      opt_background = optarg;
	      break;
	    case 16:
	      opt_border_foreground = optarg;
	      break;
	    case 17:
	      slide_border_thickness = MAX (0, atoi (optarg));
	      break;
	    case 18:
	      opt_header_foreground = optarg;
	      break;
	    case 19:
	      opt_itemize_foreground = optarg;
	      break;
	    }
	  break;
	case 'v':
	  printf (strusage (USAGE_VERSION));
	  printf (strusage (USAGE_COPYRIGHT));
	  exit (0);
	case 'h':
	  printf (strusage (-1));
	  for (i = 0; i < G_N_ELEMENTS (long_options) - 1; i++)
	    if (long_options[i].val)
	      printf ("  --%s, -%c  %s\n",
		      long_options[i].name, long_options[i].val,
		      gettext (option_description[i]));
	    else
	      printf ("  --%s  %s\n",
		      long_options[i].name,
		      gettext (option_description[i]));
	  exit (0);
	case 't':
	  opt_ggi_target = optarg;
	  break;
	case 'm':
	  opt_ggi_mode = optarg;
	  break;
#ifdef HAVE_LIBMAGICK
	case 'I':
	  opt_background_image = optarg;
	  break;
#endif
	case 'C':
	  opt_validate_dtd = optarg;
	  break;
	case 'P':
	  opt_start_page = MAX (0, atoi (optarg));
	  break;
	}
    }
}

int
main (argc, argv)
     int argc;
     char **argv;
{
  UptXMLSlideShow *slideshow;
  UptCanvas *canvas;
  gint i;

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);

  progname = g_basename (argv[0]);

  parse_options (&argc, &argv);

  if (argc < 2)
    {
      printf (strusage (-1));
      return 1;
    }

  slideshow = upt_parse_file (argv[1], opt_validate_dtd);
  if (!slideshow)
    {
      printf (_("Not a valid UltraPoint XML file.\n"));
      return 1;
    }

  if (VF_Init(opt_vflibcap, NULL) < 0)
    {
      printf (_("VFlib initialization failed.\n"));
      return 1;
    }

  canvas = upt_canvas_new (init_ggi ());
  if (opt_disable_antialias)
    canvas->disable_antialias = TRUE;
  if (opt_use_multiple_frames)
    canvas->use_multiple_frames = TRUE;
#ifdef HAVE_LIBMAGICK
  if (opt_background_image)
    upt_canvas_load_bg_image (canvas, opt_background_image);
#endif

  for (i = 0; i<opt_start_page; i++)
    upt_slideshow_next (slideshow);
  upt_slideshow_draw (canvas, slideshow);

  while (1)
    {
      struct timeval tv = { 0, 10000 };

      if (ggiEventPoll (canvas->visual, emKeyPress|emPtrButtonPress, &tv))
	{
	  ggi_event ev;
	  ggiEventRead(canvas->visual, &ev, emKeyPress|emPtrButtonPress);

	  if (ev.any.type == evPtrButtonPress)
	    switch (ev.pbutton.button)
	      {
	      case GII_PBUTTON_LEFT:
		upt_slideshow_next (slideshow);
		upt_slideshow_draw (canvas, slideshow);
		break;
	      case GII_PBUTTON_RIGHT:
		upt_slideshow_previous (slideshow);
		upt_slideshow_draw (canvas, slideshow);
		break;
	      }
	  else if (ev.any.type == evKeyPress)
	    switch (ev.key.sym)
	      {
	      case GIIUC_Space:
	      case 'n': case 'N':
		upt_slideshow_next (slideshow);
		upt_slideshow_draw (canvas, slideshow);
		break;
	      case GIIUC_Delete: case GIIUC_BackSpace:
	      case 'p': case 'P':
		upt_slideshow_previous (slideshow);
		upt_slideshow_draw (canvas, slideshow);
		break;
	      case GIIUC_Escape:
	      case 'q': case 'Q':
	      case 'x': case 'X':
		goto finish;
	      }
	}
    }

 finish:
  ggiClose (canvas->visual);
  ggiExit ();

  return 0;
}
