/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SiteconIOWorkers.h"
#include "SiteconWorkers.h"
#include "SiteconSearchDialogController.h"
#include <U2Lang/Datatype.h>
#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Designer/DelegateEditors.h>
#include <U2Lang/CoreLibConstants.h>

#include <U2Core/AppContext.h>
#include <U2Core/IOAdapter.h>
#include <U2Core/Log.h>
#include <U2Core/GUrlUtils.h>
#include <U2Core/FailTask.h>
#include "SiteconIO.h"

/* TRANSLATOR U2::SiteconIO */

namespace U2 {
namespace LocalWorkflow {

const QString SiteconReader::ACTOR_ID("sitecon-read");
const QString SiteconWriter::ACTOR_ID("sitecon-write");

static const QString SITECON_OUT_PORT_ID("out-sitecon");
static const QString SITECON_IN_PORT_ID("in-sitecon");

const QString SiteconWorkerFactory::SITECON_MODEL_TYPE_ID("sitecon.model");

DataTypePtr const SiteconWorkerFactory::SITECON_MODEL_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(SITECON_MODEL_TYPE_ID, SiteconIO::tr("Sitecon model"), "")));
        startup = false;
    }
    return dtr->getById(SITECON_MODEL_TYPE_ID);
}

const Descriptor SiteconWorkerFactory::SITECON_SLOT("sitecon-model", SiteconIO::tr("Sitecon model"), "");

const Descriptor SiteconWorkerFactory::SITECON_CATEGORY() {return Descriptor("hsitecon", SiteconIO::tr("SITECON"), "");}

SiteconIOProto::SiteconIOProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: IntegralBusActorPrototype(_desc, _ports, _attrs) {
}

bool SiteconIOProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params, const QString & urlAttrId ) const {
    if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            QString url = urls.at(0).toLocalFile();
            QString ext = GUrlUtils::getUncompressedExtension(GUrl(url, GUrl_File));
            if (SiteconIO::SITECON_EXT == ext) {
                if (params) {
                    params->insert(urlAttrId, url);
                }
                return true;
            }
        }
    }
    return false;
}

ReadSiteconProto::ReadSiteconProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: SiteconIOProto( _desc, _ports, _attrs ) {
    
    attrs << new Attribute(CoreLibConstants::URL_IN_ATTR(), CoreDataTypes::STRING_TYPE(), true);
    QMap<QString, PropertyDelegate*> delegateMap;
    delegateMap[CoreLibConstants::URL_IN_ATTR().getId()] = new URLDelegate(SiteconIO::getFileFilter(), SiteconIO::SITECON_ID, true);
    setEditor(new DelegateEditor(delegateMap));
    setIconPath(":sitecon/images/sitecon.png");
}

bool ReadSiteconProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return SiteconIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_IN_ATTR().getId() );
}

WriteSiteconProto::WriteSiteconProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: SiteconIOProto( _desc, _ports, _attrs ) {
    attrs << new Attribute(CoreLibConstants::URL_OUT_ATTR(), CoreDataTypes::STRING_TYPE(), true );
    attrs << new Attribute(BioActorLibrary::FILE_MODE_ATTR(), CoreDataTypes::NUM_TYPE(), false, SaveDoc_Roll);

    QMap<QString, PropertyDelegate*> delegateMap;
    delegateMap[CoreLibConstants::URL_OUT_ATTR().getId()] = new URLDelegate(SiteconIO::getFileFilter(), SiteconIO::SITECON_ID, false );
    delegateMap[BioActorLibrary::FILE_MODE_ATTR_ID] = new FileModeDelegate(false);
    
    setEditor(new DelegateEditor(delegateMap));
    setIconPath(":sitecon/images/sitecon.png");
    setValidator(new ScreenedParamValidator(CoreLibConstants::URL_OUT_ATTR().getId(), ports.first()->getId(), CoreLibConstants::URL_SLOT().getId()));
    setPortValidator(SITECON_IN_PORT_ID, new ScreenedSlotValidator(CoreLibConstants::URL_SLOT().getId()));
}

bool WriteSiteconProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return SiteconIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_OUT_ATTR().getId() );
}

void SiteconWorkerFactory::init() 
{
    ActorPrototypeRegistry* r = WorkflowEnv::getProtoRegistry();
    assert(r);
    {        
        QMap<Descriptor, DataTypePtr> m;
        m[CoreLibConstants::URL_SLOT()] = CoreDataTypes::STRING_TYPE();
        m[SiteconWorkerFactory::SITECON_SLOT] = SiteconWorkerFactory::SITECON_MODEL_TYPE();
        DataTypePtr t(new MapDataType(Descriptor("write.sitecon.content"), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(SITECON_IN_PORT_ID, SiteconIO::tr("Sitecon model"), SiteconIO::tr("Input Sitecon model"));
        p << new PortDescriptor(pd, t, true /*input*/);
        Descriptor desc(SiteconWriter::ACTOR_ID, SiteconIO::tr("Write SITECON model"), SiteconIO::tr("Saves all input SITECON profiles to specified location."));
        IntegralBusActorPrototype* proto = new WriteSiteconProto(desc, p, a);
        proto->setPrompter(new SiteconWritePrompter());
        r->registerProto(BioActorLibrary::CATEGORY_TRANSCRIPTION(), proto);
    }
    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(SITECON_OUT_PORT_ID, SiteconIO::tr("Sitecon model"), SiteconIO::tr("Loaded SITECON profile data."));
        
        QMap<Descriptor, DataTypePtr> outM;
        outM[SiteconWorkerFactory::SITECON_SLOT] = SiteconWorkerFactory::SITECON_MODEL_TYPE();
        p << new PortDescriptor(pd, DataTypePtr(new MapDataType("sitecon.read.out", outM)), false /*input*/, true /*multi*/);
        
        Descriptor desc(SiteconReader::ACTOR_ID, SiteconIO::tr("Read SITECON model"), SiteconIO::tr("Reads SITECON profiles from file(s). The files can be local or Internet URLs."));
        IntegralBusActorPrototype* proto = new ReadSiteconProto(desc, p, a);
        proto->setPrompter(new SiteconReadPrompter());
        r->registerProto(BioActorLibrary::CATEGORY_TRANSCRIPTION(), proto);
    }

    SiteconBuildWorker::registerProto();
    SiteconSearchWorker::registerProto();

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconReader::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconWriter::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconSearchWorker::ACTOR_ID));
    localDomain->registerEntry(new SiteconWorkerFactory(SiteconBuildWorker::ACTOR_ID));
}

Worker* SiteconWorkerFactory::createWorker(Actor* a) {
    BaseWorker* w = NULL;
    if (SiteconReader::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconReader(a);
    } 
    else if (SiteconWriter::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconWriter(a);
    }
    else if (SiteconBuildWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconBuildWorker(a);
    }
    else if (SiteconSearchWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new SiteconSearchWorker(a);
    }

    return w;    
}

QString SiteconReadPrompter::composeRichDoc() {
    return tr("Read model(s) from <u>%1</u>").arg(getURL(CoreLibConstants::URL_IN_ATTR().getId()));
}

QString SiteconWritePrompter::composeRichDoc() {
    IntegralBusPort* input = qobject_cast<IntegralBusPort*>(target->getPort(SITECON_IN_PORT_ID));
    Actor* producer = input->getProducer(SiteconWorkerFactory::SITECON_MODEL_TYPE_ID);
    if (!producer) {
        return getURL(CoreLibConstants::URL_OUT_ATTR().getId());
    }
    QString url = getScreenedURL(input, CoreLibConstants::URL_OUT_ATTR().getId(), CoreLibConstants::URL_SLOT().getId()); 
    QString doc = tr("Save the profile(s) from <u>%1</u> to %2.")
        .arg(producer->getLabel())
        .arg(url);
    return doc;
}

void SiteconReader::init() {
    output = ports.value(SITECON_OUT_PORT_ID);
    urls = WorkflowUtils::expandToUrls(actor->getParameter(CoreLibConstants::URL_IN_ATTR().getId())->getAttributeValue<QString>());
    mtype = SiteconWorkerFactory::SITECON_MODEL_TYPE();
}

Task* SiteconReader::tick() {
    Task* t = new SiteconReadTask(urls.takeFirst());
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    tasks.append(t);
    return t;
}

void SiteconReader::sl_taskFinished() {
    SiteconReadTask* t = qobject_cast<SiteconReadTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    tasks.removeAll(t);
    if (output) {
        if (!t->hasErrors()) {
            QVariant v = qVariantFromValue<SiteconModel>(t->getResult());
            output->put(Message(mtype, v));
        }
        if (urls.isEmpty() && tasks.isEmpty()) {
            output->setEnded();
        }
        algoLog.info(tr("Loaded SITECON model from %1").arg(t->getURL()));
    }
}

void SiteconWriter::init() {
    input = ports.value(SITECON_IN_PORT_ID);
}

Task* SiteconWriter::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    url = actor->getParameter(CoreLibConstants::URL_OUT_ATTR().getId())->getAttributeValue<QString>();
    fileMode = actor->getParameter(BioActorLibrary::FILE_MODE_ATTR_ID)->getAttributeValue<uint>();
    QVariantMap data = inputMessage.getData().toMap();
    SiteconModel model = data.value(SiteconWorkerFactory::SITECON_SLOT.getId()).value<SiteconModel>();
    
    QString anUrl = url;
    if (anUrl.isEmpty()) {
        anUrl = data.value(CoreLibConstants::URL_SLOT().getId()).toString();
    }
    if (anUrl.isEmpty()) {
        QString err = tr("Unspecified URL for writing Sitecon");
        //if (failFast) {
            return new FailTask(err);
        /*} else {
            algoLog.error(err);
            return NULL;
        }*/
    }
    assert(!anUrl.isEmpty());
    int count = ++counter[anUrl];
    if (count != 1) {
        anUrl = GUrlUtils::prepareFileName(anUrl, count, QStringList("sitecon"));
    } else {
        anUrl = GUrlUtils::ensureFileExt( anUrl, QStringList("sitecon")).getURLString();
    }
    ioLog.info(tr("Writing SITECON model to %1").arg(anUrl));
    return new SiteconWriteTask(anUrl, model, fileMode);
}

} //namespace LocalWorkflow
} //namespace U2

