/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <math.h>
#include <cassert>

#include "uhmm3SearchResult.h"

using namespace U2;

static UHMM3SearchSeqDomainResult getDomainRes( const P7_HIT* hit, int dom, const P7_PIPELINE* pli ) {
    assert( NULL != hit );
    assert( 0 <= dom && dom < hit->ndom );
    UHMM3SearchSeqDomainResult res;
    const P7_DOMAIN& domain = hit->dcl[dom];
    
    res.isSignificant = domain.is_included;
    res.score       = domain.bitscore;
    res.bias        = domain.dombias;
    res.ival        = domain.pvalue * pli->Z;
    res.cval        = domain.pvalue * pli->domZ;
    res.queryRegion = LRegion( domain.ad->hmmfrom, domain.ad->hmmto - domain.ad->hmmfrom );
    res.seqRegion   = LRegion( domain.ad->sqfrom - 1, domain.ad->sqto - domain.ad->sqfrom + 1 );
    res.envRegion   = LRegion( domain.ienv, domain.jenv - domain.ienv );
    res.acc         = domain.oasc / ( 1.0 + fabs( (float)( domain.jenv - domain.ienv ) ) );
    return res;
}

namespace U2 {

void UHMM3SearchSeqDomainResult::writeQualifiersToAnnotation( AnnotationData * annData ) const {
    assert( NULL != annData );
    annData->qualifiers << Qualifier( "Independent e-value", QString().sprintf( "%.5e", ival ) );
    annData->qualifiers << Qualifier( "Conditional e-value", QString().sprintf( "%.5e", cval ) );
    annData->qualifiers << Qualifier( "Score", QString().sprintf( "%1.f", score ) );
    annData->qualifiers << Qualifier( "Bias", QString().sprintf( "%.5e", bias ) );
    annData->qualifiers << Qualifier( "Accuracy per residue", QString().sprintf( "%.5e", acc ) );
    annData->qualifiers << Qualifier( "HMM region", QString().sprintf( "%d...%d", queryRegion.startPos, queryRegion.endPos()) );
    annData->qualifiers << Qualifier( "Envelope of domain location", QString().sprintf( "%d...%d", 
        envRegion.startPos, envRegion.endPos() ) );
}

void UHMM3SearchResult::fillDomainsResult( const P7_TOPHITS* th, const P7_PIPELINE* pli ) {
    assert( NULL != th && ( 0 == th->N || 1 == th->N ) );
    assert( NULL != pli );
    
    if( 0 == th->N ) {
        return;
    }
    
    P7_HIT* hit = th->hit[0];
    
    int d = 0;
    for( d = 0; d < hit->ndom; d++ ) {
        if( hit->dcl[d].is_reported ) {
            domainResList << getDomainRes( hit, d, pli );
        }
    }
}

void UHMM3SearchResult::fillFullSeqResults( const P7_TOPHITS* th, const P7_PIPELINE* pli ) {
    assert( NULL != th && ( 1 == th->N || 0 == th->N ) );
    assert( NULL != pli );
    fullSeqResult.isReported = th->nreported != 0;
    if( !th->N ) {
        assert( !fullSeqResult.isReported );
        return;
    }
    P7_HIT* hit = th->hit[0];
    if( !(hit->flags & p7_IS_REPORTED) ) {
        return;
    }
    
    fullSeqResult.eval     = hit->pvalue * pli->Z;
    fullSeqResult.score    = hit->score;
    fullSeqResult.bias     = hit->pre_score - hit->score;
    fullSeqResult.expectedDomainsNum = hit->nexpected;
    fullSeqResult.reportedDomainsNum = hit->nreported;
}

void UHMM3SearchResult::fillResults( const P7_TOPHITS* th, const P7_PIPELINE* pli ) {
    assert( NULL != th );
    assert( NULL != pli );
    fillFullSeqResults( th, pli );
    fillDomainsResult(  th, pli );
}

} // U2
