/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <cassert>

#include <U2Lang/ActorModel.h>
#include "SchemaAliasesConfigurationDialogImpl.h"

namespace U2 {
namespace Workflow {

SchemaAliasesConfigurationDialogImpl::SchemaAliasesConfigurationDialogImpl( const Schema & schema, QWidget * p )
: QDialog(p) {
    setupUi(this);
    
    connect( cancelPushButton, SIGNAL(clicked()), SLOT(reject()) );
    connect( okPushButton, SIGNAL(clicked()), SLOT(accept()));
    
    okPushButton->setDefault(true);
    paramAliasesTableWidget->verticalHeader()->hide();
    paramAliasesTableWidget->horizontalHeader()->setClickable(false);
    paramAliasesTableWidget->horizontalHeader()->setStretchLastSection( true );
    
    foreach( Actor * actor, schema.getProcesses() ) {
        assert( actor != NULL );
        int pos = procsListWidget->count();
        procsListWidget->insertItem( pos, new QListWidgetItem( actor->getLabel() ) );
        procListMap.insert( pos, actor->getId() );
    }
    connect( procsListWidget, SIGNAL(currentRowChanged( int )), SLOT(sl_procSelected( int )) );
    connect( paramAliasesTableWidget, SIGNAL( cellChanged(int, int) ), SLOT(sl_aliasChanged(int, int)) );
    
    initializeModel(schema);
}

void SchemaAliasesConfigurationDialogImpl::initializeModel( const Schema & schema ) {
    foreach( Actor * actor, schema.getProcesses() ) {
        assert( actor != NULL );
        QMap<Descriptor, QString> aliasMap;
        foreach( Attribute * attr, actor->getParameters().values() ) {
            assert( attr != NULL );
            aliasMap.insert( *attr, actor->getParamAliases().value( attr->getId() ) );
        }
        model.insert( actor->getId(), aliasMap );
    }
}

SchemaAliasesCfgDlgModel SchemaAliasesConfigurationDialogImpl::getModel() const {
    SchemaAliasesCfgDlgModel ret;
    SchemaAliasesCfgDlgModel::const_iterator it = model.constBegin();
    while( it != model.constEnd() ) {
        QMap<Descriptor, QString> aliases;
        QMap<Descriptor, QString>::const_iterator aliasesIt = it.value().constBegin();
        while( aliasesIt != it.value().constEnd() ) {
            if( !aliasesIt.value().isEmpty() ) {
                aliases.insert( aliasesIt.key(), aliasesIt.value() );
            }
            ++aliasesIt;
        }
        ret.insert( it.key(), aliases );
        ++it;
    }
    return ret;
}

bool SchemaAliasesConfigurationDialogImpl::validateModel() const {
    SchemaAliasesCfgDlgModel m = getModel();
    QStringList allAliases;
    SchemaAliasesCfgDlgModel::const_iterator it = m.constBegin();
    while( it != m.constEnd() ) {
        QMap<Descriptor, QString>::const_iterator aliasIt = it.value().constBegin();
        while( aliasIt != it.value().constEnd() ) {
            allAliases << aliasIt.value();
            ++aliasIt;
        }
        ++it;
    }
    return allAliases.removeDuplicates() == 0;
}

void SchemaAliasesConfigurationDialogImpl::clearAliasTable() {
    paramAliasesTableWidget->clearContents();
    paramAliasesTableWidget->setRowCount(0);
}

void SchemaAliasesConfigurationDialogImpl::sl_procSelected( int row ) {
    if( row == -1 ) {
        return;
    }
    clearAliasTable();
    
    assert( row >= 0 && row < procsListWidget->count() );
    ActorId currentActor = procListMap.value( row );
    assert( !currentActor.isEmpty() );
    
    int rowInd = 0;
    QMap<Descriptor, QString> aliasMap = model.value( currentActor );
    QMap<Descriptor, QString>::const_iterator it = aliasMap.constBegin();
    while( it != aliasMap.constEnd() ) {
        paramAliasesTableWidget->insertRow( rowInd );
        
        QTableWidgetItem * keyItem = new QTableWidgetItem( it.key().getDisplayName() ) ;
        paramAliasesTableWidget->setItem( rowInd, 0, keyItem );
        keyItem->setData( Qt::UserRole, qVariantFromValue<Descriptor>( it.key() ) );
        keyItem->setFlags( keyItem->flags() ^ Qt::ItemIsSelectable ^ Qt::ItemIsEditable );
        
        QTableWidgetItem * valItem = new QTableWidgetItem( it.value() );
        paramAliasesTableWidget->setItem( rowInd, 1, valItem );
        
        rowInd++;
        ++it;
    }
    paramAliasesTableWidget->resizeColumnToContents(0);
}

void SchemaAliasesConfigurationDialogImpl::sl_aliasChanged( int row, int col ) {
    assert( row >= 0 && row < paramAliasesTableWidget->rowCount() );
    if( col != 1 ) {
        return;
    }
    
    ActorId currentActor = procListMap.value( procsListWidget->currentRow() );
    assert( !currentActor.isEmpty() );
    
    Descriptor desc = qVariantValue<Descriptor>( paramAliasesTableWidget->item( row, 0 )->data( Qt::UserRole ) );
    assert( model[currentActor].contains( desc ) );
    model[currentActor][desc] = paramAliasesTableWidget->item( row, 1 )->text();
}

} // Workflow
} // U2
