/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtXml/qdom.h>
#include <QtCore/QPointF>

#include <U2Core/Log.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Core/QVariantUtils.h>
#include <U2Lang/WorkflowUtils.h>
#include <U2Lang/WorkflowSettings.h>

#include "WorkflowViewController.h"
#include "WorkflowViewItems.h"
#include "SceneSerializer.h"

/* TRANSLATOR U2::LocalWorkflow::WorkflowView */

namespace U2 {
using namespace Workflow;
static const QString WORKFLOW_EL = "workflow";
static const QString PROCESS_EL = "process";
static const QString PORT_EL = "port";
static const QString PARAMS_EL = "params";
static const QString DATAFLOW_EL = "dataflow";
static const QString POSITION_ATTR = "pos";
static const QString ID_ATTR = "id";
static const QString NAME_ATTR = "name";
static const QString NUM_IN_SCHEMA = "numberInSchema";
static const QString TYPE_ATTR = "type";
static const QString SRC_PORT_ATTR = "sprt";
static const QString SRC_PROC_ATTR = "sprc";
static const QString DST_PORT_ATTR = "dprt";
static const QString DST_PROC_ATTR = "dprc";
static const QString SCRIPT_TEXT = "scriptText";

static void saveProcess(const WorkflowProcessItem* pit, QDomElement& proj) {
    QDomElement docElement = SchemaSerializer::saveActor(pit->getProcess(), proj);
    pit->saveState(docElement);
    foreach(WorkflowPortItem* iot, pit->getPortItems()) {
        Port* port = iot->getPort();
        QDomElement portElement = SchemaSerializer::savePort(port, docElement);
        portElement.setAttribute(POSITION_ATTR, iot->getOrientarion());
    }
}

static void saveFlow(const WorkflowBusItem* dit, QDomElement& proj) {
    QDomElement el = SchemaSerializer::saveLink(dit->getBus(), proj);
    dit->saveState(el);
}

void SceneSerializer::scene2xml(const WorkflowScene* scene, QDomDocument& xmlDoc){
    QDomElement projectElement = xmlDoc.createElement(WORKFLOW_EL);
    // TODO save scale and view rect??  
    xmlDoc.appendChild(projectElement);
    saveItems(scene->items(), projectElement);
}

void SceneSerializer::saveItems(const QList<QGraphicsItem*>& items, QDomElement& proj) {
    foreach(QGraphicsItem* item, items) {
        switch (item->type()) {
        case WorkflowProcessItemType:
            saveProcess(qgraphicsitem_cast<WorkflowProcessItem*>(item), proj);
            break;
        case WorkflowBusItemType:
            saveFlow(static_cast<WorkflowBusItem*>(item), proj);
            break;
        }
    }
}

static void initProcMap(QMap<ActorId, WorkflowProcessItem*> & procMap, WorkflowScene* scene) {
    foreach(QGraphicsItem* item, scene->items()) {
        if (item->type() == WorkflowProcessItemType) {
            WorkflowProcessItem* wit = qgraphicsitem_cast<WorkflowProcessItem*>(item);
            procMap[wit->getProcess()->getId()] = wit;
        }
    }
}

QString SceneSerializer::xml2scene(const QDomElement& projectElement, WorkflowScene* scene, QMap<ActorId, ActorId>& remapping, 
                                                 bool ignoreErrors, bool select) {
    QMap<ActorId, WorkflowProcessItem*> procMap;
    QMap<ActorId, Actor*> actorMap;
    initProcMap(procMap, scene);
    
    ActorPrototypeRegistry* registry = WorkflowEnv::getProtoRegistry();
    
    QDomNodeList procNodes = projectElement.elementsByTagName(PROCESS_EL);
    for(int i=0; i<procNodes.size(); i++) {
        QDomNode n = procNodes.item(i);
        assert(n.isElement());
        if (!n.isElement()) {
            continue;
        }
        QDomElement procElement = n.toElement();
        
        const ActorId id = str2aid(procElement.attribute(ID_ATTR));
        if (!ignoreErrors && procMap.contains(id)) {
            return WorkflowView::tr("Invalid content: duplicate process %1").arg(id);
        }
        
        const QString name = SchemaSerializer::getElemType(procElement.attribute(TYPE_ATTR));
        ActorPrototype* proto = registry->getProto(name);
        if (!proto) {
            return WorkflowView::tr("Invalid content: unknown process type %1").arg(name);
        }
        
        AttributeScript *script;
        const QString scriptText = procElement.attribute(SCRIPT_TEXT);
        if(scriptText.isEmpty()) {
            script = NULL;
        }
        else {
            script = new AttributeScript();
            script->setScriptText(scriptText);
        }
        Actor* proc = proto->createInstance(script);
        actorMap[id] = proc;
        SchemaSerializer::readConfiguration(proc, procElement);
        SchemaSerializer::readParamAliases( proc->getParamAliases(), procElement );
        proc->setLabel(procElement.attribute(NAME_ATTR));
        WorkflowProcessItem* it = new WorkflowProcessItem(proc);
        it->loadState(procElement);
        scene->addItem(it);
        if (select) {
            it->setSelected(true);
        }
        procMap[id] = it;
        
        //read port params
        QDomNodeList nl = procElement.elementsByTagName(PORT_EL);
        for(int i=0; i<nl.size(); i++) {
            QDomElement el = nl.item(i).toElement();
            if (el.isNull()) {
                continue;
            }
            QString id = SchemaSerializer::getElemPort(name, el.attribute(ID_ATTR));
            WorkflowPortItem* p = it->getPort(id);
            if (!p) {
                if (!ignoreErrors) {
                    return WorkflowView::tr("Invalid content: unknown port %1 requested for %2").arg(id).arg(name);
                } else {
                    continue;
                }
            }
            SchemaSerializer::readConfiguration(p->getPort(), el);
            if (el.hasAttribute(POSITION_ATTR)) {
                p->setOrientation(el.attribute(POSITION_ATTR).toDouble());
            }
        }
    }
    
    QMapIterator<ActorId, Actor*> it(actorMap);
    while(it.hasNext()) {
        it.next();
        remapping[it.key()] = it.value()->getId();
    }
    foreach(Actor* a, actorMap) {
        a->remap(remapping);
    }


    QDomNodeList flowNodes = projectElement.elementsByTagName(DATAFLOW_EL);
    for(int i=0; i<flowNodes.size(); i++) {
        QDomNode n = flowNodes.item(i);
        assert(n.isElement());
        if (!n.isElement()) {
            continue;
        }
        QDomElement flowElement = n.toElement();
        const ActorId inId = str2aid(flowElement.attribute(DST_PROC_ATTR));
        const ActorId outId = str2aid(flowElement.attribute(SRC_PROC_ATTR));

        if (!ignoreErrors && !procMap.contains(inId)) {
            return WorkflowView::tr("Invalid content: no such process %1 to bind").arg(inId);
        }
        if (!ignoreErrors && !procMap.contains(outId)) {
            return WorkflowView::tr("Invalid content: no such process %1 to bind").arg(outId);
        }
        QString inP = flowElement.attribute(DST_PORT_ATTR);
        QString outP = flowElement.attribute(SRC_PORT_ATTR);

        WorkflowPortItem* input = procMap[inId]->getPort(inP);
        WorkflowPortItem* output = procMap[outId]->getPort(outP);
        if ((!input || !output || !input->tryBind(output)) && !ignoreErrors) {
            return WorkflowView::tr("Invalid content: cannot bind [%1 : %2] to [%3 : %4]").
                arg(inId).arg(inP).arg(outId).arg(outP);
        }
        WorkflowBusItem* dit = input->getDataFlow(output);
        if (dit) dit->loadState(flowElement);
        if (select) {
            if (dit) dit->setSelected(true);
        }
    }
    SchemaSerializer::updatePortBindings(actorMap.values());
    return QString();
}

}//namespace
