/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SWQuery.h"

#include <U2Core/AppContext.h>
#include <U2Core/DNATranslation.h>
#include <U2Core/PluginModel.h>
#include <U2Core/TaskSignalMapper.h>
#include <U2Core/FailTask.h>
#include <U2Core/L10n.h>

#include <U2Algorithm/SmithWatermanTaskFactoryRegistry.h>
#include <U2Algorithm/SubstMatrixRegistry.h>
#include <U2Algorithm/SWResultFilterRegistry.h>

#include <U2Lang/Attribute.h>
#include <U2Lang/ConfigurationEditor.h>
#include <U2Lang/CoreDataTypes.h>

#include <QtCore/qmath.h>

namespace U2 {

static const QString PATTERN_ATTR("pattern");
static const QString SCORE_ATTR("min-score");
static const QString MATRIX_ATTR("matrix");
static const QString AMINO_ATTR("translate");
static const QString STRAND_ATTR("strand");
static const QString ALGO_ATTR("algorithm");
static const QString FILTER_ATTR("filter");
static const QString GAPOPEN_ATTR("gap-open-score");
static const QString GAPEXT_ATTR("gap-ext-score");

void SWAlgoEditor::populate() {
    QStringList algoLst = AppContext::getSmithWatermanTaskFactoryRegistry()->getListFactoryNames();
    if (algoLst.isEmpty()) {
        return;
    }
    foreach(const QString& n, algoLst) {
        items.insert(n,n);
    }

    algAttr->setAttributeValue(algoLst.first());
}

/************************************************************************/
/* QDSWActor                                                              */
/************************************************************************/

QDSWActor::QDSWActor(QDActorPrototype const* proto) : QDActor(proto), algo(0) {
    units["sw"] = new QDSchemeUnit(this);
    algo = NULL;
}

int QDSWActor::getMinResultLen() const {
    return cfg->getParameter(PATTERN_ATTR)->getAttributeValue<QString>().toAscii().size()/2;
}

int QDSWActor::getMaxResultLen() const {
    return 4*getMinResultLen();
}

QString QDSWActor::getText() const {
    QMap<QString, Attribute*> params = cfg->getParameters();

    QString pattern = params.value(PATTERN_ATTR)->getAttributeValue<QString>().toAscii().toUpper();
    if (pattern.isEmpty()) {
        pattern = "unset";
    }

    pattern = QString("<a href=%1>%2</a>").arg(PATTERN_ATTR).arg(pattern);

    int percentOfScore = params.value(SCORE_ATTR)->getAttributeValue<int>();
    QString percentOfScoreStr = QString("<a href=%1>%2%</a>").arg(SCORE_ATTR).arg(percentOfScore);
    QString match = percentOfScore < 100 ? 
        QDSWActor::tr("matches with <u>at least %1 score</u>").arg(percentOfScoreStr) : QDSWActor::tr("exact matches");

    StrandOption strand = StrandOption(params.value(STRAND_ATTR)->getAttributeValue<int>());
    QString strandName;
    switch (strand) {
        case StrandOption_Both: strandName = QDSWActor::tr("both strands"); break;
        case StrandOption_DirectOnly: strandName = QDSWActor::tr("direct strand"); break;
        case StrandOption_ComplementOnly: strandName = QDSWActor::tr("complement strand"); break;
        default: break;
    }

    strandName = QString("<a href=%1>%2</a>").arg(STRAND_ATTR).arg(strandName);


    QString doc = QDSWActor::tr("Finds pattern <u>%1</u>."
        "<br>Looks for <u>%2</u> in <u>%3</u>.")
        .arg(pattern)
        .arg(match)
        .arg(strandName);
    return doc;
}

Task* QDSWActor::getAlgorithmTask(const QList<LRegion>& searchLocation) {
    Task* task = NULL;
    assert(scheme);
    QMap<QString, Attribute*> params = cfg->getParameters();

    settings.aminoTT = NULL;
    settings.complTT = NULL;
    settings.strand = StrandOption(params.value(STRAND_ATTR)->getAttributeValue<int>());
    settings.percentOfScore = params.value(SCORE_ATTR)->getAttributeValue<int>();
    if(settings.percentOfScore<0&&settings.percentOfScore>100) {
        QString err = tr("%1: percent of score out of bounds.").arg(getParameters()->getLabel());
        return new FailTask(err);
    }
    settings.gapModel.scoreGapExtd = params.value(GAPEXT_ATTR)->getAttributeValue<double>();
    settings.gapModel.scoreGapOpen = params.value(GAPOPEN_ATTR)->getAttributeValue<double>();
    mtrx = params.value(MATRIX_ATTR)->getAttributeValue<QString>();
    settings.pSm = AppContext::getSubstMatrixRegistry()->getMatrix(mtrx);
    QString filter = params.value(FILTER_ATTR)->getAttributeValue<QString>();
    settings.resultFilter = AppContext::getSWResultFilterRegistry()->getFilter(filter);
    if(!settings.resultFilter) {
        QString err = tr("%1: incorrect result filter.").arg(getParameters()->getLabel());
        return new FailTask(err);
    }
    settings.ptrn = params.value(PATTERN_ATTR)->getAttributeValue<QString>().toAscii().toUpper();
    if(settings.ptrn.isEmpty()) {
        QString err = tr("%1: pattern is empty.").arg(getParameters()->getLabel());
        return new FailTask(err);
    }
    QString algName = params.value(ALGO_ATTR)->getAttributeValue<QString>();
    algo = AppContext::getSmithWatermanTaskFactoryRegistry()->getFactory(algName);
    if(!algo) {
        QString err = tr("%1: can not find %2.")
            .arg(getParameters()->getLabel())
            .arg(algName);
        return new FailTask(err);
    }

    DNASequenceObject* dna = scheme->getDNA();
    settings.sqnc = QByteArray(dna->getSequence().constData(), dna->getSequenceLen());

    if (settings.strand != StrandOption_DirectOnly) {
        QList<DNATranslation*> compTTs = AppContext::getDNATranslationRegistry()->
            lookupTranslation(dna->getAlphabet(), DNATranslationType_NUCL_2_COMPLNUCL);
        if (!compTTs.isEmpty()) {
            settings.complTT = compTTs.first();
        } else {
            //Could not find complement translation, searching only direct strand
            settings.strand = StrandOption_DirectOnly;
        }
    }

    if (params.value(AMINO_ATTR)->getAttributeValue<bool>()) {
        DNATranslationType tt = (dna->getAlphabet()->getType() == DNAAlphabet_NUCL) ? DNATranslationType_NUCL_2_AMINO : DNATranslationType_RAW_2_AMINO;
        QList<DNATranslation*> TTs = AppContext::getDNATranslationRegistry()->lookupTranslation(dna->getAlphabet(), tt);
        if (!TTs.isEmpty()) {
            settings.aminoTT = TTs.first(); //FIXME let user choose or use hints ?
        }
    }
    assert(dna->getAlphabet()->containsAll(settings.ptrn.constData(),settings.ptrn.length()));
    assert(settings.pSm.getName().isEmpty() && mtrx.toLower() != "auto");

    if (settings.pSm.isEmpty()) {
        QString matrixName;
        QStringList lst = AppContext::getSubstMatrixRegistry()->selectMatrixNamesByAlphabet(dna->getAlphabet());
        if (!lst.isEmpty()) {
            matrixName = lst.first();
            settings.pSm = AppContext::getSubstMatrixRegistry()->getMatrix(matrixName);
        }
        assert(!settings.pSm.isEmpty());
    }

    settings.globalRegion = dna->getSequenceRange();

    //SmithWatermanReportCallbackImpl* rcb = new SmithWatermanReportCallbackImpl(NULL, resultName, QString()); //FIXME!!! where to delete?
    //settings.resultCallback = rcb;
    //settings.resultListener = new SmithWatermanResultListener(); //FIXME: where to delete??

    //task = algo->getTaskInstance(settings, QDSWActor::tr("smith_waterman_task"));
    task = new Task(tr("SSearch"), TaskFlag_NoRun);
    foreach(const LRegion& r, searchLocation) {
        SmithWatermanSettings stngs(settings);
        SmithWatermanReportCallbackImpl* rcb = new SmithWatermanReportCallbackImpl(NULL,"",QString());
        stngs.resultCallback = rcb;
        stngs.resultListener = new SmithWatermanResultListener();
        stngs.globalRegion = r;
        Task* sub = algo->getTaskInstance(stngs, tr("smith_waterman_task"));
        rcb->setParent(sub);
        task->addSubTask(sub);
        callbacks.insert(sub, rcb);
    }
    //callbacks.insert(task, rcb);
    connect(new TaskSignalMapper(task), SIGNAL(si_taskFinished(Task*)), SLOT(sl_onAlgorithmTaskFinished(Task*)));
    return task;
}

void QDSWActor::sl_onAlgorithmTaskFinished(Task*) {
    /*SmithWatermanReportCallbackImpl* rcb = callbacks.take(t);
    assert(rcb);*/
    //const QList<SharedAnnotationData>& res = rcb->getAnotations();
    QList<SharedAnnotationData> res;
    QMapIterator<Task*, SmithWatermanReportCallbackImpl*> iter(callbacks);
    while (iter.hasNext()) {
        iter.next();
        res << iter.value()->getAnotations();
    }

    foreach(const SharedAnnotationData& ad, res) {
        QDResultUnit ru(new QDResultUnitData);
        ru->complement = ad.data()->complement;
        ru->quals = ad.data()->qualifiers;
        ru->region = ad.data()->location.first();
        ru->owner = units.value("sw");
        QDResultGroup* g = new QDResultGroup;
        g->add(ru);
        results.append(g);
    }
    callbacks.clear();
}

static const QString STRAND_BOTH = "both";
static const QString STRAND_COMPL = "complement";
static const QString STRAND_DIRECT = "direct";

QList< QPair<QString,QString> > QDSWActor::saveConfiguration() const {
    QList< StringAttribute > res = QDActor::saveConfiguration();
    Attribute* a = cfg->getParameter(STRAND_ATTR);
    for(int i=0; i< res.size(); i++) {
        StringAttribute& attr = res[i];
        if (attr.first==a->getId()) {
            StrandOption strand = StrandOption(a->getAttributeValue<int>());
            switch (strand) {
            case StrandOption_Both:
                attr.second = STRAND_BOTH;
                break;
            case StrandOption_ComplementOnly:
                attr.second = STRAND_COMPL;
                break;
            case StrandOption_DirectOnly:
                attr.second = STRAND_DIRECT;
                break;
            default:
                break;
            }
        }
    }
    return res;
}

void QDSWActor::loadConfiguration(const QList< QPair<QString, QString> >& strMap) {
    QDActor::loadConfiguration(strMap);
    foreach(const StringAttribute& attr, strMap) {
        if (attr.first==STRAND_ATTR) {
            int strand;
            const QString& strandVal = attr.second;
            if (strandVal==STRAND_BOTH) {
                strand = 2;
            }
            else if (strandVal==STRAND_COMPL) {
                strand = 1;
            }
            else if (strandVal==STRAND_DIRECT) {
                strand = 0;
            }
            cfg->setParameter(STRAND_ATTR, qVariantFromValue(strand));
        }
    }
}

SWQDActorFactory::SWQDActorFactory() {
    descriptor.setId("ssearch");
    descriptor.setDisplayName(QDSWActor::tr("Smith-Waterman"));
    descriptor.setDocumentation(QDSWActor::tr("Finds regions of similarity to the specified pattern in each input sequence (nucleotide or protein one). "
        "<p>Under the hood is the well-known Smith-Waterman algorithm for performing local sequence alignment."));

    QStringList filterLst = AppContext::getSWResultFilterRegistry()->getFiltersIds();
    QString defFilter = filterLst.isEmpty() ? QString() : filterLst.first();

    Descriptor pd(PATTERN_ATTR, QDSWActor::tr("Pattern"), QDSWActor::tr("A subsequence pattern to look for."));
    Descriptor scd(SCORE_ATTR, QDSWActor::tr("Min score"), QDSWActor::tr("The search stringency."));
    Descriptor ald(ALGO_ATTR, QDSWActor::tr("Algorithm"), QDSWActor::tr("Algorithm version."));
    Descriptor amd(AMINO_ATTR, QDSWActor::tr("Search in translation"), QDSWActor::tr("Translate a supplied nucleotide sequence to protein then search in the translated sequence."));
    Descriptor sd(STRAND_ATTR, L10N::searchIn(), QDSWActor::tr("Which strands should be searched: direct, complement or both."));
    Descriptor mxd(MATRIX_ATTR, QDSWActor::tr("Scoring matrix"), QDSWActor::tr("The scoring matrix."));
    Descriptor frd(FILTER_ATTR, QDSWActor::tr("Filter results"), QDSWActor::tr("Result filtering strategy."));
    Descriptor god(GAPOPEN_ATTR, QDSWActor::tr("Gap open score"), QDSWActor::tr("Gap open score."));
    Descriptor ged(GAPEXT_ATTR, QDSWActor::tr("Gap ext score"), QDSWActor::tr("Gap extension score."));

    attributes << new Attribute(pd, CoreDataTypes::STRING_TYPE(), true);
    attributes << new Attribute(mxd, CoreDataTypes::STRING_TYPE(), true, QString("---"));
    Attribute* algAttr = new Attribute(ald, CoreDataTypes::STRING_TYPE(), true);
    attributes << algAttr;
    attributes << new Attribute(frd, CoreDataTypes::STRING_TYPE(), false, defFilter);
    attributes << new Attribute(scd, CoreDataTypes::NUM_TYPE(), false, 90);
    attributes << new Attribute(sd, CoreDataTypes::NUM_TYPE(), false, StrandOption_Both);
    attributes << new Attribute(amd, CoreDataTypes::BOOL_TYPE(), false, false);
    attributes << new Attribute(god, CoreDataTypes::NUM_TYPE(), false, -10.);
    attributes << new Attribute(ged, CoreDataTypes::NUM_TYPE(), false, -1.);

    QMap<QString, PropertyDelegate*> delegates;    
    {
        QVariantMap m; m["minimum"] = 1; m["maximum"] = 100; m["suffix"] = "%";
        delegates[SCORE_ATTR] = new SpinBoxDelegate(m);
    }    
    {
        QVariantMap m; m["maximum"] = -0.; m["minimum"]=-10000000.;
        delegates[GAPOPEN_ATTR] = new DoubleSpinBoxDelegate(m);
        m["maximum"] = -1.;
        delegates[GAPEXT_ATTR] = new DoubleSpinBoxDelegate(m);
    }    
    {
        QVariantMap strandMap; 
        strandMap[QDSWActor::tr("both strands")] = StrandOption_Both;
        strandMap[QDSWActor::tr("direct strand")] = StrandOption_DirectOnly;
        strandMap[QDSWActor::tr("complement strand")] = StrandOption_ComplementOnly;
        delegates[STRAND_ATTR] = new ComboBoxDelegate(strandMap);
    }
    {
        QVariantMap m;   
        foreach(const QString& n, filterLst) {
            m.insert(n,n);
        } 
        delegates[FILTER_ATTR] = new ComboBoxDelegate(m);
    }
    {
        QVariantMap m; m.insert(QDSWActor::tr("Auto"), QString("---"));
        QStringList lst = AppContext::getSubstMatrixRegistry()->getMatrixNames();	
        foreach(const QString& n, lst) {
            m.insert(n,n);
        } 
        delegates[MATRIX_ATTR] = new ComboBoxDelegate(m);
    }

    SWAlgoEditor* aled = new SWAlgoEditor(algAttr);
    aled->connect(AppContext::getPluginSupport(), SIGNAL(si_allStartUpPluginsLoaded()), SLOT(populate()));
    aled->populate();
    delegates[ALGO_ATTR] = aled;

    editor = new DelegateEditor(delegates);
}

}//namespace
