/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _U2_FIND_REPEATS_TASK_H_
#define _U2_FIND_REPEATS_TASK_H_

#include "RFBase.h"

#include <U2Core/Task.h>
#include <U2Core/GObjectReference.h>

#include <U2Core/DNASequence.h>
#include <U2Core/AnnotationData.h>

#include <QtCore/QMutex>

namespace U2 {


class FindRepeatsTaskSettings {
public:
    FindRepeatsTaskSettings() : minLen(2), mismatches(0), minDist(0), maxDist(0), 
        inverted(false), reportReflected(false), filterNested(true), maxResults(10*1000*100), 
        algo(RFAlgorithm_Auto), nThreads(MAX_PARALLEL_SUBTASKS_AUTO){}

    int                 minLen;
    int                 mismatches;
    int                 minDist;
    int                 maxDist;
    bool                inverted;
    bool                reportReflected;
    bool                filterNested;
    int                 maxResults;
    LRegion             seqRegion, seq2Region;
    
    //all these regions are in global sequence coordinates
    QVector<LRegion>    midRegionsToInclude;  //reported repeat must contain one of these regions
    QVector<LRegion>    midRegionsToExclude;  //reported repeat must not contain none of these regions
    QVector<LRegion>    allowedRegions;       //reported repeat must fit one of these regions

    RFAlgorithm         algo;
    int                 nThreads;

    void setIdentity(int percent) {mismatches = int((minLen / 100.0) * (100 - percent));}
    int  getIdentity() const {return qBound(50, int(100.0 - mismatches * 100. /minLen), 100);}
    bool hasRegionFilters() const {return !midRegionsToInclude.isEmpty() || !midRegionsToExclude.isEmpty() || !allowedRegions.isEmpty();}

};

//WARNING: this task is suitable only for a single sequence processing -> check addResults x/y sorting
class RevComplSequenceTask;
class FindRepeatsTask : public Task, public RFResultsListener {
Q_OBJECT
public:
    FindRepeatsTask(const FindRepeatsTaskSettings& s, const DNASequence& seq, const DNASequence& seq2);

    void run();
    ReportResult report();
    void cleanup();

    QList<Task*> onSubTaskFinished(Task* subTask);

    virtual void onResult(const RFResult& r);
    virtual void onResults(const QVector<RFResult>& v) ;
    
    QVector<RFResult> getResults() const {return results;} // used if createAnnotations == false
    const FindRepeatsTaskSettings&  getSettings() const {return settings;}

protected:
    void addResult(const RFResult& r);
    bool isFilteredByRegions(const RFResult& r);
    RFAlgorithmBase* createRFTask();
    void filterNestedRepeats();

    FindRepeatsTaskSettings     settings;
    DNASequence                 directSequence, directSequence2;
    QVector<RFResult>           results;
    QMutex                      resultsLock;
    RevComplSequenceTask*       revComplTask;
    RFAlgorithmBase*            rfTask;
    quint64                     startTime;
};

class FindRepeatsToAnnotationsTask : public Task {
    Q_OBJECT
public:
    FindRepeatsToAnnotationsTask(const FindRepeatsTaskSettings& s, const DNASequence& seq, 
                                const QString& annName, const QString& groupName, const GObjectReference& annObjRef);
    
    QList<Task*> onSubTaskFinished(Task* subTask);    
    QList<SharedAnnotationData> importAnnotations();

private:
    QString             annName;
    QString             annGroup;
    GObjectReference    annObjRef;
    FindRepeatsTask*    findTask;
};

class RevComplSequenceTask : public Task {
    Q_OBJECT
public:
    RevComplSequenceTask(const DNASequence& s, const LRegion& reg);
    
    void run();    
    void cleanup();

    DNASequence sequence;
    LRegion     region;
    DNASequence complementSequence;
};

} //namespace

#endif
