/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "QDWorker.h"
#include "QDSceneIOTasks.h"
#include "QueryDesignerPlugin.h"

#include <U2Core/L10n.h>
#include <U2Core/TaskSignalMapper.h>
#include <U2Core/MultiTask.h>
#include <U2Core/DNASequenceObject.h>
#include <U2Core/AnnotationTableObject.h>

#include <U2Lang/IntegralBusModel.h>
#include <U2Lang/WorkflowEnv.h>
#include <U2Lang/ActorPrototypeRegistry.h>
#include <U2Lang/CoreDataTypes.h>
#include <U2Lang/CoreLibConstants.h>
#include <U2Lang/BioActorLibrary.h>
#include <U2Lang/BioDatatypes.h>
#include <U2Lang/QDScheme.h>

#include <U2Designer/DelegateEditors.h>
#include <U2Designer/QDScheduler.h>

#include <U2Misc/DialogUtils.h>

#include <U2Formats/GenbankFeatures.h>


namespace U2 {
namespace LocalWorkflow {

/*************************************************************************
* QDWorkerFactory                                                      
*************************************************************************/

static const QString SCHEMA_ATTR("schema");

const QString QDWorkerFactory::ACTOR_ID("query");

void QDWorkerFactory::init() {
    QList<PortDescriptor*> p; QList<Attribute*> a;

    {
        Descriptor id(CoreLibConstants::IN_SEQ_PORT_ID,
            QDWorker::tr("Input sequences"), 
            QDWorker::tr("A nucleotide sequence to analyze."));

        Descriptor od(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID,
            QDWorker::tr("Result annotations"), 
            QDWorker::tr("A set of annotations marking found results."));

        QMap<Descriptor, DataTypePtr> inM;
        inM[BioActorLibrary::SEQ_SLOT()] = BioDataTypes::DNA_SEQUENCE_TYPE();
        p << new PortDescriptor(id, DataTypePtr(new MapDataType("query.seq", inM)), true /*input*/);

        QMap<Descriptor, DataTypePtr> outM;
        outM[BioActorLibrary::FEATURE_TABLE_SLOT()] = BioDataTypes::ANNOTATION_TABLE_TYPE();
        p << new PortDescriptor(od, DataTypePtr(new MapDataType("query.annotations", outM)), false /*input*/, true /*multi*/);
    }
    {
        Descriptor sd(SCHEMA_ATTR, QDWorker::tr("Schema"), QDWorker::tr("Schema file."));
        a << new Attribute(sd, CoreDataTypes::STRING_TYPE(), true);
    }

    Descriptor desc(ACTOR_ID, QDWorker::tr("Analyze with query"), 
        QDWorker::tr("Analyzes a nucleotide sequence using different algorithms \
                     (Repeat finder, ORF finder, etc.) imposing constraints \
                     on the positional relationship of the results."));

    ActorPrototype* proto = new IntegralBusActorPrototype(desc, p, a);

    QMap<QString, PropertyDelegate*> delegates;
    {
        delegates[SCHEMA_ATTR] = new URLDelegate(
            DialogUtils::prepareFileFilter(QDWorker::tr("Query schemes"), QStringList(QUERY_SCHEME_EXTENSION), true),
            QUERY_DESIGNER_ID,
            false);
    }

    proto->setPrompter(new QDPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":query_designer/images/query_designer.png");
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_BASIC(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new QDWorkerFactory());
}

/******************************
* QDPrompter
******************************/
QString QDPrompter::composeRichDoc() {
    IntegralBusPort* input = qobject_cast<IntegralBusPort*>(target->getPort(CoreLibConstants::IN_SEQ_PORT_ID));
    Actor* producer = input->getProducer(BioActorLibrary::SEQ_SLOT().getId());
    QString unsetStr = "<font color='red'>"+tr("unset")+"</font>";
    QString producerName = tr("from %1").arg(producer ? producer->getLabel() : unsetStr);
    QString schemaFile = getRequiredParam(SCHEMA_ATTR);

    QString doc = tr("Analyze each nucleotide sequence <u>%1</u> with <u>%2</u>.")
        .arg(producerName)
        .arg(schemaFile);

    return doc;
}

/******************************
* QDWorker
******************************/
QDWorker::QDWorker(Actor* a) : BaseWorker(a), input(NULL), output(NULL) {
}

void QDWorker::init() {
    input = ports.value(CoreLibConstants::IN_SEQ_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_ANNOTATIONS_PORT_ID);
}

bool QDWorker::isReady() {
    return (input && input->hasMessage());
}

Task* QDWorker::tick() {
    QString schemaUri = actor->getParameter(SCHEMA_ATTR)->getAttributePureValue().toString();
    QDDocument doc;

    QFileInfo fi(schemaUri);
    if (!fi.exists()) {
        QString defaultDir = QDir::searchPaths( PATH_PREFIX_DATA ).first() + QUERY_SAMPLES_PATH;
        QDir dir(defaultDir);
        QStringList names(QString("*.%1").arg(QUERY_SCHEME_EXTENSION));
        foreach(const QFileInfo& fi, dir.entryInfoList(names, QDir::Files|QDir::NoSymLinks)) {
            if (fi.fileName()==schemaUri || fi.baseName()==schemaUri) {
                schemaUri = fi.absoluteFilePath();
                break;
            }
        }
    }
    
    QFile f(schemaUri);
    if (!f.open(QIODevice::ReadOnly)) {
        coreLog.error(L10N::errorOpeningFileRead(schemaUri));
        return NULL;
    }
    QByteArray data = f.readAll();
    QString content = QString::fromUtf8(data);
    f.close();

    bool res = doc.setContent(content);
    if (!res) {
        return NULL;
    }

    scheme = new QDScheme;
    
    QList<QDDocument*> docs;
    docs << &doc;
    bool ok = QDSceneSerializer::doc2scheme(docs, scheme);
    if (!ok) {
        return NULL;
    }

    Message inputMessage = getMessageAndSetupScriptValues(input);
    QVariantMap map = inputMessage.getData().toMap();
    DNASequence seq = map.value(BioActorLibrary::SEQ_SLOT().getId()).value<DNASequence>();

    QDRunSettings settings;
    settings.annotationsObj = new AnnotationTableObject("QDResult");
    settings.scheme = scheme;
    settings.sequenceObj = new DNASequenceObject("QQQ", seq);
    settings.region = settings.sequenceObj->getSequenceRange();
    scheme->setDNA(settings.sequenceObj);

    QDScheduler* scheduler = new QDScheduler(settings);
    QDImportToTableTask* importTask = new QDImportToTableTask(scheduler->getLinker(), "QDResults", settings.groupName);
    connect(new TaskSignalMapper(importTask), SIGNAL(si_taskFinished(Task*)), SLOT(sl_taskFinished(Task*)));
    QList<Task*> tasks;
    tasks << scheduler << importTask;
    Task* task =new MultiTask(tr("QDWorker task"), tasks);
    return task;
}

bool QDWorker::isDone() {
    return !input || input->isEnded();
}

void QDWorker::cleanup() {
}

void annObjToAnnDataList(AnnotationTableObject* annObj, QList<SharedAnnotationData>& result) {
    foreach(Annotation* a, annObj->getAnnotations()) {
        foreach(AnnotationGroup* grp, a->getGroups()) {
            a->addQualifier(GBFeatureUtils::QUALIFIER_GROUP, grp->getGroupName());
        }
        result.append(a->data());
    }
}

void QDWorker::sl_taskFinished(Task* t) {
    delete scheme;
    if (output) {
        //QDScheduler* sched = qobject_cast<QDScheduler*>(t);
        QDImportToTableTask* tableTask = qobject_cast<QDImportToTableTask*>(t);

        AnnotationTableObject* annObj = tableTask->getTable();
        QList<SharedAnnotationData> res;
        annObjToAnnDataList(annObj, res);
        QVariant v = qVariantFromValue< QList<SharedAnnotationData> >(res);
        output->put(Message(BioDataTypes::ANNOTATION_TABLE_TYPE(), v));
        if (input->isEnded()) {
            output->setEnded();
        }
    }
}

}//Workflow namespace
}//U2 namespace
