/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "EnzymesIO.h"

#include <U2Core/AppContext.h>
#include <U2Core/IOAdapter.h>
#include <U2Core/Log.h>
#include <U2Core/DNAAlphabet.h>
#include <U2Core/L10n.h>
#include <U2Core/TextUtils.h>
#include <U2Misc/DialogUtils.h>
#include <memory>

namespace U2 {

static Logger log(ULOG_ENZYME_PLUGIN);

QString EnzymesIO::getFileDialogFilter() {
    return DialogUtils::prepareFileFilter(tr("Bairoch format"), QStringList("bairoch"));
}

QList<SEnzymeData> EnzymesIO::readEnzymes(const QString& url, TaskStateInfo& ti) {
    QList<SEnzymeData> res;

    IOAdapterId ioId = BaseIOAdapters::url2io(url);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(ioId);
    if (iof == NULL) {
        ti.setError(  tr("Unsupported URI type") );
        return res;
    }
    
    EnzymeFileFormat f = detectFileFormat(url);
    if (ti.hasErrors()) {
        return res;
    }
    switch(f) {
        case EnzymeFileFormat_Bairoch:
            res = readBairochFile(url, iof, ti);
            break;
        default: 
            ti.setError(  tr("Unsupported enzymes file format") );
            break;
    }
    
    //assign alphabet if needed.
    for (int i=0, n = res.count(); i<n; i++) {
        SEnzymeData& d = res[i];
        if (d->alphabet == NULL) {
            d->alphabet = AppContext::getDNAAlphabetRegistry()->findAlphabet(d->seq);
            if (!d->alphabet->isNucleic()) {
                log.error(tr("Non-nucleic enzyme alphabet: '%1', alphabet: %2, sequence '%3'")
                    .arg(d->id).arg(d->alphabet->getId()).arg(QString(d->seq)));
            }
        }
    }
    return res;
}

void EnzymesIO::writeEnzymes(const QString& url, const QString& source, const QSet<QString>& enzymes, TaskStateInfo& ti) {
    QString tmp;
    foreach(tmp, enzymes){
        log.trace(tmp);
    }

    IOAdapterId ioId = BaseIOAdapters::url2io(url);
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(ioId);
    if (iof == NULL) {
        ti.setError(  tr("Unsupported URI type") );
        return;
    }

    IOAdapterId srcioId = BaseIOAdapters::url2io(source);
    IOAdapterFactory* srciof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(srcioId);
    if (srciof == NULL) {
        ti.setError(  tr("Unsupported URI type") );
        return;
    }
    
    EnzymeFileFormat f = detectFileFormat(source);
    if (ti.hasErrors()) {
        return;
    }
    switch(f) {
        case EnzymeFileFormat_Bairoch:
            writeBairochFile(url, iof, source, srciof, enzymes, ti);
            break;
        default: 
            ti.setError(  tr("Unsupported enzymes file format") );
            break;
    }
}

EnzymeFileFormat EnzymesIO::detectFileFormat(const QString& url) {
    QByteArray data= BaseIOAdapters::readFileHeader(url, 256);
    if (data.startsWith("CC ") || data.startsWith("ID ")) {
        return EnzymeFileFormat_Bairoch;
    }
    return EnzymeFileFormat_Unknown;

}

#define READ_BUFF_SIZE 4096
QList<SEnzymeData> EnzymesIO::readBairochFile(const QString& url, IOAdapterFactory* iof, TaskStateInfo& ti) {
    QList<SEnzymeData> res;
    
    std::auto_ptr<IOAdapter> io(iof->createIOAdapter());
    if (!io->open(url, IOAdapterMode_Read)) {
        ti.setError(L10N::errorOpeningFileRead(url));
        return res;
    }
    
    SEnzymeData currentData(new EnzymeData());
    QByteArray buffArr(READ_BUFF_SIZE, 0);
    char* buff = buffArr.data();
    const QBitArray& LINE_BREAKS = TextUtils::LINE_BREAKS;
    int line = 0, len = 0;
    bool lineOk = true;
    while ((len = io->readUntil(buff, READ_BUFF_SIZE, LINE_BREAKS, IOAdapter::Term_Include, &lineOk)) > 0 && !ti.cancelFlag) {
        line++;
        if (!lineOk) {
            ti.setError(  tr("Line is too long: %1").arg(line) );
            return res;
        }
        if (len < 2) {
            continue;
        }
        if (buff[0] == '/' && buff[1] == '/') {
            if (!currentData->id.isEmpty()) {
                res.append(currentData);
                currentData = new EnzymeData();
            } else {
                log.trace(QString("Enzyme without ID, line %1, skipping").arg(line));
            }
            continue;
        }
        if (len < 4) {
            continue;
        }
        if (buff[0] == 'I' && buff[1] == 'D') {
            currentData->id = QByteArray(buff + 3, len - 3).trimmed();
            continue;
        }
        if (buff[0] == 'E' && buff[1] == 'T') {
            currentData->type = QByteArray(buff + 3, len - 3).trimmed();
            continue;
        }
        if (buff[0] == 'O' && buff[1] == 'S') {
            currentData->organizm = QByteArray(buff + 3, len - 3).trimmed();
            continue;
        }
        if (buff[0] == 'A' && buff[1] == 'C') {
            currentData->accession = QByteArray(buff + 3, len - 3).trimmed();
            int sz = currentData->accession.size();
            if (sz > 0 && currentData->accession[sz-1]==';') {
                currentData->accession.resize(sz-1);
            }
            continue;
        }
        if (buff[0] == 'R' && buff[1] == 'S') {
            QByteArray rsLine = QByteArray(buff + 3, len - 3);
            QList<QByteArray> strands = rsLine.split(';');
            for (int s = 0, n = qMin(2, strands.count()); s < n; s++) {
                const QByteArray& strandInfo = strands.at(s);
                QByteArray seq;
                int cutPos = ENZYME_CUT_UNKNOWN;
                int cutIdx = strandInfo.indexOf(',');
                if (cutIdx > 0) {
                    seq = strandInfo.left(cutIdx).trimmed();
                    QByteArray cutStr = strandInfo.mid(cutIdx+1).trimmed();
                    bool ok = true;
                    cutPos = cutStr.length() == 1 && cutStr[0] == '?' ? ENZYME_CUT_UNKNOWN : cutStr.toInt(&ok);
                    if (!ok) {
                        log.error(tr("Illegal cut pos: %1, line %2").arg(QString(cutStr)).arg(line));
                        break;
                    }
                } else {
                    seq = strandInfo.trimmed();
                }
                if (s == 0) {
                    currentData->seq= seq;
                    currentData->cutDirect = cutPos;
                    currentData->cutComplement = cutPos;
                } else if (cutPos != ENZYME_CUT_UNKNOWN) {
                    currentData->cutComplement = cutPos;
                }
            }
        }
    }
    io->close();
    return res;
}

void EnzymesIO::writeBairochFile(const QString& url, IOAdapterFactory* iof, const QString& source, IOAdapterFactory* srciof, const QSet<QString>& enzymes, TaskStateInfo& ti) {
    
    std::auto_ptr<IOAdapter> io(iof->createIOAdapter());
    if (!io->open(url, IOAdapterMode_Write)) {
        ti.setError(L10N::errorOpeningFileWrite(url));
        return;
    }

    std::auto_ptr<IOAdapter> srcio(srciof->createIOAdapter());
    if (!srcio->open(source, IOAdapterMode_Read)) {
        ti.setError(L10N::errorOpeningFileRead(source));
        return;
    }
    
    QByteArray buffArr(READ_BUFF_SIZE, 0);
    char* buff = buffArr.data();
    const QBitArray& LINE_BREAKS = TextUtils::LINE_BREAKS;
    int line = 0, len = 0;
    bool lineOk = true;
    bool writeString = true;

    while ((len = srcio->readUntil(buff, READ_BUFF_SIZE, LINE_BREAKS, IOAdapter::Term_Include, &lineOk)) > 0 && !ti.cancelFlag) {
        line++;
        if (!lineOk) {
            ti.setError(  tr("Line is too long: %1").arg(line) );
        }
        if (len < 2) {
            if (writeString) io->writeBlock(buff, len);
            continue;
        }
        
        if (buff[0] == 'I' && buff[1] == 'D') {
            QString currID(QByteArray(buff + 3, len - 3).trimmed());
            writeString = enzymes.contains(currID);
        }
        if (writeString) io->writeBlock(buff, len);
    }
    srcio->close();
    io->close();
}

SEnzymeData EnzymesIO::findEnzymeById(const QString& id, const QList<SEnzymeData>& enzymes) {
    const QString idLower = id.toLower();
    foreach(const SEnzymeData& ed, enzymes) {
        if (ed->id.toLower() == idLower) {
            return ed;
        }
    }
    return SEnzymeData();
}

//////////////////////////////////////////////////////////////////////////
// load task

LoadEnzymeFileTask::LoadEnzymeFileTask(const QString& _url) 
: Task(tr("Load enzymes from %1").arg(_url), TaskFlag_None), url(_url)
{
}

void LoadEnzymeFileTask::run() {
    enzymes = EnzymesIO::readEnzymes(url, stateInfo);
}

//////////////////////////////////////////////////////////////////////////
// save task

SaveEnzymeFileTask::SaveEnzymeFileTask(const QString& _url, const QString& _source, const QSet<QString>& _enzymes) 
: Task(tr("Save enzymes to %1").arg(_url), TaskFlag_None), url(_url), source(_source), enzymes(_enzymes)
{
}

void SaveEnzymeFileTask::run() {
    EnzymesIO::writeEnzymes(url, source, enzymes, stateInfo);
}

}//namespace
