/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _U2_REGION_H_
#define _U2_REGION_H_

#include <U2Core/global.h>

namespace U2 {

enum U2Strand {
    U2Strand_Direct, 
    U2Strand_Complement,
    U2Strand_Irrelevant,
};


/** 
    Linear 64bit region 
*/
class U2CORE_EXPORT U2Region {
public:
    U2Region(): startPos(0), length(0){}
    U2Region(qint64 s, qint64 l): startPos(s), length(l){}

    /** Region start position. */
    qint64 startPos;

    /** Region length. */
    qint64 length;


    ////////////////////////// Member functions and operators ////////////////////

    /** Region end position, exclusive. */
    qint64 endPos() const {return startPos + length;}

    /** Central point of the region. */
    qint64 center() const { return startPos + length / 2 ;}

    /** Checks if this region has zero length. */
    bool isEmpty() const {return length == 0; }

    /** Checks whether the specified point falls inside this region. */
    bool contains(qint64 pos) const {return pos>=startPos && pos < endPos();}

    /** Checks whether the specified region fits inside this region or is equal. */
    bool contains(const U2Region& r) const {return r.startPos >= startPos && r.endPos() <= endPos();}

    /** Checks whether the specified region has common points with this region. */
    bool intersects(const U2Region& r) const;

    /** Checks whether this region has common points with any region in the specified list. */
    bool intersectsWithAny(const QList<U2Region>& rs) const;

    /** Returns the intersection between 2 regions, or default (empty) value if they do not intersect. */
    U2Region intersect(const U2Region& r) const;

    /** Checks whether the specified region is equal to this region. */
    bool operator== ( const U2Region & r ) const { return r.startPos == startPos && r.length == length; }

    /** Checks whether the specified region is not equal to this region. */
    bool operator!= ( const U2Region & r ) const { return r.startPos != startPos || r.length != length; }

    /** Compares 2 regions by start position. 
    Returns true if this region starts strictly earlier than the specified one. */
    bool operator<(const U2Region &r) const {return startPos < r.startPos;}

    ///////////////////////// Class functions /////////////////////////////////

    /** Returns least common region which contains both of the 2 specified regions. */
    static U2Region containingRegion(const U2Region& r1, const U2Region& r2);

    /** Returns least common region which contains all of the specified regions. */
    static U2Region containingRegion(const QList<U2Region>& regions);

    /** Normalizes the specified list by joining overlapping regions. 
    This function sorts regions by starting position then 
    iterates through them and replaces all groups of intersecting regions by containing regions.
    */
    static void join(QList<U2Region>& regions);

private:
    static bool registerMeta;
};

//////////////////// inline implementations ///////////////////////////

inline bool U2Region::intersects(const U2Region& r) const {
    qint64 sd = startPos - r.startPos;
    return (sd >= 0) ? (sd < r.length) : (-sd < length);
}

inline U2Region U2Region::intersect(const U2Region& r) const {
    qint64 newStart = qMax(startPos, r.startPos);
    qint64 newEnd = qMin(endPos(), r.endPos());
    if (newStart > newEnd) {
        return U2Region();
    }
    return U2Region(newStart, newEnd - newStart);
}


inline bool U2Region::intersectsWithAny(const QList<U2Region>& rs) const {
    bool res = false;
    foreach(const U2Region& r, rs) {
        res = intersects(r);
        if (res) {
            break;
        }
    }
    return res;
}


inline U2Region U2Region::containingRegion(const U2Region& r1, const U2Region& r2) {
    qint64 newStart = qMin(r1.startPos, r2.startPos);
    qint64 newEnd = qMax(r1.endPos(), r2.endPos());
    return U2Region(newStart, newEnd - newStart);
}

inline U2Region U2Region::containingRegion(const QList<U2Region>& regions) {
    assert(!regions.isEmpty());

    U2Region res = regions.first();
    foreach(const U2Region& r, regions) {
        res = containingRegion(res, r);
    }
    return res;
}

U2CORE_EXPORT QDataStream &operator<<(QDataStream &out, const U2Region &myObj);
U2CORE_EXPORT QDataStream &operator>>(QDataStream &in, U2Region &myObj);

}//namespace

Q_DECLARE_TYPEINFO(U2::U2Region, Q_PRIMITIVE_TYPE);
Q_DECLARE_METATYPE( U2::U2Region)
Q_DECLARE_METATYPE( QList< U2::U2Region >)

#endif
