/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <U2Core/U2AssemblyUtils.h>

#include <U2Core/TextUtils.h>

#include <QtCore/QRegExp>

namespace U2 {

U2CigarOp U2AssemblyUtils::char2Cigar(char c, QString& err) {
    char cu = TextUtils::UPPER_CASE_MAP[c];
    switch (c) {
        case 'D':
            return U2CigarOp_D; // deleted
        case 'I':
            return U2CigarOp_I; // inserted
        case 'H':
            return U2CigarOp_H; // hard-clipped
        case 'M':
            return U2CigarOp_M; // matched
        case 'N':
            return U2CigarOp_N; // skipped
        case 'P':
            return U2CigarOp_P; // padded
        case 'S':
            return U2CigarOp_S; // soft-clipped
    }
    err = tr("Invalid CIGAR op: '%1'!").arg(c);
    return U2CigarOp_Invalid;
}

char U2AssemblyUtils::cigar2Char(U2CigarOp op) {
    char c;
    switch(op) {
            case U2CigarOp_D: c = 'D'; break;
            case U2CigarOp_I: c = 'I'; break;
            case U2CigarOp_H: c = 'H'; break;
            case U2CigarOp_M: c = 'M'; break;
            case U2CigarOp_N: c = 'N'; break;
            case U2CigarOp_P: c = 'P'; break;
            case U2CigarOp_S: c = 'S'; break;
            default: assert(0); c = '?';
    }
    return c;
}

QList<U2CigarToken> U2AssemblyUtils::parseCigar(const QByteArray& cigarString, QString& err) {
    QList<U2CigarToken> result;
    int pos = 0, len = cigarString.length();
    const char* cigar = cigarString.constData();
    int count = 0;
    // count numbers
    while (pos < len ) {
        char c = cigar[pos++];
        if (c >= '0' && c <= '9') {
            int n = c - '0';
            count = count * 10 + n;
            continue;
        } 
        U2CigarOp op = char2Cigar(c, err);
        if (!err.isEmpty()) {
            break;
        }
        result.append(U2CigarToken(op, count));
        count = 0;
    }
    return result;
}


QString U2AssemblyUtils::cigar2String(const QList<U2CigarToken>& cigar) {
    QString res;
    foreach(const U2CigarToken& t, cigar) {
        if (t.op != U2CigarOp_Invalid) {
            res = res + QString::number(t.count) + QChar(cigar2Char(t.op));
        }
    }
    return res;
}

qint64 U2AssemblyUtils::getCigarExtraLength(const QList<U2CigarToken>& cigar) {
    qint64 res = 0;
    foreach(const U2CigarToken& t, cigar) {
        //TODO: recheck and test
        switch(t.op) {
            case U2CigarOp_I: res++; break;
            case U2CigarOp_D: res--; break;
        }
    }
    return res;
}

} //namespace
