/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <U2Core/U2AnnotationUtils.h>

namespace U2 {

QList<U2Region> U2AnnotationUtils::fixLocationsForRemovedRegion( const U2Region& regionToDelete, 
                                                                QList<U2Region>& regionList, 
                                                                AnnotationStrategyForResize s )
{
    //assert (s == AnnotationStrategyForResize_Remove || s == AnnotationStrategyForResize_Resize);
    QList<U2Region> toDelete, toReplace;
    foreach(U2Region reg, regionList){
        toDelete.append(reg);
        if (s != AnnotationStrategyForResize_Remove){
            if(reg.contains(regionToDelete)){
                reg.length -= regionToDelete.length;
            }else if (regionToDelete.contains(reg)) {
                reg.length = 0;
            }else if(reg.intersects(regionToDelete)){
                if(reg.startPos <= regionToDelete.startPos){
                    reg.length -= reg.endPos() - regionToDelete.startPos;
                }else{
                    reg.length -= regionToDelete.endPos() - reg.startPos;
                    reg.startPos = regionToDelete.startPos;
                }
            }else if(reg.startPos >= regionToDelete.endPos()){
                reg.startPos -= regionToDelete.length;
            } 
        }
        else {
            if(reg.intersects(regionToDelete) || regionToDelete.contains(reg)){
                reg.length = 0;
            }else if(reg.startPos >= regionToDelete.endPos()){
                reg.startPos -= regionToDelete.length;
            }
        }
        if (reg.length != 0) {
            toDelete.removeLast();
            toReplace.append(reg);
        }
    }
    regionList.clear();
    regionList.append(toReplace);

    return toDelete;
}

QList<U2Region> U2AnnotationUtils::fixLocationsForInsertedRegion( qint64 insertPos, qint64 len, 
                                                                 QList<U2Region>& regionList, 
                                                                 AnnotationStrategyForResize s )
{
    QList<U2Region> toReplace, toDelete;
    foreach(U2Region reg, regionList){
        if(reg.endPos() <= insertPos){
            toReplace.append(reg);
        } else {
            if (s == AnnotationStrategyForResize_Resize){
                if(reg.startPos < insertPos && reg.endPos() > insertPos){
                    reg.length += len;
                    toReplace.append(reg);
                }else if(reg.startPos >= insertPos){
                    reg.startPos += len;
                    toReplace.append(reg);
                }                                
            }else if(s == AnnotationStrategyForResize_Split){
                if(reg.startPos < insertPos && reg.endPos() > insertPos){
                    U2Region firstPart, secondPart;
                    firstPart.startPos = reg.startPos;
                    firstPart.length = insertPos - reg.startPos;
                    secondPart.startPos = firstPart.endPos() + len;
                    secondPart.length = reg.length - firstPart.length;
                    toReplace.append(firstPart);
                    toReplace.append(secondPart);
                }else if(reg.startPos >= insertPos){
                    reg.startPos += len;
                    toReplace.append(reg);
                }                                
            }else if(s == AnnotationStrategyForResize_Remove){
                if(reg.startPos < insertPos && reg.endPos() > insertPos){
                    toDelete.append(reg);
                }else if(reg.startPos >= insertPos){
                    reg.startPos += len;
                    toReplace.append(reg);
                }                                
            }
        }
    }
    regionList.clear();
    regionList.append(toReplace);
    return toDelete;
}

QList<U2Region> U2AnnotationUtils::fixLocationsForReplacedRegion( const U2Region& regionToReplace, 
                                                                 qint64 newLen, QList<U2Region>& loc, 
                                                                 AnnotationStrategyForResize s )
{
    if (s == AnnotationStrategyForResize_Remove) {
        QList<U2Region> l1 = fixLocationsForRemovedRegion(regionToReplace, loc, s);
        QList<U2Region> l2 = fixLocationsForInsertedRegion(regionToReplace.startPos, newLen, loc, s);
        assert(l2.isEmpty()); Q_UNUSED(l2);
        return l1;
    } else {
        int offset = newLen - regionToReplace.length;
        if (s == AnnotationStrategyForResize_Resize && offset == 0) {
            return QList<U2Region>();
        }
        assert(s == AnnotationStrategyForResize_Resize); // FIXME do we ever need to SPLIT when replacing ???

        QList<U2Region> toReplace, toDelete;
        foreach(U2Region reg, loc){
            U2Region copy = reg;
            if(reg.endPos() <= regionToReplace.startPos){
                toReplace.append(reg);
            } else {
                if (reg.contains(regionToReplace)) {
                    reg.length += offset;
                } else if (reg.startPos >= regionToReplace.endPos()) {
                    reg.startPos += offset;
                } else {
                    // start pos and/or end pos lie inside the regionToReplace
                    // let's assume offset is applied at the region end
                    if (offset > 0) {
                        if (reg.endPos() <= regionToReplace.endPos()) {
                            // leave it as is
                        } else {
                            // append tail
                            reg.length += offset;
                        }
                    } else {
                        if (reg.endPos() <= regionToReplace.endPos() + offset) {
                            // leave it as is
                        } else if (reg.startPos < regionToReplace.endPos() + offset && reg.endPos() >= regionToReplace.endPos()) {    
                            // crop inner subregion
                            reg.length += offset;
                        } else if (reg.startPos >= regionToReplace.endPos() + offset && reg.endPos() <= regionToReplace.endPos()) {
                            // drop the region
                            reg.length = 0;
                        } else if (reg.startPos < regionToReplace.endPos() + offset) {
                            // crop tail
                            assert(reg.endPos() < regionToReplace.endPos() && reg.endPos() > regionToReplace.endPos() + offset);
                            reg.length -= reg.endPos() - regionToReplace.startPos;
                        } else {
                            // crop head
                            assert(reg.startPos >= regionToReplace.endPos() + offset && reg.startPos < regionToReplace.endPos());
                            assert(reg.endPos() > regionToReplace.endPos());
                            reg.length = reg.endPos() - regionToReplace.endPos();
                            reg.startPos = regionToReplace.endPos() + offset;
                        }
                    }
                }
                assert(reg.length >= 0);
                if (reg.length == 0) {
                    toDelete.append(reg);
                } else {
                    toReplace.append(reg);
                }
            }
        }
        loc.clear();
        loc.append(toReplace);
        return toDelete;
    }
}

} //namespace
